/* GNOME-DB Library
 * Copyright (C) 2000-2003 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libgda/gda-xml-connection.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkpaned.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtktoolbar.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomedb/gnome-db-form.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-list.h>
#include <libgnomedb/gnome-db-util.h>
#include "table-properties-dialog.h"
#include "tables-page.h"

typedef struct {
	GdaConnection *cnc;
	GtkWidget *toolbar;
	GtkWidget *tables_list;
	GtkWidget *data_grid;
	GtkWidget *new_button;
	GtkWidget *props_button;
} TablesPagePrivate;

static void
free_private_data (gpointer user_data)
{
	TablesPagePrivate *priv = user_data;

	if (priv->cnc) {
		g_object_unref (priv->cnc);
		priv->cnc = NULL;
	}

	g_free (priv);
}

static void
grid_row_selected_cb (GnomeDbGrid *grid, gint row, gpointer user_data)
{
	GList *selected;
	GdaDataModel *model;
	const GdaValue *value;
	gchar *table_name;
	TablesPagePrivate *priv = (TablesPagePrivate *) user_data;

	gtk_widget_set_sensitive (priv->props_button, TRUE);

	/* get the selection from the tables grid */
	selected = gnome_db_list_get_selection (GNOME_DB_LIST (priv->tables_list));
	if (!selected)
		return;

	/* retrieve the table name */
	model = gnome_db_list_get_model (GNOME_DB_LIST (priv->tables_list));
	value = gda_data_model_get_value_at (model, 0, GPOINTER_TO_INT (selected->data));
	table_name = gda_value_stringify (value);

	if (table_name) {
		GdaCommand *cmd;

		cmd = gda_command_new (table_name, GDA_COMMAND_TYPE_TABLE, 0);
		model = gda_connection_execute_single_command (priv->cnc, cmd, NULL);

		if (model) {
			gnome_db_grid_set_model (GNOME_DB_GRID (priv->data_grid), model);
			g_object_unref (model);
		} else
			gnome_db_grid_set_model (GNOME_DB_GRID (priv->data_grid), NULL);

		gda_command_free (cmd);
		g_free (table_name);
	} else
		gnome_db_grid_set_model (GNOME_DB_GRID (priv->data_grid), NULL);

	/* free memory */
	g_list_free (selected);
}

static void
grid_selection_cleared_cb (GnomeDbGrid *grid, gpointer user_data)
{
       	TablesPagePrivate *priv = (TablesPagePrivate *) user_data;

	gtk_widget_set_sensitive (priv->props_button, FALSE);
	gnome_db_grid_set_model (GNOME_DB_GRID (priv->data_grid), NULL);
}

static void
view_properties_cb (GtkWidget *widget, gpointer user_data)
{
	GList *selected;
	GdaDataModel *model;
	const GdaValue *value;
	gchar *table_name;
	TablesPagePrivate *priv = (TablesPagePrivate *) user_data;

	/* get the selection from the grid */
	selected = gnome_db_list_get_selection (GNOME_DB_LIST (priv->tables_list));
	if (!selected)
		return;

	/* retrieve the table name */
	model = gnome_db_list_get_model (GNOME_DB_LIST (priv->tables_list));
	value = gda_data_model_get_value_at (model, 0, GPOINTER_TO_INT (selected->data));
	table_name = gda_value_stringify (value);

	if (table_name) {
		table_properties_dialog (GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (priv->tables_list))),
					 priv->cnc, table_name);
		g_free (table_name);
	}

	/* free memory */
	g_list_free (selected);
}

GtkWidget *
tables_page_new (void)
{
	GtkWidget *page, *paned;
	TablesPagePrivate *priv;

	page = gnome_db_new_table_widget (3, 2, FALSE);
	gtk_widget_show (page);

	priv = g_new0 (TablesPagePrivate, 1);
	g_object_set_data_full (G_OBJECT (page), "TablesPagePrivate", priv, free_private_data);

	/* create the toolbar */
	priv->toolbar = gnome_db_new_toolbar_widget (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_table_attach (GTK_TABLE (page), priv->toolbar, 0, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);

	priv->new_button = gtk_toolbar_append_item (GTK_TOOLBAR (priv->toolbar), _("New"), _("Create new table"),
						    _("Create new table"),
						    gtk_image_new_from_stock (GTK_STOCK_NEW, GTK_ICON_SIZE_SMALL_TOOLBAR),
						    NULL, priv);
	gtk_widget_set_sensitive (priv->new_button, FALSE);

	priv->props_button = gtk_toolbar_append_item (GTK_TOOLBAR (priv->toolbar), _("Properties"), _("View table properties"),
						      _("View table properties"),
						      gtk_image_new_from_stock (GTK_STOCK_PROPERTIES, GTK_ICON_SIZE_SMALL_TOOLBAR),
						      GTK_SIGNAL_FUNC (view_properties_cb), priv);
	gtk_widget_set_sensitive (priv->props_button, FALSE);

	/* create the paned widget to contain the tables list and the detail */
	paned = gnome_db_new_hpaned_widget ();
	gtk_table_attach (GTK_TABLE (page), paned, 0, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, 6, 6);

	/* create the tables list */
	priv->tables_list = gnome_db_list_new ();
	g_signal_connect (G_OBJECT (priv->tables_list), "row_selected", G_CALLBACK (grid_row_selected_cb), priv);
	g_signal_connect (G_OBJECT (priv->tables_list), "selection_cleared", G_CALLBACK (grid_selection_cleared_cb), priv);
	gtk_widget_show (priv->tables_list);
	gtk_paned_add1 (GTK_PANED (paned), priv->tables_list);

	/* create the data grid */
	priv->data_grid = gnome_db_grid_new ();
	gtk_widget_show (priv->data_grid);
	gtk_paned_add2 (GTK_PANED (paned), priv->data_grid);

	return page;
}

void
tables_page_set_connection (GtkWidget *tp, GdaConnection *cnc)
{
	TablesPagePrivate *priv;

	priv = g_object_get_data (G_OBJECT (tp), "TablesPagePrivate");
	if (!priv)
		return;

	if (priv->cnc)
		g_object_unref (cnc);

	priv->cnc = cnc;
	if (GDA_IS_CONNECTION (priv->cnc)) {
		GdaDataModel *model;

		g_object_ref (priv->cnc);

		model = gda_connection_get_schema (cnc,GDA_CONNECTION_SCHEMA_TABLES, NULL);
		gnome_db_list_set_model (GNOME_DB_LIST (priv->tables_list), model, 0);

		g_object_unref (model);
	}
}
