/* GNOME-DB Components
 * Copyright (C) 2000-2003 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libgnomeui/gnome-about.h>
#include <libgnomedb/gnome-db-i18n.h>
#include <libgnomedb/gnome-db-window.h>
#include "db-shell.h"

typedef struct {
	GtkWidget *shell;
	GdaConnection *cnc;

	DbShellCallbackFunc new_func;
	DbShellCallbackFunc delete_func;
	DbShellCallbackFunc edit_properties_func;
	gpointer user_data;
} DbShellData;

static void
command_about (BonoboUIComponent *uic, void *data, const char *path)
{
	GdkPixbuf *icon;
	GtkWidget *dialog;
	const gchar *authors[] = {
		"Rodrigo Moya <rodrigo@gnome-db.org>",
		"Vivien Malerba <malerba@gnome-db.org>",
		"Carlos Perello Marin <carlos@gnome-db.org>",
		"Gonzalo Paniagua Javier <gonzalo@gnome-db.org>",
		NULL
	};
	const gchar *documenters[] = {
		"Rodrigo Moya <rodrigo@gnome-db.org>",
		NULL
	};
	const gchar *translator_credits =
		"Christian Rose <menthos@menthos.com> Swedish translations\n" \
		"Kjartan Maraas <kmaraas@online.no> Norwegian translation\n";
	DbShellData *dsd = data;

	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);

	dialog = gnome_about_new ("GNOME-DB", VERSION,
				  "(C) 1998-2003 GNOME Foundation",
				  _("Database services for the GNOME Desktop"),
				  authors,
				  documenters,
				  translator_credits,
				  icon);
	g_signal_connect (G_OBJECT (dialog), "destroy",
			  G_CALLBACK (gtk_widget_destroyed),
			  &dialog);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (dsd->shell));
	gtk_widget_show (dialog);

}

static void
command_close (BonoboUIComponent *uic, void *data, const char *path)
{
	DbShellData *dsd = data;

	gnome_db_window_close (GNOME_DB_WINDOW (dsd->shell));
}

static void
command_delete (BonoboUIComponent *uic, void *data, const char *path)
{
	DbShellData *dsd = data;

	if (dsd->delete_func)
		dsd->delete_func (dsd->shell, dsd->user_data);
}

static void
command_edit_properties (BonoboUIComponent *uic, void *data, const char *path)
{
	DbShellData *dsd = data;

	if (dsd->edit_properties_func)
		dsd->edit_properties_func (dsd->shell, dsd->user_data);
}

static void
command_new (BonoboUIComponent *uic, void *data, const char *path)
{
	DbShellData *dsd = data;

	if (dsd->new_func)
		dsd->new_func (dsd->shell, dsd->user_data);
}

static void
command_preferences (BonoboUIComponent *uic, void *data, const char *path)
{
}

static void
command_save_connection (BonoboUIComponent *uic, void *data, const char *path)
{
	DbShellData *dsd = data;

	
}

static BonoboUIVerb db_shell_verbs [] = {
	BONOBO_UI_VERB ("DatabaseClose", command_close),
	BONOBO_UI_VERB ("DatabaseNew", command_new),
	BONOBO_UI_VERB ("DatabaseSaveConnection", command_save_connection),
	BONOBO_UI_VERB ("EditDelete", command_delete),
	BONOBO_UI_VERB ("EditProperties", command_edit_properties),
	BONOBO_UI_VERB ("SettingsPreferences", command_preferences),
	BONOBO_UI_VERB ("HelpAbout", command_about),

	BONOBO_UI_VERB_END
};

static void
shell_closed_cb (GnomeDbWindow *window, gpointer user_data)
{
	DbShellData *dsd = user_data;

	dsd->shell = NULL;
	dsd->cnc = NULL;
	g_free (dsd);

	g_object_set_data (G_OBJECT (window), "GNOME_DB_Shell_PrivateData", NULL);
}

static void
sensitize_ui (DbShellData *dsd)
{
	gboolean connected;
	BonoboUIComponent *uic;

	if (GDA_IS_CONNECTION (dsd->cnc) && gda_connection_is_open (dsd->cnc))
		connected = TRUE;
	else
		connected = FALSE;

	uic = gnome_db_window_get_ui_component (GNOME_DB_WINDOW (dsd->shell));

	bonobo_ui_component_set_prop (uic, "/commands/DatabaseSaveConnection", "hidden",
				      connected ? "0" : "1",
				      NULL);
	bonobo_ui_component_set_prop (uic, "/commands/DatabaseNew", "hidden",
				      dsd->new_func ? "0" : "1",
				      NULL);
	bonobo_ui_component_set_prop (uic, "/commands/EditDelete", "hidden",
				      dsd->delete_func ? "0" : "1",
				      NULL);
	bonobo_ui_component_set_prop (uic, "/commands/EditProperties", "hidden",
				      dsd->edit_properties_func ? "0" : "1",
				      NULL);
}

GtkWidget *
db_shell_new (const gchar *title, GtkWidget *widget, GdaConnection *cnc,
	      DbShellCallbackFunc new_func,
	      DbShellCallbackFunc delete_func,
	      DbShellCallbackFunc edit_properties_func,
	      gpointer user_data)
{
	DbShellData *dsd;

	dsd = g_new0 (DbShellData, 1);
	dsd->cnc = cnc;
	dsd->new_func = new_func;
	dsd->delete_func = delete_func;
	dsd->edit_properties_func = edit_properties_func;
	dsd->user_data = user_data;

	/* create the main window */
	dsd->shell = gnome_db_window_new (title, PREFIX,
					  LIBGNOMEDB_UIDIR "/db-shell.xml",
					  db_shell_verbs, dsd);
	g_object_set_data (G_OBJECT (dsd->shell), "GNOME_DB_Shell_PrivateData", dsd);
	g_signal_connect_after (dsd->shell, "close", G_CALLBACK (shell_closed_cb), dsd);

	gnome_db_window_set_contents (GNOME_DB_WINDOW (dsd->shell), widget);
	gtk_widget_show (widget);
	gtk_window_set_default_size (GTK_WINDOW (dsd->shell), 400, 350);
	gtk_widget_show (dsd->shell);

	sensitize_ui (dsd);

	return dsd->shell;
}
