/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <libgda/gda-config.h>
#include <libgda/gda-connection.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkimage.h>
#include <gtk/gtklabel.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkpaned.h>
#include <gtk/gtktable.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtktreeview.h>
#include <libgnomedb/gnome-db-browser.h>
#include <libgnomedb/gnome-db-config.h>
#include <libgnomedb/gnome-db-list.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-util.h>
#include "gnome-db-i18n.h"
#include "gnome-db-browser-private.h"

#define PARENT_TYPE GTK_TYPE_VBOX

typedef struct {
	GtkTreeIter top;
	GdaConnectionSchema schema;
	GdkPixbuf *pixbuf;
} ObjectNodeData;

typedef struct {
	GnomeDbBrowser *browser;
	GtkWidget *widget;
	GdaConnectionSchema schema;
	gchar *name;
} ObjectDetailData;

struct _GnomeDbBrowserPrivate {
	GdaConnection *cnc;

	/* widgets */
	GtkWidget *paned;
	GtkWidget *db_selector;
	GtkWidget *scroll;
	GtkWidget *object_list;
	GtkWidget *detail_notebook;

	GList *object_nodes;
	GList *detail_widgets;
};

static void gnome_db_browser_class_init   (GnomeDbBrowserClass *klass);
static void gnome_db_browser_init         (GnomeDbBrowser *browser,
					   GnomeDbBrowserClass *klass);
static void gnome_db_browser_set_property (GObject *object,
					   guint paramid,
					   const GValue *value,
					   GParamSpec *pspec);
static void gnome_db_browser_get_property (GObject *object,
					   guint param_id,
					   GValue *value,
					   GParamSpec *pspec);
static void gnome_db_browser_finalize     (GObject *object);

enum {
	PROP_0,
	PROP_CONNECTION
};

enum {
	PROGRESS_MESSAGE,
	LAST_SIGNAL
};

static gint browser_signals[LAST_SIGNAL];
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
close_tab_cb (GtkButton *button, gpointer user_data)
{
	ObjectDetailData *detail_data = (ObjectDetailData *) user_data;

	detail_data->browser->priv->detail_widgets = g_list_remove (
		detail_data->browser->priv->detail_widgets, detail_data);
	gtk_notebook_remove_page (
		GTK_NOTEBOOK (detail_data->browser->priv->detail_notebook),
		gtk_notebook_page_num (
			GTK_NOTEBOOK (detail_data->browser->priv->detail_notebook),
			detail_data->widget));

	g_free (detail_data->name);
	g_free (detail_data);
}

static void
paned_notification_cb (GObject *object, GParamSpec *pspec, gpointer user_data)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail (G_IS_PARAM_SPEC (pspec));
	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	if (!strcmp (pspec->name, "position")) {
		gnome_db_config_set_int (GNOME_DB_CONFIG_KEY_BROWSER_PANED_POSITION,
				    gtk_paned_get_position (GTK_PANED (browser->priv->paned)));
	}
}

static void
tree_selection_changed_cb (GtkTreeSelection *selection, gpointer user_data)
{
	GtkTreeModel *tree_model;
	GtkTreeIter iter;
	gchar *object_name;
	GList *l;
	ObjectNodeData *node_data = NULL;
	ObjectDetailData *detail_data = NULL;
	GtkWidget *button;
	GtkWidget *box = NULL;
	GtkWidget *label;
	gboolean add_tab = TRUE;
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* retrieve the selection from the tree */
	if (!gtk_tree_selection_get_selected (selection, &tree_model, &iter))
		return;

	gtk_tree_model_get (tree_model, &iter, 1, &object_name, -1);

	for (l = browser->priv->object_nodes; l != NULL; l = l->next) {
		GtkTreeIter parent;
		ObjectNodeData *tmp = (ObjectNodeData *) l->data;

		if (gtk_tree_model_iter_parent (tree_model, &parent, &iter)) {
			if (parent.stamp == tmp->top.stamp &&
			    parent.user_data == tmp->top.user_data) {
				node_data = tmp;
				break;
			}
		}
	}

	if (!node_data) // Parent not found!!!
		return;

	/* now search if we're already displaying this object */
	for (l = browser->priv->detail_widgets; l != NULL; l = l->next) {
		detail_data = (ObjectDetailData *) l->data;
		if (detail_data->schema == node_data->schema &&
		    !strcmp (detail_data->name, object_name)) {
			add_tab = FALSE;
			break;
		}
	}

	if (add_tab) {
		gchar *s;

		/* not being displayed, so display it */
		detail_data = g_new0 (ObjectDetailData, 1);
		detail_data->browser = browser;
		detail_data->schema = node_data->schema;
		detail_data->name = g_strdup (object_name);

		browser->priv->detail_widgets = g_list_append (browser->priv->detail_widgets, detail_data);

		box = gnome_db_new_hbox_widget (FALSE, 1);
		button = gtk_image_new_from_pixbuf (node_data->pixbuf);
		gtk_widget_show (button);
		gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 1);
		label = gnome_db_new_label_widget ("");
		s = g_strdup_printf ("<small>%s</small>", object_name);
		gtk_label_set_markup (GTK_LABEL (label), s);
		g_free (s);
		gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 1);

		button = gtk_button_new ();
		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		g_signal_connect (G_OBJECT (button), "clicked",
				  G_CALLBACK (close_tab_cb), detail_data);
		gtk_widget_show (button);
		label = gnome_db_new_label_widget ("");
		gtk_label_set_markup (GTK_LABEL (label), "<small><b>X</b></small>");
		gtk_container_add (GTK_CONTAINER (button), label);
		gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 1);
	}

	switch (detail_data->schema) {
	case GDA_CONNECTION_SCHEMA_TABLES :
		if (add_tab)
			detail_data->widget = gnome_db_browser_tables_new ();
		gnome_db_browser_tables_show (detail_data->widget, browser->priv->cnc, object_name);
		break;
	case GDA_CONNECTION_SCHEMA_PROCEDURES :
		if (add_tab)
			detail_data->widget = gnome_db_browser_procedures_new ();
		break;
	case GDA_CONNECTION_SCHEMA_TYPES :
		if (add_tab)
			detail_data->widget = gnome_db_browser_types_new ();
		break;
	case GDA_CONNECTION_SCHEMA_VIEWS :
		if (add_tab)
			detail_data->widget = gnome_db_browser_views_new ();
		gnome_db_browser_views_show (detail_data->widget, browser->priv->cnc, object_name);
		break;
	default :
		detail_data->widget = gnome_db_new_label_widget (_("Not implemented yet!"));
		break;
	}

	if (add_tab) {
		gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
					  detail_data->widget, box);
	}

	gtk_notebook_set_current_page (
		GTK_NOTEBOOK (browser->priv->detail_notebook),
		gtk_notebook_page_num (GTK_NOTEBOOK (browser->priv->detail_notebook),
				       detail_data->widget));
}

/*
 * Private functions
 */

static void
tree_value_set_func (GtkTreeViewColumn *tree_column,
			    GtkCellRenderer *cell,
			    GtkTreeModel *model,
			    GtkTreeIter *iter,
			    gpointer user_data)
{
	GdkPixbuf *pixbuf;
	gchar *text;
	GList *renderers;
	
	gtk_tree_model_get (model, iter, 0, &pixbuf, 1, &text, -1);
	renderers = gtk_tree_view_column_get_cell_renderers (tree_column);
	cell = renderers->data;
	g_object_set (G_OBJECT (cell), "pixbuf", pixbuf, NULL);
	cell = renderers->next->data;
	g_object_set (G_OBJECT (cell), "text", text, NULL);
	g_list_free (renderers);
}

/*
 * GnomeDbBrowser class implementation
 */

static void
gnome_db_browser_class_init (GnomeDbBrowserClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_browser_set_property;
	object_class->get_property = gnome_db_browser_get_property;
	object_class->finalize = gnome_db_browser_finalize;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_CONNECTION,
		g_param_spec_object ("connection", NULL, NULL,
				     GDA_TYPE_CONNECTION,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* add class signals */
	browser_signals[PROGRESS_MESSAGE] =
		g_signal_new ("progress_message",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbBrowserClass, progress_message),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE, 1, G_TYPE_STRING);
}

static void
gnome_db_browser_init (GnomeDbBrowser *browser, GnomeDbBrowserClass *klass)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* allocate private structure */
	browser->priv = g_new0 (GnomeDbBrowserPrivate, 1);
	browser->priv->cnc = NULL;
	browser->priv->object_nodes = NULL;
	browser->priv->detail_widgets = NULL;

	/* set up widgets */
	table = gnome_db_new_table_widget (2, 8, FALSE);
	gtk_box_pack_start (GTK_BOX (browser), table, TRUE, TRUE, 0);

	label = gnome_db_new_label_widget (_("Database"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 2, 2);
	browser->priv->db_selector = gnome_db_new_combo_widget ();
	gtk_combo_set_value_in_list (GTK_COMBO (browser->priv->db_selector), FALSE, TRUE);
	gtk_table_attach (GTK_TABLE (table), browser->priv->db_selector,
			  1, 2, 0, 1, GTK_SHRINK, GTK_SHRINK, 2, 2);

	browser->priv->paned = gnome_db_new_hpaned_widget ();
	gtk_table_attach (GTK_TABLE (table), browser->priv->paned, 0, 8, 1, 2,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  2, 2);

	browser->priv->scroll = gnome_db_new_scrolled_window_widget ();
	browser->priv->object_list = gnome_db_new_tree_view_widget (NULL);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (browser->priv->object_list), FALSE);
	g_signal_connect (G_OBJECT (gtk_tree_view_get_selection (
					    GTK_TREE_VIEW (browser->priv->object_list))), "changed",
			  G_CALLBACK (tree_selection_changed_cb), browser);
			  
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (browser->priv->object_list), FALSE);
	gtk_container_add (GTK_CONTAINER (browser->priv->scroll), browser->priv->object_list);
	gtk_paned_add1 (GTK_PANED (browser->priv->paned), browser->priv->scroll);

	/* Create column */
	column = gtk_tree_view_column_new ();

	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (column,
						 renderer,
			                         tree_value_set_func,
						 NULL, NULL);
	
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (browser->priv->object_list), column);

	gnome_db_browser_refresh (browser);

	browser->priv->detail_notebook = gnome_db_new_notebook_widget ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (browser->priv->detail_notebook), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (browser->priv->detail_notebook), TRUE);
	gtk_notebook_popup_disable (GTK_NOTEBOOK (browser->priv->detail_notebook));
	gtk_paned_add2 (GTK_PANED (browser->priv->paned), browser->priv->detail_notebook);

	/* load configuration */
	if (gnome_db_config_has_key (GNOME_DB_CONFIG_KEY_BROWSER_PANED_POSITION)) {
		gtk_paned_set_position (GTK_PANED (browser->priv->paned),
					gnome_db_config_get_int (GNOME_DB_CONFIG_KEY_BROWSER_PANED_POSITION));
	}

	g_signal_connect (G_OBJECT (browser->priv->paned), "notify",
			  G_CALLBACK (paned_notification_cb), browser);

	gnome_db_browser_refresh (browser);
}

static void
gnome_db_browser_set_property (GObject *object,
			       guint param_id,
			       const GValue *value,
			       GParamSpec *pspec)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	switch (param_id) {
	case PROP_CONNECTION :
		gnome_db_browser_set_connection (browser,
						 GDA_CONNECTION (g_value_get_object (value)));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_browser_get_property (GObject *object,
			       guint param_id,
			       GValue *value,
			       GParamSpec *pspec)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	switch (param_id) {
	case PROP_CONNECTION :
		g_value_set_object (value, G_OBJECT (browser->priv->cnc));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_browser_finalize (GObject *object)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* free memory */
	if (GDA_IS_CONNECTION (browser->priv->cnc)) {
		g_object_unref (G_OBJECT (browser->priv->cnc));
		browser->priv->cnc = NULL;
	}

	if (browser->priv->object_nodes) {
		g_list_foreach (browser->priv->object_nodes, (GFunc) g_free, NULL);
		g_list_free (browser->priv->object_nodes);
		browser->priv->object_nodes = NULL;
	}

	if (browser->priv->detail_widgets) {
		GList *l;

		while ((l = browser->priv->detail_widgets)) {
			ObjectDetailData *detail_data = (ObjectDetailData *) l->data;

			browser->priv->detail_widgets = g_list_remove (browser->priv->detail_widgets, l);
			g_free (detail_data->name);
			g_free (detail_data);
		}
	}

	g_free (browser->priv);
	browser->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_browser_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbBrowserClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_browser_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbBrowser),
			0,
			(GInstanceInitFunc) gnome_db_browser_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbBrowser", &info, 0);
	}
	return type;
}

/**
 * gnome_db_browser_new
 *
 * Create a new #GnomeDbBrowser widget. This widget lets you present to your
 * users a complete database browser, which lets him/her navigate through
 * his/her database objects (tables, views, procedures, etc), in a
 * visual way.
 *
 * Returns: the newly created object.
 */
GtkWidget *
gnome_db_browser_new (void)
{
	GnomeDbBrowser *browser;

	browser = g_object_new (GNOME_DB_TYPE_BROWSER, NULL);
	return GTK_WIDGET (browser);
}

/**
 * gnome_db_browser_get_connection
 * @browser: a #GnomeDbBrowser widget.
 *
 * Retrieve the #GdaConnection object being used by the given browser
 * widget. The connection object associated with the browser makes it
 * show the objects for that connection.
 *
 * Returns: a #GdaConnection object.
 */
GdaConnection *
gnome_db_browser_get_connection (GnomeDbBrowser *browser)
{
	g_return_val_if_fail (GNOME_DB_IS_BROWSER (browser), NULL);
	return browser->priv->cnc;
}

/*
 * gnome_db_browser_set_connection
 * @browser: a #GnomeDbBrowser widget.
 * @cnc: a #GdaConnection object.
 *
 * Associates a #GdaConnection object with the given browser widget. This will
 * make the browser widget refresh its widgets and immediately display
 * the objects contained in the new connection (@cnc).
 */
void
gnome_db_browser_set_connection (GnomeDbBrowser *browser, GdaConnection *cnc)
{
	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE],
		       0, _("Loading database schemas..."));

	/* clear the current setup */
	if (GDA_IS_CONNECTION (browser->priv->cnc)) {
		g_object_unref (G_OBJECT (browser->priv->cnc));
		browser->priv->cnc = NULL;
	}

	/* refresh views */
	if (GDA_IS_CONNECTION (cnc))
		g_object_ref (G_OBJECT (cnc));
	browser->priv->cnc = cnc;

	gnome_db_browser_refresh (browser);

	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE], 0, NULL);
}

static void
add_schema_objects (GnomeDbBrowser *browser,
		    GtkTreeStore *store,
		    GdaConnectionSchema schema,
		    GdaConnectionFeature ask_feature,
		    const gchar *title,
		    const gchar *pixmap)
{
	GdaDataModel *model;
	gint rows, i;
	ObjectNodeData *node_data;

	if (!GDA_IS_CONNECTION (browser->priv->cnc))
		return;

	/* check if the underlying provider supports it */
	if (ask_feature) {
		if (!gda_connection_supports (browser->priv->cnc, ask_feature))
			return;
	}

	model = gda_connection_get_schema (browser->priv->cnc, schema, NULL);
	if (!GDA_IS_DATA_MODEL (model))
		return;
	rows = gda_data_model_get_n_rows (model);
	if (!rows) {
		g_object_unref (G_OBJECT (model));
		return;
	}

	node_data = g_new0 (ObjectNodeData, 1);
	node_data->schema = schema;

	/* create top node for these schema objects */
	node_data->pixbuf = gdk_pixbuf_new_from_file (pixmap, NULL);

	gtk_tree_store_append (store, &node_data->top, NULL);
	gtk_tree_store_set (store, &node_data->top, 0, node_data->pixbuf, 1, title, -1);

	/* fill in this node of the tree */
	for (i = 0; i < rows; i++) {
		gchar *value;
		GtkTreeIter node;

		value = gda_value_stringify (gda_data_model_get_value_at (model, 0, i));
		gtk_tree_store_append (store, &node, &node_data->top);
		gtk_tree_store_set (store, &node, 0, node_data->pixbuf, 1, value, -1);

		g_free (value);
	}

	g_object_unref (G_OBJECT (model));
	gdk_pixbuf_unref (node_data->pixbuf);

	/* update internal lists */
	browser->priv->object_nodes = g_list_append (browser->priv->object_nodes, node_data);
}

/**
 * gnome_db_browser_refresh
 * @browser: a #GnomeDbBrowser widget.
 *
 * Make the browser widget refresh its currently displayed data.
 */
void
gnome_db_browser_refresh (GnomeDbBrowser *browser)
{
	GtkTreeStore *store;
	GdaDataModel *model;
	GList *l;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE],
		       0, _("Refreshing data..."));

	/* clear up widgets and structures */
	g_list_foreach (browser->priv->object_nodes, (GFunc) g_free, NULL);
	g_list_free (browser->priv->object_nodes);
	browser->priv->object_nodes = NULL;

	l = g_list_append (NULL, "");
	gtk_combo_set_popdown_strings (GTK_COMBO (browser->priv->db_selector), l);
	g_list_free (l);
	gtk_widget_set_sensitive (GTK_WIDGET (browser->priv->db_selector), FALSE);

	while ((l = browser->priv->detail_widgets)) {
		ObjectDetailData *detail_data = (ObjectDetailData *) l->data;

		browser->priv->detail_widgets = g_list_remove (browser->priv->detail_widgets, detail_data);
		gtk_notebook_remove_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
					  gtk_notebook_page_num (
						  GTK_NOTEBOOK (browser->priv->detail_notebook),
						  detail_data->widget));
		g_free (detail_data->name);
		g_free (detail_data);
	}

	/* list of databases */
	if (GDA_IS_CONNECTION (browser->priv->cnc) &&
	    gda_connection_is_open (browser->priv->cnc)) {
		GList *strs = NULL;
		gint rows, r;

		model = gda_connection_get_schema (browser->priv->cnc,
						   GDA_CONNECTION_SCHEMA_DATABASES,
						   NULL);
		rows = gda_data_model_get_n_rows (model);
		for (r = 0; r < rows; r++) {
			gchar *s;

			s = gda_value_stringify (gda_data_model_get_value_at (model, 0, r));
			strs = g_list_append (strs, s);
		}

		gtk_combo_set_popdown_strings (GTK_COMBO (browser->priv->db_selector),
					       strs);
		gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (browser->priv->db_selector)->entry),
				    gda_connection_get_database (browser->priv->cnc));
		gtk_widget_set_sensitive (GTK_WIDGET (browser->priv->db_selector), TRUE);

		g_list_foreach (strs, (GFunc) g_free, NULL);
		g_list_free (strs);
	}

	/* create the new tree model */
	store = gtk_tree_store_new (2, GDK_TYPE_PIXBUF, G_TYPE_STRING);
	add_schema_objects (browser, store, GDA_CONNECTION_SCHEMA_TABLES,
			    0, _("Tables"), LIBGNOMEDB_ICONSDIR "/gnome-db-tables_16x16.png");
	add_schema_objects (browser, store, GDA_CONNECTION_SCHEMA_VIEWS,
			    GDA_CONNECTION_FEATURE_VIEWS, _("Views"),
			    LIBGNOMEDB_ICONSDIR "/gnome-db-views_16x16.png");
	add_schema_objects (browser, store, GDA_CONNECTION_SCHEMA_INDEXES,
			    GDA_CONNECTION_FEATURE_INDEXES, _("Indexes"),
			    LIBGNOMEDB_ICONSDIR "/gnome-db-indexes_16x16.png");
	add_schema_objects (browser, store, GDA_CONNECTION_SCHEMA_SEQUENCES,
			    GDA_CONNECTION_FEATURE_SEQUENCES, _("Sequences"),
			    LIBGNOMEDB_ICONSDIR "/gnome-db-squences_16x16.png");
	add_schema_objects (browser, store, GDA_CONNECTION_SCHEMA_PROCEDURES,
			    GDA_CONNECTION_FEATURE_PROCEDURES, _("Procedures"),
			    LIBGNOMEDB_ICONSDIR "/gnome-db-procedures_16x16.png");
	add_schema_objects (browser, store, GDA_CONNECTION_SCHEMA_TRIGGERS,
			    GDA_CONNECTION_FEATURE_TRIGGERS, _("Triggers"),
			    LIBGNOMEDB_ICONSDIR "/gnome-db-triggers_16x16.png");
	add_schema_objects (browser, store, GDA_CONNECTION_SCHEMA_AGGREGATES,
			    GDA_CONNECTION_FEATURE_AGGREGATES, _("Aggregates"),
			    LIBGNOMEDB_ICONSDIR "/gnome-db-aggregates_16x16.png");
	add_schema_objects (browser, store, GDA_CONNECTION_SCHEMA_TYPES,
			    0, _("Types"), LIBGNOMEDB_ICONSDIR "/gnome-db-types_16x16.png");

	/* display the new data */
	gtk_tree_view_set_model (GTK_TREE_VIEW (browser->priv->object_list),
				 GTK_TREE_MODEL (store));
	g_object_unref (G_OBJECT (store));
	
	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE], 0, NULL);
}
