/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtklabel.h>
#include <libgnomedb/gnome-db-gray-bar.h>
#include <libgnomedb/gnome-db-util.h>

#define PARENT_TYPE GTK_TYPE_EVENT_BOX

struct _GnomeDbGrayBarPrivate {
	GtkWidget *label;
	gchar *real_text;
};

static void gnome_db_gray_bar_class_init   (GnomeDbGrayBarClass *klass);
static void gnome_db_gray_bar_init         (GnomeDbGrayBar *bar, GnomeDbGrayBarClass *klass);
static void gnome_db_gray_bar_set_property (GObject *object,
					    guint paramid,
					    const GValue *value,
					    GParamSpec *pspec);
static void gnome_db_gray_bar_get_property (GObject *object,
					    guint param_id,
					    GValue *value,
					    GParamSpec *pspec);
static void gnome_db_gray_bar_finalize     (GObject *object);

enum {
	PROP_0,
	PROP_TEXT
};

static GObjectClass *parent_class = NULL;

/*
 * GnomeDbGrayBar class implementation
 */

static void
gnome_db_gray_bar_style_set (GtkWidget *w, GtkStyle *previous_style)
{
	static int in_style_set = 0;
	GtkRcStyle *rc_style;

	if (in_style_set > 0)
                return;

        in_style_set ++;

	rc_style = gtk_rc_style_new ();
	rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_BG;
	rc_style->bg[GTK_STATE_NORMAL].red = 0x8000;
        rc_style->bg[GTK_STATE_NORMAL].green = 0x8000;
        rc_style->bg[GTK_STATE_NORMAL].blue = 0x8000;

	gtk_widget_modify_style (w, rc_style);
        gtk_rc_style_unref (rc_style);

	in_style_set --;

	GTK_WIDGET_CLASS (parent_class)->style_set (w, previous_style);
}

static void
gnome_db_gray_bar_class_init (GnomeDbGrayBarClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_gray_bar_set_property;
	object_class->get_property = gnome_db_gray_bar_get_property;
	object_class->finalize = gnome_db_gray_bar_finalize;
	widget_class->style_set = gnome_db_gray_bar_style_set;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_TEXT,
		g_param_spec_string ("text", NULL, NULL, NULL,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_gray_bar_init (GnomeDbGrayBar *bar, GnomeDbGrayBarClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_GRAY_BAR (bar));

	bar->priv = g_new0 (GnomeDbGrayBarPrivate, 1);

	/* set up widgets */
	bar->priv->label = gnome_db_new_label_widget ("");
	gtk_misc_set_alignment (GTK_MISC (bar->priv->label), 0.0, 0.5);
        gtk_misc_set_padding (GTK_MISC (bar->priv->label), 6, 6);
	gtk_container_add (GTK_CONTAINER (bar), bar->priv->label);
	bar->priv->real_text = NULL;
}

static void
gnome_db_gray_bar_set_property (GObject *object,
				guint param_id,
				const GValue *value,
				GParamSpec *pspec)
{
	GnomeDbGrayBar *bar = (GnomeDbGrayBar *) object;

	g_return_if_fail (GNOME_DB_IS_GRAY_BAR (bar));

	switch (param_id) {
	case PROP_TEXT :
		gnome_db_gray_bar_set_text (bar, g_value_get_string (value));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_gray_bar_get_property (GObject *object,
				guint param_id,
				GValue *value,
				GParamSpec *pspec)
{
	GnomeDbGrayBar *bar = (GnomeDbGrayBar *) object;

	g_return_if_fail (GNOME_DB_IS_GRAY_BAR (bar));

	switch (param_id) {
	case PROP_TEXT :
		g_value_set_string (value, gnome_db_gray_bar_get_text (bar));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_gray_bar_finalize (GObject *object)
{
	GnomeDbGrayBar *bar = (GnomeDbGrayBar *) object;

	g_return_if_fail (GNOME_DB_IS_GRAY_BAR (bar));

	if (bar->priv->real_text) {
		g_free (bar->priv->real_text);
		bar->priv->real_text = NULL;
	}

	g_free (bar->priv);
	bar->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_gray_bar_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGrayBarClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_gray_bar_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGrayBar),
			0,
			(GInstanceInitFunc) gnome_db_gray_bar_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbGrayBar", &info, 0);
	}
	return type;
}

/**
 * gnome_db_gray_bar_new
 */
GtkWidget *
gnome_db_gray_bar_new (const gchar *label)
{
	GnomeDbGrayBar *bar;

	bar = g_object_new (GNOME_DB_TYPE_GRAY_BAR, NULL);
	gnome_db_gray_bar_set_text (bar, label);

	return GTK_WIDGET (bar);
}

/**
 * gnome_db_gray_bar_get_text
 * @bar: A #GnomeDbGrayBar widget.
 *
 * Get the text being displayed in the given gray bar widget.
 *
 * Returns: the text in the widget.
 */
const gchar *
gnome_db_gray_bar_get_text (GnomeDbGrayBar *bar)
{
	g_return_val_if_fail (GNOME_DB_IS_GRAY_BAR (bar), NULL);
	return (const gchar *) bar->priv->real_text;
}

/**
 * gnome_db_gray_bar_set_text
 */
void
gnome_db_gray_bar_set_text (GnomeDbGrayBar *bar, const gchar *text)
{
	gchar *markup;

	g_return_if_fail (GNOME_DB_IS_GRAY_BAR (bar));

	if (bar->priv->real_text)
		g_free (bar->priv->real_text);
	bar->priv->real_text = g_strdup (text);

	markup = g_strdup_printf ("<big><b><span foreground=\"#FFFFFF\">%s</span></b></big>",
				  text);
	gtk_label_set_markup (GTK_LABEL (bar->priv->label), markup);

	g_free (markup);
}
