/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <bonobo/bonobo-property-bag.h>
#include <libgda/gda-marshal.h>
#include <bonobo/bonobo-ui-util.h>
#include <libgnomedb/gnome-db-control.h>

#define PARENT_TYPE BONOBO_TYPE_CONTROL

struct _GnomeDbControlPrivate {
	BonoboPropertyBag *properties;

	/* UI properties */
	gchar *app_prefix;
	gchar *ui_xml_file;
	BonoboUIVerb *verbs;
	gpointer cb_user_data;
};

static void gnome_db_control_class_init (GnomeDbControlClass *klass);
static void gnome_db_control_init       (GnomeDbControl *control, GnomeDbControlClass *klass);
static void gnome_db_control_finalize   (GObject *object);

enum {
	ACTIVATED,
	GET_PROPERTY,
	SET_PROPERTY,
	LAST_SIGNAL
};

static gint control_signals[LAST_SIGNAL];
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
control_activated_cb (BonoboControl *bonobo_control, gboolean activate, gpointer user_data)
{
	g_return_if_fail (GNOME_DB_IS_CONTROL (bonobo_control));

        if (activate) {
                gnome_db_control_activate (GNOME_DB_CONTROL (bonobo_control));
		g_signal_emit (G_OBJECT (bonobo_control), control_signals[ACTIVATED], 0);
	}
        else
                gnome_db_control_deactivate (GNOME_DB_CONTROL (bonobo_control));
}

static void
get_prop_cb (BonoboPropertyBag *bag, BonoboArg *arg, guint arg_id, gpointer user_data)
{
        GnomeDbControl *control = (GnomeDbControl *) user_data;

        g_return_if_fail (GNOME_DB_IS_CONTROL (control));
	g_signal_emit (G_OBJECT (control), control_signals[GET_PROPERTY], 0, arg, arg_id);
}

static void
set_prop_cb (BonoboPropertyBag *bag, const BonoboArg *arg, guint arg_id, gpointer user_data)
{
        GnomeDbControl *control = (GnomeDbControl *) user_data;

        g_return_if_fail (GNOME_DB_IS_CONTROL (control));
	g_signal_emit (G_OBJECT (control), control_signals[SET_PROPERTY], 0, arg, arg_id);
}

/*
 * GnomeDbControl class implementation
 */

static void
gnome_db_control_class_init (GnomeDbControlClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_control_finalize;

	/* add class signals */
	control_signals[ACTIVATED] =
		g_signal_new ("activated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbControlClass, activated),
		              NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	control_signals[GET_PROPERTY] =
		g_signal_new ("get_property",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbControlClass, get_property),
			      NULL, NULL,
			      gda_marshal_VOID__POINTER_UINT,
			      G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_UINT);
	control_signals[SET_PROPERTY] =
		g_signal_new ("set_property",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbControlClass, set_property),
			      NULL, NULL,
			      gda_marshal_VOID__POINTER_UINT,
			      G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_UINT);
}

static void
gnome_db_control_init (GnomeDbControl *control, GnomeDbControlClass *klass)
{
	/* allocate the internal structure */
	control->priv = g_new0 (GnomeDbControlPrivate, 1);
}

static void
gnome_db_control_finalize (GObject *object)
{
	GnomeDbControl *control = (GnomeDbControl *) object;

	g_return_if_fail (GNOME_DB_IS_CONTROL (control));

	/* free memory */
	if (control->priv->app_prefix != NULL) {
		g_free (control->priv->app_prefix);
		control->priv->app_prefix = NULL;
	}
	if (control->priv->ui_xml_file != NULL) {
		g_free (control->priv->ui_xml_file);
		control->priv->ui_xml_file = NULL;
	}

	g_free (control->priv);
	control->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_control_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbControlClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_control_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbControl),
			0,
			(GInstanceInitFunc) gnome_db_control_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbControl", &info, 0);
	}
	return type;
}

/**
 * gnome_db_control_construct
 */
GnomeDbControl *
gnome_db_control_construct (GnomeDbControl *control, GtkWidget *widget)
{
	g_return_val_if_fail (GNOME_DB_IS_CONTROL (control), control);
	g_return_val_if_fail (GTK_IS_WIDGET (widget), control);

	gtk_widget_show (widget);
	control = GNOME_DB_CONTROL (bonobo_control_construct (
					    BONOBO_CONTROL (control), widget));

	/* set up the Bonobo control */
	g_signal_connect (G_OBJECT (control), "activate",
			  G_CALLBACK (control_activated_cb), NULL);

	control->priv->properties = bonobo_property_bag_new (
		(BonoboPropertyGetFn) get_prop_cb,
                (BonoboPropertySetFn) set_prop_cb,
                (gpointer) control);
	bonobo_control_set_properties (
		BONOBO_CONTROL (control),
		bonobo_object_corba_objref (BONOBO_OBJECT (control->priv->properties)),
		NULL);

	return control;
}

/**
 * gnome_db_control_new
 * @widget: the widget to be wrapped by the control
 *
 * Creates a new #GnomeDbControl object, which is an instance of the
 * Bonobo::Control CORBA interface, with some useful additions.
 *
 * When calling this function, you must specify an already created
 * #GtkWidget, which is the one that will be used for creating the
 * Bonobo control. You should use this class if you intend to write your
 * own Bonobo controls.
 *
 * Returns: the newly created #GnomeDbControl object.
 */
GnomeDbControl *
gnome_db_control_new (GtkWidget *widget)
{
	GnomeDbControl *control;

	control = g_object_new (GNOME_DB_TYPE_CONTROL, NULL);
	return gnome_db_control_construct (control, widget);
}

/**
 * gnome_db_control_set_ui
 * @control: a #GnomeDbControl object
 * @app_prefix: the prefix where your application data is installed. This is
 * used by the underlying Bonobo code, which searches for the given file in
 * that directory
 * @ui_xml_file: name of the file in @app_prefix that contains the XML
 * description of the component's UI
 * @verbs: list of verbs and corresponding actions
 * @user_data: data to be passed to the verb callback functions
 *
 * Sets the UI for the given #GnomeDbControl object. This is done by
 * specifying a file that contains the description of the UI (see Bonobo
 * documentation) and a set of verbs and their corresponding actions.
 */
void
gnome_db_control_set_ui (GnomeDbControl *control,
                         const gchar *app_prefix,
                         const gchar *ui_xml_file,
                         BonoboUIVerb *verbs,
                         gpointer user_data)
{
	g_return_if_fail (GNOME_DB_IS_CONTROL (control));

        if (control->priv->app_prefix != NULL)
                g_free (control->priv->app_prefix);
        control->priv->app_prefix = g_strdup (app_prefix);

        if (control->priv->ui_xml_file != NULL)
                g_free (control->priv->ui_xml_file);
        control->priv->ui_xml_file = g_strdup (ui_xml_file);

        control->priv->verbs = verbs;
        control->priv->cb_user_data = user_data;
}

/**
 * gnome_db_control_activate
 * @control: a #GnomeDbControl object
 *
 * Activates the given #GnomeDbControl object. This means that the UI for
 * this control is merged with the UI of the container.
 */
void
gnome_db_control_activate (GnomeDbControl *control)
{
        BonoboUIComponent *uic;

        g_return_if_fail (GNOME_DB_IS_CONTROL (control));

        uic = bonobo_control_get_ui_component (BONOBO_CONTROL (control));
        if (uic != NULL) {
                Bonobo_UIContainer remote_ui_container;

                remote_ui_container = bonobo_control_get_remote_ui_container (
			BONOBO_CONTROL (control), NULL);
                bonobo_ui_component_set_container (uic, remote_ui_container, NULL);
                bonobo_object_release_unref (remote_ui_container, NULL);

                bonobo_ui_component_add_verb_list_with_data (
                        uic, control->priv->verbs, control->priv->cb_user_data);
                bonobo_ui_component_freeze (uic, NULL);
                bonobo_ui_util_set_ui (uic,
                                       control->priv->app_prefix,
                                       control->priv->ui_xml_file,
                                       g_get_prgname(), NULL);
                bonobo_ui_component_thaw (uic, NULL);
        }
}

/**
 * gnome_db_control_deactivate
 * @control: a #GnomeDbControl object
 *
 * Deactivates the given #GnomeDbControl object, which means that the
 * control's UI is removed from the container's.
 */
void
gnome_db_control_deactivate (GnomeDbControl *control)
{
        BonoboUIComponent* uic;

        g_return_if_fail (GNOME_DB_IS_CONTROL (control));

        uic = bonobo_control_get_ui_component (BONOBO_CONTROL (control));
        if (uic) {
                bonobo_ui_component_rm (uic, "/", NULL);
                bonobo_ui_component_unset_container (uic, NULL);
        }
}

/**
 * gnome_db_control_set_status
 * @control: a #GnomeDbControl object
 * @msg: message to show on the status bar
 *
 * Displays the given message on the container's status bar. If you use
 * #GnomeDbWindow for your containers, you automatically have a status
 * bar on your window, and, with this function, you can set the message
 * to be displayed on that status bar from the components.
 */
void
gnome_db_control_set_status (GnomeDbControl *control, const gchar *msg)
{
        BonoboUIComponent* uic;

        g_return_if_fail (GNOME_DB_CONTROL (control));

        uic = bonobo_control_get_ui_component (BONOBO_CONTROL (control));
        if (uic)
                bonobo_ui_component_set_status (uic, msg, NULL);
}
