/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Gonzalo Paniagua Javier <gonzalo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <bonobo-activation/bonobo-activation.h>
#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-i18n.h>
#include <gconf/gconf.h>
#include <libgda/gda-log.h>
#include <libgnomedb/gnome-db-config.h>

static GList *
activation_property_to_list (Bonobo_ActivationProperty *prop)
{
	GList *list = NULL;

	g_return_val_if_fail (prop != NULL, NULL);

	if (prop->v._d == Bonobo_ACTIVATION_P_STRING)
		list = g_list_append (list, g_strdup (prop->v._u.value_string));
	else if (prop->v._d == Bonobo_ACTIVATION_P_STRINGV) {
		gint j;
		Bonobo_StringList strlist = prop->v._u.value_stringv;

		for (j = 0; j < strlist._length; j++) {
                        gchar *str = g_strdup (strlist._buffer[j]);
			list = g_list_append (list, str);
		}
	}

	return list;
}

static gchar *
activation_property_to_string (Bonobo_ActivationProperty *prop)
{
	g_return_val_if_fail (prop != NULL, NULL);

	if (prop->v._d == Bonobo_ACTIVATION_P_STRING)
		return g_strdup (prop->v._u.value_string);
	else if (prop->v._d == Bonobo_ACTIVATION_P_STRINGV) {
		gint j;
		GString *str = NULL;
		Bonobo_StringList strlist = prop->v._u.value_stringv;
		for (j = 0; j < strlist._length; j++) {
			if (!str)
				str = g_string_new (strlist._buffer[j]);
			else {
				str = g_string_append (str, ";");
				str = g_string_append (str, strlist._buffer[j]);
			}
		}
		if (str) {
			gchar *ret = str->str;
			g_string_free (str, FALSE);
			return ret;
		}
	}

	return NULL;
}

static GdaParameter *
activation_property_to_parameter (Bonobo_ActivationProperty *prop)
{
        GdaParameter *param;
        gchar *str;

        g_return_val_if_fail (prop != NULL, NULL);

        switch (prop->v._d) {
        case Bonobo_ACTIVATION_P_STRING :
                param = gda_parameter_new ((const gchar *) prop->name, GDA_VALUE_TYPE_STRING);
                gda_value_set_string ((GdaValue *) gda_parameter_get_value (param),
                                      prop->v._u.value_string);
                break;
        case Bonobo_ACTIVATION_P_NUMBER :
                param = gda_parameter_new ((const gchar *) prop->name, GDA_VALUE_TYPE_DOUBLE);
                gda_value_set_double ((GdaValue *) gda_parameter_get_value (param),
                                      prop->v._u.value_number);
                break;
        case Bonobo_ACTIVATION_P_BOOLEAN :
                param = gda_parameter_new ((const gchar *) prop->name, GDA_VALUE_TYPE_BOOLEAN);
                gda_value_set_boolean ((GdaValue *) gda_parameter_get_value (param),
                                       prop->v._u.value_boolean);
                break;
        case Bonobo_ACTIVATION_P_STRINGV :
                param = gda_parameter_new ((const gchar *) prop->name, GDA_VALUE_TYPE_STRING);
                str = activation_property_to_string (prop);
                if (str) {
                        gda_value_set_string ((GdaValue *) gda_parameter_get_value (param), str);
                        g_free (str);
                }
                break;
        default :
                param = NULL;
        }

	return param;
}

/**
 * gnome_db_config_get_component_list
 * @query: condition for components to be retrieved.
 *
 * Return a list of all components currently installed in
 * the system that match the given query (see
 * BonoboActivation documentation). Each of the nodes
 * in the returned GList is a #GnomeDbComponentInfo. To free
 * the returned list, call the #gnome_db_config_free_component_list
 * function.
 *
 * Returns: a GList of #GnomeDbComponentInfo structures.
 */
GList *
gnome_db_config_get_component_list (const gchar *query)
{
	CORBA_Environment ev;
	Bonobo_ServerInfoList *server_list;
	gint i;
	GList *list = NULL;

	CORBA_exception_init (&ev);
	server_list = bonobo_activation_query (query, NULL, &ev);
	if (BONOBO_EX (&ev)) {
		gda_log_error (_("Could not query CORBA components"));
		CORBA_exception_free (&ev);
		return NULL;
	}

	/* create the list to be returned from the CORBA sequence */
	for (i = 0; i < server_list->_length; i++) {
		GnomeDbComponentInfo *comp_info;
		gint j;
		Bonobo_ServerInfo *bonobo_info = &server_list->_buffer[i];

		comp_info = g_new0 (GnomeDbComponentInfo, 1);
		comp_info->id = g_strdup (bonobo_info->iid);
		comp_info->location = g_strdup (bonobo_info->location_info);
		comp_info->description = activation_property_to_string (
			(Bonobo_ActivationProperty *) bonobo_server_info_prop_find (bonobo_info, "description"));
		comp_info->repo_ids = activation_property_to_list (
			bonobo_server_info_prop_find (bonobo_info, "repo_ids"));
		comp_info->username = g_strdup (bonobo_info->username);
		comp_info->hostname = g_strdup (bonobo_info->hostname);
		comp_info->domain = g_strdup (bonobo_info->domain);
		if (!strcmp (bonobo_info->server_type, "exe"))
			comp_info->type = GNOME_DB_COMPONENT_TYPE_EXE;
		else if (!strcmp (bonobo_info->server_type, "shlib"))
			comp_info->type = GNOME_DB_COMPONENT_TYPE_SHLIB;
		else if (!strcmp (bonobo_info->server_type, "factory"))
			comp_info->type = GNOME_DB_COMPONENT_TYPE_FACTORY;
		else
			comp_info->type = GNOME_DB_COMPONENT_TYPE_INVALID;

		/* get all properties */
		comp_info->properties = gda_parameter_list_new ();
		for (j = 0; j < bonobo_info->props._length; j++) {
			GdaParameter *param;
			param = activation_property_to_parameter (
				&bonobo_info->props._buffer[j]);
			if (param != NULL) {
				gda_parameter_list_add_parameter (
					comp_info->properties, param);
			}
		}
		list = g_list_append (list, comp_info);
	}

	CORBA_free (server_list);
	
	return list;
}

/**
 * gnome_db_config_free_component_list
 */
void
gnome_db_config_free_component_list (GList *list)
{
	GList *l;

	for (l = g_list_first (list); l; l = l->next) {
		GnomeDbComponentInfo *comp_info = (GnomeDbComponentInfo *) l->data;

		if (comp_info != NULL) {
			g_free (comp_info->id);
			g_free (comp_info->location);
			g_free (comp_info->description);
                        g_free (comp_info->username);
			g_free (comp_info->hostname);
			g_free (comp_info->domain);
			g_list_foreach (comp_info->repo_ids, (GFunc) g_free, NULL);
			g_list_free (comp_info->repo_ids);
			gda_parameter_list_free (comp_info->properties);
			g_free (comp_info);
		}
	}

	g_list_free (list);
}

/*
 * Copies configuration from gda_config file to gconf
 */
static void
sync_from_gda ()
{
	GList *s;
	GList *sections;
	GList *e;
	GList *entries;
	gchar *spath;
	gchar *epath;
	gchar *type;
	
	sections = gda_config_list_sections (GDA_CONFIG_BASE);
	for (s = sections; s; s = s->next){
		spath = g_strdup_printf ("%s/%s", GDA_CONFIG_BASE,
						  (char *) s->data);
		entries = gda_config_list_keys (spath);
		for (e = entries; e; e = e->next){
			epath = g_strdup_printf ("%s/%s", spath, (char *) e->data);
			type = gda_config_get_type (epath);
			g_assert (type);
			if (!strcmp (type, "string"))
				gnome_db_config_set_string (epath,
						gda_config_get_string (epath));
			else if (!strcmp (type, "bool"))
				gnome_db_config_set_boolean (epath,
						gda_config_get_boolean (epath));
			else if (!strcmp (type, "float"))
				gnome_db_config_set_float (epath,
						gda_config_get_float (epath));
			else if (!strcmp (type, "long"))
				gnome_db_config_set_int (epath,
						gda_config_get_int (epath));
			else
				g_assert_not_reached ();
			
			g_free (type);
			g_free (epath);
		}
		gda_config_free_list (entries);
		g_free (spath);
	}
	gda_config_free_list (sections);
}

static void
sync_to_gda (GConfClient *client,
	     guint cnxn_id,
	     GConfEntry *entry,
	     gpointer user_data)
{
	const gchar *key;
	const GConfValue *value;

	g_return_if_fail (entry != NULL);

	key = gconf_entry_get_key (entry);
	value = gconf_entry_get_value (entry);
	if (value == NULL){
		gda_config_remove_key (key);
		return;
	}

	switch (value->type) {
	case GCONF_VALUE_STRING:
		gda_config_set_string (key, gconf_value_get_string (value));
		break;
	case GCONF_VALUE_INT:
		gda_config_set_int (key, gconf_value_get_int (value));
		break;
	case GCONF_VALUE_FLOAT:
		gda_config_set_float (key, gconf_value_get_float (value));
		break;
	case GCONF_VALUE_BOOL: 
		gda_config_set_boolean (key, gconf_value_get_bool (value));
		break;
	default:
		g_warning (_("Unsupported type in GconfEntry: %d"), value->type);
	}
}

static GConfClient *conf_client;

static GConfClient *
get_conf_client ()
{
	if (!conf_client) {
		/* initialize GConf */
		if (!gconf_is_initialized ())
			gconf_init (0, NULL, NULL);
		conf_client = gconf_client_get_default ();
		sync_from_gda ();
		gnome_db_config_add_listener (GDA_CONFIG_BASE, sync_to_gda, NULL);
	}
	return conf_client;
}

/**
 * gnome_db_config_get_string
 * @path: path to the configuration entry
 *
 * Gets the value of the specified configuration entry as a string. You
 * are then responsible to free the returned string
 *
 * Returns: the value stored at the given entry
 */
gchar *
gnome_db_config_get_string (const gchar * path)
{
	return gconf_client_get_string (get_conf_client (), path, NULL);
}

/**
 * gnome_db_config_get_int
 * @path: path to the configuration entry
 *
 * Gets the value of the specified configuration entry as an integer
 *
 * Returns: the value stored at the given entry
 */
gint
gnome_db_config_get_int (const gchar * path)
{
	return gconf_client_get_int (get_conf_client (), path, NULL);
}

/**
 * gnome_db_config_get_float
 * @path: path to the configuration entry
 *
 * Gets the value of the specified configuration entry as a float
 *
 * Returns: the value stored at the given entry
 */
gdouble
gnome_db_config_get_float (const gchar * path)
{
	return gconf_client_get_float (get_conf_client (), path, NULL);
}

/**
 * gnome_db_config_get_boolean
 * @path: path to the configuration entry
 *
 * Gets the value of the specified configuration entry as a boolean
 *
 * Returns: the value stored at the given entry
 */
gboolean
gnome_db_config_get_boolean (const gchar * path)
{
	return gconf_client_get_bool (get_conf_client (), path, NULL);
}

/**
 * gnome_db_config_set_string
 * @path: path to the configuration entry
 * @new_value: new value
 *
 * Sets the given configuration entry to contain a string
 */
void
gnome_db_config_set_string (const gchar * path, const gchar * new_value)
{
	gconf_client_set_string (get_conf_client (), path, new_value, NULL);
}

/**
 * gnome_db_config_set_int
 * @path: path to the configuration entry
 * @new_value: new value
 *
 * Sets the given configuration entry to contain an integer
 */
void
gnome_db_config_set_int (const gchar * path, gint new_value)
{
	gconf_client_set_int (get_conf_client (), path, new_value, NULL);
}

/**
 * gnome_db_config_set_float
 * @path: path to the configuration entry
 * @new_value: new value
 *
 * Sets the given configuration entry to contain a float
 */
void
gnome_db_config_set_float (const gchar * path, gdouble new_value)
{
	gconf_client_set_float (get_conf_client (), path, new_value, NULL);
}

/**
 * gnome_db_config_set_boolean
 * @path: path to the configuration entry
 * @new_value: new value
 *
 * Sets the given configuration entry to contain a boolean
 */
void
gnome_db_config_set_boolean (const gchar * path, gboolean new_value)
{
	g_return_if_fail (path != NULL);
	gconf_client_set_bool (get_conf_client (), path, new_value, NULL);
}

/**
 * gnome_db_config_remove_section
 * @path: path to the configuration section
 *
 * Remove the given section from the configuration database
 */
void
gnome_db_config_remove_section (const gchar * path)
{
	g_return_if_fail (path != NULL);
	gconf_client_remove_dir (get_conf_client (), path, NULL);
}

/**
 * gnome_db_config_remove_key
 * @path: path to the configuration entry
 *
 * Remove the given entry from the configuration database
 */
void
gnome_db_config_remove_key (const gchar * path)
{
	gconf_client_unset (get_conf_client (), path, NULL);
}

/**
 * gnome_db_config_has_section
 * @path: path to the configuration section
 *
 * Checks whether the given section exists in the configuration
 * system
 *
 * Returns: TRUE if the section exists, FALSE otherwise
 */
gboolean
gnome_db_config_has_section (const gchar * path)
{
	return gconf_client_dir_exists (get_conf_client (), path, NULL);
}

/**
 * gnome_db_config_has_key
 * @path: path to the configuration key
 *
 * Check whether the given key exists in the configuration system
 *
 * Returns: TRUE if the entry exists, FALSE otherwise
 */
gboolean
gnome_db_config_has_key (const gchar * path)
{
	GConfValue *value;

	g_return_val_if_fail (path != NULL, FALSE);

	value = gconf_client_get (get_conf_client (), path, NULL);
	if (value) {
		gconf_value_free (value);
		return TRUE;
	}
	return FALSE;
}

/**
 * gnome_db_config_list_sections
 * @path: path for root dir
 *
 * Return a GList containing the names of all the sections available
 * under the given root directory.
 *
 * To free the returned value, you can use #gnome_db_config_free_list
 *
 * Returns: a list containing all the section names
 */
GList *
gnome_db_config_list_sections (const gchar * path)
{
	GList *ret = NULL;
	GSList *slist;

	g_return_val_if_fail (path != NULL, NULL);

	slist = gconf_client_all_dirs (get_conf_client (), path, NULL);
	if (slist) {
		GSList *node;

		for (node = slist; node != NULL; node = g_slist_next (node)) {
			gchar *section_name =
				strrchr ((const char *) node->data, '/');
			if (section_name) {
				ret = g_list_append (ret,
						     g_strdup (section_name +
							       1));
			}
		}
		g_slist_free (slist);
	}
	return ret;
}

/**
 * gnome_db_config_list_keys
 * @path: path for root dir
 *
 * Returns a list of all keys that exist under the given path.
 *
 * To free the returned value, you can use #gnome_db_config_free_list
 *
 * Returns: a list containing all the key names
 */
GList *
gnome_db_config_list_keys (const gchar * path)
{
	GList *ret = NULL;
	GSList *slist;

	g_return_val_if_fail (path != NULL, NULL);

	slist = gconf_client_all_entries (get_conf_client (), path, NULL);
	if (slist) {
		GSList *node;

		for (node = slist; node != NULL; node = g_slist_next (node)) {
			GConfEntry *entry = (GConfEntry *) node->data;
			if (entry) {
				gchar *entry_name;

				entry_name =
					strrchr ((const char *)
						 gconf_entry_get_key (entry),
						 '/');
				if (entry_name) {
					ret = g_list_append (ret,
							     g_strdup
							     (entry_name +
							      1));
				}
				gconf_entry_free (entry);
			}
		}
		g_slist_free (slist);
	}
	return ret;
}

/**
 * gnome_db_config_free_list
 * @list: list to be freed
 *
 * Free all memory used by the given GList, which must be the return value
 * from either #gnome_db_config_list_sections and #gnome_db_config_list_keys
 */
void
gnome_db_config_free_list (GList * list)
{
	while (list != NULL) {
		gchar *str = (gchar *) list->data;
		list = g_list_remove (list, (gpointer) str);
		g_free ((gpointer) str);
	}
}


/**
 * gnome_db_config_get_data_source_list
 *
 * Retrieves information about available data sources.
 *
 * Returns: a GList of #GdaDatasourceInfo structures.
 */
GList *
gnome_db_config_get_data_source_list (void)
{
	GList *list = NULL;
	GList *sections;
	GList *l;

	sections = gnome_db_config_list_sections (GDA_CONFIG_SECTION_DATASOURCES);
	for (l = sections; l != NULL; l = l->next) {
		gchar *tmp;
		GdaDataSourceInfo *info;

		info = g_new0 (GdaDataSourceInfo, 1);
		info->name = g_strdup ((const gchar *) l->data);

		/* get the provider */
		tmp = g_strdup_printf ("%s/%s/Provider", GDA_CONFIG_SECTION_DATASOURCES, (char *) l->data);
		info->provider = gnome_db_config_get_string (tmp);
		g_free (tmp);

		/* get the connection string */
		tmp = g_strdup_printf ("%s/%s/DSN", GDA_CONFIG_SECTION_DATASOURCES, (char *) l->data);
		info->cnc_string = gnome_db_config_get_string (tmp);
		g_free (tmp);

		/* get the description */
		tmp = g_strdup_printf ("%s/%s/Description", GDA_CONFIG_SECTION_DATASOURCES, (char *) l->data);
		info->description = gnome_db_config_get_string (tmp);
		g_free (tmp);

		/* get the user name */
		tmp = g_strdup_printf ("%s/%s/Username", GDA_CONFIG_SECTION_DATASOURCES, (char *) l->data);
		info->username = gnome_db_config_get_string (tmp);
		g_free (tmp);

		/* get the password */
		tmp = g_strdup_printf ("%s/%s/Password", GDA_CONFIG_SECTION_DATASOURCES, (char *) l->data);
		info->password = gnome_db_config_get_string (tmp);
		g_free (tmp);

		list = g_list_append (list, info);
	}

	gnome_db_config_free_list (sections);

	return list;
}

/**
 * gnome_db_config_find_data_source
 * @name: name of the data source to look for.
 *
 * Looks for a data source given its name.
 * The search is case insensitive.
 *
 * Returns: a #GdaDatasourceInfo structure.
 */
GdaDataSourceInfo *
gnome_db_config_find_data_source (const gchar *name)
{
	GList *dsnlist;
	GList *l;
	GdaDataSourceInfo *info = NULL;

	g_return_val_if_fail (name != NULL, NULL);

	dsnlist = gnome_db_config_get_data_source_list ();
	for (l = dsnlist; l != NULL; l = l->next) {
		GdaDataSourceInfo *tmp_info = (GdaDataSourceInfo *) l->data;

		if (tmp_info && !g_strcasecmp (tmp_info->name, name)) {
			info = gnome_db_config_copy_data_source_info (tmp_info);
			break;
		}
	}

	gnome_db_config_free_data_source_list (dsnlist);

	return info;
}

/**
 * gnome_db_config_copy_data_source_info
 * @src: #GdaDataSourceInfo to copy
 *
 * Copies a #GdaDataSourceInfo structure.
 *
 * Returns: a copy of @src.
 */
GdaDataSourceInfo *
gnome_db_config_copy_data_source_info (GdaDataSourceInfo *src)
{
	GdaDataSourceInfo *info;

	g_return_val_if_fail (src != NULL, NULL);

	info = g_new0 (GdaDataSourceInfo, 1);
	info->name = g_strdup (src->name);
	info->provider = g_strdup (src->provider);
	info->cnc_string = g_strdup (src->cnc_string);
	info->description = g_strdup (src->description);
	info->username = g_strdup (src->username);
	info->password = g_strdup (src->password);

	return info;
}

/**
 * gnome_db_config_free_data_source_list
 * @list: a GList returned by #gnome_db_config_get_data_source_list.
 *
 * Frees the @list.
 */
void
gnome_db_config_free_data_source_list (GList *list)
{
	GList *l;
	GdaDataSourceInfo *info;

	for (l = list; l; l = l->next) {
		info = l->data;
		l->data = NULL;
		gnome_db_config_free_data_source_info (info);
	}

	g_list_remove_all (list, NULL);
}

/**
 * gnome_db_config_save_data_source
 * @name: Name for the data source to be saved.
 * @provider: Provider ID for the new data source.
 * @cnc_string: Connection string for the new data source.
 * @description: Description for the new data source.
 * @username: User name for the new data source.
 * @password: Password to use when authenticating @username.
 *
 * Adds a new data source (or update an existing one) to the GDA
 * configuration, based on the parameters given.
 */
void
gnome_db_config_save_data_source (const gchar *name,
			     const gchar *provider,
			     const gchar *cnc_string,
			     const gchar *description,
			     const gchar *username,
			     const gchar *password)
{
	GString *str;
	gint trunc_len;

	g_return_if_fail (name != NULL);
	g_return_if_fail (provider != NULL);

	str = g_string_new ("");
	g_string_printf (str, "%s/%s/", GDA_CONFIG_SECTION_DATASOURCES, name);
	trunc_len = strlen (str->str);
	
	/* set the provider */
	g_string_append (str, "Provider");
	gnome_db_config_set_string (str->str, provider);
	g_string_truncate (str, trunc_len);

	/* set the connection string */
	if (cnc_string) {
		g_string_append (str, "DSN");
		gnome_db_config_set_string (str->str, cnc_string);
		g_string_truncate (str, trunc_len);
	}

	/* set the description */
	if (description) {
		g_string_append (str, "Description");
		gnome_db_config_set_string (str->str, description);
		g_string_truncate (str, trunc_len);
	}

	/* set the username */
	if (username) {
		g_string_append (str, "Username");
		gnome_db_config_set_string (str->str, username);
		g_string_truncate (str, trunc_len);
	}

	/* set the password */
	if (password) {
		g_string_append (str, "Password");
		gnome_db_config_set_string (str->str, password);
		g_string_truncate (str, trunc_len);
	}
	g_string_free (str, TRUE);
}

/**
 * gnome_db_config_remove_data_source
 * @name: Name for the data source to be removed.
 *
 * Removes the given data source from the GDA configuration.
 */
void
gnome_db_config_remove_data_source (const gchar *name)
{
	gchar *dir;

	g_return_if_fail (name != NULL);

	dir = g_strdup_printf ("%s/%s", GDA_CONFIG_SECTION_DATASOURCES, name);
	gnome_db_config_remove_section (dir);
	g_free (dir);
}

/**
 * gnome_db_config_free_data_source_info
 * @info: the struture to deallocate.
 *
 * Frees the resources allocated by a #GdaDataSourceInfo.
 */
void
gnome_db_config_free_data_source_info (GdaDataSourceInfo *info)
{
	g_return_if_fail (info != NULL);

	g_free (info->name);
	g_free (info->provider);
	g_free (info->cnc_string);
	g_free (info->description);
	g_free (info->username);
	g_free (info->password);

	g_free (info);
}

/**
 * gnome_db_config_add_listener
 * @path: configuration path to listen to.
 * @func: callback function.
 * @user_data: data to be passed to the callback function.
 *
 * Installs a configuration listener, which is a callback function
 * which will be called every time a change occurs on a given
 * configuration entry.
 *
 * Returns: the ID of the listener, which you will need for
 * calling #gda_config_remove_listener. If an error occurs,
 * 0 is returned.
 */
guint
gnome_db_config_add_listener (const gchar *path, 
			      GConfClientNotifyFunc func, 
			      gpointer user_data)
{
	gconf_client_add_dir (get_conf_client (),
			      path,
			      GCONF_CLIENT_PRELOAD_NONE,
			      NULL);

	return gconf_client_notify_add (get_conf_client (),
					path,
					func,
					user_data,
					NULL,
					NULL);
}

/**
 * gnome_db_config_remove_listener
 * @id: the value returned from the call to #gnome_db_config_add_listener
 *
 * Removes a previously installed configuration listener.
 */
void gnome_db_config_remove_listener (guint id)
{
	gconf_client_notify_remove (get_conf_client (), id);
}

