/* GNOME DB library
 * Copyright (C) 1999-2001, The Free Software Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-util.h>
#include <gobject/gmarshal.h>
#include <gtk/gtktreeview.h>
#include "libgnomedb-private.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbGridPrivate {
	GdaDataModel *model;

	/* widgets */
	GtkWidget *scroll;
	GtkWidget *tree_view;
};

static void gnome_db_grid_class_init   (GnomeDbGridClass *klass);
static void gnome_db_grid_init         (GnomeDbGrid *grid, GnomeDbGridClass *klass);
static void gnome_db_grid_set_property (GObject *object,
					guint paramid,
					const GValue *value,
					GParamSpec *pspec);
static void gnome_db_grid_get_property (GObject *object,
					guint param_id,
					GValue *value,
					GParamSpec *pspec);
static void gnome_db_grid_finalize   (GObject *object);

enum {
	ROW_SELECTED,
	SELECTION_CLEARED,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_MODEL
};

static gint gnome_db_grid_signals[LAST_SIGNAL] = { 0 };
static GdkAtom clipboard_atom = GDK_NONE;
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
selection_changed_cb (GtkTreeSelection *selection, gpointer user_data)
{
	GtkTreeModel *tree_model;
	GtkTreeIter iter;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	if (gtk_tree_selection_get_selected (selection, &tree_model, &iter)) {
		DataModelInfo *iter_info;

		gtk_tree_model_get (tree_model, &iter, 0, &iter_info, -1);
		g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[ROW_SELECTED],
			       0, iter_info->row);
	}
	else
		g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[SELECTION_CLEARED], 0);
}

/*
 * Private functions
 */

static void
setup_grid (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	/* setup selection stuff */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (selection), "changed",
			  G_CALLBACK (selection_changed_cb), grid);
}

/*
 * GnomeDbGrid class implementation
 */

static void
gnome_db_grid_class_init (GnomeDbGridClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_grid_set_property;
	object_class->get_property = gnome_db_grid_get_property;
	object_class->finalize = gnome_db_grid_finalize;
	klass->row_selected = NULL;
	klass->row_unselected = NULL;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_MODEL,
		g_param_spec_object ("model", NULL, NULL,
				     GDA_TYPE_DATA_MODEL,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* add class signals */
	gnome_db_grid_signals[ROW_SELECTED] =
		g_signal_new ("row_selected",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, row_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);
	gnome_db_grid_signals[SELECTION_CLEARED] =
		g_signal_new ("selection_cleared",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, row_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	/* clipboard atom */
	if (!clipboard_atom)
		clipboard_atom = gdk_atom_intern ("CLIPBOARD", FALSE);
}

static void
gnome_db_grid_init (GnomeDbGrid *grid, GnomeDbGridClass *klass)
{
	/* allocate the internal structure */
	grid->priv = g_new0 (GnomeDbGridPrivate, 1);

	/* set up widgets */
	grid->priv->scroll = gnome_db_new_scrolled_window_widget ();
	gtk_box_pack_start (GTK_BOX (grid), grid->priv->scroll, 1, 1, 0);

	grid->priv->tree_view = gnome_db_new_tree_view_widget (NULL);
	gtk_container_add (GTK_CONTAINER (grid->priv->scroll), grid->priv->tree_view);
	setup_grid (grid);
}

static void
gnome_db_grid_set_property (GObject *object,
			    guint param_id,
			    const GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	switch (param_id) {
	case PROP_MODEL :
		gnome_db_grid_set_model (grid, GDA_DATA_MODEL (g_value_get_object (value)));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_grid_get_property (GObject *object,
			    guint param_id,
			    GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	switch (param_id) {
	case PROP_MODEL :
		g_value_set_object (value, G_OBJECT (grid->priv->model));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_grid_finalize (GObject *object)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	/* free memory */
	g_free (grid->priv);
	grid->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_grid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGridClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_grid_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGrid),
			0,
			(GInstanceInitFunc) gnome_db_grid_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbGrid", &info, 0);
	}
	return type;
}

/**
 * gnome_db_grid_new
 */
GtkWidget *
gnome_db_grid_new (void)
{
	GnomeDbGrid *grid;

	grid = g_object_new (GNOME_DB_TYPE_GRID, NULL);
	return GTK_WIDGET (grid);
}

/**
 * gnome_db_grid_new_with_model
 */
GtkWidget *
gnome_db_grid_new_with_model (GdaDataModel *model)
{
	GnomeDbGrid *grid;

	grid = g_object_new (GNOME_DB_TYPE_GRID, NULL);
	gnome_db_grid_set_model (grid, model);
	return GTK_WIDGET (grid);
}

/**
 * gnome_db_grid_get_model
 */
GdaDataModel *
gnome_db_grid_get_model (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);
	return grid->priv->model;
}

/**
 * gnome_db_grid_set_model
 */
void
gnome_db_grid_set_model (GnomeDbGrid *grid, GdaDataModel *model)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	if (GDA_IS_DATA_MODEL (grid->priv->model))
		g_object_unref (G_OBJECT (grid->priv->model));

	grid->priv->model = model;
	if (GDA_IS_DATA_MODEL (model))
		g_object_ref (G_OBJECT (model));
	else {
		/* just empty the grid and return */
		gtk_tree_view_set_model (GTK_TREE_VIEW (grid->priv->tree_view), NULL);
		return;
	}

	/* redisplay the GtkTreeView */
	gtk_widget_destroy (grid->priv->tree_view);
	grid->priv->tree_view = gnome_db_model_to_gtk_tree_view (grid->priv->model);
	gtk_container_add (GTK_CONTAINER (grid->priv->scroll), grid->priv->tree_view);
	gtk_widget_show (grid->priv->tree_view);
	setup_grid (grid);
}

/**
 * gnome_db_grid_hide_column_titles
 */
void
gnome_db_grid_hide_column_titles (GnomeDbGrid *grid)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (grid->priv->tree_view), FALSE);
}

/**
 * gnome_db_grid_show_column_titles
 */
void
gnome_db_grid_show_column_titles (GnomeDbGrid *grid)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (grid->priv->tree_view), TRUE);
}
