/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgnomedb/gnome-db-dsn-config-druid.h>
#include <libgnomedb/gnome-db-util.h>
#include <bonobo/bonobo-i18n.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <libgnomeui/gnome-druid-page-edge.h>
#include <libgnomeui/gnome-druid-page-standard.h>

#define PARENT_TYPE GNOME_TYPE_DRUID

struct _GnomeDbDsnConfigDruidPrivate {
	GdaDataSourceInfo *dsn_info;

	GdkPixbuf *logo;

	/* widgets */
	GtkWidget *start_page;
	GtkWidget *general_page;
	GtkWidget *end_page;
};

static void gnome_db_dsn_config_druid_class_init (GnomeDbDsnConfigDruidClass *klass);
static void gnome_db_dsn_config_druid_init       (GnomeDbDsnConfigDruid *druid,
						  GnomeDbDsnConfigDruidClass *klass);
static void gnome_db_dsn_config_druid_finalize   (GObject *object);

enum {
	FINISHED,
	LAST_SIGNAL
};

static gint config_druid_signals[LAST_SIGNAL] = { 0, };
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
druid_cancelled_cb (GnomeDruidPage *druid_page, GtkWidget *druid_widget, gpointer user_data)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));
	g_signal_emit (G_OBJECT (druid), config_druid_signals[FINISHED], 0, TRUE);
}

static void
druid_finished_cb (GnomeDruidPage *druid_page, GtkWidget *druid_widget, gpointer user_data)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* FIXME: set the internal GdaDataSourceInfo with the input from the user */
	g_signal_emit (G_OBJECT (druid), config_druid_signals[FINISHED], 0, FALSE);
}

/*
 * GnomeDbDsnConfigDruid class implementation
 */

static void
gnome_db_dsn_config_druid_class_init (GnomeDbDsnConfigDruidClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	config_druid_signals[FINISHED] =
		g_signal_new ("finished",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbDsnConfigDruidClass, finished),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);

	object_class->finalize = gnome_db_dsn_config_druid_finalize;
}

static void
gnome_db_dsn_config_druid_init (GnomeDbDsnConfigDruid *druid,
				GnomeDbDsnConfigDruidClass *klass)
{
	GtkWidget *label;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* create private structure */
	druid->priv = g_new0 (GnomeDbDsnConfigDruidPrivate, 1);
	druid->priv->dsn_info = g_new0 (GdaDataSourceInfo, 1);

	/* load icons */
	druid->priv->logo = gdk_pixbuf_new_from_file (LIBGNOMEDB_PIXMAPSDIR "/gnome-db.png", NULL);

	/* set up widgets */
	gnome_druid_set_show_help (GNOME_DRUID (druid), FALSE);

	/* start page */
	druid->priv->start_page = gnome_druid_page_edge_new_with_vals (
		GNOME_EDGE_START, TRUE, _("Add a new data source..."),
		_("This druid will guide you through the process of\n"
		  "creating a new data source for your GNOME-DB\n"
		  "installation. Just follow the steps!"),
		druid->priv->logo, NULL, NULL);
	gtk_widget_show (druid->priv->start_page);
	gnome_druid_append_page (GNOME_DRUID (druid), druid->priv->start_page);

	g_signal_connect (G_OBJECT (druid->priv->start_page), "cancel", G_CALLBACK (druid_cancelled_cb), druid);

	/* general info page */
	druid->priv->general_page = gnome_druid_page_standard_new_with_vals (
		_("General Information"), druid->priv->logo, NULL);

	label = gnome_db_new_label_widget (NULL);
	gtk_label_set_markup (GTK_LABEL (label),
		_("<b>The following fields represent the basic information\n"
		  "items for your new data source. Some of them (description,\n"
		  "username) are optional, whereas the others are mandatory.\n"
		  "If you don't know which provider to use, just select\n"
		  "the default one</b>"));
	gnome_druid_page_standard_append_item (GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page), NULL, label, NULL);

	gtk_widget_show (druid->priv->general_page);
	gnome_druid_append_page (GNOME_DRUID (druid), druid->priv->general_page);

	g_signal_connect (G_OBJECT (druid->priv->general_page), "cancel", G_CALLBACK (druid_cancelled_cb), druid);

	/* end page */
	druid->priv->end_page = gnome_druid_page_edge_new_with_vals (
		GNOME_EDGE_FINISH, TRUE, _("All information retrieved"),
		_("All information needed to create a new data source\n"
		  "has been retrieved. Now, press 'Finish' to close\n"
		  "this dialog. To open your newly created data source,\n"
		  "just select it when asked for a data source"),
		  druid->priv->logo, NULL, NULL);
	gtk_widget_show (druid->priv->end_page);
	gnome_druid_append_page (GNOME_DRUID (druid), druid->priv->end_page);

	g_signal_connect (G_OBJECT (druid->priv->end_page), "cancel", G_CALLBACK (druid_cancelled_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->end_page), "finish", G_CALLBACK (druid_finished_cb), druid);
}

static void
gnome_db_dsn_config_druid_finalize (GObject *object)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) object;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* free memory */
	gdk_pixbuf_unref (druid->priv->logo);
	gda_config_free_data_source_info (druid->priv->dsn_info);

	g_free (druid->priv);
	druid->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_dsn_config_druid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDsnConfigDruidClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_dsn_config_druid_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDsnConfigDruid),
			0,
			(GInstanceInitFunc) gnome_db_dsn_config_druid_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbDsnConfigDruid",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_dsn_config_druid_new
 */
GtkWidget *
gnome_db_dsn_config_druid_new (void)
{
	GnomeDbDsnConfigDruid *druid;

	druid = g_object_new (GNOME_DB_TYPE_DSN_CONFIG_DRUID, NULL);
	return GTK_WIDGET (druid);
}

/**
 * gnome_db_dsn_config_druid_get_dsn
 */
GdaDataSourceInfo *
gnome_db_dsn_config_druid_get_dsn (GnomeDbDsnConfigDruid *druid)
{
	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), NULL);
	return druid->priv->dsn_info;
}
