/* GNOME-DB Components
 * Copyright (C) 2000-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-persist-stream.h>
#include <bonobo/bonobo-stream-client.h>
#include <libgnomedb/gnome-db-control.h>
#include <libgnomedb/gnome-db-editor.h>
#include "connection-viewer.h"

/*
 * Private functions
 */

static char *
read_stream (Bonobo_Stream stream)
{
        Bonobo_Stream_iobuf *buffer;
        CORBA_Environment ev;
        gchar *data = NULL;
        gint length = 0;

        CORBA_exception_init (&ev);
        do {
#define READ_CHUNK_SIZE 65536
                Bonobo_Stream_read (stream, READ_CHUNK_SIZE, &buffer, &ev);

                if (BONOBO_EX (&ev)) {
                        CORBA_exception_free (&ev);
                        return NULL;
                }

                if (buffer->_length <= 0)
                        break;

                data = g_realloc (data, length + buffer->_length + 1);
                memcpy (data + length, buffer->_buffer, buffer->_length);
                length += buffer->_length;
                data[length] = '\0';

                CORBA_free (buffer);
#undef READ_CHUNK_SIZE
        } while (1);

        CORBA_free (buffer);
        CORBA_exception_free (&ev);

        if (data == NULL)
                data = g_strdup("");

        return data;
}

/*
 * Implementation of the Bonobo::PersistStream methods
 */

static void
pstream_load (BonoboPersistStream *ps, const Bonobo_Stream stream,
              Bonobo_Persist_ContentType type, void *user_data,
              CORBA_Environment *ev)
{
	gchar *text;
	GnomeDbEditor *sql = (GnomeDbEditor *) user_data;

	g_return_if_fail (GNOME_DB_IS_EDITOR (sql));

	if (type && g_strcasecmp (type, "text/x-sql") != 0) {
		bonobo_exception_set (ev, ex_Bonobo_Persist_WrongDataType);
                return;
	}

	text = read_stream (stream);
	if (!text) {
		bonobo_exception_set (ev, ex_Bonobo_Persist_FileNotFound);
                return;
	}

	gnome_db_editor_set_text (sql, text, strlen (text));
	g_free (text);
}

static void
pstream_save (BonoboPersistStream *ps, const Bonobo_Stream stream,
              Bonobo_Persist_ContentType type, void *user_data,
              CORBA_Environment *ev)
{
        GnomeDbEditor *sql = (GnomeDbEditor *) user_data;
        gchar *text;
        gint len;

	g_return_if_fail (GNOME_DB_IS_EDITOR (sql));

        if (type && g_strcasecmp (type, "text/x-sql") != 0) {
                bonobo_exception_set (ev, ex_Bonobo_Persist_WrongDataType);
                return;
        }

	text = gnome_db_editor_get_all_text (sql);
	if (text) {
		len = strlen (text);
		bonobo_stream_client_write (stream, text, len, ev);
		g_free (text);
	}
}

static CORBA_long
pstream_get_max_size (BonoboPersistStream *ps, void *user_data,
                      CORBA_Environment *ev)
{
	GnomeDbEditor *sql = (GnomeDbEditor *) user_data;
        gchar *text;
	CORBA_long len;

	g_return_val_if_fail (GNOME_DB_IS_EDITOR (sql), 0L);

	text = gnome_db_editor_get_all_text (sql);
	len = (CORBA_long) text ? strlen (text) : 0;
	g_free (text);

	return len;
}

static Bonobo_Persist_ContentTypeList *
pstream_get_content_types (BonoboPersistStream *ps,
			   void *closure,
                           CORBA_Environment *ev)
{
        return bonobo_persist_generate_content_types (1, "text/x-sql");
}

/*
 * Public functions
 */

BonoboObject *
sql_viewer_new (void)
{
	GtkWidget *sql;
	GnomeDbControl *control;
	BonoboPersistStream *stream;

	/* create the Bonobo control */
	sql = gnome_db_editor_new ();
	control = gnome_db_control_new (sql);

	/* create the persist stream object */
	stream = bonobo_persist_stream_new (pstream_load, pstream_save,
                                            pstream_get_max_size,
                                            pstream_get_content_types,
                                            sql);

        if (stream == NULL) {
                bonobo_object_unref (BONOBO_OBJECT (control));
                return NULL;
        }

        bonobo_object_add_interface (BONOBO_OBJECT (control),
                                     BONOBO_OBJECT (stream));

	return BONOBO_OBJECT (control);
}
