/*
 * GNetwork: libgnetwork/gnetwork-tcp-connection.c
 *
 * Copyright (C) 2001-2003 James M. Cape
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/*
 * GLib 2.0/GObject-based TCP/IP networking sockets wrapper.
 * 
 * Notes on editing: Tab size: 4 
 */


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include "gnetwork-tcp-connection.h"

#include "gnetwork-connection.h"
#include "gnetwork-ssl-private.h"

#include "gnetwork-type-builtins.h"

#include "marshal.h"
#include "gnetwork-tcp-proxy-private.h"
#include "gnetwork-dns.h"
#include "gnetwork-threads.h"
#include "gnetwork-utils.h"

#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <glib/gi18n.h>


#ifndef DEFAULT_BUFFER_SIZE
#	define DEFAULT_BUFFER_SIZE 2048
#endif /* DEFAULT_BUFFER_SIZE */

#define GINT(ptr)	((gint *) (ptr))


enum
{
  PROP_0,

  /* Object Properties */
  TCP_STATUS,
  PROXY_TYPE,
  ADDRESS,
  PORT,
  LOCAL_ADDRESS,
  LOCAL_PORT,
  IP_ADDRESS,

  /* SSL Properties */
  SSL_ENABLED,
  SSL_AUTH_TYPE,
  SSL_CA_FILE,
  SSL_CERT_FILE,
  SSL_KEY_FILE,

  /* Shortcut for Servers */
  SOCKET_FD,

  /* GNetworkConnection Properties */
  CXN_TYPE,
  CXN_STATUS,
  CXN_BYTES_RECEIVED,
  CXN_BYTES_SENT,
  CXN_BUFFER_SIZE
};

enum
{
  CERTIFICATE_ERROR,
  LAST_SIGNAL
};


struct _GNetworkTcpConnectionPrivate
{
  /* Properties */
  /* GNetworkTcpConnection */
  gchar *address;
  guint16 port;

  gchar *local_address;
  guint16 local_port;
  gchar *ip_address;

  /* GNetworkConnectionIface */
  guint buffer_size;
  gulong bytes_received;
  gulong bytes_sent;

  /* GNetworkSslIface */
  gchar *ca_file;
  gchar *cert_file;
  gchar *key_file;

  /* DNS Lookups In Progress */
  GNetworkDnsHandle dns_handle;
  GNetworkDnsHandle proxy_dns_handle;

  /* proxy_dns_handle Return */
  gchar *proxy_ip_address;

  /* The socket itself :-) */
  GIOChannel *channel;
  guint source_id;
  gint sockfd;

  /* Property Bits */
  /* GNetworkTcpConnection */
  GNetworkTcpConnectionStatus tcp_status:3;
  GNetworkTcpProxyType tcp_proxy_type:3;

  /* GNetworkSslIface */
  GNetworkSslAuthType auth_type:2;
  gboolean ssl_enabled:1;

  /* GNetworkConnectionIface */
  GNetworkConnectionType cxn_type:2;
  GNetworkConnectionStatus cxn_status:3;
};


/* ****************** *
 *  Global Variables  *
 * ****************** */

static gpointer parent_class = NULL;
static gint signals[LAST_SIGNAL] = { 0 };


/* ******************* *
 *  Utility Functions  *
 * ******************* */

static GError *
get_connection_error_from_errno (gint en, const gchar * address)
{
  GError *error = NULL;

  switch (en)
    {
    case EINPROGRESS:
      g_assert_not_reached ();
      break;

    case ECONNREFUSED:
      error = g_error_new (GNETWORK_CONNECTION_ERROR, GNETWORK_CONNECTION_ERROR_CONNECTION_REFUSED,
			   _("The connection to %s could not be completed because the server "
			     "refused to allow it."), address);
      break;

    case ETIMEDOUT:
      error = g_error_new (GNETWORK_CONNECTION_ERROR, GNETWORK_CONNECTION_ERROR_TIMEOUT,
			   _("The connection to %s took too long to complete. The server may be "
			     "down, your network connection may be down, or your network "
			     "connection may be improperly configured."), address);
      break;

    case ENETUNREACH:
      error = g_error_new (GNETWORK_CONNECTION_ERROR, GNETWORK_CONNECTION_ERROR_NETWORK_UNREACHABLE,
			   _("The network that %s is on could not be reached. Your network "
			     "connection may be down or improperly configured."), address);
      break;

    case EACCES:
    case EPERM:
      error = g_error_new (GNETWORK_CONNECTION_ERROR,
			   GNETWORK_CONNECTION_ERROR_BAD_BROADCAST_OR_FIREWALL,
			   _("You cannot connect to %s, because your computer or firewall is "
			     "configured to prevent it."), address);
      break;

    default:
      error = g_error_new (GNETWORK_CONNECTION_ERROR, GNETWORK_CONNECTION_ERROR_INTERNAL,
			   _("The connection to %s could not be completed because an  error "
			     "occured inside the GNetwork library."), address);
      break;
    }

  return error;
}

/* ********************** *
 *  Connection Functions  *
 * ********************** */

static gboolean
io_channel_handler (GIOChannel * channel, GIOCondition cond, GNetworkTcpConnection * connection)
{
  if (connection->_priv->tcp_status <= GNETWORK_TCP_CONNECTION_CLOSED)
    return FALSE;

  switch (cond)
    {
    case G_IO_IN:
      {
	GIOStatus status;
	guchar *buffer;
	gsize bytes_read;
	GError *error = NULL;

	buffer = g_new0 (guchar, connection->_priv->buffer_size + 1);
	do
	  {
	    status = g_io_channel_read_chars (channel, buffer, connection->_priv->buffer_size,
					      &bytes_read, &error);
	  }
	while (status == G_IO_STATUS_AGAIN);

	switch (status)
	  {
	  case G_IO_STATUS_NORMAL:
	    if (bytes_read > 0)
	      {
		/* Make sure the read buffer is 0-terminated. */
		buffer[bytes_read] = 0;
		gnetwork_connection_received (GNETWORK_CONNECTION (connection), buffer, bytes_read);
		g_free (buffer);
	      }
	    return TRUE;
	    break;

	  case G_IO_STATUS_ERROR:
	    gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
	    g_error_free (error);
	    /* Fall through to notify about the closed state. */
	  case G_IO_STATUS_EOF:
	    g_free (buffer);
	    g_io_channel_shutdown (connection->_priv->channel, FALSE, NULL);
	    g_io_channel_unref (connection->_priv->channel);
	    connection->_priv->channel = NULL;

	    connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
	    connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
	    connection->_priv->sockfd = -1;

	    g_object_freeze_notify (G_OBJECT (connection));
	    g_object_notify (G_OBJECT (connection), "tcp-status");
	    g_object_notify (G_OBJECT (connection), "status");
	    g_object_notify (G_OBJECT (connection), "socket-fd");
	    g_object_thaw_notify (G_OBJECT (connection));
	    break;
	  default:
	    break;
	  }
      }
      break;

    case G_IO_ERR:
      {
	g_warning (G_STRLOC ": FIXME: Handle G_IO_ERR -- getsockopt?");

	g_io_channel_shutdown (connection->_priv->channel, FALSE, NULL);
	g_io_channel_unref (connection->_priv->channel);
	connection->_priv->channel = NULL;

	connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
	connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
	connection->_priv->sockfd = -1;

	g_object_freeze_notify (G_OBJECT (connection));
	g_object_notify (G_OBJECT (connection), "tcp-status");
	g_object_notify (G_OBJECT (connection), "status");
	g_object_notify (G_OBJECT (connection), "socket-fd");
	g_object_thaw_notify (G_OBJECT (connection));
      }
      break;

    case G_IO_HUP:
      g_io_channel_shutdown (connection->_priv->channel, FALSE, NULL);
      g_io_channel_unref (connection->_priv->channel);
      connection->_priv->channel = NULL;

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
      connection->_priv->sockfd = -1;

      g_object_freeze_notify (G_OBJECT (connection));
      g_object_notify (G_OBJECT (connection), "tcp-status");
      g_object_notify (G_OBJECT (connection), "status");
      g_object_notify (G_OBJECT (connection), "socket-fd");
      g_object_thaw_notify (G_OBJECT (connection));
      break;

    default:
      g_assert_not_reached ();
      break;
    }

  return FALSE;
}


static void
open_ssl_connection (GNetworkTcpConnection * connection)
{
  GIOChannel *channel;
  GNetworkSslAuth *auth;
  GError *error;

  if (connection->_priv->tcp_status <= GNETWORK_TCP_CONNECTION_CLOSED)
    return;

  error = NULL;

  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_AUTHENTICATING;

  g_object_notify (G_OBJECT (connection), "tcp-status");

  if (connection->_priv->tcp_status <= GNETWORK_TCP_CONNECTION_CLOSED)
    return;

  auth = _gnetwork_ssl_auth_new (connection->_priv->auth_type, connection->_priv->cxn_type);

  if (connection->_priv->auth_type == GNETWORK_SSL_AUTH_CERTIFICATE)
    {
      _gnetwork_ssl_auth_set_server_hostname (auth, connection->_priv->address);

      if (connection->_priv->ca_file != NULL)
	_gnetwork_ssl_auth_set_authority_file (auth, connection->_priv->ca_file);

      if (connection->_priv->cert_file != NULL && connection->_priv->key_file != NULL)
	_gnetwork_ssl_auth_set_certs_and_keys_files (auth, connection->_priv->cert_file,
						     connection->_priv->key_file);
    }

  channel = _gnetwork_io_channel_ssl_new (connection->_priv->channel, auth, &error);
  _gnetwork_ssl_auth_unref (auth);

  g_io_channel_unref (connection->_priv->channel);
  connection->_priv->channel = channel;

  if (error != NULL)
    {
      if (error->domain == GNETWORK_SSL_CERT_ERROR)
	{
	  g_signal_emit (connection, signals[CERTIFICATE_ERROR], 0,
			 GNETWORK_SSL_CERT (error->message), error->code);
	  gnetwork_ssl_cert_free (GNETWORK_SSL_CERT (error->message));
	  error->message = NULL;
	}
      else
	{
	  gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
	}
      g_error_free (error);

      if (connection->_priv->tcp_status <= GNETWORK_TCP_CONNECTION_CLOSED)
	return;
    }

  connection->_priv->source_id = gnetwork_thread_io_add_watch (connection->_priv->channel,
							       (G_IO_IN | G_IO_PRI | G_IO_ERR |
								G_IO_HUP),
							       (GIOFunc) io_channel_handler,
							       connection);

  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_OPEN;
  connection->_priv->cxn_status = GNETWORK_CONNECTION_OPEN;

  g_object_freeze_notify (G_OBJECT (connection));
  g_object_notify (G_OBJECT (connection), "tcp-status");
  g_object_notify (G_OBJECT (connection), "status");
  g_object_thaw_notify (G_OBJECT (connection));
}


static void
proxy_done_cb (GIOChannel * channel, GError * error, GNetworkTcpConnection * connection)
{
  if (connection->_priv->tcp_status <= GNETWORK_TCP_CONNECTION_CLOSED)
    return;

  g_io_channel_unref (connection->_priv->channel);
  g_io_channel_ref (channel);
  connection->_priv->channel = channel;

  if (error != NULL)
    {
      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);

      g_io_channel_shutdown (connection->_priv->channel, FALSE, NULL);
      g_io_channel_unref (connection->_priv->channel);
      connection->_priv->channel = NULL;

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
      connection->_priv->sockfd = -1;

      g_object_freeze_notify (G_OBJECT (connection));
      g_object_notify (G_OBJECT (connection), "tcp-status");
      g_object_notify (G_OBJECT (connection), "status");
      g_object_notify (G_OBJECT (connection), "socket-fd");
      g_object_thaw_notify (G_OBJECT (connection));
      return;
    }

  /* We still need SSL. */
  if (connection->_priv->ssl_enabled)
    {
      open_ssl_connection (connection);
    }
  else
    {
      connection->_priv->source_id = gnetwork_thread_io_add_watch (connection->_priv->channel,
								   (G_IO_IN | G_IO_PRI | G_IO_ERR |
								    G_IO_HUP),
								   (GIOFunc) io_channel_handler,
								   connection);

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_OPEN;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_OPEN;

      g_object_freeze_notify (G_OBJECT (connection));
      g_object_notify (G_OBJECT (connection), "tcp-status");
      g_object_notify (G_OBJECT (connection), "status");
      g_object_thaw_notify (G_OBJECT (connection));
    }
}


static void
open_proxy_connection (GNetworkTcpConnection * connection)
{
  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_PROXYING;
  g_object_notify (G_OBJECT (connection), "tcp-status");

  _gnetwork_io_channel_proxy_new (connection->_priv->channel, connection->_priv->tcp_proxy_type,
				  connection->_priv->ip_address, connection->_priv->port,
				  (GNetworkIOCallback) proxy_done_cb, connection, NULL);
}


static void
connect_suceeded (GNetworkTcpConnection * connection)
{
  gchar local_address[INET_ADDRSTRLEN + 1];
  struct sockaddr_in sin;
  socklen_t sin_size;

  getsockname (connection->_priv->sockfd, (struct sockaddr *) &sin, &sin_size);

  inet_ntop (AF_INET, &(sin.sin_addr), local_address, INET_ADDRSTRLEN);
  connection->_priv->local_address = g_strdup (local_address);
  connection->_priv->local_port = g_ntohs (sin.sin_port);

  g_object_freeze_notify (G_OBJECT (connection));
  g_object_notify (G_OBJECT (connection), "local-address");
  g_object_notify (G_OBJECT (connection), "local-port");
  g_object_thaw_notify (G_OBJECT (connection));

  /* We need a proxy. */
  if (connection->_priv->proxy_ip_address != NULL)
    {
      open_proxy_connection (connection);
    }
  else if (connection->_priv->ssl_enabled)
    {
      open_ssl_connection (connection);
    }
  /* We're good to go. */
  else
    {
      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_OPEN;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_OPEN;

      g_object_freeze_notify (G_OBJECT (connection));
      g_object_notify (G_OBJECT (connection), "tcp-status");
      g_object_notify (G_OBJECT (connection), "status");
      g_object_thaw_notify (G_OBJECT (connection));

      connection->_priv->source_id =
	gnetwork_thread_io_add_watch (connection->_priv->channel,
				      (G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP),
				      (GIOFunc) io_channel_handler, connection);
    }
}


/* connect() Is Done */
static gboolean
connect_done_handler (GIOChannel * channel, GIOCondition cond, GNetworkTcpConnection * connection)
{
  GError *error;
  gint result, error_val, length;

  /* Remove this here so we don't get called for proxy/SSL stuff */
  gnetwork_thread_source_remove (connection->_priv->source_id);

  errno = 0;
  error_val = 0;
  result = getsockopt (connection->_priv->sockfd, SOL_SOCKET, SO_ERROR, &error_val, &length);

  /* Couldn't get socket options */
  if (result != 0)
    {
      error = g_error_new (GNETWORK_CONNECTION_ERROR, GNETWORK_CONNECTION_ERROR_INTERNAL,
			   _("The connection to %s could not be completed because an error "
			     "occured inside the GNetwork library."), connection->_priv->address);
      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
      g_error_free (error);

      g_io_channel_shutdown (connection->_priv->channel, FALSE, NULL);
      g_io_channel_unref (connection->_priv->channel);
      connection->_priv->channel = NULL;

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
      connection->_priv->sockfd = -1;

      g_object_freeze_notify (G_OBJECT (connection));
      g_object_notify (G_OBJECT (connection), "tcp-status");
      g_object_notify (G_OBJECT (connection), "status");
      g_object_notify (G_OBJECT (connection), "socket-fd");
      g_object_thaw_notify (G_OBJECT (connection));
      return FALSE;
    }

  if (error_val != 0)
    {
      error = get_connection_error_from_errno (errno, connection->_priv->address);
      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
      g_error_free (error);

      g_io_channel_shutdown (connection->_priv->channel, FALSE, NULL);
      g_io_channel_unref (connection->_priv->channel);
      connection->_priv->channel = NULL;

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
      connection->_priv->sockfd = -1;

      g_object_freeze_notify (G_OBJECT (connection));
      g_object_notify (G_OBJECT (connection), "tcp-status");
      g_object_notify (G_OBJECT (connection), "status");
      g_object_notify (G_OBJECT (connection), "socket-fd");
      g_object_thaw_notify (G_OBJECT (connection));
    }
  else
    {
      connect_suceeded (connection);
    }

  return FALSE;
}


/* **************************** *
 *  Basic Connection Functions  *
 * **************************** */

static void
open_client_connection (GNetworkTcpConnection * connection)
{
  GError *error;
  const gchar *address;
  struct sockaddr_in sin;
  gint flags, result;
  GObject *object = G_OBJECT (connection);

  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_OPENING;
  connection->_priv->cxn_status = GNETWORK_CONNECTION_OPENING;

  g_object_freeze_notify (object);
  g_object_notify (object, "tcp-status");
  g_object_notify (object, "status");
  g_object_thaw_notify (object);

  /* Create the socket */
  errno = 0;
  connection->_priv->sockfd = socket (PF_INET, SOCK_STREAM, 0);
  g_object_notify (object, "socket-fd");

  if (connection->_priv->sockfd < 0)
    {
      error = g_error_new (GNETWORK_CONNECTION_ERROR, GNETWORK_CONNECTION_ERROR_INTERNAL,
			   _("The connection to %s could not be completed because an error "
			     "occured inside the GNetwork library."), connection->_priv->address);
      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
      g_error_free (error);

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;

      g_object_freeze_notify (object);
      g_object_notify (object, "tcp-status");
      g_object_notify (object, "status");
      g_object_thaw_notify (object);
      return;
    }

  memset (&sin, 0, sizeof (sin));

  if (connection->_priv->proxy_ip_address != NULL)
    {
      address = connection->_priv->proxy_ip_address;
      sin.sin_port = g_htons (_gnetwork_tcp_proxy_get_port (connection->_priv->tcp_proxy_type));
    }
  else
    {
      address = connection->_priv->ip_address;
      sin.sin_port = g_htons (connection->_priv->port);
    }

  inet_pton (AF_INET, address, &(sin.sin_addr));

  sin.sin_family = AF_INET;

  /* Retrieve the current socket flags */
  flags = fcntl (connection->_priv->sockfd, F_GETFL, 0);
  if (flags == -1)
    {
      error = g_error_new (GNETWORK_CONNECTION_ERROR, GNETWORK_CONNECTION_ERROR_INTERNAL,
			   _("The connection to %s could not be completed because an error "
			     "occured inside the GNetwork library."), connection->_priv->address);
      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
      g_error_free (error);

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
      close (connection->_priv->sockfd);
      connection->_priv->sockfd = -1;

      g_object_freeze_notify (object);
      g_object_notify (object, "tcp-status");
      g_object_notify (object, "status");
      g_object_notify (object, "socket-fd");
      g_object_thaw_notify (object);
      return;
    }

  /* Add the non-blocking flag */
  if (fcntl (connection->_priv->sockfd, F_SETFL, flags | O_NONBLOCK) == -1)
    {
      error = g_error_new (GNETWORK_CONNECTION_ERROR, GNETWORK_CONNECTION_ERROR_INTERNAL,
			   _("The connection to %s could not be completed because an error "
			     "occured inside the GNetwork library."), connection->_priv->address);
      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
      g_error_free (error);

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
      close (connection->_priv->sockfd);
      connection->_priv->sockfd = -1;

      g_object_freeze_notify (object);
      g_object_notify (object, "tcp-status");
      g_object_notify (object, "status");
      g_object_notify (object, "socket-fd");
      g_object_thaw_notify (object);
      return;
    }

  errno = 0;
  result = connect (connection->_priv->sockfd, (struct sockaddr *) &(sin), sizeof (sin));

  if (result != 0)
    {
      if (errno == EINPROGRESS)
	{
	  connection->_priv->channel = g_io_channel_unix_new (connection->_priv->sockfd);
	  g_io_channel_set_encoding (connection->_priv->channel, NULL, NULL);
	  g_io_channel_set_buffered (connection->_priv->channel, FALSE);

	  connection->_priv->source_id =
	    gnetwork_thread_io_add_watch (connection->_priv->channel, GNETWORK_IO_ANY,
					  (GIOFunc) connect_done_handler, connection);
	  return;
	}
      else if (connection->_priv->proxy_ip_address != NULL)
	{
	  GNetworkTcpProxyError error_type = _gnetwork_tcp_proxy_error_from_errno (errno);

	  error = g_error_new_literal (GNETWORK_TCP_PROXY_ERROR, error_type, NULL);
	  error->message = _gnetwork_tcp_proxy_strerror (error_type,
							 connection->_priv->tcp_proxy_type,
							 connection->_priv->address);
	}
      else
	{
	  error = get_connection_error_from_errno (errno, connection->_priv->address);
	}

      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
      g_error_free (error);

      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
      close (connection->_priv->sockfd);
      connection->_priv->sockfd = -1;

      g_object_freeze_notify (object);
      g_object_notify (object, "tcp-status");
      g_object_notify (object, "status");
      g_object_notify (object, "socket-fd");
      g_object_thaw_notify (G_OBJECT (connection));
    }
  else
    {
      connection->_priv->channel = g_io_channel_unix_new (connection->_priv->sockfd);
      g_io_channel_set_encoding (connection->_priv->channel, NULL, NULL);
      g_io_channel_set_buffered (connection->_priv->channel, FALSE);

      connect_suceeded (connection);
    }
}


/* *************** *
 *  DNS Callbacks  *
 * *************** */

static void
dns_callback (const GNetworkDnsEntry * entry, GNetworkTcpConnection * connection)
{
  /* We're done with the handle */
  connection->_priv->dns_handle = GNETWORK_DNS_INVALID_HANDLE;

  /* If we're still interested in connecting :-) */
  if (connection->_priv->tcp_status > GNETWORK_TCP_CONNECTION_CLOSED)
    {
      /* Lookup Failed */
      if (entry->error != GNETWORK_DNS_ERROR_NONE)
	{
	  GError *error = g_error_new (GNETWORK_DNS_ERROR, entry->error,
				       gnetwork_dns_strerror (entry->error));
	  gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
	  g_error_free (error);

	  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
	  connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;

	  g_object_freeze_notify (G_OBJECT (connection));
	  g_object_notify (G_OBJECT (connection), "tcp-status");
	  g_object_notify (G_OBJECT (connection), "status");
	  g_object_thaw_notify (G_OBJECT (connection));
	}
      else
	{
	  connection->_priv->ip_address = g_strdup (entry->ip_addresses->data);
	  g_object_notify (G_OBJECT (connection), "ip-address");

	  /* Lookup succeeded, we need a proxy, and we've already gotten the proxy return... */
	  if (gnetwork_tcp_proxy_get_use_proxy (connection->_priv->tcp_proxy_type,
						connection->_priv->address))
	    {
	      if (connection->_priv->proxy_ip_address != NULL)
		{
		  open_client_connection (connection);
		}
	    }
	  /* Lookup suceeded, and we don't need a proxy */
	  else
	    {
	      open_client_connection (connection);
	    }
	}
    }
}


static void
proxy_dns_callback (const GNetworkDnsEntry * proxy_entry, GNetworkTcpConnection * connection)
{
  /* We're done with the proxy lookup */
  connection->_priv->proxy_dns_handle = GNETWORK_DNS_INVALID_HANDLE;

  /* If we're still interested in connecting :-) */
  if (connection->_priv->tcp_status > GNETWORK_TCP_CONNECTION_CLOSED)
    {
      /* Lookup Failed */
      if (proxy_entry->error != GNETWORK_DNS_ERROR_NONE)
	{
	  GError *error = g_error_new (GNETWORK_DNS_ERROR, proxy_entry->error,
				       gnetwork_dns_strerror (proxy_entry->error));
	  gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
	  g_error_free (error);

	  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
	  connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;

	  g_object_freeze_notify (G_OBJECT (connection));
	  g_object_notify (G_OBJECT (connection), "tcp-status");
	  g_object_notify (G_OBJECT (connection), "status");
	  g_object_thaw_notify (G_OBJECT (connection));
	}
      /* Otherwise, it suceeded, and if the (real) destination lookup is done
         as well, start the connection */
      else if (connection->_priv->ip_address != NULL)
	{
	  open_client_connection (connection);
	}
    }
}


/* *********************************** *
 *  GNetworkConnectionIface Functions  *
 * *********************************** */

/* Signal Callbacks */
static void
gnetwork_tcp_connection_sent (GNetworkConnection * connection, gconstpointer data, gulong length)
{
  g_return_if_fail (GNETWORK_IS_TCP_CONNECTION (connection));

  GNETWORK_TCP_CONNECTION (connection)->_priv->bytes_sent += length;
  g_object_notify (G_OBJECT (connection), "bytes-sent");
}


static void
gnetwork_tcp_connection_received (GNetworkConnection * connection, gconstpointer data,
				  gulong length)
{
  g_return_if_fail (GNETWORK_IS_TCP_CONNECTION (connection));

  GNETWORK_TCP_CONNECTION (connection)->_priv->bytes_received += length;
  g_object_notify (G_OBJECT (connection), "bytes-received");
}


static void
gnetwork_tcp_connection_open (GNetworkTcpConnection * connection)
{
  GError *error;
  GNetworkDnsEntry *entry;
  GObject *object;

  g_return_if_fail (GNETWORK_IS_TCP_CONNECTION (connection));
  g_return_if_fail (connection->_priv->cxn_status == GNETWORK_CONNECTION_CLOSED);

  object = G_OBJECT (connection);

  connection->_priv->bytes_received = 0;
  connection->_priv->bytes_sent = 0;
  connection->_priv->cxn_status = GNETWORK_CONNECTION_OPENING;
  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_LOOKUP;

  g_object_freeze_notify (object);
  g_object_notify (object, "bytes-received");
  g_object_notify (object, "bytes-sent");
  g_object_notify (object, "tcp-status");
  g_object_notify (object, "status");
  g_object_thaw_notify (object);

  connection->_priv->proxy_dns_handle = GNETWORK_DNS_INVALID_HANDLE;
  connection->_priv->dns_handle = GNETWORK_DNS_INVALID_HANDLE;

  if (connection->_priv->cxn_type == GNETWORK_CONNECTION_CLIENT)
    {
      /* If we need to use a proxy... */
      if (gnetwork_tcp_proxy_get_use_proxy (connection->_priv->tcp_proxy_type,
					    connection->_priv->address))
	{
	  gchar *proxy_host;

	  proxy_host = _gnetwork_tcp_proxy_get_host (connection->_priv->tcp_proxy_type);

	  /* If the proxy host is already an IP address, just fake a return and call the callback */
	  if (connection->_priv->proxy_ip_address != NULL)
	    {
	      entry = gnetwork_dns_entry_new ();

	      entry->error = GNETWORK_DNS_ERROR_NONE;
	      entry->ip_addresses = g_slist_append (entry->ip_addresses, proxy_host);
	      proxy_dns_callback (entry, connection);

	      gnetwork_dns_entry_free (entry);
	    }
	  /* Otherwise, start the lookup */
	  else
	    {
	      connection->_priv->proxy_dns_handle =
		gnetwork_dns_get (proxy_host, (GNetworkDnsCallbackFunc) proxy_dns_callback,
				  connection, NULL, &error);

	      g_free (proxy_host);

	      /* Lookup couldn't start for some reason. */
	      if (connection->_priv->proxy_dns_handle == GNETWORK_DNS_INVALID_HANDLE)
		{
		  gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
		  g_error_free (error);

		  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
		  connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;

		  g_object_freeze_notify (object);
		  g_object_notify (object, "tcp-status");
		  g_object_notify (object, "status");
		  g_object_thaw_notify (object);
		  return;
		}
	    }
	}

      /* If the (real) destination address is already an IP address, just fake a
         return and call the callback */
      if (connection->_priv->ip_address != NULL)
	{
	  entry = gnetwork_dns_entry_new ();

	  entry->error = GNETWORK_DNS_ERROR_NONE;
	  entry->ip_addresses = g_slist_append (entry->ip_addresses,
						g_strdup (connection->_priv->address));

	  dns_callback (entry, connection);

	  gnetwork_dns_entry_free (entry);
	}
      /* Othwerise, start a lookup for that too */
      else
	{
	  connection->_priv->dns_handle =
	    gnetwork_dns_get (connection->_priv->address, (GNetworkDnsCallbackFunc) dns_callback,
			      connection, NULL, &error);

	  /* The lookup couldn't start */
	  if (connection->_priv->dns_handle == GNETWORK_DNS_INVALID_HANDLE)
	    {
	      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
	      g_error_free (error);

	      connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
	      connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;

	      g_object_freeze_notify (object);
	      g_object_notify (object, "tcp-status");
	      g_object_notify (object, "status");
	      g_object_thaw_notify (object);
	    }
	}
    }
  else
    {
      if (connection->_priv->sockfd < 0)
	{
	  g_warning ("%s: You cannot open a server connection without first setting the socket-fd "
		     "property on the connection to the accepted socket.", G_STRLOC);
	  return;
	}

      connection->_priv->channel = g_io_channel_unix_new (connection->_priv->sockfd);
      g_io_channel_set_encoding (connection->_priv->channel, NULL, NULL);
      g_io_channel_set_buffered (connection->_priv->channel, FALSE);

#ifdef _USE_SSL
      if (connection->_priv->ssl_enabled)
	{
	  open_ssl_connection (connection);
	}
      else
#endif /* _USE_SSL */
	{

	  connection->_priv->source_id =
	    gnetwork_thread_io_add_watch (connection->_priv->channel,
					  (G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP),
					  (GIOFunc) io_channel_handler, connection);

	  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_OPEN;
	  connection->_priv->cxn_status = GNETWORK_CONNECTION_OPEN;

	  g_object_freeze_notify (object);
	  g_object_notify (object, "tcp-status");
	  g_object_notify (object, "status");
	  g_object_thaw_notify (object);
	}
    }
}


static void
gnetwork_tcp_connection_close (GNetworkTcpConnection * connection)
{
  GNetworkTcpConnectionStatus status;

  g_return_if_fail (GNETWORK_IS_TCP_CONNECTION (connection));

  if (connection->_priv->tcp_status == GNETWORK_TCP_CONNECTION_CLOSING ||
      connection->_priv->tcp_status == GNETWORK_TCP_CONNECTION_CLOSED)
    return;

  /* Save the old status */
  status = connection->_priv->tcp_status;

  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSING;
  connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSING;

  g_object_freeze_notify (G_OBJECT (connection));
  g_object_notify (G_OBJECT (connection), "tcp-status");
  g_object_notify (G_OBJECT (connection), "status");
  g_object_thaw_notify (G_OBJECT (connection));

  switch (status)
    {
      /* Fully Open & Ready. */
    case GNETWORK_TCP_CONNECTION_OPENING:
    case GNETWORK_TCP_CONNECTION_PROXYING:
    case GNETWORK_TCP_CONNECTION_AUTHENTICATING:
    case GNETWORK_TCP_CONNECTION_OPEN:
      if (connection->_priv->channel != NULL)
	{
	  g_io_channel_shutdown (connection->_priv->channel, FALSE, NULL);
	  g_io_channel_unref (connection->_priv->channel);
	}
      else if (connection->_priv->sockfd > 0)
	{
	  shutdown (connection->_priv->sockfd, SHUT_RDWR);
	  close (connection->_priv->sockfd);
	  connection->_priv->sockfd = -1;
	}
      break;

    case GNETWORK_TCP_CONNECTION_LOOKUP:
      if (connection->_priv->dns_handle != GNETWORK_DNS_INVALID_HANDLE)
	{
	  gnetwork_dns_cancel (connection->_priv->dns_handle);
	  connection->_priv->dns_handle = GNETWORK_DNS_INVALID_HANDLE;
	}

      if (connection->_priv->proxy_dns_handle != GNETWORK_DNS_INVALID_HANDLE)
	{
	  gnetwork_dns_cancel (connection->_priv->proxy_dns_handle);
	  connection->_priv->proxy_dns_handle = GNETWORK_DNS_INVALID_HANDLE;
	}
      break;

    default:
      g_assert_not_reached ();
      break;
    }

  connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;
  connection->_priv->sockfd = -1;

  g_object_freeze_notify (G_OBJECT (connection));
  g_object_notify (G_OBJECT (connection), "status");
  g_object_notify (G_OBJECT (connection), "tcp-status");
  g_object_notify (G_OBJECT (connection), "socket-fd");
  g_object_thaw_notify (G_OBJECT (connection));
}


static void
gnetwork_tcp_connection_send (GNetworkTcpConnection * connection, gconstpointer data, gulong length)
{
  GIOStatus status;
  gsize bytes_written;
  GError *error;

  g_return_if_fail (GNETWORK_IS_TCP_CONNECTION (connection));
  g_return_if_fail (connection->_priv->tcp_status == GNETWORK_TCP_CONNECTION_OPEN);

  error = NULL;

  do
    {
      status = g_io_channel_write_chars (connection->_priv->channel, (const gchar *) data,
					 length, &bytes_written, &error);
    }
  while (status == G_IO_STATUS_AGAIN);

  if (error != NULL)
    {
      gnetwork_connection_error (GNETWORK_CONNECTION (connection), error);
      g_free (error);
    }
}


static void
gnetwork_tcp_connection_connection_iface_init (GNetworkConnectionIface * iface)
{
  iface->sent = gnetwork_tcp_connection_sent;
  iface->received = gnetwork_tcp_connection_received;

  iface->open = (GNetworkConnectionFunc) gnetwork_tcp_connection_open;
  iface->close = (GNetworkConnectionFunc) gnetwork_tcp_connection_close;
  iface->send = (GNetworkConnectionSendFunc) gnetwork_tcp_connection_send;
}


/* ******************* *
 *  GObject Functions  *
 * ******************* */

static void
gnetwork_tcp_connection_set_property (GObject * object, guint property, const GValue * value,
				      GParamSpec * param_spec)
{
  GNetworkTcpConnection *connection = GNETWORK_TCP_CONNECTION (object);

  switch (property)
    {
    case ADDRESS:
      {
	struct in_addr addr;
	const gchar *address = g_value_get_string (value);

	g_return_if_fail (address == NULL || address[0] != '\0');
	g_return_if_fail (connection->_priv->tcp_status == GNETWORK_TCP_CONNECTION_CLOSED ||
			  connection->_priv->cxn_type == GNETWORK_CONNECTION_SERVER);

	g_free (connection->_priv->address);
	connection->_priv->address = g_strdup (address);

	g_object_freeze_notify (object);
	g_object_notify (object, "address");

	if (connection->_priv->cxn_type == GNETWORK_CONNECTION_CLIENT)
	  {
	    if (address != NULL && inet_pton (AF_INET, address, &addr))
	      {
		connection->_priv->ip_address = g_strdup (address);
	      }
	    else
	      {
		connection->_priv->ip_address = NULL;
	      }

	    g_object_notify (object, "ip-address");
	  }

	g_object_thaw_notify (object);
      }
      break;
    case PORT:
      g_return_if_fail (connection->_priv->tcp_status < GNETWORK_TCP_CONNECTION_OPENING);

      connection->_priv->port = g_value_get_uint (value);
      g_object_notify (object, "port");
      break;
    case PROXY_TYPE:
      g_return_if_fail (connection->_priv->tcp_status < GNETWORK_TCP_CONNECTION_PROXYING);

      connection->_priv->tcp_proxy_type = g_value_get_enum (value);

      g_free (connection->_priv->proxy_ip_address);
      connection->_priv->proxy_ip_address = NULL;

      g_object_notify (object, "proxy-type");
      break;

    case SOCKET_FD:
      g_return_if_fail (connection->_priv->tcp_status == GNETWORK_TCP_CONNECTION_CLOSED);
      {
	connection->_priv->sockfd = g_value_get_int (value);

	g_free (connection->_priv->local_address);

	if (connection->_priv->sockfd > 0)
	  {
	    gchar local_address[INET_ADDRSTRLEN + 1];
	    struct sockaddr_in sin;
	    socklen_t sin_size;

	    getsockname (connection->_priv->sockfd, (struct sockaddr *) &sin, &sin_size);

	    connection->_priv->local_address = g_strdup (inet_ntop (AF_INET, &(sin.sin_addr),
								    local_address,
								    INET_ADDRSTRLEN));
	    connection->_priv->local_port = g_ntohs (sin.sin_port);
	  }
	else
	  {
	    connection->_priv->local_address = NULL;
	    connection->_priv->local_port = 0;
	  }

	g_object_freeze_notify (object);
	g_object_notify (object, "socket-fd");
	g_object_notify (object, "local-address");
	g_object_notify (object, "local-port");
	g_object_thaw_notify (object);
      }
      break;

    case CXN_TYPE:
      connection->_priv->cxn_type = g_value_get_enum (value);
      g_object_notify (object, "connection-type");
      break;
    case CXN_BUFFER_SIZE:
      g_return_if_fail (connection->_priv->tcp_status < GNETWORK_TCP_CONNECTION_OPENING);
      connection->_priv->buffer_size = g_value_get_uint (value);
      g_object_notify (object, "buffer-size");
      break;

#ifdef _USE_SSL
    case SSL_ENABLED:
      g_return_if_fail (connection->_priv->tcp_status < GNETWORK_TCP_CONNECTION_AUTHENTICATING);

      connection->_priv->ssl_enabled = g_value_get_boolean (value);

      g_object_notify (object, "ssl-enabled");
      break;
    case SSL_AUTH_TYPE:
      g_return_if_fail (connection->_priv->tcp_status < GNETWORK_TCP_CONNECTION_AUTHENTICATING);

      connection->_priv->auth_type = g_value_get_enum (value);

      g_object_notify (object, "authentication-type");
      break;
    case SSL_CA_FILE:
      g_return_if_fail (connection->_priv->tcp_status < GNETWORK_TCP_CONNECTION_AUTHENTICATING);

      g_free (connection->_priv->ca_file);
      connection->_priv->ca_file = g_value_dup_string (value);

      g_object_notify (object, "authority-file");
      break;
    case SSL_CERT_FILE:
      g_return_if_fail (connection->_priv->tcp_status < GNETWORK_TCP_CONNECTION_AUTHENTICATING);

      g_free (connection->_priv->cert_file);
      connection->_priv->cert_file = g_value_dup_string (value);

      g_object_notify (object, "certificate-file");
      break;
    case SSL_KEY_FILE:
      g_return_if_fail (connection->_priv->tcp_status < GNETWORK_TCP_CONNECTION_AUTHENTICATING);

      g_free (connection->_priv->key_file);
      connection->_priv->key_file = g_value_dup_string (value);

      g_object_notify (object, "key-file");
      break;
#else /* !_USE_SSL */
    case SSL_ENABLED:
    case SSL_AUTH_TYPE:
    case SSL_CA_FILE:
    case SSL_CERT_FILE:
    case SSL_KEY_FILE:
      g_warning (G_STRLOC ": SSL properties cannot be set because this version of the GNetwork "
		 "library was compiled without SSL support.");
      break;
#endif /* _USE_SSL */

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property, param_spec);
      break;
    }
}


static void
gnetwork_tcp_connection_get_property (GObject * object, guint property, GValue * value,
				      GParamSpec * param_spec)
{
  GNetworkTcpConnection *connection = GNETWORK_TCP_CONNECTION (object);

  switch (property)
    {
      /* GNetworkTcpConnection */
    case TCP_STATUS:
      g_value_set_enum (value, connection->_priv->tcp_status);
      break;
    case ADDRESS:
      g_value_set_string (value, connection->_priv->address);
      break;
    case PORT:
      g_value_set_uint (value, connection->_priv->port);
      break;
    case LOCAL_ADDRESS:
      g_value_set_string (value, connection->_priv->local_address);
      break;
    case LOCAL_PORT:
      g_value_set_uint (value, connection->_priv->local_port);
      break;
    case IP_ADDRESS:
      g_value_set_string (value, connection->_priv->ip_address);
      break;
      /* SSL */
    case SSL_ENABLED:
      g_value_set_boolean (value, connection->_priv->ssl_enabled);
      break;
    case SSL_AUTH_TYPE:
      g_value_set_enum (value, connection->_priv->auth_type);
      break;
    case SSL_CA_FILE:
      g_value_set_string (value, connection->_priv->ca_file);
      break;
    case SSL_CERT_FILE:
      g_value_set_string (value, connection->_priv->cert_file);
      break;
    case SSL_KEY_FILE:
      g_value_set_string (value, connection->_priv->key_file);
      break;

      /* GNetworkConnectionIface */
    case CXN_TYPE:
      g_value_set_enum (value, connection->_priv->cxn_type);
      break;
    case CXN_STATUS:
      g_value_set_enum (value, connection->_priv->cxn_status);
      break;
    case CXN_BUFFER_SIZE:
      g_value_set_uint (value, connection->_priv->buffer_size);
      break;
    case CXN_BYTES_RECEIVED:
      g_value_set_ulong (value, connection->_priv->bytes_received);
      break;
    case CXN_BYTES_SENT:
      g_value_set_ulong (value, connection->_priv->bytes_sent);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property, param_spec);
      break;
    }
}


static void
gnetwork_tcp_connection_dispose (GObject * object)
{
  GNetworkTcpConnection *connection = GNETWORK_TCP_CONNECTION (object);

  if (connection->_priv->tcp_status > GNETWORK_TCP_CONNECTION_CLOSED)
    gnetwork_tcp_connection_close (connection);

  if (G_OBJECT_CLASS (parent_class)->dispose)
    (*G_OBJECT_CLASS (parent_class)->dispose) (object);
}


static void
gnetwork_tcp_connection_finalize (GObject * object)
{
  GNetworkTcpConnection *connection = GNETWORK_TCP_CONNECTION (object);

  g_free (connection->_priv->address);
  g_free (connection->_priv->local_address);
  g_free (connection->_priv->ip_address);

  g_free (connection->_priv->ca_file);
  g_free (connection->_priv->cert_file);
  g_free (connection->_priv->key_file);

  g_free (connection->_priv);

  if (G_OBJECT_CLASS (parent_class)->finalize)
    (*G_OBJECT_CLASS (parent_class)->finalize) (object);
}


/* ***************** *
 *  GType Functions  *
 * ***************** */

static void
gnetwork_tcp_connection_base_init (gpointer class)
{
#if 0
#ifdef _USE_SSL
  _gnetwork_ssl_initialize ();
#endif /* _USE_SSL */
#endif /* 0 */
  _gnetwork_tcp_proxy_initialize ();
}


static void
gnetwork_tcp_connection_base_finalize (gpointer class)
{
#if 0
#ifdef _USE_SSL
  _gnetwork_ssl_shutdown ();
#endif /* _USE_SSL */
#endif /* 0 */
  _gnetwork_tcp_proxy_shutdown ();
}


static void
gnetwork_tcp_connection_class_init (GNetworkTcpConnectionClass * class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  object_class->get_property = gnetwork_tcp_connection_get_property;
  object_class->set_property = gnetwork_tcp_connection_set_property;
  object_class->dispose = gnetwork_tcp_connection_dispose;
  object_class->finalize = gnetwork_tcp_connection_finalize;

  g_object_class_install_property (object_class, TCP_STATUS,
				   g_param_spec_enum ("tcp-status", _("TCP/IP Connection Status"),
						      _("The current status of the TCP/IP "
							"connection."),
						      GNETWORK_TYPE_TCP_CONNECTION_STATUS,
						      GNETWORK_TCP_CONNECTION_CLOSED,
						      G_PARAM_READABLE));

  g_object_class_install_property (object_class, PROXY_TYPE,
				   g_param_spec_enum ("proxy-type", _("Proxy Type"),
						      _("The type of proxy to use, depending on "
							"the protocol."),
						      GNETWORK_TYPE_TCP_PROXY_TYPE,
						      GNETWORK_TCP_PROXY_HTTP,
						      (G_PARAM_READWRITE | G_PARAM_CONSTRUCT)));
  g_object_class_install_property (object_class, ADDRESS,
				   g_param_spec_string ("address", _("Address"),
							_("The hostname or IP address to connect "
							  "to."),
							NULL,
							(G_PARAM_READWRITE | G_PARAM_CONSTRUCT)));
  g_object_class_install_property (object_class, PORT,
				   g_param_spec_uint ("port", _("Port Number"),
						      _("The port number to connect to."),
						      0, 65535, 0,
						      (G_PARAM_READWRITE | G_PARAM_CONSTRUCT)));
  g_object_class_install_property (object_class, LOCAL_ADDRESS,
				   g_param_spec_string ("local-address", _("Local Address"),
							_("The hostname or IP address of this "
							  "computer."), NULL, G_PARAM_READABLE));
  g_object_class_install_property (object_class, LOCAL_PORT,
				   g_param_spec_uint ("local-port",
						      _("Local Port Number"),
						      _("The local port number to connect "
							"through."),
						      0, 65535, 0, G_PARAM_READABLE));
  g_object_class_install_property (object_class, IP_ADDRESS,
				   g_param_spec_string ("ip-address", _("IP Address"),
							_("The IP address connected to."), NULL,
							G_PARAM_READABLE));
  g_object_class_install_property (object_class, SOCKET_FD,
				   g_param_spec_int ("socket-fd", _("Socket File Descriptor"),
						     _("The socket file descriptor. For use by "
						       "#GNetworkTcpServer."),
						     -1, G_MAXINT, -1,
						     (G_PARAM_PRIVATE | G_PARAM_WRITABLE)));

  /* SSL Properties */
  g_object_class_install_property (object_class, SSL_ENABLED,
				   g_param_spec_boolean ("ssl-enabled", _("SSL Enabled"),
							 _("Whether or not SSL will be used with "
							   "this connection."), FALSE,
							 (G_PARAM_READWRITE | G_PARAM_CONSTRUCT)));
  g_object_class_install_property (object_class, SSL_AUTH_TYPE,
				   g_param_spec_enum ("authentication-type",
						      _("Authentication Type"),
						      _("What type of SSL authentication should be "
							"performed with this connection."),
						      GNETWORK_TYPE_SSL_AUTH_TYPE,
						      GNETWORK_SSL_AUTH_ANONYMOUS,
						      (G_PARAM_READWRITE | G_PARAM_CONSTRUCT)));
  g_object_class_install_property (object_class, SSL_CA_FILE,
				   g_param_spec_string ("authority-file",
							_("Certificate Authorities File"),
							_("The path to a file which contains the "
							  "X.509 certificates of trusted signers."),
							NULL,
							(G_PARAM_READWRITE | G_PARAM_CONSTRUCT)));
  g_object_class_install_property (object_class, SSL_CERT_FILE,
				   g_param_spec_string ("certificate-file", _("Certificate File"),
							_("The path to a file which contains X.509 "
							  "certificates to distribute to clients."),
							NULL,
							(G_PARAM_READWRITE | G_PARAM_CONSTRUCT)));
  g_object_class_install_property (object_class, SSL_KEY_FILE,
				   g_param_spec_string ("key-file", _("Key File"),
							_("The path to a file which contains X.509 "
							  "keys to distribute to clients."), NULL,
							(G_PARAM_READWRITE | G_PARAM_CONSTRUCT)));

  /* GNetworkConnectionIface Properties */
  g_object_class_override_property (object_class, CXN_TYPE, "connection-type");
  g_object_class_override_property (object_class, CXN_STATUS, "status");
  g_object_class_override_property (object_class, CXN_BYTES_RECEIVED, "bytes-sent");
  g_object_class_override_property (object_class, CXN_BYTES_SENT, "bytes-received");
  g_object_class_override_property (object_class, CXN_BUFFER_SIZE, "buffer-size");

  /* Signals */
  signals[CERTIFICATE_ERROR] =
	g_signal_new ("certificate-error",
		      GNETWORK_TYPE_TCP_CONNECTION,
		      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (GNetworkTcpConnectionClass, certificate_error),
		      NULL, NULL, _gnetwork_marshal_VOID__BOXED_FLAGS,
		      G_TYPE_NONE, 2, GNETWORK_TYPE_SSL_CERT, GNETWORK_TYPE_SSL_CERT_ERROR_FLAGS);
}


static void
gnetwork_tcp_connection_instance_init (GNetworkTcpConnection * connection)
{
  connection->_priv = g_new (GNetworkTcpConnectionPrivate, 1);

  connection->_priv->cxn_type = GNETWORK_CONNECTION_CLIENT;
  connection->_priv->cxn_status = GNETWORK_CONNECTION_CLOSED;
  connection->_priv->tcp_status = GNETWORK_TCP_CONNECTION_CLOSED;

  connection->_priv->address = NULL;
  connection->_priv->ip_address = NULL;

  connection->_priv->proxy_dns_handle = GNETWORK_DNS_INVALID_HANDLE;
  connection->_priv->dns_handle = GNETWORK_DNS_INVALID_HANDLE;

  connection->_priv->local_address = NULL;
  connection->_priv->proxy_ip_address = NULL;

  connection->_priv->ca_file = NULL;
  connection->_priv->cert_file = NULL;
  connection->_priv->key_file = NULL;

  connection->_priv->channel = NULL;
  connection->_priv->source_id = 0;
  connection->_priv->sockfd = -1;
}


/* ************ *
 *  PUBLIC API  *
 * ************ */

GType
gnetwork_tcp_connection_get_type (void)
{
  static GType type = G_TYPE_INVALID;

  if (type == G_TYPE_INVALID)
    {
      static const GTypeInfo info = {
	sizeof (GNetworkTcpConnectionClass),
	(GBaseInitFunc) gnetwork_tcp_connection_base_init,
	(GBaseFinalizeFunc) gnetwork_tcp_connection_base_finalize,
	(GClassInitFunc) gnetwork_tcp_connection_class_init,
	(GClassFinalizeFunc) NULL,
	NULL,			/* class data */
	sizeof (GNetworkTcpConnection),
	0,			/* number of pre-allocs */
	(GInstanceInitFunc) gnetwork_tcp_connection_instance_init,
	NULL			/* value table */
      };
      static const GInterfaceInfo cxn_info = {
	(GInterfaceInitFunc) gnetwork_tcp_connection_connection_iface_init,
	NULL, NULL
      };

      type = g_type_register_static (G_TYPE_OBJECT, "GNetworkTcpConnection", &info, 0);

      g_type_add_interface_static (type, GNETWORK_TYPE_CONNECTION, &cxn_info);
    }

  return type;
}
