/* LibGTcpSocket: src/gtcp-dns.c
 *
 * Copyright (C) 2001 James M. Cape
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/*

GLib 2.0 DNS lookup backend.

Notes on editing:
	Tab size: 4
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include "gnetwork-dns.h"

#include "gnetwork-threads.h"

#include <time.h>
#include <netdb.h>
#include <string.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <glib/gi18n.h>


#define CLEANUP_INTERVAL 2400000


#define set_handle(lookup,retval) \
G_STMT_START{ \
	retval = lookup->handle = dns->current_handle; \
	dns->current_handle++; \
} G_STMT_END


typedef struct
{
  GSList *cache;
  GStaticMutex cache_mutex;

  gint cache_cleanup_id;
  GNetworkDnsHandle current_handle;

  guint ref;
}
GNetworkDns;


typedef struct
{
  GSList *hashes;

  gchar *lookup_addr;

  GStaticMutex in_progress_mutex;

  GSList *lookups;
  GStaticMutex lookups_mutex;
  GMainLoop *lookup_loop;

  guint thread_done_id;
  GTime mod_time;

  GNetworkDnsEntry *dns_entry;	/* Glib-esque hostent */
}
GNetworkDnsCacheEntry;


typedef struct
{
  GNetworkDnsHandle handle;
  GNetworkDnsCallbackFunc callback;
  gpointer user_data;
  GDestroyNotify destroy_data;
}
GNetworkDnsWaitingLookup;


typedef struct _EnumString
{
  const gint value;
  const gchar *const str;
}
EnumString;


static GNetworkDns *dns = NULL;


static GNetworkDnsCacheEntry *
gnetwork_dns_cache_entry_new (void)
{
  GNetworkDnsCacheEntry *entry = NULL;

  entry = g_new0 (GNetworkDnsCacheEntry, 1);

  entry->hashes = NULL;
  entry->lookups = NULL;

  g_static_mutex_init (&(entry->lookups_mutex));
  g_static_mutex_init (&(entry->in_progress_mutex));

  entry->dns_entry = gnetwork_dns_entry_new ();

  entry->mod_time = time (NULL);

  return entry;
}


static void
gnetwork_dns_cache_entry_free (GNetworkDnsCacheEntry * entry)
{
  GNetworkDnsWaitingLookup *lookup;

  g_return_if_fail (entry != NULL);

  g_static_mutex_lock (&(dns->cache_mutex));
  dns->cache = g_slist_remove (dns->cache, entry);
  g_static_mutex_unlock (&(dns->cache_mutex));

  g_slist_free (entry->hashes);

  while (entry->lookups != NULL)
    {
      lookup = (GNetworkDnsWaitingLookup *) entry->lookups->data;

      lookup->callback = NULL;

      if (lookup->destroy_data != NULL && lookup->user_data != NULL)
	(*lookup->destroy_data) (lookup->user_data);

      lookup->user_data = NULL;

      g_free (lookup);

      entry->lookups = g_slist_remove (entry->lookups, lookup);
    }

  gnetwork_dns_entry_free (entry->dns_entry);

  g_free (entry);
}


static void
gnetwork_dns_shutdown ()
{
  GNetworkDns *tmp_dns = dns;

  if (dns == NULL)
    return;

  dns->ref--;

  if (dns->ref > 0)
    return;

  gnetwork_thread_source_remove (tmp_dns->cache_cleanup_id);
  dns = NULL;

  while (tmp_dns->cache != NULL)
    {
      gnetwork_dns_cache_entry_free (tmp_dns->cache->data);
      tmp_dns->cache = g_slist_remove (tmp_dns->cache, tmp_dns->cache->data);
    }

  g_free (tmp_dns);
}


static gboolean
cache_cleanup (gpointer user_data)
{
  GNetworkDnsCacheEntry *entry;
  GSList *list;
  GTime current_time;

  current_time = time (NULL);

  for (list = dns->cache; list != NULL; list = list->next)
    {
      entry = (GNetworkDnsCacheEntry *) list->data;

      if ((current_time - entry->mod_time) > CLEANUP_INTERVAL)
	{
	  if (g_slist_length (dns->cache) == 1)
	    gnetwork_dns_shutdown ();
	  else
	    gnetwork_dns_cache_entry_free (entry);
	}
    }

  return TRUE;
}


static void
gnetwork_dns_init ()
{
  if (dns != NULL)
    {
      dns->ref++;
      return;
    }

  dns = g_new0 (GNetworkDns, 1);

  dns->cache = NULL;

  g_static_mutex_init (&(dns->cache_mutex));

  dns->current_handle = 0;

  dns->ref = 1;
  dns->cache_cleanup_id = gnetwork_thread_timeout_add (CLEANUP_INTERVAL, cache_cleanup, NULL);

#ifdef G_THREADS_ENABLED
  if (!g_thread_supported ())
    g_thread_init (NULL);
#endif /* G_THREADS_ENABLED */
}


static void
parse_hostent (GNetworkDnsCacheEntry * entry, struct hostent *host, gint host_error_num) 
{
  guint i;
  gchar *dotted_quad;

  if (!host)
    {
      switch (host_error_num)
	{
	case HOST_NOT_FOUND:
	  entry->dns_entry->error = GNETWORK_DNS_ERROR_NOT_FOUND;
	  break;

	case TRY_AGAIN:
	  entry->dns_entry->error = GNETWORK_DNS_ERROR_TRY_AGAIN;
	  break;

	case NO_ADDRESS:
	  entry->dns_entry->error = GNETWORK_DNS_ERROR_IPV4_IPV6_MISMATCH;
	  break;

	  /* We make NO_RECOVERY the default, since something evil has
	     happened if it gets to default: by it's own, and NO_RECOVERY is
	     pretty evil, and indicates something is seriously wrong. */
	default:
	  entry->dns_entry->error = GNETWORK_DNS_ERROR_NO_RECOVERY;
	  break;
	}
    }
  /* host != NULL */
  else
    {
      entry->dns_entry->error = GNETWORK_DNS_ERROR_NONE;

      /* Cannonical name */
      entry->dns_entry->hostname = g_strdup (host->h_name);

      entry->hashes = g_slist_append (entry->hashes, GUINT_TO_POINTER (g_str_hash (host->h_name)));

      for (i = 0; host->h_aliases != NULL && host->h_aliases[i] != NULL; i++)
	{
	  entry->dns_entry->aliases =
	    g_slist_append (entry->dns_entry->aliases, g_strdup (host->h_aliases[i]));

	  entry->hashes =
	    g_slist_append (entry->hashes, GUINT_TO_POINTER (g_str_hash (host->h_aliases[i])));
	}

      for (i = 0; host->h_addr_list[i] != NULL; i++)
	{
	  dotted_quad =
	    g_strdup_printf ("%u.%u.%u.%u", (guint8) host->h_addr_list[i][0],
			     (guint8) host->h_addr_list[i][1], (guint8) host->h_addr_list[i][2],
			     (guint8) host->h_addr_list[i][3]);
	  entry->dns_entry->ip_addresses =
	    g_slist_append (entry->dns_entry->ip_addresses, dotted_quad);

	  entry->hashes =
	    g_slist_append (entry->hashes, GUINT_TO_POINTER (g_str_hash (dotted_quad)));
	}
    }

}


static gpointer
fwd_lookup_thread (GNetworkDnsCacheEntry * entry)
{
  struct hostent *host = NULL;

  g_assert (entry != NULL);

  g_static_mutex_lock (&(entry->in_progress_mutex));
  host = gethostbyname (entry->lookup_addr);

  parse_hostent (entry, host, h_errno);

  g_static_mutex_unlock (&(entry->in_progress_mutex));

  return NULL;
}


static gpointer
rev_lookup_thread (GNetworkDnsCacheEntry * entry)
{
  struct hostent *host = NULL;

  g_assert (entry != NULL);

  g_static_mutex_lock (&(entry->in_progress_mutex));
  host = gethostbyaddr (entry->lookup_addr, strlen (entry->lookup_addr), AF_INET);

  parse_hostent (entry, host, h_errno);

  g_static_mutex_unlock (&(entry->in_progress_mutex));

  return NULL;
}


static gboolean
run_callbacks (GNetworkDnsCacheEntry * entry)
{
  GNetworkDnsWaitingLookup *lookup;

  g_static_mutex_lock (&(entry->lookups_mutex));

  while (entry->lookups != NULL)
    {
      lookup = (GNetworkDnsWaitingLookup *) entry->lookups->data;
      lookup->callback (entry->dns_entry, lookup->user_data);

      if (lookup->destroy_data != NULL && lookup->user_data != NULL)
	(*lookup->destroy_data) (lookup->user_data);

      g_free (lookup);

      entry->lookups = g_slist_remove (entry->lookups, lookup);
    }

  g_static_mutex_unlock (&(entry->lookups_mutex));

  return FALSE;
}


static gboolean
thread_joiner (gpointer data)
{
  GNetworkDnsCacheEntry *entry = (GNetworkDnsCacheEntry *) data;

  if (g_static_mutex_trylock (&(entry->in_progress_mutex)))
    {
      gnetwork_thread_timeout_add (25, (GSourceFunc) run_callbacks, data);
      g_static_mutex_unlock (&(entry->in_progress_mutex));

      return FALSE;
    }

  return TRUE;
}


/**
 * gnetwork_dns_get:
 * @address: the hostname or IP address to find.
 * @callback: the callback to be called when the lookup has completed.
 * @user_data: the user data to pass to @callback.
 * @destroy_data: a function capable of freeing @user_data, or %NULL.
 * @error: a location to store threading errors, or %NULL.
 *
 * This function performs an asynchronous DNS lookup (or reverse lookup) on the
 * hostname or IP address in @address. After @callback has been called,
 * @user_data will be destroyed using @destroy_data. If there an error occurred
 * trying to create the DNS lookup thread, @error will be set, and
 * #GNETWORK_DNS_INVALID_HANDLE will be returned.
 *
 * Note: @callback may be called before this function returns if the DNS record
 * for @address is already in the DNS cache.
 *
 * Returns: a #GNetworkDnsHandle used to cancel the lookup.
 *
 * Since: 1.0
 **/
GNetworkDnsHandle
gnetwork_dns_get (const gchar * address, GNetworkDnsCallbackFunc callback, gpointer user_data,
		  GDestroyNotify destroy_data, GError ** error)
{
  GNetworkDnsCacheEntry *entry;
  GNetworkDnsWaitingLookup *lookup;
  GSList *cache_list, *hash_list;
  GThreadFunc func;
  guint addr_hash;
  struct in_addr addr;
  GNetworkDnsHandle retval = GNETWORK_DNS_INVALID_HANDLE;

  g_return_val_if_fail (address != NULL && address[0] != '\0', GNETWORK_DNS_INVALID_HANDLE);
  g_return_val_if_fail (strlen (address) < 1024, GNETWORK_DNS_INVALID_HANDLE);
  g_return_val_if_fail (callback != NULL, GNETWORK_DNS_INVALID_HANDLE);
  g_return_val_if_fail (user_data != NULL || (user_data == NULL && destroy_data == NULL),
			GNETWORK_DNS_INVALID_HANDLE);

  gnetwork_dns_init ();

  addr_hash = g_str_hash (address);

  for (cache_list = dns->cache; cache_list != NULL; cache_list = cache_list->next)
    {
      entry = (GNetworkDnsCacheEntry *) (cache_list->data);

      for (hash_list = entry->hashes; hash_list != NULL; hash_list = hash_list->next)
	{
	  if (addr_hash == GPOINTER_TO_UINT (hash_list->data))
	    {
	      if (entry->dns_entry->error >= GNETWORK_DNS_ERROR_NONE)
		{
		  (*callback) (entry->dns_entry, user_data);

		  if (destroy_data != NULL && user_data != NULL)
		    (*destroy_data) (user_data);
		}
	      else
		{
		  lookup = NULL;
		  lookup = g_new0 (GNetworkDnsWaitingLookup, 1);

		  set_handle (lookup, retval);
		  lookup->callback = callback;
		  lookup->user_data = user_data;
		  lookup->destroy_data = destroy_data;

		  g_static_mutex_lock (&(entry->lookups_mutex));
		  entry->lookups = g_slist_append (entry->lookups, lookup);
		  g_static_mutex_unlock (&(entry->lookups_mutex));

		  entry->mod_time = time (NULL);
		}

	      return retval;
	    }
	}
    }

  entry = gnetwork_dns_cache_entry_new ();
  entry->lookup_addr = g_strdup (address);

  lookup = NULL;
  lookup = g_new0 (GNetworkDnsWaitingLookup, 1);

  set_handle (lookup, retval);
  lookup->callback = callback;
  lookup->user_data = user_data;
  entry->lookups = g_slist_append (entry->lookups, lookup);

  g_static_mutex_lock (&(dns->cache_mutex));
  dns->cache = g_slist_append (dns->cache, entry);
  g_static_mutex_unlock (&(dns->cache_mutex));

  if (inet_aton (address, &addr) != 0)
    {
      entry->dns_entry->hostname = g_strdup (address);

      func = (GThreadFunc) rev_lookup_thread;
    }
  else
    {
      func = (GThreadFunc) fwd_lookup_thread;
    }

  if (!gnetwork_thread_new (func, entry, NULL, NULL, error))
    {
      return GNETWORK_DNS_INVALID_HANDLE;
    }

  entry->thread_done_id = gnetwork_thread_timeout_add (500, thread_joiner, entry);

  return retval;
}


/**
 * gnetwork_dns_cancel:
 * @handle: a #GNetworkDnsHandle for a running lookup.
 *
 * This function prevents an asynchronous DNS lookup (or reverse lookup)
 * from completing.
 * 
 * Note: This function will not actually stop a DNS lookup, only prevent the
 * callbacks associated with @handle from being called. (The lookup will still
 * finish and the results will still be cached.)
 *
 * Since: 1.0
 **/
void
gnetwork_dns_cancel (GNetworkDnsHandle handle)
{
  GSList *list, *lookups_list;
  GNetworkDnsCacheEntry *entry;
  GNetworkDnsWaitingLookup *lookup;

  g_static_mutex_lock (&(dns->cache_mutex));
  for (list = dns->cache; list != NULL; list = list->next)
    {
      entry = (GNetworkDnsCacheEntry *) list->data;

      g_static_mutex_lock (&(entry->lookups_mutex));
      for (lookups_list = entry->lookups; lookups_list != NULL; lookups_list = lookups_list->next)
	{
	  lookup = (GNetworkDnsWaitingLookup *) entry->lookups->data;

	  if (lookup->handle == handle)
	    {
	      entry->lookups = g_slist_remove (entry->lookups, lookup);

	      if (lookup->destroy_data != NULL)
		(*lookup->destroy_data) (lookup->user_data);

	      g_free (lookup);
	    }
	}
      g_static_mutex_unlock (&(entry->lookups_mutex));
    }
  g_static_mutex_unlock (&(dns->cache_mutex));
}


/**
 * gnetwork_dns_entry_new:
 *
 * Allocates a new #GNetworkDnsEntry structure. The returned data
 * should be freed with gnetwork_dns_entry_free().
 *
 * Returns: an empty DNS entry reply structure.
 *
 * Since: 1.0
 **/
GNetworkDnsEntry *
gnetwork_dns_entry_new (void)
{
  GNetworkDnsEntry *entry = NULL;

  entry = g_new0 (GNetworkDnsEntry, 1);

  entry->error = GNETWORK_DNS_ERROR_IN_PROGRESS;
  entry->hostname = NULL;
  entry->aliases = NULL;
  entry->ip_addresses = NULL;

  return entry;
}


/**
 * gnetwork_dns_entry_free:
 * @entry: the #GNetworkDnsEntry to free.
 *
 * This function frees a #GNetworkDnsEntry.
 *
 * Since: 1.0
 **/
void
gnetwork_dns_entry_free (GNetworkDnsEntry * entry)
{
  if (entry == NULL)
    return;

  g_free (entry->hostname);

  for (; entry->aliases != NULL;
       entry->aliases = g_slist_remove_link (entry->aliases, entry->aliases))
    {
      g_free (entry->aliases->data);
    }


  for (; entry->ip_addresses != NULL;
       entry->ip_addresses = g_slist_remove_link (entry->ip_addresses, entry->ip_addresses))
    {
      g_free (entry->ip_addresses->data);
    }
}


/**
 * gnetwork_dns_entry_copy:
 * @src: the entry to copy.
 *
 * Copies an existing a #GNetworkDnsEntry. The returned data should be freed
 * with gnetwork_dns_entry_free() when no longer needed.
 *
 * Returns: a copy of @src.
 *
 * Since: 1.0
 **/
GNetworkDnsEntry *
gnetwork_dns_entry_copy (const GNetworkDnsEntry * src)
{
  GSList *list;
  GNetworkDnsEntry *dest;

  g_return_val_if_fail (src != NULL, NULL);

  dest = gnetwork_dns_entry_new ();

  dest->error = src->error;
  dest->hostname = g_strdup (src->hostname);
  dest->aliases = NULL;
  dest->ip_addresses = NULL;

  for (list = src->aliases; list != NULL; list = list->next)
    dest->aliases = g_slist_append (dest->aliases, g_strdup (list->data));

  for (list = src->ip_addresses; list != NULL; list = list->next)
    dest->ip_addresses = g_slist_append (dest->ip_addresses, g_strdup (list->data));

  return dest;
}


GType
gnetwork_dns_entry_get_type (void)
{
  static GType type = 0;

  if (type == 0)
    {
      type = g_boxed_type_register_static ("GNetworkDnsEntry",
					   (GBoxedCopyFunc) gnetwork_dns_entry_copy,
					   (GBoxedFreeFunc) gnetwork_dns_entry_free);
    }

  return type;
}


/**
 * gnetwork_dns_strerror:
 * @error: the DNS error code to use.
 * 
 * Retrieves a string message corresponding to @error. The returned data should
 * not be modified or freed.
 * 
 * Returns: a string message.
 *
 * Since: 1.0
 **/
G_CONST_RETURN gchar *
gnetwork_dns_strerror (GNetworkDnsError error)
{
  static const EnumString msgs[] = {
    {GNETWORK_DNS_ERROR_IN_PROGRESS,
     N_("Search in progress...")},
    {GNETWORK_DNS_ERROR_NONE,
     N_("The host was found")},
    {GNETWORK_DNS_ERROR_NOT_FOUND,
     N_("The host could not be found. The name might be misspelled, or it may not exist.")},
    {GNETWORK_DNS_ERROR_NO_RECOVERY,
     N_("The DNS lookup server could not be contacted. The network may be down, or the DNS server "
	"may be broken.")},
    {GNETWORK_DNS_ERROR_TRY_AGAIN,
     N_("The DNS lookup server is too busy to respond right now, try connecting again in a few "
	"minutes.")},
    {GNETWORK_DNS_ERROR_IPV4_IPV6_MISMATCH,
     N_("The network is misconfigured, contact your network administrator or ISP for more "
	"information.")},
    {GNETWORK_DNS_ERROR_INTERNAL,
     N_("There is a problem with your networking library, please verify that a thread "
	"implementation is installed, and GLib is configured to use it.")},
    {0, NULL}
  };
  guint i;

  g_return_val_if_fail (error >= GNETWORK_DNS_ERROR_IN_PROGRESS &&
			error <= GNETWORK_DNS_ERROR_INTERNAL, NULL);

  for (i = 0; i < G_N_ELEMENTS (msgs); i++)
    {
      if (error == msgs[i].value)
	{
	  return _(msgs[i].str);
	}
    }

  return NULL;
}


G_LOCK_DEFINE_STATIC (quark);

GQuark
gnetwork_dns_error_get_quark (void)
{
  static volatile GQuark quark = 0;

  G_LOCK (quark);

  if (quark == 0)
    {
      quark = g_quark_from_static_string ("gnetwork-dns-error");
    }

  G_UNLOCK (quark);

  return quark;
}
