/* $Id: kbdraw.c,v 1.6 2003/08/18 17:16:36 nlevitt Exp $ */
/*
 * kbdraw.c: main program file for kbdraw
 *
 * Copyright (c) 2003 Noah Levitt
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <gtk/gtk.h>
#include <popt.h>
#include <stdlib.h>
#include <string.h>
#include "keyboard-drawing.h"


static gchar *groups = NULL;
static gchar *levels = NULL;
static gchar *symbols = NULL;
static gchar *geometry = NULL;
static struct poptOption options[] = 
{ 
  { "groups", '\0', POPT_ARG_STRING, &groups, 0, 
    "Keyboard groups to display, from 1-4. One or two groups only may be displayed. Examples: --groups=3 or --groups=1,2", "group1[,group2]" },
  { "levels", '\0', POPT_ARG_STRING, &levels, 0, 
    "Keyboard shift levels to display, from 1-64. One or two shift levels only may be displayed. Examples: --levels=3 or --levels=1,2", "level1[,level2]" },
  { "symbols", '\0', POPT_ARG_STRING, &symbols, 0, 
    "Symbols component of the keyboard. If you omit this option, it is obtained from the X server; that is, the keyboard that is currently configured is drawn. Examples: --symbols=us or --symbols=us(pc104)+iso9995-3+group(switch)+ctrl(nocaps)",
    NULL },
  { "geometry", '\0', POPT_ARG_STRING, &geometry, 0, 
    "Geometry xkb component. If you omit this option, it is obtained from the X server; that is, the keyboard that is currently configured is drawn. Example: --geometry=kinesis",
    NULL },
  { "version", 'v', POPT_ARG_NONE, NULL, 1, NULL, NULL },
  POPT_AUTOHELP
  { NULL, '\0', 0, NULL, 0 }
};


static gboolean
set_groups (gchar *groups_option,
            gint *group1,
            gint *group2)
{
  *group1 = -1;
  *group2 = -1;

  if (groups_option == NULL)
    return TRUE;

  if (strlen (groups_option) != 1 && strlen (groups_option) != 3)
    return FALSE;

  *group1 = groups_option[0] - '1';
  if (*group1 < 0 || *group1 > 3)
    return FALSE;

  if (groups_option[1] == '\0')
    return TRUE;

  if (groups_option[1] != ',')
    return FALSE;

  *group2 = groups_option[2] - '1';
  if (*group2 < 0 || *group1 > 3)
    return FALSE;

  return TRUE;
}


static gboolean
set_levels (gchar *levels_option,
            gint *level1,
            gint *level2)
{
  gchar *p;

  *level1 = -1;
  *level2 = -1;

  if (levels_option == NULL)
    return TRUE;

  if (strlen (levels_option) < 1 || strlen (levels_option) > 5)
    return FALSE;

  *level1 = (gint) strtol (levels_option, &p, 10) - 1;
  if (*level1 < 0 || *level1 >= 64)
    return FALSE;

  if (*p == '\0')
    return TRUE;

  if (*p != ',')
    return FALSE;

  p++;

  *level2 = (gint) strtol (p, &p, 10) - 1;
  if (*level2 < 0 || *level2 >= 64 || *p != '\0')
    return FALSE;

  return TRUE;
}


static void
bad_keycode (KeyboardDrawing *drawing,
             guint keycode)
{
  g_warning ("got keycode %u, which is not on your keyboard according to your configuration", keycode);
}


gint
main (gint argc, gchar **argv)
{
  GtkWidget *window;
  GtkWidget *keyboard_drawing;
  GdkScreen *screen;
  gint monitor;
  GdkRectangle rect;
  poptContext popt_context;
  gint rc;
  gint group1, group2;
  gint level1, level2;

  gtk_init (&argc, &argv);

  popt_context = poptGetContext ("kbdraw", argc, (const gchar **) argv, options, 0);

  for (rc = poptGetNextOpt (popt_context);  rc > 0;  rc = poptGetNextOpt (popt_context))
    {
      if (rc == 1)
        {
          g_print ("kbdraw %s\n", VERSION);
          exit (0);
        }
    }

  if (rc != -1)
    {
       g_printerr ("%s: %s\n", 
                   poptBadOption (popt_context, POPT_BADOPTION_NOALIAS),
                   poptStrerror (rc));
       exit (1);
    }

  if (! set_groups (groups, &group1, &group2))
    {
       g_printerr ("--groups: invalid argument\n");
       exit (1);
    }

  if (! set_levels (levels, &level1, &level2))
    {
       g_printerr ("--levels: invalid argument\n");
       exit (1);
    }

  if (group1 == -1 && group2 == -1)
    {
      group1 = 0;
      group2 = 1;
    }

  if (level1 == -1 && level2 == -1)
    {
      level1 = 0;
      level2 = 1;
    }

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  g_signal_connect (G_OBJECT (window), "destroy",
                    G_CALLBACK (gtk_main_quit), NULL);

  screen = gtk_window_get_screen (GTK_WINDOW (window));
  monitor = gdk_screen_get_monitor_at_point (screen, 0, 0);
  gdk_screen_get_monitor_geometry (screen, monitor, &rect);
  gtk_window_set_default_size (GTK_WINDOW (window),
                               rect.width * 4/5,
                               rect.height * 1/2);

  gtk_widget_show (window);

  keyboard_drawing = keyboard_drawing_new ();
  gtk_widget_show (keyboard_drawing);
  gtk_container_add (GTK_CONTAINER (window), keyboard_drawing);
  keyboard_drawing_set_groups (KEYBOARD_DRAWING (keyboard_drawing), group1, group2);
  keyboard_drawing_set_levels (KEYBOARD_DRAWING (keyboard_drawing), level1, level2);
  g_signal_connect (G_OBJECT (keyboard_drawing), "bad-keycode",
                    G_CALLBACK (bad_keycode), NULL);

  if (symbols || geometry)
    {
      XkbComponentNamesRec names;
      gint success;

      memset (&names, '\0', sizeof (names));
      names.keycodes = (gchar *) keyboard_drawing_get_keycodes (KEYBOARD_DRAWING (keyboard_drawing));
      names.types = (gchar *) keyboard_drawing_get_types (KEYBOARD_DRAWING (keyboard_drawing));
      names.compat = (gchar *) keyboard_drawing_get_compat (KEYBOARD_DRAWING (keyboard_drawing));

      if (symbols)
        names.symbols = symbols;
      else
        names.symbols = (gchar *) keyboard_drawing_get_symbols (KEYBOARD_DRAWING (keyboard_drawing));

      if (geometry)
        names.geometry = geometry;
      else
        names.geometry = (gchar *) keyboard_drawing_get_geometry (KEYBOARD_DRAWING (keyboard_drawing));

      success = keyboard_drawing_set_keyboard (KEYBOARD_DRAWING (keyboard_drawing), &names);
      if (!success)
        {
          g_printerr ("\nError loading new keyboard description with components:\n\n"
                      "  keycodes:  %s\n"
                      "  types:     %s\n"
                      "  compat:    %s\n"
                      "  symbols:   %s\n"
                      "  geometry:  %s\n\n",
                      names.keycodes, names.types, names.compat, names.symbols, names.geometry);
          exit (1);
        }
    }

  gtk_widget_grab_focus (keyboard_drawing);

  gtk_main ();

  return 0;
}
