/*
 * Copyright (C) 2003-2004 Imendio HB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <math.h>
#include <glib/gi18n.h>
#include <gst/gst.h>
#include <gst/gconf/gconf.h>
#include <gst/control/dparam_smooth.h>
#include <gst/control/control.h>
#include "jamboree-marshal.h"
#include "utils.h"
#include "types.h"
#include "player.h"

/* Only used for the esd hack below. */
#include <esd.h>

#define TICK_TIMEOUT 250
#define d(x) 

static void     player_class_init   (PlayerClass  *klass);
static void     player_init         (Player       *player);
static void     player_finalize     (GObject      *object);
static void     player_update_state (void);
static gboolean player_setup        (GError      **error);
static gboolean tick_timeout_cb     (void);
static void     start_tick          (void);
static void     stop_tick           (void);
static void     timer_start         (void);
static void     timer_pause         (void);
static void     timer_stop          (void);
static int      timer_get_time      (void);
static gboolean workaround_esd      (const char *sink_name);
static char    *get_sink_name       (void);

enum {
	EOS,
	TICK,
	STATE_CHANGED,
	ERROR,
	LAST_SIGNAL
};

struct _Player {
	GObject        parent;

	GstElement    *thread;
	GstElement    *volume;
	GstElement    *sink;

	int            volume_level;
	GstDParam     *volume_dparam;

	gboolean       playing;
	Song          *current_song;

	GList         *songs;
	GList         *current;
	
	guint          tick_timeout_id;

	GTimer        *timer;
	int            timer_add;

	gboolean       has_error;
};

static guint signals[LAST_SIGNAL];
static Player *player = NULL;

G_DEFINE_TYPE (Player, player, G_TYPE_OBJECT);


static void
player_class_init (PlayerClass *klass)
{
	GObjectClass *object_class;

	object_class = (GObjectClass*) klass;

	object_class->finalize = player_finalize;

	signals[EOS] =
		g_signal_new ("eos",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      jamboree_marshal_VOID__BOXED,
			      G_TYPE_NONE, 1, TYPE_SONG);
	
	signals[TICK] =
		g_signal_new ("tick",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      jamboree_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);
	
	signals[STATE_CHANGED] =
		g_signal_new ("state_changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      jamboree_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);
	
	signals[ERROR] =
		g_signal_new ("error",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      jamboree_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1, G_TYPE_POINTER);
}

static void
player_init (Player *instance)
{
	player = instance;
	
	player->timer = g_timer_new ();
	timer_stop ();
	
	player->volume_dparam = gst_dpsmooth_new (G_TYPE_DOUBLE);

	g_object_set (player->volume_dparam,
		      "slope_time", (gint64) 1000000000,
		      "slope_delta_float", (float) 1.0,
		      NULL);
}

static void
player_finalize (GObject *object)
{
	Player *instance;

	instance = PLAYER (object);

	gconf_client_set_int (gconf_client,
			      GCONF_PATH "/volume",
			      instance->volume_level,
			      NULL);

	stop_tick ();

	G_OBJECT_CLASS (player_parent_class)->finalize (object);
}

GQuark
player_error_quark (void)
{
	static GQuark quark = 0;
	if (!quark) {
		quark = g_quark_from_static_string ("player_error");
	}
	
	return quark;
}

PlayingState
player_get_state (void)
{
	if (!player->thread) {
		return PLAYING_STATE_STOPPED;
	}
	
	if (player->playing) {
		return PLAYING_STATE_PLAYING;
	}
	
	return PLAYING_STATE_PAUSED;
}

static gboolean
eos_idle_cb (void)
{
	d(g_print ("eos_idle_cb\n"));

	if (player->has_error) {
		return FALSE;
	}
	
	stop_tick ();
		
	g_signal_emit (player, signals[TICK], 0, (int) 0);
	g_signal_emit (player, signals[EOS], 0, player->current_song, NULL);

	return FALSE;
}

static void
eos_cb (GstElement *element, gpointer data)
{
	g_idle_add_full (G_PRIORITY_HIGH, (GSourceFunc) eos_idle_cb, player, NULL);
}

static gboolean
error_idle_cb (GError *error)
{
	d(g_print ("error_idle_cb\n"));
	
	if (!player->has_error) {
		return FALSE;
	}
		
	player_stop ();

	g_signal_emit (player, signals[TICK], 0, (int) 0);
	g_signal_emit (player, signals[ERROR], 0, error);

	g_error_free (error);

	return FALSE;
}

static void
error_cb (GObject    *object,
	  GstElement *origin,
	  GError     *error,
	  char       *debug,
	  Player     *player)
{
	GError *new_error;
	
	if (player->has_error) {
		return;
	}

	if (g_getenv ("JAM_DEBUG_GST")) {
		g_print ("error origin: %p, domain: %d, msg: %s\n", origin, error->domain, error->message);
	}	

	if (origin == player->sink && error->domain == GST_RESOURCE_ERROR) {
		new_error = g_error_new (PLAYER_ERROR,
					 PLAYER_ERROR_RESOURCE_BUSY,
					 _("The audio device is busy."));
	} else {
		new_error = g_error_copy (error);
	}

	player->has_error = TRUE;

	g_idle_add_full (G_PRIORITY_HIGH, (GSourceFunc) error_idle_cb, new_error, NULL);
}

static gboolean 
tick_timeout_cb (void)
{
	if (!player->has_error && player->playing && player->sink) {
		if (gst_element_get_state (player->sink) == GST_STATE_PLAYING) {
			g_signal_emit (player, signals[TICK], 0, (int) timer_get_time ());
		}
	}
	
	return TRUE;
}
                                                                                
gboolean
player_set_song (Song *song, GError **error)
{
	gboolean was_playing;
	gboolean success;
  
	was_playing = player->playing;
  
	if (!song) {
		player_stop ();
		return TRUE;
	}

	player->current_song = song;
	success = player_setup (error);
  
	player->playing = was_playing && success;
	player_update_state ();

	return success;
}

gboolean
player_play (GError **error)
{
	if (!player->current_song) {
		return FALSE;
	}
  
	if (!player->thread) {
		return player_play_song (player->current_song, error);
	}
	
	player->playing = TRUE;

	player_update_state ();

	return TRUE;
}

gboolean
player_play_song (Song    *song,
		  GError **error)
{
	player->current_song = song;

	if (song) {
		player->playing = player_setup (error);
	} else {
		player->playing = FALSE;
	}
	
	player_update_state ();

	return player->playing;
}

void
player_stop (void)
{
	gboolean    emit;
	GstElement *source;

	stop_tick ();
	timer_stop ();
  
	emit = player->playing;
	player->playing = FALSE;
	
	if (player->thread) {
		/* Reset the location so we close the file as soon as possible
		 * if something below fails. This seems to help against Jamboree
		 * keeping lots of played files open indefinitely.
		 */
		source = gst_bin_get_by_name (GST_BIN (player->thread), "source");
		if (source) {
			g_object_set (source, "location", NULL, NULL);
		}
		
		gst_element_set_state (player->thread, GST_STATE_NULL);
			
		gst_object_unref (GST_OBJECT (player->thread));
		
		player->thread = NULL;
		player->volume = NULL;
		player->sink = NULL;

		/* If we had a thread, we're either playing or paused. */
		emit = TRUE;
	}

	if (emit) {
		g_signal_emit (player, signals[STATE_CHANGED], 0, PLAYING_STATE_STOPPED);
	}
}

void
player_pause (void)
{
	player->playing = FALSE;
	player_update_state ();
}

void
player_set_volume (int volume)
{
	double d;

	g_return_if_fail (volume >= 0 && volume <= 100);

	player->volume_level = volume;
   
	d = volume / 100.0;

	if (player->volume_dparam) {
		g_object_set (player->volume_dparam, "value_double", d, NULL);
	}
}

int
player_get_volume (void)
{
	return player->volume_level;
}

/* t is in milliseconds. */
void
player_seek (int t)
{
	if (!player->thread || !player->sink) {
		return;
	}
  
	if (gst_element_set_state (player->thread, GST_STATE_PAUSED) != GST_STATE_SUCCESS) {
		return;
	}

	timer_stop ();

	gst_element_seek (player->sink, GST_SEEK_METHOD_SET | GST_FORMAT_TIME, t * GST_SECOND);

	if (player->playing) {
		gst_element_set_state (player->thread, GST_STATE_PLAYING);

		timer_start ();
	}

	player->timer_add = t;
}

/* Return value is in seconds. */
int
player_tell (void)
{
	if (!player->has_error && player->playing && player->sink) {
		return timer_get_time ();
	}

	return 0;
}

gboolean
player_is_playing (Song *song)
{
	g_return_val_if_fail (song != NULL, FALSE);

	return player->playing && song == player->current_song;
}

Song *
player_get_song (void)
{
	return player->current_song;
}

static void
player_update_state (void)
{
	if (player->playing) {
		gst_element_set_state (player->thread, GST_STATE_PLAYING);
		timer_start ();
		start_tick ();
	} else {
		stop_tick ();
      
		if (player->thread) {
			gst_element_set_state (player->thread, GST_STATE_PAUSED);
			gst_element_set_state (player->sink, GST_STATE_NULL);

			timer_pause ();
		} else {
			timer_stop ();
		}
	}
}

static GstElement *
create_pipeline (const char *sink_name, GError **error)
{
	GstElement *thread;
	GstElement *source;
	GstElement *spider;
	GstElement *volume;
	GstElement *audioconvert;
	GstElement *audioscale;
	GstElement *sink;

	/* src -> spider -> volume -> audioconvert -> audioscale -> sink */
	
	thread = gst_element_factory_make ("thread", "thread");
	source = gst_element_factory_make ("filesrc", "source");
	spider = gst_element_factory_make ("spider", "decoder");

	audioconvert = gst_element_factory_make ("audioconvert", "audioconvert");
	audioscale = gst_element_factory_make ("audioscale", "audioscale");
  
	volume = gst_element_factory_make ("volume", "volume");

	sink = gst_element_factory_make (sink_name, "sink");
	
	if (g_getenv ("JAM_DEBUG_GST")) {
		g_print ("thread:       %s\n", thread ? "OK" : "failed");
		g_print ("source:       %s\n", source ? "OK" : "failed");
		g_print ("spider:       %s\n", spider ? "OK" : "failed");
		g_print ("audioconvert: %s\n", audioconvert ? "OK" : "failed");
		g_print ("audioscale:   %s\n", audioscale ? "OK" : "failed");
		g_print ("volume:       %s\n", volume ? "OK" : "failed");
		g_print ("sink:         %s (%s)\n", sink ? "OK" : "failed", sink_name);
	}
	
	if (!thread || !source || !spider || !audioconvert || !audioscale || !volume || !sink) {
		goto fail;
	}
  
	gst_bin_add_many (GST_BIN (thread),
			  source, spider, volume, audioscale, audioconvert, sink,
			  NULL);

	if (!gst_element_link_many (source, spider, volume, audioconvert, audioscale, sink, NULL)) {
		goto fail;
	}
	
	return thread;
  
 fail:
	/* This is pretty much fatal so don't bother too much with cleaning
	 * up.
	 */

	if (thread) {
		g_object_unref (thread);
	}
	
	g_set_error (error,
		     PLAYER_ERROR,
		     PLAYER_ERROR_INTERNAL,
		     _("Internal error, please check your GStreamer installation."));
  
	return NULL;
}

static gboolean
player_setup (GError **error)
{
	char             *sink_name;
	GstElement       *source;
	GstDParamManager *dpman;

	player->has_error = FALSE;

	if (player->thread) {
		player_stop ();
	}
	
	if (!player->current_song) {
		return FALSE;
	}

	if (!g_str_has_suffix (song_get_path (player->current_song), ".mp3") &&
	    !g_str_has_suffix (song_get_path (player->current_song), ".MP3") &&
	    !g_str_has_suffix (song_get_path (player->current_song), ".ogg") &&
	    !g_str_has_suffix (song_get_path (player->current_song), ".OGG")) {
		g_set_error (error,
			     PLAYER_ERROR,
			     PLAYER_ERROR_FORMAT,
			     _("Unrecognized music format."));
		
		goto bail;
	}

	sink_name = get_sink_name ();
	if (!workaround_esd (sink_name)) {
		g_set_error (error, PLAYER_ERROR,
			     PLAYER_ERROR_RESOURCE_BUSY,
			     _("The audio device is busy."));

		return FALSE;
	}

	player->thread = create_pipeline (sink_name, error);

	g_free (sink_name);
      
	if (!player->thread) {
		goto bail;
	}
 
	source = gst_bin_get_by_name (GST_BIN (player->thread), "source");
	g_object_set (source, "location", song_get_path (player->current_song), NULL);

	g_signal_connect (player->thread,
			  "error",
			  G_CALLBACK (error_cb),
			  player);

	player->volume = gst_bin_get_by_name (GST_BIN (player->thread), "volume");
	dpman = gst_dpman_get_manager (player->volume);
	gst_dpman_set_mode (dpman, "synchronous");
	gst_dpman_attach_dparam (dpman, "volume", player->volume_dparam);
	player_set_volume (player->volume_level);
  
	player->sink = gst_bin_get_by_name (GST_BIN (player->thread), "sink");
	g_signal_connect (player->sink,
			  "eos",
			  G_CALLBACK (eos_cb),
			  player);
  
	player_update_state ();

	return TRUE;

 bail:

	player_stop ();

	return FALSE;
}

static void
start_tick (void)
{
	if (player->tick_timeout_id) {
		return;
	}
  
	player->tick_timeout_id = g_timeout_add (TICK_TIMEOUT,
					       (GSourceFunc) tick_timeout_cb,
					       player);
}

static void
stop_tick (void)
{
	if (!player->tick_timeout_id) {
		return;
	}
  
	g_source_remove (player->tick_timeout_id);
	player->tick_timeout_id = 0;
}

static void
timer_start (void)
{
	g_timer_reset (player->timer);
	g_timer_start (player->timer);
}

static void
timer_pause (void)
{
	player->timer_add = player->timer_add + floor (0.5 + g_timer_elapsed (player->timer, NULL));
  
	g_timer_stop (player->timer);
	g_timer_reset (player->timer);
}

static void
timer_stop (void)
{
	g_timer_stop (player->timer);
	g_timer_reset (player->timer);

	player->timer_add = 0;
}

static int
timer_get_time (void)
{
	return player->timer_add + floor (0.5 + g_timer_elapsed (player->timer, NULL));
}

Player *
player_get (void)
{
	if (!player) {
		g_object_new (TYPE_PLAYER, NULL);
	}   
	
	return player;
}

void
player_shutdown (void)
{
	if (player) {
		g_object_unref (player);
		player = NULL;
	}
}

/* Hacky workaround for esd/esdsink suckyness. */
static gboolean
workaround_esd (const char *sink_name)
{
	static int esd_fd;

	if (strcmp (sink_name, "esdsink") == 0) {
		if (!esd_fd) {
			d(g_print ("hack: opening esd connection\n"));
			esd_fd = esd_open_sound (NULL);

			if (esd_fd == -1) {
				esd_fd = 0;
				return FALSE;
			}
		}
	}
	else if (esd_fd) {
		d(g_print ("hack: closing esd connection\n"));
		esd_close (esd_fd);
		esd_fd = 0;
	}
	
	return TRUE;
}

static char *
get_sink_name (void)
{
	char *sink_name;

	sink_name = gst_gconf_get_string ("default/audiosink");
  	if (!sink_name) {
		g_warning ("GStreamer gconf setup broken, please check your installation. "
			   "Falling back to OSS output.");
		sink_name = g_strdup ("osssink");
	}

	return sink_name;
}

