/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003-2004 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003      Anders Carlsson <andersca@gnome.org>
 * Copyright (C) 2003      Johan Dahlin <johan@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <vorbis/vorbisfile.h>
#include <glib/gi18n.h>
#include "song-db.h"
#include "song.h"
#include "id3-tag.h"
#include "mp3bitrate.h"

#define VERSION_KEY "jamboree-version"
#define VERSION_KEY_LENGTH (16)

#define _ALIGN_VALUE(this, boundary) \
  (( ((unsigned long)(this)) + (((unsigned long)(boundary)) -1)) & (~(((unsigned long)(boundary))-1)))
	
#define _ALIGN_ADDRESS(this, boundary) \
  ((void*)_ALIGN_VALUE(this, boundary))


static void song_db_class_init (SongDBClass *klass);
static void song_db_init       (SongDB      *db);
static void song_db_finalize   (GObject     *object);


static GObjectClass *parent_class;


GType
song_db_get_type (void)
{
  static GType type = 0;
	
  if (!type)
    {
      static const GTypeInfo info =
	{
	  sizeof (SongDBClass),
	  NULL,           /* base_init */
	  NULL,           /* base_finalize */
	  (GClassInitFunc) song_db_class_init,
	  NULL,           /* class_finalize */
	  NULL,           /* class_data */
	  sizeof (SongDB),
	  0,
	  (GInstanceInitFunc) song_db_init,
	};

      type = g_type_register_static (G_TYPE_OBJECT,
				     "SongDB",
				     &info, 0);
    }

  return type;
}

static void
song_db_class_init (SongDBClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass*) klass;

  object_class->finalize = song_db_finalize;
}

static void
song_db_init (SongDB *db)
{
}

static void
song_db_finalize (GObject *object)
{
  SongDB *db = SONG_DB (object);

  g_list_foreach (db->songs, (GFunc) song_free, NULL);
  g_list_free (db->songs);

  if (db->dbf)
    gdbm_close (db->dbf);
  
  if (G_OBJECT_CLASS (parent_class)->finalize)
    (* G_OBJECT_CLASS (parent_class)->finalize) (object);
}

/* Note: str is set to NULL for empty strings. */
static gpointer
unpack_string (gpointer p, char **str, gpointer endp)
{
  int len;
  
  p = _ALIGN_ADDRESS (p, 4);
  
  len = *(int *)p;

  p += 4;

  /* Try to catch corrup database :/ */
  if (p + len > endp)
    {
      g_warning ("String length seems to be corrupt.");
      if (*str)
	*str = NULL;
      
      return endp + 1;
    }
      
  if (len == 0)
    {
      if (str)
	*str = NULL;
      return p;      
    }
  
  if (str != NULL)
    {
      *str = g_malloc (len + 1);

      memcpy (*str, p, len);
      (*str)[len] = 0;
    }
  
  return p + len + 1;
}

static gpointer
unpack_int (gpointer p, int *val)
{
  p = _ALIGN_ADDRESS (p, 4);

  if (val)
    *val = *(int *)p;
   
  p += 4;
  
  return p;
}

static gpointer
unpack_uint64 (gpointer p, guint64 *val)
{
  p = _ALIGN_ADDRESS (p, 8);

  if (val)
    *val = *(guint64 *)p;

  p += 8;

  return p;
}

static gpointer
unpack_playlists (gpointer p, GList **playlists, gpointer endp)
{
  char *str;
  char **strv;
  int i = 0;
  int id;

  p = unpack_string (p, &str, endp);
  if (p > endp)
    {
      g_free (str);
      return endp + 1;
    }

  if (str == NULL)
    return p;
  
  strv = g_strsplit (str, ",", -1);

  while (strv[i])
    {
      id = atoi (strv[i]);
      *playlists = g_list_append (*playlists, GINT_TO_POINTER (id));

      i++;
    }

  g_free (str);
  g_strfreev (strv);

  return p;
}

static Song *
unpack_song (gpointer p, int len)
{
  gpointer end;
  Song *song;
  char *str;
  
  end = p + len;
  
  song = g_new0 (Song, 1);

  p = unpack_string (p, &str, end);
  if (p > end)
    goto fail;
  song->title = string_entry_add (str);
  
  p = unpack_string (p, &str, end);
  if (p > end)
    goto fail;
  song->artist = string_entry_add (str);
  
  p = unpack_string (p, &str, end);
  if (p > end)
    goto fail;
  song->album = string_entry_add (str);

  p = unpack_int (p, &song->genre);
  p = unpack_int (p, &song->year);
  p = unpack_int (p, &song->length); 
  p = unpack_int (p, &song->bitrate);
  p = unpack_int (p, &song->samplerate);
  p = unpack_int (p, &song->track_number);
  p = unpack_int (p, (int*)&song->date_added);
  p = unpack_int (p, (int*)&song->date_modified);
  p = unpack_int (p, &song->rating);
  p = unpack_int (p, &song->play_count);
  p = unpack_int (p, (int*)&song->last_played);
  p = unpack_uint64 (p, &song->filesize);
  p = unpack_playlists (p, &song->playlists, end);
  if (p > end)
    goto fail;

  return song;

 fail:
  song_free (song);
  return NULL;
}

static void
string_align (GString *string, int boundary)
{
  gpointer p;
  int padding;
  int i;

  p = string->str + string->len;

  padding = _ALIGN_ADDRESS (p, boundary) - p;

  for (i = 0; i < padding; i++)
    g_string_append_c (string, 0);
}

static void
pack_int (GString *string, int val)
{
  string_align (string, 4);

  g_string_append_len (string, (char *)&val, 4);
}

static void
pack_uint64 (GString *string, guint64 val)
{
  string_align (string, 8);

  g_string_append_len (string, (char *)&val, 8);
}

static void
pack_string (GString *string, const char *str)
{
  int len;
  
  if (str)
    len = strlen (str);
  else
    len = 0;
  
  pack_int (string, len);

  if (len > 0)
    {
      g_string_append (string, str);
      g_string_append_c (string, 0);
    }
}

static void
pack_playlists (GString *string, GList *playlists)
{
  GString *str;
  GList *l;

  str = g_string_new (NULL);

  for (l = playlists; l; l = l->next)
    {
      g_string_append_printf (str, "%d", GPOINTER_TO_INT (l->data));
      g_string_append_c (str, ',');
    }

  pack_string (string, str->str);

  g_string_free (str, TRUE);
}

gpointer
_song_db_pack_song (Song *song, int *len)
{
  GString *string;

  string = g_string_new ("");

  pack_string (string, string_entry_get_str (song->title));
  pack_string (string, string_entry_get_str (song->artist));
  pack_string (string, string_entry_get_str (song->album));

  pack_int (string, song->genre);
  pack_int (string, song->year);
  pack_int (string, song->length);
  pack_int (string, song->bitrate);
  pack_int (string, song->samplerate);
  pack_int (string, song->track_number);
  pack_int (string, song->date_added);
  pack_int (string, song->date_modified);
  pack_int (string, song->rating);
  pack_int (string, song->play_count);
  pack_int (string, song->last_played); 
  pack_uint64 (string, song->filesize);
  pack_playlists (string, song->playlists);

  if (len)
    *len = string->len;
  
  return g_string_free (string, FALSE);
}

static void
read_songs (SongDB *db)
{
  datum key, next_key, data;
  Song *song;
  
  memset (&key, 0, sizeof (key));
  memset (&data, 0, sizeof (data));

  key = gdbm_firstkey (db->dbf);
  while (key.dptr)
    {
      next_key = gdbm_nextkey (db->dbf, key);
      
      if (((char*)key.dptr)[0] == VERSION_KEY[0] && strncmp (key.dptr, VERSION_KEY, VERSION_KEY_LENGTH) == 0)
	{
	  free (key.dptr);

	  key = next_key;
	  continue;
	}
      
      data = gdbm_fetch (db->dbf, key);

      if (data.dptr == NULL)
	{
	  free (key.dptr);

	  key = next_key;
	  continue;
	}

      song = unpack_song (data.dptr, data.dsize);
      if (song != NULL)
	{	
	  song->filename = g_strndup (key.dptr, key.dsize);
	  db->songs = g_list_prepend (db->songs, song);
	}
      else
	{
	  gchar *tmp;

	  tmp = g_strndup (key.dptr, key.dsize);
	  g_warning ("Corrupt db, skipping song %s\n", tmp);
	  g_free (tmp);
	}
      
      free (key.dptr);
      free (data.dptr);

      key = next_key;
    }
}

SongDB *
song_db_new (const char *filename, gboolean readonly)
{
  SongDB *db;
  int rw_flag;

  db = g_object_new (TYPE_SONG_DB, NULL);

  if (readonly)
    {
      rw_flag = GDBM_READER;
      db->readonly = TRUE;
    }
  else
    rw_flag = GDBM_WRCREAT;

  db->dbf = gdbm_open ((char*) filename, 4096, rw_flag | GDBM_SYNC, 04644, NULL);
  if (!db->dbf)
    goto fail;
  
  return db;
  
 fail:
  
  g_object_unref (db);
  return NULL;
}

void
song_db_read_songs (SongDB *db)
{
  g_return_if_fail (IS_SONG_DB (db));

  g_list_foreach (db->songs, (GFunc) song_free, NULL);
  g_list_free (db->songs);
  db->songs = NULL;
  
  read_songs (db);
}

/*
static int
unpack_int32 (const unsigned char *data)
{
  return (data[0] << 24) +
    (data[1] << 16) +
    (data[2] << 8) +
    data[3];
}
*/

static char *
get_title_from_filename (const char *filename)
{
  char *utf8;
  char *tmp;
  char *p;
  gunichar c;
  
  utf8 = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (utf8)
    {
      tmp = g_path_get_basename (utf8);
      g_free (utf8);

      if (g_str_has_suffix (tmp, ".mp3") || g_str_has_suffix (tmp, ".ogg"))
	{
	  p = strrchr (tmp, '.'); 
	  if (p)
	    *p = 0;
	}
      
      p = tmp;
      while (*p)
	{
	  c = g_utf8_get_char (p);
	  
	  if (c == '_')
	    *p = ' ';
	  
	  p = g_utf8_next_char (p);
	}
      
      return tmp;
    }

  return NULL;
}

static void
assign_song_info_mp3 (Song *song)
{
  FILE *file;
  struct id3_file *id3_file;
  struct id3_tag *tag;
  guchar buf[8192];
  int i, bytes_read;
  int bitrate = 0, samplerate = 0, time = 0, version = 0, vbr = 0, channels = 0;
  int tries = 0;
  const char *title = NULL, *artist = NULL, *album = NULL;

  file = fopen (song->filename, "r");
  if (!file)
    goto no_info;
  
  id3_file = id3_file_fdopen (fileno (file), ID3_FILE_MODE_READONLY);
  if (id3_file)
    {
      tag = id3_file_tag (id3_file);
      if (tag)
	{
	  title = id3_tag_get_title (tag);
	  artist = id3_tag_get_artist (tag);
	  album = id3_tag_get_album (tag);

	  song->genre = id3_tag_get_genre (tag);
	  song->year = id3_tag_get_year (tag);
	  song->track_number = id3_tag_get_number (tag);
	}
    }

  while (tries < 2)
    {
      bytes_read = fread (buf, 1, sizeof (buf), file);
      if (bytes_read < 8192)
	break;
      
      for (i = 0; i + 4 < bytes_read; i++)
	{
	  if (mp3_bitrate_parse_header (buf + i, bytes_read - i,
					&bitrate, &samplerate,
					&time, &version,
					&vbr, &channels))
	    goto done;
	}

      tries++;
    }
  
 done:

  song->bitrate = bitrate;
  song->samplerate = samplerate;

  if (!vbr)
    {
      if (song->bitrate > 0)
	song->length = ((double) song->filesize) / ((double) song->bitrate / 8000.0f);
      else
	song->length = 0;
    }
  else
    song->length = time;
      
  if (id3_file)
    id3_file_close (id3_file);
  
  fclose (file);

 no_info:
  
  if (title == NULL || title[0] == 0)
    song->title = string_entry_add (get_title_from_filename (song->filename));
  else 
    song->title = string_entry_add_const (title);

  if (artist == NULL || artist[0] == 0)
    artist = _("Unknown");
  song->artist = string_entry_add_const (artist);
  
  if (album == NULL || album[0] == 0)
    album = _("Unknown");
  song->album = string_entry_add_const (album);
}

static void
assign_song_info_ogg (Song *song)
{
  FILE *f;
  OggVorbis_File vf;
  vorbis_comment *comment;
  vorbis_info *info;
  int i;
  const char *title = NULL, *artist = NULL, *album = NULL;
  gboolean clear = FALSE;
  
  f = fopen (song->filename, "r");
  if (!f)
    goto no_info;
  
  if (ov_open (f, &vf, NULL, 0) < 0)
    {
      fclose (f);
      goto no_info;
    }

  comment = ov_comment (&vf, -1);
  if (!comment)
    {
      ov_clear (&vf);
      goto no_info;
    }

  clear = TRUE;

  song->genre = -1;
  
  for (i = 0; i < comment->comments; i++)
    {
      if (title == NULL && strncasecmp (comment->user_comments[i], "title=", 6) == 0)
	title = comment->user_comments[i] + 6;
      else if (artist == NULL && strncasecmp (comment->user_comments[i], "artist=", 7) == 0)
	artist = comment->user_comments[i] + 7;
      else if (album == NULL && strncasecmp (comment->user_comments[i], "album=", 6) == 0)
	album = comment->user_comments[i] + 6;
      else if (strncasecmp (comment->user_comments[i], "tracknumber=", 12) == 0)
	song->track_number = strtol (comment->user_comments[i] + 12, NULL, 10);
      else if (strncasecmp (comment->user_comments[i], "date=", 5) == 0)
	song->year = strtol (comment->user_comments[i] + 5, NULL, 10);
    }

  info = ov_info (&vf, -1);
  
  song->length = ov_time_total (&vf, -1) * 1000;
  song->bitrate = info->bitrate_nominal;
  song->samplerate = info->rate;

 no_info:
  
  if (title == NULL || title[0] == 0)
    song->title = string_entry_add (get_title_from_filename (song->filename));
  else 
    song->title = string_entry_add_const (title);

  if (artist == NULL || artist[0] == 0)
    artist = _("Unknown");
  song->artist = string_entry_add_const (artist);
  
  if (album == NULL || album[0] == 0)
    album = _("Unknown");
  song->album = string_entry_add_const (album);

  if (clear)
    ov_clear (&vf);
}

static gboolean
add_song (SongDB* db, const char *filename, gboolean overwrite)
{
  datum key, data;
  int ret;
  gpointer p;
  gsize len;
  Song *song;
  struct stat buf;
  time_t now;

  if (db->readonly)
    return FALSE;
  
  if (access (filename, R_OK))
    return FALSE;
  
  song = g_new0 (Song, 1);
  song->filename = g_strdup (filename);

  memset (&key, 0, sizeof (key));
  key.dptr = song->filename;
  key.dsize = strlen (key.dptr);

  if (!overwrite)
    if (gdbm_exists (db->dbf, key))
      {
	g_free (song->filename);
	g_free (song);
	return FALSE;
      }

  if (stat (filename, &buf) == 0)
    song->filesize = buf.st_size;
  else
    song->filesize = 0;
  
  if (g_str_has_suffix (filename, ".mp3"))
    assign_song_info_mp3 (song);    
  else if (g_str_has_suffix (filename, ".ogg"))
    assign_song_info_ogg (song);

  now = time (NULL);
  song->date_added = now;
  song->date_modified = now; 
  
  p = _song_db_pack_song (song, &len);

  memset (&data, 0, sizeof (data));
  data.dptr = p;
  data.dsize = len;

  ret = gdbm_store (db->dbf, key, data, overwrite ? GDBM_REPLACE : GDBM_INSERT);
  if (ret == 0)
    db->songs = g_list_prepend (db->songs, song);

  g_free (p);

  return ret == 0;
}

static gboolean
add_dir (SongDB                *db,
	 const char            *path,
	 SongDBAddProgressFunc  progress_callback,
	 gpointer               user_data)
{
  GDir *dir;
  const char *name;
  char *full;

  dir = g_dir_open (path, 0, NULL);

  if (!dir)
    return FALSE;
  
  while ((name = g_dir_read_name (dir)))
    {
      full = g_build_filename (path, name, NULL);
      
      if (g_file_test (full, G_FILE_TEST_IS_DIR))
	add_dir (db, full, progress_callback, user_data);
      else
	{
	  /* FIXME: Perhaps sniff this info from the song data. */
	  if (g_str_has_suffix (name, ".mp3") || g_str_has_suffix (name, ".ogg"))
	    {
	      if (progress_callback)
		if (!progress_callback (db, full, user_data))
		  return FALSE;

	      add_song (db, full, FALSE);
	    }
	}

      g_free (full);
    }

  g_dir_close (dir);

  return TRUE;
}

void
song_db_add_dir (SongDB                *db,
		 const char            *path,
		 SongDBAddProgressFunc  progress_callback,
		 gpointer               user_data)
{
  if (g_file_test (path, G_FILE_TEST_IS_DIR))
    add_dir (db, path, progress_callback, user_data);
  else if (g_file_test (path, G_FILE_TEST_IS_REGULAR))
    {
      if (progress_callback)
	if (!progress_callback (db, path, user_data))
	  return;

      add_song (db, path, FALSE);
    }
  else
    {
      g_warning ("Don't know how to handle: %s", path);
      return;
    }
}

void
song_db_add_file (SongDB     *db,
		  const char *filename)
{
  g_return_if_fail (IS_SONG_DB (db));

  add_song (db, filename, FALSE);
}

void
song_db_update_song (SongDB *db, Song *song)
{
  int ret;
  datum key, data;
  gpointer p;
  int len;
  
  g_return_if_fail (IS_SONG_DB (db));
  g_return_if_fail (song != NULL);

  song->date_modified = time (NULL);

  if (db->readonly)
    return;
  
  p = _song_db_pack_song (song, &len);

  memset (&key, 0, sizeof (key));
  key.dptr = song->filename;
  key.dsize = strlen (key.dptr);
  
  memset (&data, 0, sizeof (data));
  data.dptr = p;
  data.dsize = len;

  ret = gdbm_store (db->dbf, key, data, GDBM_REPLACE);

  g_free (p);
}

gboolean
song_db_remove_song (SongDB *db,
		     Song   *song)
{
  datum key;
  int ret;
    
  g_return_val_if_fail (IS_SONG_DB (db), FALSE);

 if (!db->readonly)
   {
     memset (&key, 0, sizeof (key));
     key.dptr = song->filename;
     key.dsize = strlen (key.dptr);
     
     ret = gdbm_delete (db->dbf, key);
   }
 
  db->songs = g_list_remove (db->songs, song);
  
  song_free (song);
  
  return TRUE;
}

int
song_db_get_version (SongDB *db)
{
  datum key, data;
  int ret;

  g_return_val_if_fail (IS_SONG_DB (db), -1);

  memset (&key, 0, sizeof (key));
  key.dptr = VERSION_KEY;
  key.dsize = strlen (key.dptr);
  
  data = gdbm_fetch (db->dbf, key);
  if (data.dptr == NULL)
    return -1;
  
  unpack_int (data.dptr, &ret);

  free (data.dptr);
  
  return ret;
}

void
song_db_set_version (SongDB *db, int version)
{
  GString *string;
  datum key, data;
  int ret;
  
  g_return_if_fail (IS_SONG_DB (db));

  if (db->readonly)
    return;
  
  string = g_string_new (NULL);

  pack_int (string, version);
  
  memset (&key, 0, sizeof (key));
  key.dptr = VERSION_KEY;
  key.dsize = strlen (key.dptr);
    
  memset (&data, 0, sizeof (data));
  data.dptr = string->str;
  data.dsize = string->len;

  ret = gdbm_store (db->dbf, key, data, GDBM_REPLACE);
  if (ret)
    g_warning ("Could not update database version");

  g_string_free (string, TRUE);
}

void
song_db_rebuild (SongDB *db)
{
  datum key, data;
  GList *songs = NULL;
  GList *l;

  g_return_if_fail (IS_SONG_DB (db));

  memset (&key, 0, sizeof (key));
  memset (&data, 0, sizeof (data));
  
  while (1)
    {
      key = gdbm_nextkey (db->dbf, key);
      if (key.dptr == NULL)
	break;

      if (((char*)key.dptr)[0] == VERSION_KEY[0] &&
	  strncmp (key.dptr, VERSION_KEY, VERSION_KEY_LENGTH) == 0)
	continue;
      
      songs = g_list_prepend (songs, g_strndup (key.dptr, key.dsize));
    }

  for (l = songs; l; l = l->next)
    add_song (db, l->data, TRUE);

  g_list_foreach (songs, (GFunc) g_free, NULL);
  g_list_free (songs);

}

