/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <unistd.h>
#include <glib/gi18n.h>
#include <gtk/gtkliststore.h>
#include "playlist.h"
#include "playlist-xml.h"

static int maxid = 100;

void
playlist_free (Playlist *playlist)
{
  switch (playlist->type)
    {
      case PLAYLIST_TYPE_LIBRARY:
	break;
      case PLAYLIST_TYPE_SMART:
	query_unref (playlist->v.smart.query);
	break;
      case PLAYLIST_TYPE_REGULAR:
	query_unref (playlist->v.regular.query);
	break;
      case PLAYLIST_TYPE_IRADIO:
	g_free (playlist->v.iradio.url);
	break;
    }

  g_free (playlist->name);  
  g_free (playlist);
}

static void
playlist_update_query (Playlist* playlist)
{
  if (playlist->type == PLAYLIST_TYPE_REGULAR)
    {
      Expr *v, *c, *e;
      Query *query;
  
      v = expr_variable_new (VARIABLE_PLAYLISTS);
      c = expr_constant_new (constant_int_new (playlist->id));

      e = expr_binary_new (EXPR_OP_HAS_PLAYLIST, v, c);

      query = query_new (e);
      playlist_set_query (playlist, query);
      query_unref (query);
    }
}

Playlist *
playlist_new (PlaylistType  type,
	      const char   *name)
{
  Playlist *playlist;

  playlist = g_new0 (Playlist, 1);

  playlist->type = type;
  playlist->name = g_strdup (name);

  if (type == PLAYLIST_TYPE_LIBRARY)
    playlist->id = 0;
  else 
    playlist->id = maxid++;
  
  playlist_update_query (playlist);
  
  return playlist;
}

Query *
playlist_get_query (Playlist *playlist)
{
  g_return_val_if_fail (playlist != NULL, NULL);

  switch (playlist->type) {
  case PLAYLIST_TYPE_SMART:
    return playlist->v.smart.query;
  case PLAYLIST_TYPE_REGULAR:
    return playlist->v.regular.query;
  default:
    g_assert_not_reached ();
    return NULL;
  }
}

void
playlist_set_query (Playlist *playlist,
		    Query    *query)
{
  g_return_if_fail (playlist != NULL);

  switch (playlist->type) {
  case PLAYLIST_TYPE_SMART:
    if (playlist->v.smart.query)
      query_unref (playlist->v.smart.query);
    
    playlist->v.smart.query = query_ref (query);
    break;
  case PLAYLIST_TYPE_REGULAR:
    if (playlist->v.regular.query)
      query_unref (playlist->v.regular.query);

    playlist->v.regular.query = query_ref (query);
    break;
  default:
    g_assert_not_reached ();
    break;
  }    
}

gboolean
playlist_is_editable (Playlist *playlist)
{
  g_return_val_if_fail (playlist != NULL, FALSE);

  switch (playlist->type) {
  case PLAYLIST_TYPE_SMART:
  case PLAYLIST_TYPE_REGULAR:
    return TRUE;
  case PLAYLIST_TYPE_LIBRARY:
    return FALSE;
  default:
    g_assert_not_reached ();
    return FALSE;
  }
}

const char *
playlist_get_name (Playlist *playlist)
{
  g_return_val_if_fail (playlist != NULL, NULL);

  return playlist->name;
}

void
playlist_set_name (Playlist *playlist, const char *name)
{
  char *tmp;
  
  g_return_if_fail (playlist != NULL);
  g_return_if_fail (name != NULL);

  tmp = g_strdup (name);
  g_free (playlist->name);
  playlist->name = tmp;
}

void
playlist_set_id (Playlist *playlist, int id)
{
  if (id > maxid)
    id = maxid;
  
  playlist->id = id;

  maxid = id + 1;

  playlist_update_query (playlist);
}

int
playlist_get_id (Playlist *playlist)
{
  return playlist->id;
}
