/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2004 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003 Jorn Baayen <jorn@nl.linux.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <math.h>
#include <xine.h>
#include <glib/gi18n.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include "jamboree-marshal.h"
#include "song-list-model.h"
#include "utils.h"
#include "player.h"


static void     player_class_init (PlayerClass *klass);
static void     player_init       (Player      *player);
static void     player_finalize   (GObject     *object);
static long     player_get_time   (Player      *player);
static gboolean player_playing    (Player      *player);
static void     player_close      (Player      *player);
static void     start_fade        (Player      *player,
				   gboolean     fade_in);


typedef enum {
  PLAYER_ERROR_NO_INPUT_PLUGIN,
  PLAYER_ERROR_NO_QUEUE_PLUGIN,
  PLAYER_ERROR_NO_DEMUX_PLUGIN,
  PLAYER_ERROR_NO_VOLUME_PLUGIN,
  PLAYER_ERROR_DEMUX_FAILED,
  PLAYER_ERROR_NO_AUDIO,
  PLAYER_ERROR_GENERAL,
  PLAYER_ERROR_INTERNAL
} PlayerError;
 
#define PLAYER_ERROR player_error_quark ()

#define FADE_LOW 15
#define FADE_HI 100

GQuark player_error_quark (void);


enum {
  END_OF_STREAM,
  TICK,
  STATE_CHANGED,
  ERROR,
  LAST_SIGNAL
};

typedef struct {
  int signal;
  Song *song;
} SignalData;

struct _PlayerPriv {
  char *uri;

  Song *current_song;
  
  xine_t *xine;
  xine_ao_driver_t *audio_driver;
  xine_vo_driver_t *video_driver;

  xine_stream_t *stream;
  xine_event_queue_t *event_queue;

  char *configfile;

  int volume;
  gboolean mute;

  GTimer *timer;
  long timer_add;

  int mute_level;

  int fade_level;
  float mute_factor;

  guint fade_timeout_id;

  guint tick_timeout_id;
  GAsyncQueue *queue;
};

static GObjectClass *parent_class;
static guint signals[LAST_SIGNAL];

GType
player_get_type (void)
{
  static GType type = 0;
	
  if (!type)
    {
      static const GTypeInfo info =
	{
	  sizeof (PlayerClass),
	  NULL,           /* base_init */
	  NULL,           /* base_finalize */
	  (GClassInitFunc) player_class_init,
	  NULL,           /* class_finalize */
	  NULL,           /* class_data */
	  sizeof (Player),
	  0,
	  (GInstanceInitFunc) player_init,
	};
      
      type = g_type_register_static (G_TYPE_OBJECT,
				     "Player",
				     &info, 0);
    }

  return type;
}

static void
player_class_init (PlayerClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass*) klass;

  object_class->finalize = player_finalize;

  signals[END_OF_STREAM] =
    g_signal_new ("end_of_stream",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  0,
		  NULL, NULL,
		  jamboree_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);
  
  signals[TICK] =
    g_signal_new ("tick",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  0,
		  NULL, NULL,
		  g_cclosure_marshal_VOID__LONG,
		  G_TYPE_NONE, 1, G_TYPE_LONG);
  
  signals[STATE_CHANGED] =
    g_signal_new ("state_changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  0,
		  NULL, NULL,
		  g_cclosure_marshal_VOID__INT,
		  G_TYPE_NONE, 1, G_TYPE_INT);

  signals[ERROR] =
    g_signal_new ("error",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  0,
		  NULL, NULL,
		  g_cclosure_marshal_VOID__STRING,
		  G_TYPE_NONE, 1, G_TYPE_STRING);
}

static gboolean
tick_timeout (Player *player)
{
  if (!player_playing (player))
    return TRUE;

  g_signal_emit (player, signals[TICK], 0, player_get_time (player));
  
  return TRUE;
}

static void
player_init (Player *player)
{
  PlayerPriv *priv;

  priv = g_new0 (PlayerPriv, 1);
  player->priv = priv;

  priv->tick_timeout_id = g_timeout_add (200, (GSourceFunc) tick_timeout, player);
}

static void
player_finalize (GObject *object)
{
  Player *player;
  PlayerPriv *priv;

  player = PLAYER (object);
  priv = player->priv;
  
  g_source_remove (priv->tick_timeout_id);
  
  if (priv->fade_timeout_id)
    g_source_remove (priv->fade_timeout_id);

  if (priv->stream != NULL)
    {
      xine_stop (priv->stream);
      xine_close (priv->stream);
      xine_event_dispose_queue (priv->event_queue);
      xine_dispose (priv->stream);
    }
  
  if (priv->audio_driver != NULL)
    xine_close_audio_driver (priv->xine, priv->audio_driver);
  
  if (priv->video_driver != NULL)
    xine_close_video_driver (priv->xine, priv->video_driver);
  
  xine_config_save (priv->xine, priv->configfile);
  g_free (priv->configfile);

  xine_exit (priv->xine);

  g_timer_destroy (priv->timer);
  g_free (priv->uri);
  g_async_queue_unref (priv->queue);

  g_free (priv);
  
  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
player_set_time (Player *player,
		 long    time)
{
  PlayerPriv *priv;
  
  g_return_if_fail (IS_PLAYER (player));
  g_return_if_fail (time >= 0);

  priv = player->priv;
  
  if (priv->stream != NULL)
    {
      xine_play (priv->stream, 0, time * 1000);
      
      if (player_playing (player))
	xine_set_param (priv->stream, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
      else
	xine_set_param (priv->stream, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
      
      g_timer_reset (priv->timer);
      priv->timer_add = time;
    }
}

static long
player_get_time (Player *player)
{
  PlayerPriv *priv;
  
  g_return_val_if_fail (IS_PLAYER (player), -1);

  priv = player->priv;
  
  if (priv->stream != NULL)
    return (long) floor (g_timer_elapsed (priv->timer, NULL) + 0.5) + priv->timer_add;
  else
    return -1;
}

static gboolean
signal_idle (Player *player)
{
  PlayerPriv *priv = player->priv;
  int queue_length;
  SignalData *data;

  data = g_async_queue_try_pop (priv->queue);
  if (data == NULL)
    return FALSE;

  switch (data->signal)
    {
    case END_OF_STREAM:
      g_signal_emit (player, signals[END_OF_STREAM], 0, data->song);
      break;
    }
  
  g_object_unref (player);
  g_free (data);
  queue_length = g_async_queue_length (priv->queue);
  
  return (queue_length > 0);
}

static void
xine_event (Player *player,
	    const xine_event_t *event)
{
  PlayerPriv *priv = player->priv;
  SignalData *data;
  xine_progress_data_t *prg;

  switch (event->type)
    {
    case XINE_EVENT_UI_PLAYBACK_FINISHED:
      g_object_ref (player);
      data = g_new0 (SignalData, 1);
      data->signal = END_OF_STREAM;
      data->song = priv->current_song;
      g_async_queue_push (priv->queue, data);
      g_idle_add ((GSourceFunc) signal_idle, player);
      break;
    case XINE_EVENT_PROGRESS:
      prg = event->data;
      /*
	if (prg->percent == 0 || prg->percent == 100)
	{
	g_object_ref (player);
	data = g_new0 (SignalData, 1);
	data->signal = prg->percent ? BUFFERING_END : BUFFERING_BEGIN;
	g_idle_add ((GSourceFunc) signal_idle, player);
	break;
	}
      */
    }
}

static void
player_construct (Player *player,
		  GError **error)
{
  PlayerPriv *priv = player->priv;
  const char *audio_driver;

  priv->xine = xine_new ();

  priv->configfile = g_build_filename (g_get_home_dir (),
				       ".gnome2", "jamboree",
				       "xine-config",
				       NULL);
  xine_config_load (priv->xine, priv->configfile);

  xine_init (priv->xine);

  audio_driver = "auto";

  if (strcmp (audio_driver, "null") == 0)
    priv->audio_driver = NULL;
  else
    {
      if (strcmp (audio_driver, "auto") != 0) {
	/* first try the requested driver */
	priv->audio_driver = xine_open_audio_driver (priv->xine,
						     audio_driver, NULL);
      }
      
      /* autoprobe */
      if (priv->audio_driver == NULL)
	priv->audio_driver = xine_open_audio_driver (priv->xine, NULL, NULL);
    }
  
  if (priv->audio_driver == NULL)
    g_set_error (error,
		 PLAYER_ERROR,
		 PLAYER_ERROR_NO_AUDIO,
		 _("Failed to set up an audio driver; check your installation"));
  
  priv->video_driver = NULL;

  priv->stream = xine_stream_new (priv->xine,
				  priv->audio_driver,
				  priv->video_driver);
  priv->event_queue = xine_event_new_queue (priv->stream);
  priv->queue = g_async_queue_new ();

  xine_event_create_listener_thread (priv->event_queue,
				     (xine_event_listener_cb_t) xine_event, player);

  xine_config_register_range (priv->xine,
			      "misc.aplayer_level",
			      50, 0, 100, "aplayer volume level",
			      NULL, 10, NULL, NULL);
  priv->volume = -1;

  priv->timer = g_timer_new ();
  g_timer_stop (priv->timer);
  g_timer_reset (priv->timer);
  priv->timer_add = 0;
}

Player *
player_new (void)
{
  Player *player;

  player = g_object_new (TYPE_PLAYER, NULL);
  player_construct (player, NULL);

  return player;
}

GQuark
player_error_quark (void)
{
  static GQuark quark = 0;

  if (!quark)
    quark = g_quark_from_static_string ("player_error");

  return quark;
}

void
player_open (Player *player,
	     const char *uri,
	     GError **error)
{
  PlayerPriv *priv = player->priv;
  int xine_error;
  char *unesc;

  g_return_if_fail (IS_PLAYER (player));

  player_close (player);

  if (uri == NULL)
    return;

  if (!xine_open (priv->stream, uri))
    xine_error = xine_get_error (priv->stream);
  else
    xine_error = XINE_ERROR_NONE;

  if (xine_error != XINE_ERROR_NONE)
    {
      switch (xine_error)
	{
	case XINE_ERROR_NO_INPUT_PLUGIN:
	  unesc = gnome_vfs_unescape_string_for_display (uri);
	  g_set_error (error,
		       PLAYER_ERROR,
		       PLAYER_ERROR_NO_INPUT_PLUGIN,
		       _("No input plugin available for %s; check your installation."),
		       unesc);
	  g_free (unesc);
	  break;
	case XINE_ERROR_NO_DEMUX_PLUGIN:
	  unesc = gnome_vfs_unescape_string_for_display (uri);
	  g_set_error (error,
		       PLAYER_ERROR,
		       PLAYER_ERROR_NO_DEMUX_PLUGIN,
		       _("No demux plugin available for %s; check your installation."),
		       unesc);
	  g_free (unesc);
	  break;
	case XINE_ERROR_DEMUX_FAILED:
	  unesc = gnome_vfs_unescape_string_for_display (uri);
	  g_set_error (error,
		       PLAYER_ERROR,
		       PLAYER_ERROR_DEMUX_FAILED,
		       _("Demuxing for %s failed; check your installation."),
		       unesc);
	  g_free (unesc);
	  break;
	default:
	  g_set_error (error,
		       PLAYER_ERROR,
		       PLAYER_ERROR_INTERNAL,
		       _("Internal error; check your installation."));
	  break;
	}
    }
  else if (!xine_get_stream_info (priv->stream, XINE_STREAM_INFO_AUDIO_HANDLED))
    {
      unesc = gnome_vfs_unescape_string_for_display (uri);
      g_set_error (error,
		   PLAYER_ERROR,
		   PLAYER_ERROR_NO_AUDIO,
		   _("Audio of %s not handled; check your installation."),
		   unesc);
      g_free (unesc);
    }
  else
    {
      g_timer_stop (priv->timer);
      g_timer_reset (priv->timer);
      priv->timer_add = 0;
    }
  
  priv->uri = g_strdup (uri);
}

void
player_close (Player *player)
{
  PlayerPriv *priv = player->priv;
    
  g_return_if_fail (IS_PLAYER (player));

  if (priv->stream != NULL)
    {
      xine_stop (priv->stream);
      xine_close (priv->stream);
    }
  
  g_free (priv->uri);
  priv->uri = NULL;
}

const char *
player_get_uri (Player *player)
{
  PlayerPriv *priv = player->priv;
    
  g_return_val_if_fail (IS_PLAYER (player), NULL);

  return priv->uri;
}

gboolean
player_play (Player *player)
{
  PlayerPriv *priv = player->priv;
  int speed, status;

  g_return_val_if_fail (IS_PLAYER (player), FALSE);

  if (priv->stream == NULL)
    return FALSE;

  speed = xine_get_param (priv->stream, XINE_PARAM_SPEED);
  status = xine_get_status (priv->stream);

  if (speed != XINE_SPEED_NORMAL && status == XINE_STATUS_PLAY)
    xine_set_param (priv->stream, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
  else
    xine_play (priv->stream, 0, 0);

  g_timer_start (priv->timer);

  return TRUE;
}

void
player_pause (Player *player)
{
  PlayerPriv *priv = player->priv;

  g_return_if_fail (IS_PLAYER (player));

  if (priv->stream != NULL)
    {
      xine_set_param (priv->stream, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
      
      /* Close the audio device when on pause */
      xine_set_param (priv->stream, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
    }

  priv->timer_add += floor (g_timer_elapsed (priv->timer, NULL) + 0.5);
  g_timer_stop (priv->timer);
  g_timer_reset (priv->timer);
}

gboolean
player_is_playing (Player *player, Song *song)
{
  return player_playing (player) && song == player->priv->current_song;
}

gboolean
player_playing (Player *player)
{
  PlayerPriv *priv = player->priv;

  g_return_val_if_fail (IS_PLAYER (player), FALSE);

  if (priv->stream == NULL)
    return FALSE;

  return (xine_get_status (priv->stream) == XINE_STATUS_PLAY &&
	  xine_get_param (priv->stream, XINE_PARAM_SPEED) == XINE_SPEED_NORMAL);
}

static gboolean
can_set_volume (Player *player)
{
  PlayerPriv *priv = player->priv;

  if (priv->audio_driver == NULL)
    return FALSE;
  if (xine_get_param (priv->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL) == -2)
    return FALSE;

  return TRUE;
}

void
player_set_volume (Player *player,
		   int     volume)
{
  PlayerPriv *priv = player->priv;

  g_return_if_fail (IS_PLAYER (player));
  g_return_if_fail (volume >= 0 && volume <= 100);

  if (priv->stream != NULL)
    {
      int real_vol;
      
      if (!can_set_volume (player))
	return;

      real_vol = volume;
      
      if (priv->mute_level > 0 || priv->fade_timeout_id != 0)
	real_vol = volume * priv->fade_level / FADE_HI;
      
      xine_set_param (priv->stream, XINE_PARAM_AUDIO_AMP_LEVEL, CLAMP (real_vol, 0, 100));
    }
  
  priv->volume = volume;
}

int
player_get_volume (Player *player)
{
  PlayerPriv *priv = player->priv;

  g_return_val_if_fail (IS_PLAYER (player), 0);

  return priv->volume;
}

void
player_set_mute (Player *player,
		 gboolean mute)
{
  PlayerPriv *priv;

  g_return_if_fail (IS_PLAYER (player));

  priv = player->priv;

  if (priv->mute == mute)
    return;
  
  priv->mute = mute;
  start_fade (player, !mute);
}

gboolean
player_get_mute (Player *player)
{
  PlayerPriv *priv = player->priv;

  g_return_val_if_fail (IS_PLAYER (player), FALSE);

  return priv->mute;
}

gboolean
player_is_seekable (Player *player)
{
  PlayerPriv *priv = player->priv;

  g_return_val_if_fail (IS_PLAYER (player), FALSE);

  if (priv->stream != NULL)
    return xine_get_stream_info (priv->stream, XINE_STREAM_INFO_SEEKABLE);
  else
    return FALSE;
}

/* Return value is in milliseconds. */
guint64
player_tell (Player *player)
{
  return player_get_time (player) * 1000;
}

/* t is in milliseconds. */
void
player_seek (Player *player, guint64 t)
{
  return player_set_time (player, t / (1000));
}

PlayerState
player_get_state (Player *player)
{
  PlayerPriv *priv;

  g_return_val_if_fail (IS_PLAYER (player), PLAYER_STATE_STOPPED);

  priv = player->priv;
  
  if (priv->stream == NULL)
    return PLAYER_STATE_STOPPED;

  if (xine_get_status (priv->stream) == XINE_STATUS_PLAY &&
      xine_get_param (priv->stream, XINE_PARAM_SPEED) == XINE_SPEED_NORMAL)
    return PLAYER_STATE_PLAYING;

  if (xine_get_status (priv->stream) == XINE_STATUS_PLAY)
    return PLAYER_STATE_PAUSED;

  return PLAYER_STATE_STOPPED;
}


gboolean
player_set_song (Player *player, Song *song)
{
  PlayerPriv *priv;
  gboolean start;

  g_return_val_if_fail (IS_PLAYER (player), FALSE);

  priv = player->priv;

  priv->current_song = song;

  start = player_playing (player);
 
  player_open (player, song->filename, NULL);

  if (start)
    player_play (player);
  
  return TRUE;
}

Song *
player_get_song (Player *player)
{
  PlayerPriv *priv;

  g_return_val_if_fail (IS_PLAYER (player), NULL);

  priv = player->priv;

  return priv->current_song;
}

void
player_play_song (Player *player,
		  Song   *song)
{
  PlayerPriv *priv;
  
  g_return_if_fail (IS_PLAYER (player));
  g_return_if_fail (song != NULL);  

  priv = player->priv;

  player_set_song (player, song);
  player_play (player);
}

void
player_stop (Player *player)
{
  PlayerPriv *priv = player->priv;
    
  g_return_if_fail (IS_PLAYER (player));

  if (priv->stream != NULL)
    {
      xine_stop (priv->stream);
      g_signal_emit (player, signals[STATE_CHANGED], 0, PLAYER_STATE_STOPPED);
    }
}

void
player_push_mute (Player *player)
{
  player->priv->mute_level ++;
  
  if (player->priv->mute_level == 1)
    start_fade (player, FALSE);
}

void
player_pop_mute (Player *player)
{
  if (player->priv->mute_level == 0)
    return;
  
  player->priv->mute_level--;

  if (player->priv->mute_level == 0)
    start_fade (player, TRUE);
}

static gboolean
fade_in_cb (Player *player)
{
  PlayerPriv *priv = player->priv;

  priv->fade_level++;

  player_set_volume (player, player_get_volume (player));

  if (priv->fade_level >= 100)
    {
      priv->fade_timeout_id = 0;
      return FALSE;
    }
  
  return TRUE;
}

static gboolean
fade_out_cb (Player *player)
{
  PlayerPriv *priv = player->priv;

  priv->fade_level--;

  player_set_volume (player, player_get_volume (player));

  if (priv->fade_level < FADE_LOW)
    {
      priv->fade_timeout_id = 0;
      return FALSE;
    }
  
  return TRUE;
}

static void
start_fade (Player *player, gboolean fade_in)
{
  PlayerPriv *priv = player->priv;
  
  if (priv->fade_timeout_id)
    g_source_remove (priv->fade_timeout_id);
  
  if (fade_in)
    {
      priv->fade_level = FADE_LOW;
      priv->fade_timeout_id = g_timeout_add (10, (GSourceFunc) fade_in_cb, player);
    }
  else
    {
      priv->fade_level = FADE_HI;
      priv->fade_timeout_id = g_timeout_add (10, (GSourceFunc) fade_out_cb, player);
    }
}

