/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtkmain.h>
#include "main-window.h"
#include "player.h"
#include "dbus.h"

static void              server_unregistered_func (DBusConnection *connection,
						   gpointer        user_data);
static DBusHandlerResult server_message_func      (DBusConnection *connection,
						   DBusMessage    *message,
						   gpointer        user_data);
static gboolean          dbus_message_append_song (DBusMessage    *message,
						   Song           *song);


static DBusConnection *bus_conn;
static const char *server_path[] = { "org", "imendio", "jamboree", "Player", NULL };

static DBusObjectPathVTable
server_vtable = {
  server_unregistered_func,
  server_message_func,
  NULL,
};

static gboolean break_ongoing = FALSE;

static DBusHandlerResult
handler_message_filter (DBusConnection   *dbus_conn,
			DBusMessage      *message,
			gpointer          user_data)
{
  if (dbus_message_is_signal (message,
			      "org.gnome.TypingMonitor",
			      "BreakStarted"))
    {
      player_push_mute (main_window_get_player (main_window_get ()));
      
      break_ongoing = TRUE;
    }
  else if (dbus_message_is_signal (message,
				   "org.gnome.TypingMonitor",
				   "BreakFinished"))
    {
      player_pop_mute (main_window_get_player (main_window_get ()));
      
      break_ongoing = FALSE;
    }
  else if (dbus_message_is_signal (message,
				   "org.freedesktop.DBus",
				   "ServiceDeleted"))
    {
      char *name;

      dbus_message_get_args (message,
			     NULL,
			     DBUS_TYPE_STRING, &name,
			     DBUS_TYPE_INVALID);


      if (strcmp (name, "org.gnome.TypingMonitor") == 0 && break_ongoing)
	{
	  player_pop_mute (main_window_get_player (main_window_get ()));
	  
	  break_ongoing = FALSE;
	}
	
      dbus_free (name);
    }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}


void
dbus_init_handler (void)
{
  dbus_init_service ();

  dbus_bus_add_match (bus_conn, "type='signal',sender='org.gnome.TypingMonitor',interface='org.gnome.TypingMonitor'",
		      NULL);
  dbus_bus_add_match (bus_conn, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus'",
		      NULL);
  
  dbus_connection_add_filter (bus_conn, handler_message_filter,
			      NULL, NULL);
}

gboolean
dbus_init_service (void)
{
  DBusError error;

  if (bus_conn)
    return TRUE;
  
  dbus_error_init (&error);
  bus_conn = dbus_bus_get (DBUS_BUS_SESSION, &error);
  
  if (!bus_conn) 
    {
      g_warning ("Failed to connect to the D-BUS daemon: %s", error.message);
      dbus_error_free (&error);
      return FALSE;
    }
  
  dbus_bus_acquire_service (bus_conn, JAMBOREE_DBUS_SERVICE, 0, &error);
  if (dbus_error_is_set (&error)) 
    {
      g_warning ("Failed to acquire player service");
      dbus_error_free (&error);
      return FALSE;
    }

  if (!dbus_connection_register_object_path (bus_conn,
					     server_path,
					     &server_vtable,
					     NULL))
    {
      g_warning ("Failed to register server object with the D-BUS bus daemon");
      return FALSE;
    }
  
  dbus_connection_setup_with_g_main (bus_conn, NULL);
  
  return TRUE;
}

static gboolean
dbus_init_client (void)
{
  DBusError error;

  if (bus_conn)
    return TRUE;
  
  dbus_error_init (&error);
  bus_conn = dbus_bus_get (DBUS_BUS_SESSION, &error);
  
  if (!bus_conn) 
    {
      g_warning ("Failed to connect to the D-BUS daemon: %s", error.message);
      dbus_error_free (&error);
      return FALSE;
    }
  
  if (!dbus_bus_service_exists (bus_conn, JAMBOREE_DBUS_SERVICE, NULL))
    {
      g_printerr ("No jamboree service is running.\n");
      return FALSE;
    }

  dbus_connection_setup_with_g_main (bus_conn, NULL);
  
  return TRUE;
}

static void
server_unregistered_func (DBusConnection *connection, void *user_data)
{
}

static void
send_ok_reply (DBusConnection *connection,
	       DBusMessage    *message)
{
  DBusMessage *reply;

  reply = dbus_message_new_method_return (message);
                                                                                
  dbus_connection_send (connection, reply, NULL);
  dbus_message_unref (reply);
}

static DBusHandlerResult
handle_play (DBusConnection *connection,
	     DBusMessage    *message,
	     MainWindow     *window)
{
  main_window_handle_play (window);
  send_ok_reply (connection, message);
  
  return DBUS_HANDLER_RESULT_HANDLED;
}


static DBusHandlerResult
handle_stop (DBusConnection *connection,
	     DBusMessage    *message,
	     MainWindow     *window)
{
  main_window_handle_stop (window);
  send_ok_reply (connection, message);

  return DBUS_HANDLER_RESULT_HANDLED;
}


static DBusHandlerResult
handle_prev (DBusConnection *connection,
	     DBusMessage    *message,
	     MainWindow     *window)
{
  main_window_handle_prev (window);
  send_ok_reply (connection, message);
  
  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
handle_next (DBusConnection *connection,
	     DBusMessage    *message,
	     MainWindow     *window)
{
  main_window_handle_next (window);
  send_ok_reply (connection, message);
  
  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
handle_push_mute (DBusConnection *connection,
		  DBusMessage    *message,
		  MainWindow     *window)
{
  main_window_push_mute (window);
  send_ok_reply (connection, message);
  
  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
handle_pop_mute (DBusConnection *connection,
		 DBusMessage    *message,
		 MainWindow     *window)
{
  main_window_pop_mute (window);
  send_ok_reply (connection, message);
  
  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
handle_get_current_song (DBusConnection *connection,
			 DBusMessage    *message,
			 MainWindow     *window)
{
  DBusMessage *reply;
  Song *song;
  
  song = main_window_get_current_song (window);
  
  reply = dbus_message_new_method_return (message);

  dbus_message_append_song (reply, song);
  
  dbus_connection_send (connection, reply, NULL);
  dbus_message_unref (reply);
  
  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
handle_quit (DBusConnection *connection,
	     DBusMessage    *message,
	     MainWindow     *window)
{
  send_ok_reply (connection, message);

  dbus_connection_flush (connection);
  
  gtk_main_quit ();
  
  return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
server_message_func (DBusConnection *connection,
		     DBusMessage    *message,
		     gpointer        user_data)
{
  MainWindow *window;
  
  window = main_window_get ();

  if (dbus_message_is_method_call (message,
				   JAMBOREE_DBUS_PLAYER_INTERFACE,
				   JAMBOREE_DBUS_PLAYER_PLAY))
    return handle_play (connection, message, window);
  else if (dbus_message_is_method_call (message,
					JAMBOREE_DBUS_PLAYER_INTERFACE,
					JAMBOREE_DBUS_PLAYER_STOP))
    return handle_stop (connection, message, window);
  else if (dbus_message_is_method_call (message,
					JAMBOREE_DBUS_PLAYER_INTERFACE,
					JAMBOREE_DBUS_PLAYER_PREV))
    return handle_prev (connection, message, window);
  else if (dbus_message_is_method_call (message,
					JAMBOREE_DBUS_PLAYER_INTERFACE,
					JAMBOREE_DBUS_PLAYER_NEXT))
    return handle_next (connection, message, window);
  else if (dbus_message_is_method_call (message,
					JAMBOREE_DBUS_PLAYER_INTERFACE,
					JAMBOREE_DBUS_PLAYER_PUSH_MUTE))
    return handle_push_mute (connection, message, window);
  else if (dbus_message_is_method_call (message,
					JAMBOREE_DBUS_PLAYER_INTERFACE,
					JAMBOREE_DBUS_PLAYER_POP_MUTE))
    return handle_pop_mute (connection, message, window);
  else if (dbus_message_is_method_call (message,
					JAMBOREE_DBUS_PLAYER_INTERFACE,
					JAMBOREE_DBUS_PLAYER_GET_CURRENT_SONG))
    return handle_get_current_song (connection, message, window);
  else if (dbus_message_is_method_call (message,
					JAMBOREE_DBUS_PLAYER_INTERFACE,
					JAMBOREE_DBUS_PLAYER_QUIT))
    return handle_quit (connection, message, window);
  
  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

gboolean
dbus_send_remote_cmd (const gchar *arg, gboolean *invalid_arg)
{
  DBusMessage *message;
  DBusMessage *reply;
  DBusError error;
  const char *msg = NULL;

  *invalid_arg = FALSE;
  
  if (!dbus_init_client ())
    return FALSE;

  if (strcmp (arg, "play") == 0)
    msg = JAMBOREE_DBUS_PLAYER_PLAY;
  else if (strcmp (arg, "stop") == 0)
    msg = JAMBOREE_DBUS_PLAYER_STOP;
  else if (strcmp (arg, "prev") == 0)
    msg = JAMBOREE_DBUS_PLAYER_PREV;
  else if (strcmp (arg, "next") == 0)
    msg = JAMBOREE_DBUS_PLAYER_NEXT;
  else if (strcmp (arg, "pop-mute") == 0)
    msg = JAMBOREE_DBUS_PLAYER_POP_MUTE;
  else if (strcmp (arg, "push-mute") == 0)
    msg = JAMBOREE_DBUS_PLAYER_PUSH_MUTE;
  else if (strcmp (arg, "current-song") == 0)
    {
      Song *song;
	   
      reply = dbus_send_remote_cmd_with_reply (JAMBOREE_DBUS_PLAYER_GET_CURRENT_SONG);
	   
      if (!reply)
	{
	  g_printerr ("Couldn't contact Jamboree service.\n");
	  exit (1);
	}
      
      song = dbus_message_to_song (reply);
      g_print ("%s - %s\n",
	       string_entry_get_str (song->title),
	       string_entry_get_str (song->artist));
      
      dbus_message_unref (reply);
    }
  else if (strcmp (arg, "quit") == 0)
    msg = JAMBOREE_DBUS_PLAYER_QUIT;
  else
    {
      *invalid_arg = TRUE;
      return FALSE;
    }
  
  g_assert (msg != NULL);
  
  message = dbus_message_new_method_call (JAMBOREE_DBUS_SERVICE,
					  JAMBOREE_DBUS_PLAYER_OBJECT,
					  JAMBOREE_DBUS_PLAYER_INTERFACE,
					  msg);
      
  dbus_error_init (&error);
  
  reply = dbus_connection_send_with_reply_and_block (bus_conn,
						     message, 2000,
						     &error);
  
  dbus_message_unref (message);

  if (dbus_error_is_set (&error)) 
    {
      g_warning ("Failed to send message '%s': %s", msg, error.message);
      dbus_error_free (&error);
      return FALSE;
    }
  
  dbus_message_unref (reply);
  
  return TRUE;
}

DBusMessage *
dbus_send_remote_cmd_with_reply (const gchar *msg)
{
  DBusMessage *message;
  DBusMessage *reply;
  DBusError error;

  if (!dbus_init_client ())
    return NULL;
  
  message = dbus_message_new_method_call (JAMBOREE_DBUS_SERVICE,
					  JAMBOREE_DBUS_PLAYER_OBJECT,
					  JAMBOREE_DBUS_PLAYER_INTERFACE,
					  msg);
      
  dbus_error_init (&error);
  
  reply = dbus_connection_send_with_reply_and_block (bus_conn,
						     message, 1000,
						     &error);
  
  dbus_message_unref (message);

  if (dbus_error_is_set (&error)) 
    {
      g_warning ("Failed to send message '%s': %s", msg, error.message);
      dbus_error_free (&error);
      return NULL;
    }
  
  return reply;
}

Song *
dbus_message_to_song (DBusMessage *message)
{
  DBusMessageIter iter, dict;
  Song *song;
  char *title = NULL, *filename = NULL, *artist = NULL;
  char *key;
    
  dbus_message_iter_init (message, &iter);
  dbus_message_iter_init_dict_iterator (&iter, &dict);

  song = g_new0 (Song, 1);

  while ((key = dbus_message_iter_get_dict_key (&dict)))
    {
      if (!title && strcmp (key, "title") == 0)
	title = dbus_message_iter_get_string (&dict);
      else if (!filename && strcmp (key, "filename") == 0)
	filename = dbus_message_iter_get_string (&dict);
      else if (!artist && strcmp (key, "artist") == 0)
	artist = dbus_message_iter_get_string (&dict);
      
      dbus_free (key);

      if (!dbus_message_iter_next (&dict))
	break;
    }      

  song->title = string_entry_add_const (title);
  song->filename = g_strdup (filename);
  song->artist = string_entry_add_const (artist);

  dbus_free (title);
  dbus_free (filename);
  dbus_free (artist);
  
  return song;
}

static gboolean
dbus_message_append_song (DBusMessage *message, Song *song)
{
  DBusMessageIter iter, dict;
  
  dbus_message_append_iter_init (message, &iter);

  dbus_message_iter_append_dict (&iter, &dict);  

  dbus_message_iter_append_dict_key (&dict, "title");
  dbus_message_iter_append_string (&dict, string_entry_get_str (song->title));

  dbus_message_iter_append_dict_key (&dict, "filename");
  dbus_message_iter_append_string (&dict, song->filename);

  dbus_message_iter_append_dict_key (&dict, "artist");
  dbus_message_iter_append_string (&dict, string_entry_get_str (song->artist));

  dbus_message_iter_append_dict_key (&dict, "length");
  dbus_message_iter_append_int32 (&dict, song->length);
  
  return TRUE;
}

