/* -*- tab-width: 4; c-basic-offset: 4 -*- */

/* helpful GUI utilities */

using System;
using System.Threading;
using System.IO;
using Cairo;
using Gtk;

public class GUtils {
    static public Cairo.Color SaturateColor (Cairo.Color color, double factor)
    {
		double h, s, v;
		double r, g, b;
		
		Gtk.Global.RgbToHsv (color.R, color.G, color.B,
							 out h, out s, out v);
		
		Gtk.HSV.ToRgb (h, Math.Min (s * factor, 1), v, out r, out g, out b);
		return new Cairo.Color (r, g, b);
    }

    static public Cairo.Color LightenColor (Cairo.Color color, double factor)
    {
		return new Cairo.Color (color.R * factor,
                                color.G * factor,
                                color.B * factor,
                                color.A);
    }

	static public void SpreadColor (Cairo.Color color, out Cairo.Color light,
									out Cairo.Color dark)
	{
		light = new Cairo.Color ((color.R + 1.0) / 2,
								 (color.G + 1.0) / 2,
								 (color.B + 1.0) / 2);

		dark = new Cairo.Color (color.R / 2,
								color.G / 2,
								color.B / 2);
	}

	public class Palette
	{
		Cairo.Color [] colors;

		public Palette(int numColors)
		{
			this.colors = new Cairo.Color [numColors];

			double delta = 1.0 / numColors;

			for (int i = 0; i < numColors; i++) {
				double h, s, v;
				double r, g, b;

				h = i * delta;
				s = 0.5;
				v = 1.0;
				
				Gtk.HSV.ToRgb (h, s, v, out r, out g, out b);
				
				this.colors[i].R = r;
				this.colors[i].G = g;
				this.colors[i].B = b;
				this.colors[i].A = 1.0;
			}
		}

		public Palette() : this(20) {}

		public Cairo.Color GetColor(int idx)
		{
			return this.colors[idx % this.colors.Length];
		}
	}

	static Gtk.Widget AlignWrap (Gtk.Widget widget)
	{
		Gtk.Alignment alignment;
		alignment = new Gtk.Alignment (1, 0, 0, 0);
		alignment.Add (widget);
		return alignment;
	}

	static public Gtk.Button AddButton (string iconName, Gtk.Box box, EventHandler handler)
	{
		Gtk.Button button;

		button = new Gtk.Button ();
		button.Image = new Gtk.Image (iconName, IconSize.SmallToolbar);
		box.PackStart (AlignWrap (button), false, false, 0);
		button.Clicked += handler;

		return button;
	}
}

public class GuiStatusFileStream : StatusFileStream
{
    Gtk.ProgressBar bar;
    
    public GuiStatusFileStream (string fileName)
        : base (fileName)
    {}

    public override void Init()
    {
        if (this.doStatus) {
            Gtk.Window window = new Gtk.Window("Loading FS Model");
            window.WindowPosition = Gtk.WindowPosition.Center;
            this.bar = new Gtk.ProgressBar();
            this.bar.Text = "Loading ...";
            this.bar.Fraction = 0;
            window.Add (this.bar);
            window.ShowAll();
        }
    }

    public override void UpdateStatus (double newFract)
    {
        if (this.doStatus)
            GLib.Idle.Add (delegate { this.bar.Fraction = newFract; return false; } );
    }

    public override void Close()
    {
        if (this.doStatus)
            this.bar.Parent.Destroy();
        this.doStatus = false;
        base.Close();
    }

    public override void PerformIO (DoReadFSModel doRead)
    {
        if (!this.doStatus) {
            doRead ();
            Close();
        } else {
            // Process GUI / progress bar updates
            Thread thread = new Thread (new ThreadStart (
                                            delegate {
                                                doRead ();
                                                Close();
                                                GLib.Idle.Add (delegate {
                                                    Application.Quit();
                                                    return false;
                                                } );
                                            }
                                            ));
            thread.Start();
            Application.Run();
        }
    }
}

class EventMotion
{
    // Motion event details can't be cloned
    public int X;
    public int Y;
    public int rootX;
    public int rootY;
	public uint time;
	public EventMotion (Gdk.EventMotion src)
	{
		X = (int)src.X;
		Y = (int)src.Y;
		rootX = (int)src.XRoot;
		rootY = (int)src.YRoot;
		time = src.Time;
	}
}

public abstract class MouseOverHelper
{
	uint mouseoverTimeout;
	Gtk.Window popupWindow;
    Gtk.Widget widget;

    object lastObj;
	EventMotion lastEvent;
    
	bool PopUp ()
	{
		this.popupWindow = new Gtk.Window (Gtk.WindowType.Popup);
		this.popupWindow.TransientFor = (Gtk.Window)this.widget.Toplevel;
        PopulatePopup (this.popupWindow, this.lastObj);
        if (this.popupWindow.Child != null)
            this.popupWindow.Child.Show();
		this.popupWindow.Move (this.lastEvent.rootX, this.lastEvent.rootY);
		this.popupWindow.Child.Show();
		this.popupWindow.Show();
		mouseoverTimeout = 0;

		return false;
	}
    void PopDown ()
    {
        // FIXME: should we have a delayed pop-down ? ...  1 sec ?
        if (this.popupWindow != null)
            this.popupWindow.Destroy();
        this.popupWindow = null;
		if (this.mouseoverTimeout != 0)
			GLib.Source.Remove (this.mouseoverTimeout);
        this.mouseoverTimeout = 0;
    }

    void MotionEvent (object obj, Gtk.MotionNotifyEventArgs args)
    {
		// Set cursor to drag / hand if ctrl pressed ?
		lastEvent = new EventMotion (args.Event);

        object item;
		if ((item = GetItemAt (this.lastEvent.X, this.lastEvent.Y)) == null) {
			PopDown();
            this.lastObj = null;
            return;
        }
        if (item == this.lastObj)
            return;
        this.lastObj = item;

        PopDown();
		this.mouseoverTimeout = GLib.Timeout.Add
            (10, new GLib.TimeoutHandler (PopUp));
		args.RetVal = false;
    }

    public MouseOverHelper (Gtk.Widget widget)
    {
        this.widget = widget;
        widget.AddEvents ((int) Gdk.EventMask.PointerMotionMask);
        widget.MotionNotifyEvent += MotionEvent;
        widget.DestroyEvent += delegate {
            if (this.popupWindow != null)
                this.popupWindow.Destroy();
            this.popupWindow = null;
        };
    }

    // Override these two to get simple popup behavior.
    public abstract object GetItemAt (long x, long y);
    public abstract void PopulatePopup (Gtk.Window popup, object obj);
}

class BackTraceView : Gtk.TreeView {
    enum StoreFields : int { Name, Frame };
    public BackTraceView (Trace.Stack.Frame frame)
    {
        Gtk.ListStore store = new Gtk.ListStore (typeof (string),
                                                 typeof (Trace.Stack.Frame));
        while (frame != null) {
            store.AppendValues (frame.Name, frame);
            frame = frame.Parent;
        }
        Selection.Mode = Gtk.SelectionMode.None;

		Gtk.CellRendererText rendererText = new Gtk.CellRendererText ();
		rendererText.Xalign = 0.0f;
        InsertColumn (0, "frame", rendererText, "text", 0);
        Model = store;

        Show();
    }
}
