/*
 * Tool to dump file-system data using either
 * ext2fs or ioctl FIBMAP, as appropriate.
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <limits.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>

#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/vfs.h>
#include <linux/fs.h>

#include <ext2fs/ext2fs.h>

#define MAX_PATH_BUF 65536

static int pretty_depth = 0;

typedef struct {
    long  *blocks;
    long   block_len;
    long   block_alloc;
} BlockList;

struct file_info {
    char *path;
    struct stat st;
    ext2_filsys fs;
    struct ext2_inode inode;
};

struct dump_context_t {
    ext2_filsys fs;
    int blocksize;
    int depth;
};

static void
print_append (const char *format, ...)
{
    va_list args;
    va_start (args, format);
    vfprintf (stdout, format, args);
    va_end (args);
}

static void
print_tag (int depthinc, const char *format, ...)
{
    va_list args;
    int i;
    
    va_start (args, format);
    if (depthinc < 0)
        pretty_depth += depthinc;
    for (i = 0; i < pretty_depth; i++)
        fprintf (stdout, " ");
    vfprintf (stdout, format, args);
    if (depthinc > 0)
        pretty_depth += depthinc;
    va_end (args);
}

static void
blocks_init (BlockList *blist)
{
    blist->blocks = NULL;
    blist->block_len = 0;
    blist->block_alloc = 0;
}

static void
blocks_free (BlockList *blist)
{
    if (blist->blocks)
        free (blist->blocks);
}

static void
blocks_push (BlockList *blist, long block)
{
    if (blist->block_len >= blist->block_alloc) {
        if (!blist->block_alloc)
            blist->block_alloc = 16;
        blist->block_alloc *= 2;
        blist->blocks = realloc (blist->blocks,
                                  sizeof (long) * blist->block_alloc);
    }
    blist->blocks[blist->block_len++] = block;
}

static void
blocks_reset (BlockList *blist)
{
    blist->block_len = 0;
}

// resets queue ...
static char *
blocks_as_string (BlockList *blist)
{
#define ELEM_SIZE (sizeof ("i0x, ") + sizeof (long) * 2)
    char *buf;
    char *p;
    int i;

    buf = malloc (blist->block_len * ELEM_SIZE + 1);
    buf[0] = '\0';

    // FIXME: we need to check if some blocks are contiguous
    // and crunch them into a span ... (+5) ...
    for (i = 0, p = buf; i < blist->block_len; i++)
    {
        int indirect = 0;
        long j, blk = blist->blocks[i];

        // look for spans
        if (blk >= 0)
        {
            for (j = 0; i + j < blist->block_len; j++) {
                if (blist->blocks[i+j] != blk + j)
                    break;
            }
        } else {
            blk = -blk;
            indirect = 1;
            j = 1;
        }
        if (j <= 1)
            p += sprintf (p, "%s0x%lx", indirect ? "i" : "", blk);
        else
            p += sprintf (p, "0x%lx-0x%lx", blk, blk + j - 1);
        i += j - 1;
        if (i < blist->block_len - 1)
            p = strcat (p, ", ") + 2;
    }
    blocks_reset (blist);
    return buf;
#undef ELEM_SIZE
}

static char *
read_inode_attrs (struct file_info *fi)
{
    char *ret;
    char *blocks = NULL;
    int grp;
    BlockList item_list;

    if (fi->fs) {
        grp = ext2fs_group_of_ino(fi->fs, fi->st.st_ino);
        blocks_init(&item_list);
        blocks_push(&item_list, fi->fs->group_desc[grp].bg_inode_table + (fi->st.st_ino -
          grp * EXT2_INODES_PER_GROUP(fi->fs->super)) / EXT2_INODES_PER_BLOCK(fi->fs->super));
        blocks = blocks_as_string (&item_list);
        blocks_free(&item_list);
    }
    ret = malloc ((blocks ? strlen (blocks) : 0)+ sizeof ("inode=\"0x00000000\" "
                                            "iblk=\"\"") + 1);
    sprintf (ret, " inode=\"0x%x\" iblk=\"%s\"", (unsigned)fi->st.st_ino, blocks ? blocks : "");
    if (blocks)
        free (blocks);

    return ret;
}

static int
block_account_func (ext2_filsys fs, blk_t *blocknr,
                    int	blockcnt, void *priv_data)
{
    BlockList *blist = priv_data;
    long blk = *blocknr;

//    fprintf (stdout, "act: 0x%lx (%d)\n", (long)*blocknr, blockcnt);
    // -N is N'th indirection & the block is an indirect block
    if (blockcnt < 0)
        blk = -blk;
    blocks_push (blist, blk);

    return 0;
}

static char *
read_link (struct file_info *fi, char **blocks)
{
    char *str;
    char pathname[PATH_MAX];
    BlockList blist;
    ssize_t ret;

    *blocks = NULL;

    if (fi->fs) {
        blocks_init(&blist);
        if (ext2fs_inode_data_blocks (fi->fs, &fi->inode))
            blocks_push(&blist, fi->inode.i_block[0]);
        *blocks = blocks_as_string (&blist);
        blocks_free(&blist);
    }
    if ((ret = readlink(fi->path, pathname, sizeof(pathname)-1)) < 0) {
        fprintf(stderr, "Cannot read link %s: %s\n", fi->path, strerror(errno));
        *blocks = strdup ("");
	return strdup("");
    }
    pathname[ret] = 0;

    str = malloc (ret + sizeof (" link=\"\"") + 1);
    strcpy (str, " link=\"");
    strcat (str, pathname);
    strcat (str, "\"");

    // FIXME: - we need to dump the symlink's block data here ...
    
    return str;
}

static void
dump_item (struct dump_context_t *dc, char *path, char *name)
{
    char *blocks = NULL;
    const char *node_type;
    char *link = NULL;
    BlockList item_list;
    struct file_info fi;
    char *inode_str;

    fi.path = path;
    if (lstat(path, &fi.st) < 0) {
        fprintf(stderr, "Cannot stat %s: %s\n", path, strerror(errno));
        return;
    }
    fi.fs = dc->fs;
    if (fi.fs)
        ext2fs_read_inode (dc->fs, fi.st.st_ino, &fi.inode);
    inode_str = read_inode_attrs (&fi);

    if (S_ISDIR (fi.st.st_mode))
        node_type = "dir";
    else if (S_ISREG (fi.st.st_mode))
        node_type = "file";
    else if (S_ISLNK (fi.st.st_mode)) {
        node_type = "symlink";
        link = read_link (&fi, &blocks);
    } else {
        node_type = "special";
    }

    if (S_ISDIR(fi.st.st_mode) || S_ISREG(fi.st.st_mode)) {
        blocks_init (&item_list);
        if (fi.fs)
            ext2fs_block_iterate (dc->fs, fi.st.st_ino, 0, NULL, block_account_func, &item_list);

        else if (S_ISREG(fi.st.st_mode)) {
            int fd = open(path, O_RDONLY);
	    unsigned long num;
            unsigned long blk;

            if (fd < 0) {
                fprintf(stderr, "Cannot open %s: %s\n", path, strerror(errno));
                goto out_scan;
            }
            for (blk = 0; blk < (fi.st.st_size + dc->blocksize - 1) / dc->blocksize; blk++) {
                num = blk;
                if (ioctl (fd, FIBMAP, &num) < 0) {
                    fprintf (stderr, "Cannot map block %lu in %s: %s\n", blk, path, strerror(errno));
		    goto out_scan;
                }
                if (num) /* No hole? */
                    blocks_push (&item_list, num);
            }
            close (fd);
        }
out_scan:
        blocks = blocks_as_string (&item_list);
        blocks_free (&item_list);
    }
    print_tag (0, "<%s name=\"%s\" size=\"0x%lx\"%s%s",
               node_type, name, (long)fi.st.st_size,
               link ? link : "", inode_str);
    if (!S_ISDIR (fi.st.st_mode)) {
        if (blocks == NULL || blocks[0] == '\0')
            print_append ("/>\n");
        else
            print_append (">%s</%s>\n", blocks, node_type);
    } else {
        DIR *dp;
        struct dirent *de;
        char *endptr;

        print_append ("><blks>%s</blks>\n", blocks);
        pretty_depth++;
        dc->depth++;
        endptr = path + strlen(path) - 1;
        if (*endptr != '/')
            strcat(++endptr, "/");
        if ((dp = opendir(path)) == NULL) {
            fprintf(stderr, "Cannot open directory %s: %s\n", path, strerror(errno));
            goto out_dir;
	}
        while ((de = readdir(dp)) != NULL) {
            if (!strcmp(de->d_name, ".") || !strcmp(de->d_name, ".."))
                continue;
            strcat(endptr+1, de->d_name);
            dump_item(dc, path, endptr+1);
            endptr[1] = 0;
        }
        closedir(dp);
        *endptr = 0;
out_dir:
        dc->depth--;
        print_tag (-1, "</%s>\n", node_type);
    }
    if (link)
        free (link);
    if (blocks)
        free (blocks);
}

static void usage_exit (const char *warning)
{
    if (warning)
        fprintf (stderr, "Warning: %s\n\n", warning);
    fprintf (stderr, "Usage: blockdump [-d /dev/path] path\n");
    fprintf (stderr, "dump the contents of a filesystem as an XML block description.\n");
    fprintf (stderr, "optionally just a sub-tree of that file-system\n");
    exit (warning != NULL);
}

int main (int argc, char **argv)
{
    ext2_filsys fs;
    errcode_t ret;
    char *dev_name, *last_name;
    struct dump_context_t dc = { 0, };
    int i;
    struct statfs st;
    char fname[MAX_PATH_BUF];

    dev_name = NULL;
    dc.fs = NULL;
    for (i = 1; i < argc; i++) {
        if (!strcmp (argv[i], "-h") ||
            !strcmp (argv[i], "--help"))
            usage_exit (NULL);
        if (!strcmp (argv[i], "-d")) {
            if (i + 1 == argc)
                usage_exit("-d requires an argument");
            dev_name = argv[++i];
        }
        else {
            struct stat st;

            strcpy(fname, argv[i]);
            if (stat(fname, &st) < 0) {
                fprintf(stderr, "Cannot stat() given path %s: %s\n", fname, strerror(errno));
                usage_exit(NULL);
            }
	    if (S_ISDIR(st.st_mode)) {
                if (fname [strlen (fname) - 1] != '/')
                    strcat (fname, "/");
		last_name = fname + strlen(fname) - 1;
	    }
	    else {
                last_name = strrchr(fname, '/');
                if (!last_name)
                    last_name = fname;
	    }
        }
    }

    if (dev_name) {
        ret = ext2fs_open (dev_name, EXT2_FLAG_JOURNAL_DEV_OK, 0, 0, unix_io_manager, &fs);
        if (ret) {
            fprintf (stderr, "Failed to open '%s'\n", fname);
            return 1;
        }
        dc.fs = fs;
    }
    if (statfs(fname, &st) < 0) {
        fprintf(stderr, "Failed to stat filesystem %s: %s\n", fname, strerror(errno));
        return 1;
    }
    dc.blocksize = st.f_bsize;
    print_tag (1, "<root name=\"%s\" blksize=\"0x%x\">\n",
               fname, dc.blocksize);
    dc.depth = 0;
    dump_item (&dc, fname, last_name);
    print_tag (-1, "</root>\n");

    ext2fs_close (fs);
    return 0;
}
