/* -*- tab-width: 4; c-basic-offset: 4 -*- */

/* helpful non-GUI utilities */

using System;
using System.IO;

public class Utils {
	static public long ReadHex (string hex, int offset, out int endOffset)
    {
        int len;
        if (hex == null || (len = hex.Length) < offset + 1) {
            endOffset = 0;
            return 0;
        }

        int i = offset;

        // prune leading whitespace
        while (Char.IsWhiteSpace (hex [i]) && i < len)
            i++;

        // prune optional '0x' prefix.
        if (len - i >= 2 && hex[i] == '0' && hex[i+1] == 'x')
            i += 2;

        ulong val = 0;
        for (; i < len; i++) {
            char digit = hex[i];
            uint dval;

            if (Char.IsDigit (digit))
                dval = (uint) (digit - '0');
            else if (digit >= 'a' && digit <= 'f')
                dval = (uint) (digit - 'a' + 10);
            else if (digit >= 'A' && digit <= 'F')
                dval = (uint) (digit - 'A' + 10);
            else
                break;
            val = (val * 16ul) + dval;
        }
        endOffset = i;
//		Console.WriteLine ("Parse hex '" + hex + "[{0}] - 0x{1:x}", offset, val);
        return (long) val;
    }
	static public long ReadHex (string hex)
	{
        int ignore;
        return ReadHex (hex, 0, out ignore);
	}

    static public string PrettyTime (long ticks)
    {
        if (ticks == 0)
            return "0";
		double time = (double)ticks;
		string unit = "us";
		if (time > 1000) {
			time /= 1000;
			unit = "ms";
		}
		if (time > 10000) {
			time /= 1000;
			unit = "s";
		}
		string pretty = String.Format ("{0:##.#}", time) + unit;
//        Console.WriteLine ("time " + ticks + " -> " + pretty);
        return pretty;
    }
    static public string PrettyBytes (long val)
    {	
		double bytes = (double) val;
		string unit = "bytes";
		if (bytes > 1024) {
			bytes /= 1024;
			unit = "kb";
		}
		if (bytes > 1024) {
			bytes /= 1024;
			unit = "Mb";
		}
		if (bytes > 1024) {
			bytes /= 1024;
			unit = "Gb";
		}
		return String.Format ("{0:###.#}", bytes) + unit;
    }
	static public string PrettyBW (long bytes, long ticks)
	{
		double secs = (double) ticks / 1e6;
		double bw = (double) bytes / (1000.0 * secs);
		string unit = "kb/s";
		if (bw > 1024) {
			bw /= 1024;
			unit = "Mb/s";
		}
		return String.Format ("{0:###}", bw) + unit;
	}
    static public string Percentage (double percent)
    {
		return String.Format ("{0:##.#}%", percent * 100.0);
    }

	static public string PathDirName (string path)
	{
		if (path.Length == 0 || path[0] == '<') // special name
			return "/";
		return Path.GetDirectoryName (path);
	}

	static public string PathBaseName (string path)
	{
		if (path.Length > 0 && path[0] == '<') // special name
			return path;
		return Path.GetFileName (path);
	}

    // Strip redundencies & flatten '..'s
    static int warnedNormalize;
    static public string PathNormalize (string path)
    {
        // Remove sillies such as ///tmp///foo.txt
        while (path.IndexOf ("//") >= 0)
            path = path.Replace ("//", "/");

        if (warnedNormalize++ == 0 && path.IndexOf ("/..") >= 0)
            Console.WriteLine ("FIXME: need to normalize " + path);
        return path;
    }
}

public class StatusFileStream : System.IO.FileStream
{
    long lastPos;
    long step;
    protected internal bool doStatus;
    
    public StatusFileStream (string fileName)
        : base (fileName, System.IO.FileMode.Open)
    {
        long length = Length;
        this.doStatus = length > 1024*1024;
        this.step = (2 * length) / 100;
        Init();
    }
    public virtual void Init()
    {
        if (this.doStatus)
            Console.Write ("Loading fsmodel: ");
    }
    public virtual void UpdateStatus (double fract)
    {
        Console.Write (".");
    }
    public delegate void DoReadFSModel ();
    public virtual void PerformIO (DoReadFSModel doRead)
    {
        doRead ();
        Close();
    }

    // Overridden stream bits ...
    public override int Read (byte[] dest, int dest_offset, int count)
    {
        if (this.doStatus) {
            long pos = Position;
            if (pos - this.lastPos > step) {
                UpdateStatus ((double) pos / Length);
                this.lastPos = pos;
            }
        }
        return base.Read (dest, dest_offset, count);
    }
    public override void Close()
    {
        if (this.doStatus)
            Console.WriteLine (" done");
        base.Close();
    }
}
