/* -*- tab-width: 4; c-basic-offset: 4 -*- */

/* Model of a file-system API */

using System;
using System.Collections.Generic;
using FS;

public class FSModelLookup : FSPathResolver.Context {
    public SimEvent Ev;
    public FSModel  Model;
    public FSModelLookup (FSModel model, SimEvent ev)
        : base ()
    {
        this.Ev = ev;
        this.Model = model;
    }
}

public class FSModelResolver : FSPathResolver {
    public FSModelResolver (Directory root, string cwd)
        : base (root, cwd) {}
    public override void VisitLink (Context ctx, SymLink link)
    {
        if (!(ctx is FSModelLookup))
            return;
        FSModelLookup fsml = (FSModelLookup) ctx;

        fsml.Model.ReadINode (fsml.Ev, link);
        long targetBlk = link.TargetBlock;
        if (targetBlk >= 0)
            fsml.Model.ReadWholeFile (fsml.Ev, link);
    }

    public override void VisitDirectory (Context ctx, Directory dir)
    {
        if (!(ctx is FSModelLookup))
            return;
        FSModelLookup fsml = (FSModelLookup) ctx;
        fsml.Model.ReadWholeFile (fsml.Ev, dir);
    }
}

public class FSModel {
    PageCache     pageCache;
    DiskModel     diskModel;
    SimProcess    proc;
    public FSData fsData;
    FSModelResolver fsResolver;

    public int BlockSize { get { return this.fsData.BlockSize; } }

    static bool GetBlock (BIO bio, File fi, long blkInFile)
    {
        if (fi.Blocks == null)
            return false;

        long curBlk = blkInFile;
        for (int i = 0; i < fi.Blocks.Count; i++) {
            if (fi.Blocks[i].IsIndirect) {
                bio.AddBlk (fi.Blocks[i].Start);
                bio.Submit ();
                continue;
            }
            if (curBlk < fi.Blocks[i].Length) {
                bio.AddBlk (fi.Blocks[i].Start + curBlk);
/*                Console.WriteLine ("Got block in " + fi.Name +
                                   " blk: 0x{0:x} for offset {1}",
                                   fi.Blocks[i].Start + curBlk,
                                   blkInFile); */
                return true;
            }
                                   
            curBlk -= fi.Blocks[i].Length;
        }
//        Console.WriteLine ("Ran over block length " + fi.GetPath() +
//                           " blk: " + blkInFile + " left: " + curBlk);
        return false;
    }
    public void ReadBlocks (SimEvent ev, File fi, long offset, int count)
    {
//        Console.WriteLine ("file blocks read " + fi.Name + " offset "
//                           + offset + " count " + count);
        BIO bio = new BIO (this.proc, this, this.diskModel, ev);
        this.pageCache.ReadPages (bio, fi, GetBlock,
                                  offset / this.pageCache.PageSize,
                                  ((count + this.pageCache.PageSize - 1) /
                                   this.pageCache.PageSize));
    }
    public void ReadINode (SimEvent ev, File fi)
    {
//        Console.WriteLine ("inode read " + fi.Name);
        BIO bio = new BIO (this.proc, this, this.diskModel, ev);
        this.pageCache.ReadINode (bio, fi);
    }
    public void ReadWholeFile (SimEvent ev, File fi)
    {
        ReadINode (ev, fi);
        ReadBlocks (ev, fi, 0, (int)fi.Size);
    }

    class FileDescr {
        public File File;
        public long Pos;
        public long Size   { get { return this.File.Size; } }
        public string Path { get { return this.File.GetPath(); } }
        
        public FileDescr (File file)
        {
            this.File = file;
            this.Pos = 0;
        }
    }
    Dictionary<int,FileDescr> openFiles;

    public FSModel (SimProcess proc, FSData fsData, DiskModel diskModel)
    {
        this.pageCache = proc.PageCache;
        this.proc = proc;
        this.fsData = fsData;
        this.diskModel = diskModel;
        this.openFiles = new Dictionary<int,FileDescr>();
        this.fsResolver = new FSModelResolver (this.fsData.Root, "/home/michael");
    }

    public string GetFilePath (int fd)
    {
        if (!this.openFiles.ContainsKey (fd))
            return null;
        return this.openFiles[fd].Path;
    }

    // FIXME: hack for now ... - cf. SimFile constructor
    public File GetFileFromPath (string fname)
    {
        return this.fsResolver.LookupName (null, fname);
    }

    public string GetCanonicalPath (string fname)
    {
        File file = this.fsResolver.LookupName (null, fname);
        if (file == null)
            return null;
        return file.GetPath();
    }

    static int FIXME_warn_missing_files = 0;
    static int FIXME_warn_extra_files = 0;
    public void Open (SimEvent ev, string filename, int asHandle, bool birect, bool sync)
    {
//        Console.WriteLine ("Open " + filename);
        this.proc.AddNotice (ev);

        File file = this.fsResolver.LookupName (new FSModelLookup (this, ev), filename);
        if (asHandle < 0) { // the open failed ...
            if (file != null && FIXME_warn_extra_files++ < 16)
                Console.WriteLine ("FS image mismatch: extra files in image: " +
                                   "opened non-existent file " + filename);
            return;
        } else if (file == null) {
            if (FIXME_warn_missing_files++ < 16)
                Console.WriteLine ("FS image mismatch: missing files in image: " +
                                   "failed to open file " + filename);
            return;
        }
        
        // needed for permission checks etc.
        ReadINode (ev, file);

        FileDescr desc = new FileDescr (file);
        if (openFiles.ContainsKey (asHandle)) {
//            Console.WriteLine ("FIXME: failed to process close on " + asHandle);
            openFiles.Remove (asHandle);
        }
        openFiles[asHandle] = desc;
//        Console.WriteLine ("Open '" + filename + "' = " + asHandle);
    }

    public void Close (SimEvent ev, int handle)
    {
        // ~ no I/O cost
        openFiles.Remove (handle);
    }

    public void Stat (SimEvent ev, string filename)
    {
        File file = this.fsResolver.LookupName (new FSModelLookup (this, ev), filename);
        if (file != null)
            ReadINode (ev, file);
    }

    public void Read (SimEvent ev, int fd, long address, int length)
    {
        if (!this.openFiles.ContainsKey (fd))
            return;
        FileDescr fh = this.openFiles[fd];

//        Console.WriteLine ("Read " + length + " at " + fh.Pos + " size " + fh.Size);
        if (length == 0 || fh.Pos >= fh.Size)
            return;

        ReadBlocks (ev, fh.File, fh.Pos, length);
        fh.Pos += length;
    }
    public void Seek (SimEvent ev, int fd, long offset)
    {
        if (!this.openFiles.ContainsKey (fd))
            return;
        FileDescr fh = this.openFiles[fd];
        fh.Pos = Math.Min (fh.Pos + offset, fh.Size);
    }
    public void Write (SimEvent ev, int fd, long address, long length)
    {
        Console.WriteLine ("Write " + fd);
    }
    public void ReadAhead (SimEvent ev, int fd, long fileOffset, long length)
    {
        Console.WriteLine ("Readahead " + fd);
    }
    public void MMapRead (SimEvent ev, File fi, long offset)
    {
        if (fi == null) {
            Console.WriteLine ("Bogus - mmap!");
            return;
        }
        ReadBlocks (ev, fi, offset, this.pageCache.PageSize);
    }
    public void Chdir (SimEvent ev, string newDir)
    {
        this.fsResolver.Chdir (new FSModelLookup (this, ev), newDir);
    }
}
