/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.Collections.Generic;

class FileColorModel : IHistColorModel {
    int curColor;
	Dictionary<string,Cairo.Color> colors;
	GUtils.Palette  palette;
	Cairo.Color    unknown;
	Cairo.Color    shlib;
	Cairo.Color    locale;
	Cairo.Color    dir;
	Cairo.Color    heap;
	Cairo.Color    tmp;

	public FileColorModel()
	{
		this.colors = new Dictionary<string,Cairo.Color>();
		this.palette = new GUtils.Palette(9);

		this.unknown = this.palette.GetColor(curColor++);
		this.shlib = this.palette.GetColor(curColor++);
		this.dir = this.palette.GetColor(curColor++);
		this.heap = this.palette.GetColor(curColor++);
		this.shlib = this.palette.GetColor(curColor++);
		this.tmp = this.palette.GetColor(curColor++);

		// fonts
		this.colors.Add (".cache-2", this.palette.GetColor(curColor));
		this.colors.Add (".conf", this.palette.GetColor(curColor));
		this.colors.Add (".ttf", this.palette.GetColor(curColor));
		this.colors.Add (".ttc", this.palette.GetColor(curColor++));

		// xml
		this.colors.Add (".xml", this.palette.GetColor(curColor));
		this.colors.Add (".xcu", this.palette.GetColor(curColor));
		this.colors.Add (".xcs", this.palette.GetColor(curColor++));

		// icons
		this.colors.Add (".bmp", this.palette.GetColor(curColor));
		this.colors.Add (".gif", this.palette.GetColor(curColor));
		this.colors.Add (".jpeg", this.palette.GetColor(curColor));
		this.colors.Add (".png", this.palette.GetColor(curColor));
		this.colors.Add (".cache", this.palette.GetColor(curColor++));

		// translation
		this.colors.Add (".mo", this.palette.GetColor(curColor++));

        // desktop bits
		this.colors.Add (".theme", this.palette.GetColor(curColor));
		this.colors.Add (".desktop", this.palette.GetColor(curColor++));
	}
	public Cairo.Color GetColor (IHistNode node)
	{
		if (!(node is FileNode)) // a directory
			return this.dir;

		string fname = ((FileNode)node).FileName;

		if (fname.Length == 0)
			return this.unknown;
		if (fname[0] == '<')
			return this.heap;
        if (fname.StartsWith ("/tmp"))
            return this.tmp;

		// shlibs are a pain ...
		if (fname.IndexOf (".so.") > 0 ||
			fname.EndsWith (".so"))
			return this.shlib;

		// locale files are a pain ...
		if (fname.StartsWith ("/usr/lib/locale"))
			return this.locale;

		string suffix = fname.ToLower();
		int idx;
		idx = suffix.LastIndexOf('.');
		if (idx < 0 || idx > suffix.Length)
			return this.unknown;

		suffix = suffix.Substring(idx);

		if (!this.colors.ContainsKey (suffix)) {
            string dirName = Utils.PathDirName (fname);
            if (!this.colors.ContainsKey (dirName)) {
                Console.WriteLine ("unknown suffix '" + fname +
                                   "' -> '" + suffix + "' - adding directory");
                this.colors.Add (dirName, this.palette.GetColor (this.curColor++));
            }
            return this.colors[dirName];
		}
		return this.colors[suffix];
	}
}

class FileNode : Node {
	SimFile simFile;
    public FileNode (Node parent, SimFile simFile, double weight)
		: base( parent, "", weight)
	{
		this.simFile = simFile;
	}
	public override string Label {
		get {
			return Utils.PathBaseName (this.simFile.FileNameAtOpen) + " " +
				Utils.PrettyTime (this.simFile.TotalTime) + " " +
				Utils.PrettyBytes (this.simFile.TotalBytes) + " " +
				Utils.PrettyBW (this.simFile.TotalBytes, this.simFile.TotalTime);
		}
	}
	public override string ShortLabel {
        get {
            return Utils.PathBaseName (this.simFile.FileNameAtOpen);
        }
    }
    public override string TreeElemName {
        get {
            return Utils.PathBaseName (this.simFile.FileNameAtOpen);
        }
    }
    public string FileName {
		get { return this.simFile.FileNameAtOpen; }
	}
}

class FileView : BaseView {

	Node CreateNode (Node parent, SimFile simFile)
	{
		return new FileNode (parent, simFile,
                             this.type == ViewType.Time ? simFile.TotalTime 
                             : simFile.TotalBytes);
	}

	Node GetParentAtPath (Dictionary<string,Node> fsMap, string path)
	{
		if (fsMap.ContainsKey (path))
			return fsMap[path];;

		string dirName = Utils.PathDirName (path);
		string baseName = Utils.PathBaseName (path);
//		Console.WriteLine ("path " + path + " -> " + dirName + " + " + baseName);
        if (dirName == path)
            Console.WriteLine ("Infinite recursion " + path + " => " + dirName);
		Node parent = GetParentAtPath (fsMap, dirName);
		Node node = new Node (parent, baseName, 0);
		fsMap[path] = node;
		return node;
	}

	public override Node UpdateView ()
	{
		Node root = new Node (null, "", 0);
		if (!hierarchical) { // Flat tree of times ...
			foreach (SimFile simFile in this.sd.FilesSortedByIO())
				CreateNode (root, simFile);
		} else {
			Dictionary<string,Node> fsMap = new Dictionary<string,Node>();
			fsMap.Add("/", root);

			foreach (SimFile simFile in this.sd.FilesSortedByIO()) {
				string path = simFile.FileNameAtOpen;
				Node parent = GetParentAtPath (fsMap, path);
				CreateNode (parent, simFile);
			}
		}
        return root;
	}

	public FileView (SimulationData sd)
        : base (sd, new FileColorModel(), new HistogramSettings (false))
	{
	}
}
