/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using C5;
using SCG = System.Collections.Generic;
using MemMap = C5.TreeSet< AddrLayout.LayoutPos >;

public interface IVerticalLayout {
	long    Extent { get; }
    PosSpan GetPos( MemRegion r );
    long    GetAddressAtOffset(long offset);
}

public class BandLayout : IVerticalLayout {
    long   overallSize;

    public long Extent {
        get { return this.overallSize; }
    }
   
	class BandPos {
		public PosSpan Span;
		public SimMap  SimMap;
	}
	Dictionary<string,BandPos> bandLayout;

	const int BandWidth = 4;

    public BandLayout( int spacing, SimulationData sd )
    {
		this.bandLayout = new Dictionary<string, BandPos>();

		List<BandPos> sorted = new List<BandPos>();

        foreach (SimMap simMap in sd.allMaps) {
			if (this.bandLayout.ContainsKey (simMap.FileName))
				continue;

			BandPos pos = new BandPos();
			pos.SimMap = simMap;
			this.bandLayout.Add(simMap.FileName, pos);
			sorted.Add(pos);
        }

		sorted.Sort( delegate (BandPos a, BandPos b)
						{ return b.SimMap.Percentage.CompareTo(a.SimMap.Percentage); } );

		long nextSpace = 0;
		long offset = 0;
        foreach (BandPos pos in sorted) {
			offset += nextSpace;
			pos.Span = new PosSpan (offset, offset + spacing * BandLayout.BandWidth);
            offset += spacing * BandLayout.BandWidth;
			nextSpace = spacing;
        }
        this.overallSize = offset;
	}

    public PosSpan GetPos( MemRegion r )
	{
		// FIXME: use a tree by address [!]
		if (r is Trace.MapEvent) {
			Trace.MapEvent meme = (Trace.MapEvent) r;

			BandPos pos = this.bandLayout[meme.FileName];
			if (pos != null)
				return pos.Span;
		}

		foreach (BandPos pos in this.bandLayout.Values) {
			if (r.Overlaps (pos.SimMap.Map))
				return pos.Span;
		}
		return new PosSpan();
	}

    public long GetAddressAtOffset(long offset)
	{
		return -1;
	}
}

public class AddrLayout : IVerticalLayout {
    MemMap memMap;
    int spacing;
    long   overallSize;

    class MapComparer : SCG.IComparer<LayoutPos>
    {
        public int Compare(LayoutPos a, LayoutPos b)
        {
            if (a.Start > b.Start)
                return 1;
            else if (a.Start < b.Start)
                return -1;
            return 0;
        }
    }

    public AddrLayout( int spacing, SimulationData sd )
    {
        this.spacing = spacing;
        memMap = new MemMap(new MapComparer());

        layoutMemory(sd);
    }
    LayoutPos Find(long addr)
    {
        LayoutPos start = new LayoutPos( new MemRegion(addr, addr) );
        LayoutPos val;
        try {
            val = memMap.WeakPredecessor(start);
        } catch (NoSuchItemException) {
            val = null;
        }
        return val;
    }

    LayoutPos GetNext(LayoutPos pos)
    {
        LayoutPos val;
        try {
            val = memMap.Successor(pos);
        } catch (NoSuchItemException) {
            val = null;
        }
        return val;
    }

    static bool WarnedDisasterAlready = false;
    public PosSpan GetPos( MemRegion r )
    {
//        Console.WriteLine ("Find " + r + " in mem map !");
		LayoutPos val = Find (r.Start);

        if (val == null || !val.Overlaps (r)) {
			if (r.Start == 0) /* Test spacing */
				return new PosSpan();

            string str = "<null>";
            if (val != null)
                str = val.ToString();
            if (!WarnedDisasterAlready)
                Console.WriteLine ("ERROR: Disaster area - no overlap " + r + " val " + str);
            WarnedDisasterAlready = true;
            return new PosSpan();
        }

        PosSpan range = new PosSpan();

        long adjust = r.Start - val.Start;

        range.start = val.offset + adjust;
        range.end   = val.offset + r.Size + adjust;
        return range;
    }

    public long GetAddressAtOffset(long offset)
	{
		foreach (LayoutPos p in memMap) {
			if (offset >= p.offset && offset < p.offset + p.Size)
				return p.Start + offset - p.offset;
		}
		return -1;
	}

    public long Extent {
        get { return this.overallSize; }
    }
    
    class LayoutPos : MemRegion {
		public long offset; // vertical offset on screen
        public LayoutPos(MemRegion r) : base(r) {}
    }

    void layoutMemory (SimulationData sd)
    {
        foreach (SimMap se in sd.allMaps) {
            Trace.MapEvent seMap = se.Map;

			LayoutPos val = Find (seMap.Start);
            if (val == null || !val.Overlaps (seMap)) {
				// FIXME: check I don't overlap with the next segment either (?)
                LayoutPos pos = new LayoutPos(seMap);
//                Console.WriteLine ("Insert " + pos);
                memMap.Add (pos);
            } else { // We know they overlap ...
				val.Collate (seMap);
				LayoutPos next = GetNext (val);
				if (next != null && next.Overlaps (val) ) {
					val.Collate (next);
					memMap.Remove (next);
				}
            }
        }
        long offset = 0, nextSpace = 0;
        LayoutPos last = null;
        foreach (LayoutPos pos in memMap) {
			offset += nextSpace;
            pos.offset = offset;
            offset += (pos.End - pos.Start);

            if (last != null && pos.Adjacent (last))
                nextSpace = this.spacing;
            else
                nextSpace = this.spacing * 2;
        }
        this.overallSize = offset;
    }
}
