/* GStreamer
 * Copyright (C) 1999 Erik Walthinsen <omega@cse.ogi.edu>
 * Copyright (C) 2005 Tim-Philipp Müller <tim centricular net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include "gst/gst-i18n-plugin.h"

#include <sys/types.h>
#include <string.h>
#include <errno.h>

#include <cdio/cdio.h>

#define GST_TYPE_CDDASRC          (cddasrc_get_type())
#define CDDASRC(obj)              (G_TYPE_CHECK_INSTANCE_CAST((obj), GST_TYPE_CDDASRC, GstCddaSrc))
#define CDDASRC_CLASS(klass)      (G_TYPE_CHECK_CLASS_CAST((klass),  GST_TYPE_CDDASRC, GstCddaSrcClass))
#define GST_IS_CDDASRC(obj)       (G_TYPE_CHECK_INSTANCE_TYPE((obj), GST_TYPE_CDDASRC))
#define GST_IS_CDDASRC_CLASS(obj) (G_TYPE_CHECK_CLASS_TYPE((klass),  GST_TYPE_CDDASRC))

#define SECTORS_PER_SECOND                   (75)
#define SAMPLES_PER_SECTOR                   (CDIO_CD_FRAMESIZE_RAW >> 2)
#define TIME_INTERVAL_FROM_SECTORS(sectors)  ((SAMPLES_PER_SECTOR * sectors * GST_SECOND) / 44100)
#define SECTORS_FROM_TIME_INTERVAL(dtime)    (dtime * 44100 / (SAMPLES_PER_SECTOR * GST_SECOND))

#define GST_TYPE_CDDASRC_MODE (cddasrc_mode_get_type())

enum
{
  ARG_0,
  ARG_MODE,
  ARG_DEVICE,
  ARG_DISCID,
  ARG_EXTENDED_DISCID,
  ARG_TRACK
};

typedef enum
{
  CDDASRC_MODE_NORMAL,          /* stream = one track  */
  CDDASRC_MODE_CONTINUOUS       /* stream = whole disc */
} GstCddaSrcMode;

typedef struct _GstCddaSrc GstCddaSrc;
typedef struct _GstCddaSrcClass GstCddaSrcClass;

struct _GstCddaSrc
{
  GstElement element;

  GstPad *srcpad;

  gchar *device;

  CdIo *cdio;                   /* NULL if not open */

  gint num_tracks;
  gint *track_starts;           /* sector where track begins                       */
  gint *track_ends;             /* first sector not part of track, ie. [start;end[ */
  gint total_sectors;

  gboolean need_discont;
  gint cur_sector;
  gint cur_track;

  gint uri_track;
  gchar *uri;

  gint need_seek;               /* if not -1, we need to seek to the given sector */

  GstCddaSrcMode mode;

  gchar discid[20];             /* this is the short disc ID */
  gchar *discid_full;           /* this is the full disc ID  */
};

struct _GstCddaSrcClass
{
  GstElementClass parent_class;
};

static GstElementDetails cddasrc_details = {
  "CD Audio (cdda) Source",
  "Source/File",
  "Read audio from CD",
  "Tim-Philipp Müller <tim centricular net>",
};

static GstStaticPadTemplate cddasrc_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-raw-int, "
        "endianness = (int) BYTE_ORDER, "
        "signed = (boolean) true, "
        "width = (int) 16, "
        "depth = (int) 16, " "rate = (int) 44100, " "channels = (int) 2")
    );

static void cddasrc_base_init (gpointer g_class);
static void cddasrc_class_init (GstCddaSrcClass * klass);
static void cddasrc_init (GstCddaSrc * cddasrc);
static void cddasrc_finalize (GObject * obj);

static void cddasrc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void cddasrc_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

static GstData *cddasrc_get (GstPad * pad);
static gboolean cddasrc_event (GstPad * pad, GstEvent * event);
static const GstEventMask *cddasrc_get_event_mask (GstPad * pad);
static const GstFormat *cddasrc_get_formats (GstPad * pad);
static gboolean cddasrc_query (GstPad * pad, GstQueryType type,
    GstFormat * format, gint64 * value);
static const GstQueryType *cddasrc_get_query_types (GstPad * pad);

static void cddasrc_uri_handler_init (gpointer g_iface, gpointer iface_data);

static gint cddasrc_get_track_from_sector (GstCddaSrc * src, gint sector);

static void
_do_init (GType cddasrc_type)
{
  static const GInterfaceInfo urihandler_info = {
    cddasrc_uri_handler_init,
    NULL,
    NULL,
  };

  g_type_add_interface_static (cddasrc_type, GST_TYPE_URI_HANDLER,
      &urihandler_info);
}

GST_BOILERPLATE_FULL (GstCddaSrc, cddasrc, GstElement, GST_TYPE_ELEMENT,
    _do_init);

static GstElementStateReturn cddasrc_change_state (GstElement * element);

/* our two formats */
static GstFormat track_format;
static GstFormat sector_format;



/***************************************************************************
 *
 *   cddasrc_mode_get_type
 *
 ***************************************************************************/

static GType
cddasrc_mode_get_type (void)
{
  static GType mode_type;       /* 0 */
  static GEnumValue modes[] = {
    {CDDASRC_MODE_NORMAL, "0", "stream consists of a single track"},
    {CDDASRC_MODE_CONTINUOUS, "1", "stream consists of the whole disc"},
    {0, NULL, NULL}
  };

  if (mode_type == 0)
    mode_type = g_enum_register_static ("GstCddaSrcMode", modes);

  return mode_type;
}

/***************************************************************************
 *
 *   cddasrc_base_init
 *
 ***************************************************************************/

static void
cddasrc_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&cddasrc_src_template));
  gst_element_class_set_details (element_class, &cddasrc_details);

  /* Register the track and sector format */
  track_format = gst_format_register ("track", "CD track");
  sector_format = gst_format_register ("sector", "CD sector");
}

/***************************************************************************
 *
 *   cddasrc_class_init
 *
 ***************************************************************************/

static void
cddasrc_class_init (GstCddaSrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_DEVICE,
      g_param_spec_string ("device", "Device", "CD device location",
          NULL, G_PARAM_READWRITE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_DISCID,
      g_param_spec_string ("discid", "discid", "The disc ID", NULL,
          G_PARAM_READABLE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_EXTENDED_DISCID,
      g_param_spec_string ("extended-discid", "extended discid",
          "The extended disc ID", NULL, G_PARAM_READABLE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_MODE,
      g_param_spec_enum ("mode", "Mode", "Mode", GST_TYPE_CDDASRC_MODE,
          CDDASRC_MODE_NORMAL, G_PARAM_READWRITE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_TRACK,
      g_param_spec_uint ("track", "Track", "Track", 1, 999, 1,
          G_PARAM_READWRITE));

  /* tags */
  gst_tag_register ("discid", GST_TAG_FLAG_META, G_TYPE_STRING,
      _("discid"), _("CDDA discid for metadata retrieval"),
      gst_tag_merge_use_first);

  gobject_class->set_property = cddasrc_set_property;
  gobject_class->get_property = cddasrc_get_property;
  gobject_class->finalize = cddasrc_finalize;

  gstelement_class->change_state = cddasrc_change_state;
}

/***************************************************************************
 *
 *   cddasrc_init
 *
 ***************************************************************************/

static void
cddasrc_init (GstCddaSrc * cddasrc)
{
  char *dev;

  cddasrc->srcpad =
      gst_pad_new_from_template (gst_static_pad_template_get
      (&cddasrc_src_template), "src");
  gst_pad_set_get_function (cddasrc->srcpad, cddasrc_get);
  gst_pad_set_event_function (cddasrc->srcpad, cddasrc_event);
  gst_pad_set_event_mask_function (cddasrc->srcpad, cddasrc_get_event_mask);
  gst_pad_set_query_function (cddasrc->srcpad, cddasrc_query);
  gst_pad_set_query_type_function (cddasrc->srcpad, cddasrc_get_query_types);
  gst_pad_set_formats_function (cddasrc->srcpad, cddasrc_get_formats);

  gst_element_add_pad (GST_ELEMENT (cddasrc), cddasrc->srcpad);

  dev = cdio_get_default_device (NULL);
  cddasrc->device = g_strdup ((dev) ? dev : "/dev/cdrom");

  cddasrc->mode = CDDASRC_MODE_NORMAL;
  cddasrc->need_seek = -1;
  cddasrc->uri_track = -1;

  if (dev)
    free (dev);
}

/***************************************************************************
 *
 *   cddasrc_finalize
 *
 ***************************************************************************/

static void
cddasrc_finalize (GObject * obj)
{
  GstCddaSrc *cddasrc = CDDASRC (obj);

  g_free (cddasrc->uri);
  g_free (cddasrc->device);
  g_free (cddasrc->discid_full);

  G_OBJECT_CLASS (parent_class)->finalize (obj);
}

/***************************************************************************
 *
 *   cddasrc_set_property
 *
 ***************************************************************************/

static void
cddasrc_set_property (GObject * object, guint prop_id, const GValue * value,
    GParamSpec * pspec)
{
  GstCddaSrc *src = CDDASRC (object);

  switch (prop_id) {
    case ARG_MODE:
      src->mode = g_value_get_enum (value);
      src->need_discont = TRUE;
      break;
    case ARG_DEVICE:
    {
      const gchar *dev = g_value_get_string (value);

      g_free (src->device);
      if (dev && *dev)
        src->device = g_strdup (dev);
      else
        src->device = NULL;
      break;
    }
    case ARG_TRACK:
    {
      guint track = g_value_get_uint (value);

      if (src->num_tracks > 0 && track > src->num_tracks) {
        g_warning ("Invalid track %u\n", track);
      } else if (track > 0 && src->track_starts != NULL) {
        src->need_seek = src->track_starts[track - 1];
        src->uri_track = track;
      } else {
        src->uri_track = track; /* seek will be done in state change */
      }
      break;
    }
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

/***************************************************************************
 *
 *   cddasrc_get_property
 *
 ***************************************************************************/

static void
cddasrc_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstCddaSrc *src = CDDASRC (object);

  switch (prop_id) {
    case ARG_MODE:
      g_value_set_enum (value, src->mode);
      break;
    case ARG_DEVICE:
      g_value_set_string (value, src->device);
      break;
    case ARG_EXTENDED_DISCID:
      g_value_set_string (value, src->discid_full);
      break;
    case ARG_DISCID:
      g_value_set_string (value, src->discid);
      break;
    case ARG_TRACK:
      if (src->num_tracks <= 0 && src->uri_track > 0) {
        g_value_set_uint (value, src->uri_track);
      } else {
        g_value_set_uint (value, src->cur_track + 1);
      }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

/***************************************************************************
 *
 *   cddasrc_get_sectors_from_start
 *
 ***************************************************************************/

static gboolean
cddasrc_get_sectors_from_start (GstCddaSrc * src, gint * p_num_sectors,
    guint cur_sec)
{
  if (src->mode == CDDASRC_MODE_NORMAL) {
    gint cur_track = cddasrc_get_track_from_sector (src, cur_sec);

    if (cur_track < 0)
      return FALSE;

    *p_num_sectors = cur_sec - src->track_starts[cur_track];
  } else {
    *p_num_sectors = src->cur_sector - src->track_starts[0];
  }
  return TRUE;
}

/***************************************************************************
 *
 *   cddasrc_do_seek
 *
 ***************************************************************************/

static gboolean
cddasrc_do_seek (GstCddaSrc * src)
{
  guint new_track;

  GST_DEBUG ("Seeking to sector %d", src->need_seek);

  if (src->need_seek < 0
      || src->need_seek >= src->track_ends[src->num_tracks - 1]) {
    GST_ELEMENT_ERROR (src, RESOURCE, SEEK, (_("Invalid seek")),
        ("Sector out of bounds"));
    src->need_seek = -1;
    return FALSE;
  }

  new_track = cddasrc_get_track_from_sector (src, src->need_seek);
  /* shouldn't happen */
  if (new_track < 0) {
    src->need_seek = -1;
    GST_ELEMENT_ERROR (src, RESOURCE, SEEK,
        (_("Invalid seek")), ("Sector out of bounds"));
    return FALSE;
  }

  src->cur_sector = src->need_seek;

  if (src->cur_track != new_track) {
    src->cur_track = new_track;
    g_object_notify (G_OBJECT (src), "track");
  }

  GST_DEBUG ("New current track: %d", src->need_seek, src->cur_track);

  src->need_seek = -1;
  src->need_discont = TRUE;

  GST_DEBUG ("Seek done");

  return TRUE;
}

/***************************************************************************
 *
 *   cddasrc_get
 *
 ***************************************************************************/

static GstData *
cddasrc_get (GstPad * pad)
{
  GstCddaSrc *src;
  GstBuffer *buf;
  gint sectors = -1;

  src = CDDASRC (gst_pad_get_parent (pad));

  g_return_val_if_fail (src->cdio != NULL, NULL);

  if (src->need_seek >= 0) {
    if (!cddasrc_do_seek (src))
      return NULL;
    /* will set need_discont which is handled below */
  }

  g_return_val_if_fail (src->cur_track >= 0, NULL);
  g_return_val_if_fail (src->cur_track < src->num_tracks, NULL);

  if (!cddasrc_get_sectors_from_start (src, &sectors, src->cur_sector)) {
    GST_DEBUG ("end of disc, setting EOS");
    gst_element_set_eos (GST_ELEMENT (src));
    return GST_DATA (gst_event_new (GST_EVENT_EOS));
  }

  if (src->need_discont) {
    GstEvent *discont_ev;
    guint64 timestamp;

    timestamp = TIME_INTERVAL_FROM_SECTORS (sectors);
    discont_ev = gst_event_new_discontinuous (FALSE,
        GST_FORMAT_TIME, (guint64) timestamp,
        GST_FORMAT_DEFAULT, (guint64) sectors * SAMPLES_PER_SECTOR,
        GST_FORMAT_UNDEFINED);

    GST_DEBUG ("Sending discont %.6fs\n", (double) timestamp / GST_SECOND);
    src->need_discont = FALSE;
    return GST_DATA (discont_ev);
  }

  if (src->cur_sector >= src->track_ends[src->cur_track]) {
    /* stream = one track */
    if (src->mode == CDDASRC_MODE_NORMAL) {
      GST_DEBUG ("end of track, setting EOS");
      gst_element_set_eos (GST_ELEMENT (src));
      return GST_DATA (gst_event_new (GST_EVENT_EOS));
    }
    /* stream = whole disc */
    if (src->cur_sector >= src->track_ends[src->num_tracks - 1]) {
      GST_DEBUG ("end of disc, setting EOS");
      gst_element_set_eos (GST_ELEMENT (src));
      return GST_DATA (gst_event_new (GST_EVENT_EOS));
    }
    GST_DEBUG ("end of track, continuing with next track");
    ++src->cur_track;
    g_object_notify (G_OBJECT (src), "track");
    g_assert (src->cur_track <= src->track_starts[src->cur_track]);
  }

  buf = gst_pad_alloc_buffer (pad, 0, CDIO_CD_FRAMESIZE_RAW);

  if (cdio_read_audio_sector (src->cdio, GST_BUFFER_DATA (buf),
          (lsn_t) src->cur_sector) != 0) {
    gst_data_unref (GST_DATA (buf));
    return NULL;
  }

  GST_BUFFER_TIMESTAMP (buf) = TIME_INTERVAL_FROM_SECTORS (sectors);
  GST_BUFFER_DURATION (buf) = TIME_INTERVAL_FROM_SECTORS (1);
  GST_BUFFER_OFFSET (buf) = sectors * SAMPLES_PER_SECTOR;

  ++src->cur_sector;

  return GST_DATA (buf);
}

/***************************************************************************
 *
 *   cddasrc_cddb_sum
 *
 *   2344 becomes 2+3+4+4 = 13
 *
 ***************************************************************************/

static guint 
cddasrc_cddb_sum (guint n) 
{
  
guint ret = 0;
  

while (n > 0) {
    
ret = ret + (n % 10);
    
n = n / 10;
  
}
  

return ret;

}


/***************************************************************************
 *
 *   cddasrc_calculate_discid
 *
 ***************************************************************************/
    static void
cddasrc_calculate_discid (GstCddaSrc * src)
{
  GstTagList *taglist;
  GString *s;
  guint64 total_secs;
  guint32 disc_id, check_sum;
  gint i;

  s = g_string_new (NULL);

  check_sum = 0;
  for (i = 0; i < src->num_tracks; ++i) {
    guint sector = src->track_starts[i] + 150;  /* 150 = MSF_OFFSET */

    
check_sum += cddasrc_cddb_sum (sector / SECTORS_PER_SECOND);
  
}
  
total_secs = src->total_sectors / SECTORS_PER_SECOND;
  disc_id =
      ((check_sum & 0xFF) << 24) | ((guint16) total_secs << 8) | src->
      num_tracks;
  
g_string_append_printf (s, "%08x", disc_id);
  
g_snprintf (src->discid, sizeof (src->discid), "%08x", disc_id);
  
      /* print number of tracks */ 
      g_string_append_printf (s, " %d", src->num_tracks);
  

      /* print frame offsets of all tracks */ 
      for (i = 0; i < src->num_tracks; ++i) {
    
g_string_append_printf (s, " %d", src->track_starts[i] + 150);     /* 150 = MSF_OFFSET */
  
}
  

      /* print length of disc in seconds */ 
      g_string_append_printf (s, " %u",
      (guint) total_secs + (150 / SECTORS_PER_SECOND));

  g_free (src->discid_full);
  src->discid_full = g_string_free (s, FALSE);
  
GST_DEBUG ("discid: %s", src->discid_full);

  g_object_notify (G_OBJECT (src), "discid");
  g_object_notify (G_OBJECT (src), "extended-discid");

  taglist = gst_tag_list_new ();
  gst_tag_list_add (taglist, GST_TAG_MERGE_APPEND, "discid", src->discid, NULL);
  gst_element_found_tags (GST_ELEMENT (src), taglist);
  /* not much point in sending a tag event downstream */
}

/***************************************************************************
 *
 *   cddasrc_open
 *
 ***************************************************************************/

static gboolean
cddasrc_open (GstCddaSrc * src)
{
  gint first_track, i;

  g_return_val_if_fail (src->cdio == NULL, FALSE);

  GST_DEBUG_OBJECT (src, "trying to open device...");

  /* find the device */
  if (src->device)
    src->cdio = cdio_open (src->device, DRIVER_UNKNOWN);

  if (src->cdio == NULL) {
    GST_ELEMENT_ERROR (src, RESOURCE, OPEN_READ,
        (_("Could not open CD device for reading.")), ("cdio_open() failed"));
    return FALSE;
  }

  first_track = cdio_get_first_track_num (src->cdio);
  src->num_tracks = cdio_get_num_tracks (src->cdio);
  if (src->num_tracks <= 0 || first_track < 0) {
    GST_ELEMENT_ERROR (src, RESOURCE, OPEN_READ,
        (_("This CD has no tracks")), (NULL));
    return FALSE;
  }

  src->track_starts = g_new0 (gint, src->num_tracks);
  src->track_ends = g_new0 (gint, src->num_tracks);

  for (i = 0; i < src->num_tracks; ++i) {
    gint len_sectors;

    src->track_starts[i] = cdio_get_track_lsn (src->cdio, i + first_track);
    len_sectors = cdio_get_track_sec_count (src->cdio, i + first_track);
    src->track_ends[i] = src->track_starts[i] + len_sectors;
  }

  src->total_sectors =
      src->track_ends[src->num_tracks - 1] - src->track_starts[0];
  src->cur_sector = src->track_starts[0];
  src->cur_track = 0;
  src->need_discont = TRUE;
  src->need_seek = -1;

  GST_DEBUG_OBJECT (src, "device successfully openend");

  cddasrc_calculate_discid (src);

  return TRUE;
}

/***************************************************************************
 *
 *   cddasrc_close
 *
 ***************************************************************************/

static void
cddasrc_close (GstCddaSrc * src)
{
  g_return_if_fail (src->cdio != NULL);

  GST_DEBUG_OBJECT (src, "closing device...");

  /* kill the paranoia state */
  cdio_destroy (src->cdio);
  src->cdio = NULL;

  g_free (src->track_starts);
  g_free (src->track_ends);
  src->track_starts = NULL;
  src->track_ends = NULL;
  src->total_sectors = 0;
  src->cur_track = 0;
  src->num_tracks = 0;
}

/***************************************************************************
 *
 *   cddasrc_change_state
 *
 ***************************************************************************/

static GstElementStateReturn
cddasrc_change_state (GstElement * element)
{
  GstCddaSrc *src;

  g_return_val_if_fail (GST_IS_CDDASRC (element), GST_STATE_FAILURE);

  src = CDDASRC (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_NULL_TO_READY:
      if (!cddasrc_open (src))
        return GST_STATE_FAILURE;
      break;
    case GST_STATE_READY_TO_PAUSED:
      if (src->uri_track > 0) {
        if (src->uri_track > src->num_tracks) {
          GST_ELEMENT_ERROR (src, RESOURCE, SEEK,
              (_("Invalid track")), ("Track out of bounds"));
          return GST_STATE_FAILURE;
        }
        src->need_seek = src->track_starts[src->uri_track - 1];
      }
      break;
    case GST_STATE_PAUSED_TO_PLAYING:
      break;
    case GST_STATE_PLAYING_TO_PAUSED:
      break;
    case GST_STATE_PAUSED_TO_READY:
      src->need_seek = src->track_starts[0];
      break;
    case GST_STATE_READY_TO_NULL:
      cddasrc_close (src);
      src->need_seek = -1;
      break;
    default:
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}

/***************************************************************************
 *
 *   cddasrc_get_event_mask
 *
 ***************************************************************************/

static const GstEventMask *
cddasrc_get_event_mask (GstPad * pad)
{
  static const GstEventMask masks[] = {
    {GST_EVENT_SEEK, GST_SEEK_METHOD_SET | GST_SEEK_FLAG_FLUSH},
    {0,}
  };

  return masks;
}

/***************************************************************************
 *
 *   cddasrc_event
 *
 ***************************************************************************/

static gboolean
cddasrc_event (GstPad * pad, GstEvent * event)
{
  GstCddaSrc *src;
  GstSeekType method;
  GstFormat format;
  gint64 offset, sector;

  src = CDDASRC (gst_pad_get_parent (pad));

  if (src->cdio == NULL) {
    GST_DEBUG ("device not open, cannot handle event");
    gst_event_unref (event);
    return FALSE;
  }

  if (GST_EVENT_TYPE (event) != GST_EVENT_SEEK) {
    GST_DEBUG ("dropping event of type %u", GST_EVENT_TYPE (event));
    gst_event_unref (event);
    return FALSE;
  }

  format = GST_EVENT_SEEK_FORMAT (event);
  offset = GST_EVENT_SEEK_OFFSET (event);
  method = GST_EVENT_SEEK_METHOD (event);

  gst_event_unref (event);
  event = NULL;

  if (offset < 0 || method != GST_SEEK_METHOD_SET)
    return FALSE;

  /* we can only seek on sectors, so we convert 
   * the requested offsets to sectors first */
  if (format == GST_FORMAT_TIME) {
    if (src->mode == CDDASRC_MODE_NORMAL) {
      sector =
          src->track_starts[src->cur_track] +
          SECTORS_FROM_TIME_INTERVAL (offset);
    } else {
      sector = src->track_starts[0] + SECTORS_FROM_TIME_INTERVAL (offset);
    }
  } else if (format == track_format) {
    sector = src->track_starts[CLAMP (offset, 0, src->num_tracks - 1)];
  } else if (format == sector_format) {
    sector = offset;
  } else if (format == GST_FORMAT_BYTES) {
    sector = (offset >> 2) / SAMPLES_PER_SECTOR;
  } else if (format == GST_FORMAT_DEFAULT) {
    sector = offset / SAMPLES_PER_SECTOR;
  } else {
    return FALSE;
  }

  src->need_seek = (gint) sector;
  GST_DEBUG ("Trying to seek to sector %d", src->need_seek);

  return cddasrc_do_seek (src);
}

/***************************************************************************
 *
 *   cddasrc_get_formats
 *
 ***************************************************************************/

static const GstFormat *
cddasrc_get_formats (GstPad * pad)
{
  static GstFormat formats[] = {
    GST_FORMAT_TIME,
    GST_FORMAT_BYTES,
    GST_FORMAT_DEFAULT,
    0,                          /* filled later */
    0,                          /* filled later */
    0
  };

  formats[3] = track_format;
  formats[4] = sector_format;

  return formats;
}

/***************************************************************************
 *
 *   cddasrc_get_track_from_sector
 *
 ***************************************************************************/

static gint
cddasrc_get_track_from_sector (GstCddaSrc * src, gint sector)
{
  gint i;

  for (i = 0; i < src->num_tracks; ++i) {
    if (sector >= src->track_starts[i] && sector < src->track_ends[i])
      return (gint64) i;
  }
  return -1;
}


/***************************************************************************
 *
 *   cddasrc_get_query_types
 *
 ***************************************************************************/

static const GstQueryType *
cddasrc_get_query_types (GstPad * pad)
{
  static const GstQueryType src_query_types[] = {
    GST_QUERY_TOTAL,
    GST_QUERY_POSITION,
    0
  };

  return src_query_types;
}

/***************************************************************************
 *
 *   cddasrc_query
 *
 ***************************************************************************/

static gboolean
cddasrc_query (GstPad * pad, GstQueryType type,
    GstFormat * format, gint64 * value)
{
  GstCddaSrc *src;
  gint64 val;

  src = CDDASRC (gst_pad_get_parent (pad));

  if (src->cdio == NULL || src->cur_track < 0
      || src->cur_track >= src->num_tracks)
    return FALSE;

  switch (type) {
    case GST_QUERY_TOTAL:
      if (src->mode == CDDASRC_MODE_NORMAL) {
        val =
            src->track_ends[src->cur_track] - src->track_starts[src->cur_track];
      } else {
        val = src->total_sectors;
      }
      break;

    case GST_QUERY_POSITION:
      if (src->mode == CDDASRC_MODE_NORMAL) {
        val = src->cur_sector - src->track_starts[src->cur_track];
      } else {
        val = src->cur_sector - src->track_starts[0];
      }
      break;

    default:
      return FALSE;
  }

  if (*format == GST_FORMAT_TIME) {
    *value = TIME_INTERVAL_FROM_SECTORS (val);
  } else if (*format == track_format) {
    if (type == GST_QUERY_TOTAL)
      *value = src->num_tracks;
    else
      *value = src->cur_track;
  } else if (*format == sector_format) {
    *value = val;
  } else if (*format == GST_FORMAT_BYTES) {
    *value = (val * SAMPLES_PER_SECTOR) << 2;
  } else if (*format == GST_FORMAT_DEFAULT) {
    *value = val * SAMPLES_PER_SECTOR;
  } else {
    return FALSE;
  }

  return TRUE;
}

/***************************************************************************
 *
 *   plugin_init
 *
 ***************************************************************************/

static gboolean
plugin_init (GstPlugin * plugin)
{
  if (!gst_element_register (plugin, "cddasrc", GST_RANK_PRIMARY,
          GST_TYPE_CDDASRC))
    return FALSE;

  return TRUE;
}

/***************************************************************************
 *
 *   cddasrc_uri_get_type
 *
 ***************************************************************************/

static guint
cddasrc_uri_get_type (void)
{
  return GST_URI_SRC;
}

/***************************************************************************
 *
 *   cddasrc_uri_get_protocols
 *
 ***************************************************************************/

static gchar **
cddasrc_uri_get_protocols (void)
{
  static gchar *protocols[] = { "cdda", NULL };

  return protocols;
}

/***************************************************************************
 *
 *   cddasrc_uri_get_uri
 *
 ***************************************************************************/

static const gchar *
cddasrc_uri_get_uri (GstURIHandler * handler)
{
  GstCddaSrc *src = CDDASRC (handler);

  src->uri = g_realloc (src->uri, 64);
  g_snprintf (src->uri, 64, "cdda://%d", src->uri_track);

  return src->uri;
}

/***************************************************************************
 *
 *   cddasrc_uri_set_uri
 *
 ***************************************************************************/

static gboolean
cddasrc_uri_set_uri (GstURIHandler * handler, const gchar * uri)
{
  gchar *protocol, *location;

  GstCddaSrc *cddasrc = CDDASRC (handler);

  protocol = gst_uri_get_protocol (uri);
  if (protocol && strcmp (protocol, "cdda") != 0) {
    g_free (protocol);
    return FALSE;
  }
  g_free (protocol);

  location = gst_uri_get_location (uri);
  cddasrc->uri_track = strtol (location, NULL, 10);
  g_free (location);

  if (cddasrc->uri_track == 0)
    return FALSE;

  if (cddasrc->num_tracks > 0 && cddasrc->uri_track > cddasrc->num_tracks)
    return FALSE;

  if (cddasrc->uri_track > 0 && cddasrc->track_starts != NULL) {
    cddasrc->need_seek = cddasrc->track_starts[cddasrc->uri_track - 1];
  } else {
    /* seek will be done in state change */
  }

  return TRUE;
}

/***************************************************************************
 *
 *   cddasrc_uri_handler_init
 *
 ***************************************************************************/

static void
cddasrc_uri_handler_init (gpointer g_iface, gpointer iface_data)
{
  GstURIHandlerInterface *iface = (GstURIHandlerInterface *) g_iface;

  iface->get_type = cddasrc_uri_get_type;
  iface->get_protocols = cddasrc_uri_get_protocols;
  iface->get_uri = cddasrc_uri_get_uri;
  iface->set_uri = cddasrc_uri_set_uri;
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "cddasrc",
    "Read audio from CD", plugin_init, VERSION, "LGPL", GST_PACKAGE, GST_ORIGIN)
