/* GStreamer SPC plugin
 * Copyright (C) 2003 Chris Lee <clee@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <fcntl.h>

#include "gstspc.h"

struct _GstSpc
{
  GstElement element;
  GstPad *sinkpad, *srcpad;

  GstBuffer *buf;
  gboolean eos, initialized;
  GstClockTime samples;
};

struct _GstSpcClass
{
  GstElementClass parent_class;
};

static GstElementDetails gst_spc_details =
GST_ELEMENT_DETAILS ("SNESAPU SPC interpreter",
    "Codec/Decoder/Audio",
    "Uses SNESAPU to emulate an SPC processor",
    "Chris Lee <clee@kde.org>");

static GstStaticPadTemplate sink_factory =
GST_STATIC_PAD_TEMPLATE ("sink", GST_PAD_SINK, GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-spc"));

static GstStaticPadTemplate src_factory =
GST_STATIC_PAD_TEMPLATE ("src", GST_PAD_SRC, GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-raw-int, "
        "endianness = (int) BYTE_ORDER, "
        "signed = (boolean) TRUE, "
        "width = (int) 16, "
        "depth = (int) 16, " "rate = (int) 32000, " "channels = (int) 2"));

static void gst_spc_base_init (gpointer g_class);
static void gst_spc_class_init (GstSpcClass * klass);
static void gst_spc_init (GstSpc * spc);
static gboolean gst_spc_query (GstPad * pad,
    GstQueryType query, GstFormat * format, gint64 * value);
static void gst_spc_loop (GstElement * element);
static void spc_setup (GstSpc * spc);
static GstElementStateReturn gst_spc_change_state (GstElement * element);

static GstElementClass *parent_class = NULL;

GType
gst_spc_get_type (void)
{
  static GType spc_t = 0;

  if (!spc_t) {
    static const GTypeInfo spc_info = {
      sizeof (GstSpcClass),
      gst_spc_base_init, NULL,
      (GClassInitFunc) gst_spc_class_init, NULL, NULL,
      sizeof (GstSpc), 0,
      (GInstanceInitFunc) gst_spc_init
    };

    spc_t = g_type_register_static (GST_TYPE_ELEMENT, "GstSpc", &spc_info, 0);
  }

  return spc_t;
}

static void
gst_spc_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&sink_factory));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_factory));
  gst_element_class_set_details (element_class, &gst_spc_details);
}

static void
gst_spc_class_init (GstSpcClass * klass)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (klass);

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  gstelement_class->change_state = gst_spc_change_state;
}

static const GstFormat *
gst_spc_get_formats (GstPad * pad)
{
  static GstFormat formats[] = {
    GST_FORMAT_DEFAULT,
    GST_FORMAT_TIME,
    0
  };

  return formats;
}

static const GstQueryType *
gst_spc_get_query_types (GstPad * pad)
{
  static const GstQueryType query_types[] = {
    GST_QUERY_POSITION,
    0
  };

  return query_types;
}

static void
gst_spc_init (GstSpc * spc)
{
  spc->sinkpad =
      gst_pad_new_from_template (gst_element_class_get_pad_template
      (GST_ELEMENT_GET_CLASS (spc), "sink"), "sink");

  spc->srcpad =
      gst_pad_new_from_template (gst_element_class_get_pad_template
      (GST_ELEMENT_GET_CLASS (spc), "src"), "src");
  gst_pad_set_query_type_function (spc->srcpad, gst_spc_get_query_types);
  gst_pad_set_query_function (spc->srcpad, gst_spc_query);
  gst_pad_set_formats_function (spc->srcpad, gst_spc_get_formats);

  gst_element_add_pad (GST_ELEMENT (spc), spc->sinkpad);
  gst_element_add_pad (GST_ELEMENT (spc), spc->srcpad);
  gst_element_set_loop_function (GST_ELEMENT (spc), gst_spc_loop);

  spc->buf = NULL;
  spc->initialized = FALSE;
  spc->samples = 0;

  GST_FLAG_SET (spc, GST_ELEMENT_EVENT_AWARE);
}

static gboolean
gst_spc_query (GstPad * pad,
    GstQueryType query, GstFormat * format, gint64 * value)
{
  GstSpc *spc = GST_SPC (gst_pad_get_parent (pad));

  switch (query) {
    case GST_QUERY_POSITION:
      switch (*format) {
        case GST_FORMAT_TIME:
          *value = spc->samples * GST_SECOND / 32000;
          break;
        case GST_FORMAT_DEFAULT:
          *value = spc->samples;
          break;
        default:
          return FALSE;
      }
      break;
    default:
      return FALSE;
  }

  return TRUE;
}

static void
gst_spc_loop (GstElement * element)
{
  GstSpc *spc;

  spc = GST_SPC (element);

  if (spc->initialized == TRUE) {
    GstBuffer *audio = gst_buffer_new_and_alloc (1600 * 4);

    GST_BUFFER_TIMESTAMP (audio) = GST_SECOND * spc->samples / 32000;
    GST_BUFFER_DURATION (audio) = GST_SECOND / 20;
    spc->samples += 1600;
    EmuAPU (GST_BUFFER_DATA (audio), APU_CLK / 20, 1600);
    gst_pad_push (spc->srcpad, GST_DATA (audio));
  } else {
    GstBuffer *buffer_in;

    while ((buffer_in = GST_BUFFER (gst_pad_pull (spc->sinkpad)))) {
      if (GST_IS_EVENT (buffer_in)) {
        GstEvent *event = GST_EVENT (buffer_in);

        if (GST_EVENT_TYPE (event) == GST_EVENT_EOS) {
          gst_event_unref (event);
          break;
        } else if (GST_EVENT_TYPE (event) == GST_EVENT_INTERRUPT) {
          gst_event_unref (event);
          return;
        } else {
          gst_pad_event_default (spc->sinkpad, event);
        }
      } else {
        if (spc->buf == NULL) {
          spc->buf = buffer_in;
        } else {
          GstBuffer *m = gst_buffer_join (spc->buf, buffer_in);

          spc->buf = m;
        }
      }
    }

    spc_setup (spc);
  }
}

static void
spc_setup (GstSpc * spc)
{
  gchar *data = (gchar *) GST_BUFFER_DATA (spc->buf);

  if (data[0x23] == 26) {
    GstEvent *tagevent;
    GstTagList *taglist = gst_tag_list_new ();

    gchar spctitle[0x21];
    gchar spcartist[0x21];
    gchar spcgame[0x21];

    strncpy (spctitle, &data[0x2E], 32);
    strncpy (spcartist, &data[0xB0], 32);
    strncpy (spcgame, &data[0x4E], 32);

    spctitle[0x20] = 0;
    spcartist[0x20] = 0;
    spcgame[0x20] = 0;

    gst_tag_list_add (taglist, GST_TAG_MERGE_APPEND,
        GST_TAG_TITLE, spctitle,
        GST_TAG_ARTIST, spcartist, GST_TAG_ALBUM, spcgame, NULL);

    tagevent = gst_event_new_tag (taglist);
    gst_element_found_tags_for_pad (GST_ELEMENT (spc), spc->srcpad, 0, taglist);
  }

  InitAPU ();

  LoadSPCFile (GST_BUFFER_DATA (spc->buf));
  gst_buffer_unref (spc->buf);
  spc->buf = NULL;
  spc->initialized = TRUE;
  spc->samples = 0;
}

static GstElementStateReturn
gst_spc_change_state (GstElement * element)
{
  GstSpc *spc = GST_SPC (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_PAUSED_TO_READY:
      spc->initialized = FALSE;
      spc->samples = 0;
      if (spc->buf) {
        gst_buffer_unref (spc->buf);
        spc->buf = NULL;
      }
      break;
    default:
      break;
  }

  return parent_class->change_state (element);
}

static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "spcdec", GST_RANK_PRIMARY,
      GST_TYPE_SPC);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "spcdec",
    "SPC Sound Filter",
    plugin_init, VERSION, "LGPL", "Chris Lee", "http://c133.org/");
