/* GStreamer
 * Copyright (C) <2001> Steve Baker <stevebaker_org@yahoo.co.uk>
 * Copyright (C) 2002, 2003 Andy Wingo <wingo at pobox dot com>
 *
 * float2_2_int.c
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/floatcast/floatcast.h>

#include "float22int.h"

/* elementfactory information */
static GstElementDetails float2_2_int_details = {
  "Float to Integer effect",
  "Filter/Converter/Audio",
  "Convert from floating point to integer audio data",
  "Benjamin Otte <in7y118@public.uni-hamburg.de>"
};


static GstStaticPadTemplate float2_2_int_sink_factory =
GST_STATIC_PAD_TEMPLATE (
  "sink",
  GST_PAD_SINK,
  GST_PAD_REQUEST,
  GST_STATIC_CAPS ("audio/x-raw-float, "
    "width = (int) 32, "
    "endianness = (int) BYTE_ORDER, "
    "buffer-frames = (int) [ 0, MAX ], "
    /* "layout = (string) { \"interleaved\", \"sequential\" }, " was an idea... */
    "channels = (int) [ 1, MAX ], "
    "rate = (int) [ 1, MAX ]"
  )
);
static GstStaticPadTemplate float2_2_int_src_factory =
GST_STATIC_PAD_TEMPLATE (
  "src",
  GST_PAD_SRC,
  GST_PAD_ALWAYS,
  GST_STATIC_CAPS ("audio/x-raw-int, "
    "endianness = (int) BYTE_ORDER, "
    "width = (int) 16, "
    "depth = (int) 16, "
    "signed = (boolean) true, "
    "rate = (int) [ 1, MAX ], "
    "channels = (int) [1, MAX ]"
  )
);

static GstCaps *	gst_float2_2_int_getcaps		(GstPad *pad);
static GstPadLinkReturn	gst_float2_2_int_link			(GstPad *pad, const GstCaps *caps);

static void		gst_float2_2_int_chain			(GstPad *pad, GstData *data);

			
GST_BOILERPLATE (GstFloat22Int, gst_float2_2_int, GstElement, GST_TYPE_ELEMENT)

static void
gst_float2_2_int_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details (gstelement_class, &float2_2_int_details);
  gst_element_class_add_pad_template (gstelement_class,
	gst_static_pad_template_get (&float2_2_int_src_factory));
  gst_element_class_add_pad_template (gstelement_class,
	gst_static_pad_template_get (&float2_2_int_sink_factory));
}

static void
gst_float2_2_int_class_init (GstFloat22IntClass *klass)
{
}

static void
gst_float2_2_int_init (GstFloat22Int *plugin)
{
  plugin->sinkpad = gst_pad_new_from_template (
			gst_static_pad_template_get (&float2_2_int_sink_factory), "sink");
  gst_pad_set_link_function (plugin->sinkpad, gst_float2_2_int_link);
  gst_pad_set_getcaps_function (plugin->sinkpad, gst_float2_2_int_getcaps);
  gst_pad_set_chain_function (plugin->sinkpad, gst_float2_2_int_chain);
  gst_element_add_pad (GST_ELEMENT (plugin), plugin->sinkpad);

  plugin->srcpad = gst_pad_new_from_template (
			gst_static_pad_template_get (&float2_2_int_src_factory), "src");
  gst_pad_set_getcaps_function (plugin->srcpad, gst_float2_2_int_getcaps);
  gst_pad_set_link_function (plugin->srcpad, gst_float2_2_int_link);
  gst_element_add_pad (GST_ELEMENT (plugin), plugin->srcpad);
}

static GstCaps *
gst_float2_2_int_getcaps (GstPad *pad)
{
  GstFloat22Int *filter;
  GstPad *otherpad;
  GstCaps *ret;
  GstCaps *copy;
  guint i;

  filter = GST_FLOAT2_2_INT (GST_PAD_PARENT (pad));
  otherpad = (pad == filter->sinkpad ? filter->srcpad : filter->sinkpad);
  copy = gst_pad_get_allowed_caps (otherpad);
  for (i = 0; i < gst_caps_get_size (copy); i++) {
    GstStructure *structure = gst_caps_get_structure (copy, i);
    gst_structure_set_name (structure, pad == filter->srcpad ? 
	"audio/x-raw-int" : "audio/x-raw-float");
    gst_structure_remove_fields (structure, "width", "endianness", 
	"layout", "buffer-frames", "depth", "signed", NULL);
  }
  
  ret = gst_caps_intersect (copy, gst_pad_get_pad_template_caps (pad));
  gst_caps_free (copy);
  return ret;
}

static GstPadLinkReturn
gst_float2_2_int_link (GstPad *pad, const GstCaps *caps)
{
  GstFloat22Int *filter;
  GstPad *otherpad;
  GstStructure *structure;
  gint channels, rate;
  GstPadLinkReturn ret;
  GstCaps *copy;
  
  filter = GST_FLOAT2_2_INT (GST_PAD_PARENT (pad));
  
  structure = gst_caps_get_structure (caps, 0);
  if (!gst_structure_get_int  (structure, "channels", &channels) ||
      !gst_structure_get_int  (structure, "rate", &rate)) {
    g_assert_not_reached ();
  }

  otherpad = (pad == filter->sinkpad ? filter->srcpad : filter->sinkpad);
  copy = gst_pad_get_allowed_caps (otherpad);
  gst_caps_set_simple (copy, 
      "rate",	  G_TYPE_INT,	rate,
      "channels", G_TYPE_INT,	channels,
      NULL);
  ret = gst_pad_try_set_caps (otherpad, copy);
  gst_caps_free (copy);
  if (GST_PAD_LINK_FAILED (ret))
    return ret;
  
  filter->channels = channels;
  filter->rate = rate;
#if 0
  if (pad == filter->sinkpad) {
    filter->interleaved = 
      (strcmp (gst_structure_get_string (structure, "layout"), "interleaved") == 0) ?
      TRUE : FALSE;
  }
#else
  filter->interleaved = TRUE;
#endif
  return ret;
}

static void
gst_float2_2_int_chain (GstPad *pad, GstData *data)
{
  GstFloat22Int *f2i = GST_FLOAT2_2_INT (gst_pad_get_parent (pad));
  GstBuffer *buf_out;
  gint num_frames, i, j;

  if (f2i->channels == 0)
  {
    GST_ELEMENT_ERROR (f2i, CORE, NEGOTIATION, (NULL), (NULL));
    return;
  }
  /* get new output buffer */
  buf_out = gst_pad_alloc_buffer (f2i->srcpad,
      GST_BUFFER_OFFSET (data), GST_BUFFER_SIZE (data) / 2);
  gst_buffer_stamp (buf_out, GST_BUFFER (data));
  num_frames = GST_BUFFER_SIZE (data) / f2i->channels / sizeof (float);
  g_return_if_fail (GST_BUFFER_SIZE (buf_out) == num_frames * f2i->channels * sizeof (gint16));
  
  if (f2i->interleaved) {
    gfloat *data_in = ((gfloat *) GST_BUFFER_DATA (data));
    gint16 *data_out = ((gint16 *) GST_BUFFER_DATA (buf_out));
    for (i = 0; i < f2i->channels * num_frames; i++) {
      *data_out = (gint16) CLAMP (gst_cast_float(*data_in * 32767.0F), -32768, 32767);
      data_out++;
      data_in++;
    }
  } else {
    for (i = 0; i < f2i->channels; i++) {
      gfloat *data_in = ((gfloat *) GST_BUFFER_DATA (data)) + i * num_frames;
      gint16 *data_out = ((gint16 *) GST_BUFFER_DATA (buf_out)) + i;

      for (j = 0; j < num_frames; j++) {
	*data_out = (gint16) CLAMP (gst_cast_float(*data_in * 32767.0F), -32768, 32767);
	data_out += f2i->channels;
	data_in++;
      }
    }
  }
  gst_data_unref (data);
  gst_pad_push (f2i->srcpad, GST_DATA (buf_out));
}
