/* GStreamer
 * Copyright (C) <2001> Steve Baker <stevebaker_org@yahoo.co.uk>
 * Copyright (C) 2002, 2003 Andy Wingo <wingo at pobox dot com>
 *
 * float2int.c
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/floatcast/floatcast.h>

#include "gstfloat2int.h"
#include <gst/audio/audio.h>

/* elementfactory information */
static GstElementDetails float2int_details = {
  "Float to Integer effect",
  "Filter/Converter/Audio",
  "Convert from floating point to integer audio data",
  "Steve Baker <stevebaker_org@yahoo.co.uk>, Andy Wingo <wingo at pobox dot com>"
};


#define GST_FLOAT2INT_DEFAULT_BUFFER_FRAMES 256
#if 0
#define F2I_DEBUG(...) g_message (__VA_ARGS__)
#else
#define F2I_DEBUG(...)
#endif


GST_PAD_TEMPLATE_FACTORY (float2int_sink_factory,
  "sink%d",
  GST_PAD_SINK,
  GST_PAD_REQUEST,
  GST_CAPS_NEW (
    "float2int_sink",
    "audio/x-raw-float",
    "rate",		GST_PROPS_INT_RANGE (1, G_MAXINT),
    "width",		GST_PROPS_INT (32),
    "endianness",	GST_PROPS_INT (G_BYTE_ORDER),
    "buffer-frames",	GST_PROPS_INT_RANGE (1, G_MAXINT),
    "channels",		GST_PROPS_INT (1)
  )
);

GST_PAD_TEMPLATE_FACTORY (float2int_src_factory,
  "src",
  GST_PAD_SRC,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "float2int_src",
    "audio/x-raw-int",
    "channels",   GST_PROPS_INT_RANGE (1, G_MAXINT),
    "rate",       GST_PROPS_INT_RANGE (1, G_MAXINT),
    "endianness", GST_PROPS_INT (G_BYTE_ORDER),
    "width",      GST_PROPS_INT (16),
    "depth",      GST_PROPS_INT (16),
    "signed",     GST_PROPS_BOOLEAN (TRUE)
  )
);

static void	gst_float2int_class_init		(GstFloat2IntClass *klass);
static void	gst_float2int_base_init			(GstFloat2IntClass *klass);
static void	gst_float2int_init			(GstFloat2Int *plugin);

static GstPadLinkReturn gst_float2int_link		(GstPad *pad, GstCaps *caps);

static GstPad*  gst_float2int_request_new_pad		(GstElement *element, GstPadTemplate *temp,
                                                         const gchar *unused);
static void     gst_float2int_pad_removed               (GstElement *element, GstPad *pad);

static void	gst_float2int_loop			(GstElement *element);

static GstElementStateReturn gst_float2int_change_state (GstElement *element);

static GstElementClass *parent_class = NULL;

GType
gst_float2int_get_type(void) {
  static GType float2int_type = 0;

  if (!float2int_type) {
    static const GTypeInfo float2int_info = {
      sizeof(GstFloat2IntClass),
      (GBaseInitFunc)gst_float2int_base_init,
      NULL,
      (GClassInitFunc)gst_float2int_class_init,
      NULL,
      NULL,
      sizeof(GstFloat2Int),
      0,
      (GInstanceInitFunc)gst_float2int_init,
    };
    float2int_type = g_type_register_static(GST_TYPE_ELEMENT, "GstFloat2Int", &float2int_info, 0);
  }
  return float2int_type;
}

static void
gst_float2int_base_init (GstFloat2IntClass *klass)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details (gstelement_class, &float2int_details);
  gst_element_class_add_pad_template (gstelement_class,
	GST_PAD_TEMPLATE_GET (float2int_src_factory));
  gst_element_class_add_pad_template (gstelement_class,
	GST_PAD_TEMPLATE_GET (float2int_sink_factory));
}

static void
gst_float2int_class_init (GstFloat2IntClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gstelement_class->request_new_pad = gst_float2int_request_new_pad;
  gstelement_class->pad_removed = gst_float2int_pad_removed;
  gstelement_class->change_state = gst_float2int_change_state;
}

static void
gst_float2int_init (GstFloat2Int *plugin)
{
  gst_element_set_loop_function (GST_ELEMENT (plugin), gst_float2int_loop);

  plugin->srcpad = gst_pad_new_from_template (
			GST_PAD_TEMPLATE_GET (float2int_src_factory), "src");
  gst_element_add_pad (GST_ELEMENT (plugin), plugin->srcpad);
  gst_pad_set_link_function (plugin->srcpad, gst_float2int_link);

  plugin->numchannels = 0;
  plugin->channelcount = 0;
  plugin->channels = NULL;
}

static GstBufferPool*
gst_float2int_get_bufferpool (GstPad *pad)
{
  gint frames;
  GstFloat2Int *filter = (GstFloat2Int *) gst_pad_get_parent (pad);

  frames = filter->buffer_frames;
  if (!frames)
    frames = GST_FLOAT2INT_DEFAULT_BUFFER_FRAMES;
  
  return gst_buffer_pool_get_default (frames * sizeof (float), 5);
}

/* this is a very brittle function. it's hard to come up with a "correct
   behavior". what I (wingo) have chosen is as follows:
   * if the rate of the float2int plugin has already been set by a previous connection
     * if the new rate is different from the old rate, bail
     * otherwise return 'ok' (don't set other caps, no need -- only the rate needs
       setting, and it's the same)
   * otherwise set all of the caps (except for on the connecting pad, of course)
     * bail if that fails, possibly leaving the plugin in an inconsistent state

   FIXME: we really should do something about the inconsistent state case,
   possibly unlinking the offending pad. the assumption with this element,
   though, is that the pipelines on the float side will be parallel (more or
   less the same) and so we should never run into that case. that might be a
   programmer error more than anything.
 */
static GstPadLinkReturn
gst_float2int_link (GstPad *pad, GstCaps *caps)
{
  GstFloat2Int *filter;
  GstCaps *intcaps, *floatcaps;
  GSList *l;
  gint rate, channels, buffer_frames;
  
  filter = GST_FLOAT2INT (GST_PAD_PARENT (pad));
  
  /* we have three variables to play with:
     "rate": both sides
     "buffer-frames": float side
     "channels": int side. */
  if (GST_CAPS_IS_FIXED (caps)) {
    if (pad == filter->srcpad) {
      /* it's the int pad */
      if (!filter->intcaps) {
        gst_caps_get_int (caps, "channels", &channels);

        if (channels != filter->numchannels) {
          GST_DEBUG ("Tried to set caps with %d channels but there are %d source pads",
                     channels, filter->numchannels);
          return GST_PAD_LINK_REFUSED;
        }
        
        if (!filter->buffer_frames)
          filter->buffer_frames = GST_FLOAT2INT_DEFAULT_BUFFER_FRAMES;
        
        floatcaps = gst_caps_copy (gst_pad_template_get_caps (float2int_sink_factory ()));
        gst_caps_get_int (caps, "rate", &filter->rate);
        gst_caps_set (floatcaps, "rate", GST_PROPS_INT (filter->rate), NULL);
        gst_caps_set (floatcaps, "buffer-frames", GST_PROPS_INT (filter->buffer_frames), NULL);
        
        for (l=filter->channels; l; l=l->next)
          if (gst_pad_try_set_caps (GST_FLOAT2INT_CHANNEL (l)->sinkpad, floatcaps) <= 0)
            return GST_PAD_LINK_REFUSED;
        
        filter->floatcaps = gst_caps_ref (floatcaps);
        filter->intcaps = gst_caps_ref (caps);
        return GST_PAD_LINK_OK;
      } else if (gst_caps_intersect (caps, filter->intcaps)) {
        /* they have to be the same since both are fixed */
        return GST_PAD_LINK_OK;
      } else {
        GST_DEBUG ("Tried to connect the source of float2int, but with bad caps");
        gst_caps_debug (filter->intcaps, "float2int internal int source pad caps");
        return GST_PAD_LINK_REFUSED;
      }
    } else {
      /* it's a float pad */
      if (!filter->intcaps) {
        intcaps = gst_caps_copy (gst_pad_template_get_caps (float2int_src_factory ()));
        gst_caps_get_int (caps, "rate", &rate);
        gst_caps_set (intcaps, "rate", GST_PROPS_INT (rate));
        gst_caps_set (intcaps, "channels", GST_PROPS_INT (filter->numchannels));
        gst_caps_debug (intcaps, "int source pad caps going into try_set_caps()");
        
        if (gst_pad_try_set_caps (filter->srcpad, intcaps) <= 0) {
          return GST_PAD_LINK_REFUSED;
	}

        if (!gst_caps_get_int (caps, "buffer-frames", &buffer_frames)) {
          g_warning ("You have an old float plugin that needs to be converted "
                     "to the new caps format. Please file a bug!");
          return GST_PAD_LINK_REFUSED;
        }
        if (filter->buffer_frames && filter->buffer_frames != buffer_frames) {
          GST_DEBUG ("Tried to set buffer_frames=%d but it was already set to %d",
                     buffer_frames, filter->buffer_frames);
          return GST_PAD_LINK_REFUSED;
        }

        filter->floatcaps = gst_caps_ref (caps);
        filter->intcaps = gst_caps_ref (intcaps);
        filter->rate = rate;
        filter->buffer_frames = buffer_frames;

        for (l=filter->channels; l; l=l->next) {
	  if (GST_FLOAT2INT_CHANNEL (l)->sinkpad == pad) {
	    continue;
	  }
	  
          if (gst_pad_try_set_caps (GST_FLOAT2INT_CHANNEL (l)->sinkpad, caps) == FALSE) {
	    /* We don't have the caps set, so remove them */
            if (filter->floatcaps) {
              gst_caps_unref (filter->floatcaps);
              filter->floatcaps = NULL;
            }
            if (filter->intcaps) {
              gst_caps_unref (filter->intcaps);
              filter->intcaps = NULL;
            }
            return GST_PAD_LINK_REFUSED;
	  }
	}
        	
        return GST_PAD_LINK_OK;
      } else if (gst_caps_intersect (caps, filter->floatcaps)) {
        /* they have to be the same since both are fixed */
        return GST_PAD_LINK_OK;
      } else {
        return GST_PAD_LINK_REFUSED;
      }
    }
  } else {
    return GST_PAD_LINK_DELAYED;
  }
}

static GstPad*
gst_float2int_request_new_pad (GstElement *element, GstPadTemplate *templ, const gchar *unused) 
{
  gchar *name;
  GstFloat2Int *plugin;
  GstFloat2IntInputChannel *channel;

  plugin = GST_FLOAT2INT(element);
  
  g_return_val_if_fail(plugin != NULL, NULL);
  g_return_val_if_fail(GST_IS_FLOAT2INT(plugin), NULL);

  if (templ->direction != GST_PAD_SINK) {
    g_warning ("float2int: request new pad that is not a SINK pad\n");
    return NULL;
  }

  channel = g_new0 (GstFloat2IntInputChannel, 1);
  name = g_strdup_printf ("sink%d", plugin->channelcount);
  channel->sinkpad = gst_pad_new_from_template (templ, name);
  gst_element_add_pad (GST_ELEMENT (plugin), channel->sinkpad);
  gst_pad_set_link_function (channel->sinkpad, gst_float2int_link);
  gst_pad_set_bufferpool_function (channel->sinkpad, gst_float2int_get_bufferpool);
  
  plugin->channels = g_slist_append (plugin->channels, channel);
  plugin->numchannels++;
  plugin->channelcount++;
  
  if (plugin->intcaps) {
    gst_caps_set (plugin->intcaps, "channels", GST_PROPS_INT (plugin->numchannels));
    if (gst_pad_try_set_caps (plugin->srcpad, plugin->intcaps) <= 0) {
      g_warning ("float2int: added a pad, but failed to update src caps");
    }
  }

  GST_DEBUG ("float2int added pad %s\n", name);

  g_free (name);
  return channel->sinkpad;
}

static void
gst_float2int_pad_removed (GstElement *element,
			   GstPad *pad)
{
  GstFloat2Int *plugin;
  GSList *p;
  
  GST_DEBUG ("float2int removed pad %s\n", GST_OBJECT_NAME (pad));
  
  plugin = GST_FLOAT2INT (element);

  /* Find our channel for this pad */
  for (p = plugin->channels; p;) {
    GstFloat2IntInputChannel *channel = p->data;
    GSList *p_copy;

    if (channel->sinkpad == pad) {
      p_copy = p;

      p = p->next;
      plugin->channels = g_slist_remove_link (plugin->channels, p_copy);
      plugin->numchannels--;
      
      g_slist_free (p_copy);

      if (channel->buffer)
        gst_buffer_unref (channel->buffer);
      g_free (channel);
    } else {
      p = p->next;
    }
  }
}

static void
gst_float2int_loop (GstElement *element)
{
  GstFloat2Int *plugin = (GstFloat2Int*)element;
  GstBuffer *buf_out;
  gfloat *data_in;
  gint16 *data_out;
  gint num_frames, i, j;
  GSList *l;
  GstFloat2IntInputChannel *channel;
  gint frames_to_write;
  gint numchannels = plugin->numchannels;
    
  if (!plugin->channels) {
    gst_element_error (element, "float2int: at least one sink pad needs to be connected");
    return;
  }

  if (!plugin->pool)
    if (! (plugin->pool = gst_pad_get_bufferpool (plugin->srcpad)))
      plugin->pool = gst_buffer_pool_get_default (sizeof(gint16) * 1024 * numchannels, 1);

  g_assert (plugin->pool);

  /* get new output buffer */
  buf_out = gst_buffer_new_from_pool (plugin->pool, 0, 0);
  num_frames = GST_BUFFER_SIZE (buf_out) / numchannels / sizeof (gint16);
  g_return_if_fail (GST_BUFFER_SIZE (buf_out) == num_frames * numchannels * sizeof (gint16));
  
  data_out = (gint16*)GST_BUFFER_DATA(buf_out);
  
  i = 0;
  frames_to_write = num_frames;

  while (frames_to_write) {
    if (!plugin->frames_remaining) {
      /* the buffer-frames GstPropsEntry ensures us that all buffers coming in
         will be the same size. if they are not, that pad will send EOS on the
         next pull. */
      F2I_DEBUG ("pulling buffers...");
      plugin->frames_remaining = G_MAXINT;

      for (l = plugin->channels; l; l=l->next) {
        channel = GST_FLOAT2INT_CHANNEL (l);

        if (channel->buffer)
          gst_buffer_unref (channel->buffer);
        channel->buffer = NULL;
      get_buffer:
        channel->buffer = GST_BUFFER (gst_pad_pull (channel->sinkpad));
        g_return_if_fail (channel->buffer != NULL);

        if (GST_IS_EVENT (channel->buffer)) {
          F2I_DEBUG ("got an event");
          if (GST_EVENT_TYPE (channel->buffer) == GST_EVENT_EOS) {
            plugin->frames_remaining = 0;
            GST_INFO ("float2int got eos from pad %s%s", GST_DEBUG_PAD_NAME (channel->sinkpad));
          } else {
            gst_pad_event_default (plugin->srcpad, GST_EVENT (channel->buffer));
            goto get_buffer; /* goto, hahaha */
          }
	} else {
          F2I_DEBUG ("got a buffer %d frames big", GST_BUFFER_SIZE (channel->buffer)/4);
          plugin->frames_remaining = MIN (plugin->frames_remaining,
                                          GST_BUFFER_SIZE (channel->buffer) / sizeof (gfloat));
        }
      }
    }

    if (plugin->frames_remaining) {
      gint to_process;

      F2I_DEBUG ("processing %d%s", plugin->frames_remaining,
                 frames_to_write >= plugin->frames_remaining ? " (last time)" : "");
      to_process = MIN (frames_to_write, plugin->frames_remaining);
      for (l=plugin->channels, i=0; l; l=l->next, i++) {
        F2I_DEBUG ("processing for %d frames on channel %d", to_process, i);
        channel = GST_FLOAT2INT_CHANNEL (l);
        g_return_if_fail (channel->buffer != NULL);
        data_in = (gfloat*) GST_BUFFER_DATA (channel->buffer);
        data_in += GST_BUFFER_SIZE (channel->buffer) / sizeof (float) - plugin->frames_remaining;
        
        for (j = 0; j < to_process; j++)
          data_out[(j*numchannels) + i] = (gint16)(gst_cast_float(data_in[j] * 32767.0F));
      }
      data_out += to_process * numchannels;
      
      if (to_process < plugin->frames_remaining) {
        /* we're finished with this output buffer */
        plugin->frames_remaining -= to_process;
        frames_to_write = 0;
      } else {
        /* note there's still the possibility that one of the buffers had less
           data, so we're possibly discarding some data from other float pads;
           that's ok though, because the pad that didn't give us buffer_frames
           worth of data will send EOS on the next pull */
        frames_to_write -= plugin->frames_remaining;
        plugin->frames_remaining = 0;
      }
    } else {
      /* a pad has run out of data */
      GST_BUFFER_SIZE (buf_out) = GST_BUFFER_SIZE (buf_out)
        - frames_to_write * numchannels * sizeof (gint16);

      if (GST_BUFFER_SIZE (buf_out)) {
        GST_BUFFER_TIMESTAMP (buf_out) = plugin->offset * GST_SECOND / plugin->rate;
        gst_pad_push (plugin->srcpad, GST_DATA (buf_out));
      } else {
        /* we got the eos event on one of the pads */
        gst_buffer_unref (buf_out);
        gst_pad_push (plugin->srcpad, GST_DATA ((GstBuffer*)gst_event_new (GST_EVENT_EOS)));
        gst_element_set_eos (element);
      }
      
      return;
    }
  }
  
  GST_BUFFER_TIMESTAMP (buf_out) = plugin->offset * GST_SECOND / plugin->rate;
  
  plugin->offset += num_frames;
  gst_pad_push (plugin->srcpad, GST_DATA (buf_out));
}

static GstElementStateReturn
gst_float2int_change_state (GstElement *element)
{
  GstFloat2Int *plugin = (GstFloat2Int *) element;
  GSList *p;

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_READY_TO_PAUSED:
      plugin->offset = 0;
      break;

    case GST_STATE_PAUSED_TO_PLAYING:
      for (p = plugin->channels; p; p = p->next) {
	GstFloat2IntInputChannel *c = p->data;

	c->eos = FALSE;
      }
      break;
      
    case GST_STATE_PAUSED_TO_READY:
      if (plugin->floatcaps) {
        gst_caps_unref (plugin->floatcaps);
        plugin->floatcaps = NULL;
      }
      if (plugin->intcaps) {
        gst_caps_unref (plugin->intcaps);
        plugin->intcaps = NULL;
      }
      for (p = plugin->channels; p; p = p->next) {
	GstFloat2IntInputChannel *c = p->data;

        if (c->buffer)
          gst_buffer_unref (c->buffer);
        c->buffer = NULL;
	c->eos = FALSE;
      }
      plugin->frames_remaining = 0;
      break;

    default:
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}
