/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_MAD
#include <id3tag.h>
#endif

#include <string.h>
#include "gstid3types.h"

/* elementfactory information */
static GstElementDetails id3types_details = {
  "ID3v1/v2 tag parser",
  "Codec/Parser",
#ifdef HAVE_MAD
  "GPL", /* grmbl, where's a good C LGPL ID3v1/2 reader? */
#else
  "LGPL",
#endif
  "Parses ID3v1 and ID3v2 tags",
  VERSION,
  "Erik Walthinsen <omega@cse.ogi.edu>\n"
  "Ronald Bultje <rbultje@ronald.bitfreak.net>",
  "(C) 1999-2003",
};

GST_PAD_TEMPLATE_FACTORY (sink_templ,
  "sink",
  GST_PAD_SINK,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "id3types_sink",
    "audio/x-id3",
      "id3version", GST_PROPS_INT_RANGE (1, 2)
  )
);

GST_PAD_TEMPLATE_FACTORY (src_templ,
  "src",
  GST_PAD_SRC,
  GST_PAD_SOMETIMES,
  NULL
);

static GstCaps* id3_type_find (GstByteStream *bs, gpointer private);

static GstTypeDefinition id3type_definition = {
  "id3_audio/x-id3",
  "audio/x-id3",
  ".mp3 .mp2 .mp1 .mpga .ogg .flac", /* probably more... *sigh* */
  id3_type_find,
};

/* signals */
enum {
  /* FILL ME */
  LAST_SIGNAL,
};

/* args */
enum {
  ARG_0,
  ARG_METADATA,
  /* FILL ME */
};


static void	gst_id3types_class_init		(GstID3TypesClass *klass);
static void	gst_id3types_init		(GstID3Types  *mp3parse);

static const GstEventMask *
		gst_id3types_event_mask		(GstPad       *pad);
static gboolean	gst_id3types_event		(GstPad       *pad,
						 GstEvent     *event);

static const GstQueryType *
		gst_id3types_get_query_types	(GstPad       *pad);
static gboolean gst_id3types_handle_query 	(GstPad       *pad,
						 GstQueryType  type, 
						 GstFormat    *format,
						 gint64       *value);

static void	gst_id3types_chain		(GstPad       *pad,
						 GstData      *_data);
static void	gst_id3types_loop		(GstElement   *element);

static void	gst_id3types_get_property	(GObject      *object,
						 guint         prop_id,
						 GValue       *value,
						 GParamSpec   *pspec);
static GstElementStateReturn
		gst_id3types_change_state	(GstElement   *element);

static GstElementClass *parent_class = NULL;
/*static guint gst_id3types_signals[LAST_SIGNAL] = { 0 };*/

GType
gst_id3types_get_type (void)
{
  static GType id3types_type = 0;

  if (!id3types_type) {
    static const GTypeInfo id3types_info = {
      sizeof (GstID3TypesClass),
      NULL,
      NULL,
      (GClassInitFunc) gst_id3types_class_init,
      NULL,
      NULL,
      sizeof (GstID3Types),
      0,
      (GInstanceInitFunc) gst_id3types_init,
    };

    id3types_type = g_type_register_static (GST_TYPE_ELEMENT,
					    "GstID3Types",
					    &id3types_info, 0);
  }

  return id3types_type;
}

static gboolean
gst_id3types_detect (GstByteStream *bs,
		     guint         *version,
		     guint         *size)
{
  gboolean res = FALSE;
  GstBuffer *buf = NULL;

  if (gst_bytestream_peek (bs, &buf, 10) == 10) {
    guint8 *data = GST_BUFFER_DATA (buf);

    /* gracefully ripped from libid3 */
    if (data[0] == 'T' && data[1] == 'A' && data[2] == 'G') {
      /* ID3v1 tags */
      res = TRUE;
      if (size)
        *size = 128;
      if (version)
        *version = 1;
    } else if (data[0] == 'I' && data[1] == 'D' && data[2] == '3') {
      /* ID3v2 tags */
      if (data[3] < 0xff && data[4] < 0xff &&
	  data[6] < 0x80 && data[7] < 0x80 &&
	  data[8] < 0x80 && data[9] < 0x80) {
        guint32 skip = ((data[6] & 0x7f) << 21) |
		       ((data[7] & 0x7f) << 14) |
		       ((data[8] & 0x7f) << 7) |
		       ((data[9] & 0x7f) << 0);

        /* include size of header */
        skip += 10;

        /* footer present? (only available since version 4) */
        if (data[3] >= 4 && (data[5] & 0x10))
          skip += 10;

        res = TRUE;
        if (size)
          *size = skip;
        if (version)
          *version = 2;
      }
    }
  }

  if (buf != NULL) {
    gst_buffer_unref (buf);
  }

  return res;
}

static GstCaps *
id3_type_find (GstByteStream *bs,
	       gpointer       private) 
{
  GstCaps *new = NULL;
  guint version, size;

  if (gst_id3types_detect (bs, &version, &size)) {
    switch (version) {
      case 1: case 2:
        /* ID3v1/v2 tags */
        GST_DEBUG ("id3typefind: detected ID3 Tag V%d", version);
        new = GST_CAPS_NEW ("id3_type_find",
			    "audio/x-id3",
			      "id3version", GST_PROPS_INT (version));
        break;
      default:
        break;
    }
  }

  return new;
}

static void
gst_id3types_class_init (GstID3TypesClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  g_object_class_install_property (gobject_class, ARG_METADATA,
    g_param_spec_boxed ("metadata", "Metadata", "Metadata",
                        GST_TYPE_CAPS, G_PARAM_READABLE));

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  gobject_class->get_property = gst_id3types_get_property;

  gstelement_class->change_state = gst_id3types_change_state;
}

static inline void
gst_id3types_start (GstID3Types *id3)
{
  id3->pass_through = FALSE;
  id3->id3_tag_size = 0;
  id3->metadata = NULL;
}

static inline void
gst_id3types_cont (GstID3Types *id3)
{
  id3->pass_through = TRUE;
}

static inline void
gst_id3types_stop (GstID3Types *id3)
{
  gst_caps_replace (&id3->metadata, NULL);
  id3->pass_through = TRUE;
}

static void
gst_id3types_init (GstID3Types *id3)
{
  GST_FLAG_SET (id3, GST_ELEMENT_EVENT_AWARE);

  id3->sinkpad = gst_pad_new_from_template (GST_PAD_TEMPLATE_GET (sink_templ),
					    "sink");
  gst_element_set_loop_function (GST_ELEMENT (id3), gst_id3types_loop);
  gst_pad_set_event_function (id3->sinkpad, gst_id3types_event);
  gst_pad_set_event_mask_function (id3->sinkpad, gst_id3types_event_mask);
  gst_id3types_start (id3);
  gst_element_add_pad (GST_ELEMENT (id3), id3->sinkpad);

  id3->metadata = NULL;
  id3->id3_tag_size = 0;
}

static const GstEventMask *
gst_id3types_event_mask (GstPad *pad)
{
  GstID3Types *id3 = GST_ID3TYPES (gst_pad_get_parent (pad));
  const GstEventMask *masks = NULL;

  if (pad == id3->sinkpad) {
    static const GstEventMask my_masks[] = {
      { GST_EVENT_DISCONTINUOUS, 0 },
      { 0, }
    };
    masks = my_masks;
  } else if (pad == id3->srcpad) {
    static const GstEventMask my_masks[] = {
      { GST_EVENT_SEEK, GST_SEEK_METHOD_SET },
      { 0, }
    };
    masks = my_masks;
  } else {
    g_assert (0);
  }

  return masks;
}

static gboolean
gst_id3types_event (GstPad   *pad,
		    GstEvent *event)
{
  GstID3Types *id3 = GST_ID3TYPES (gst_pad_get_parent (pad));
  gboolean res = TRUE;

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_DISCONTINUOUS: {
      /* comes from the sink pad */
      GstEvent *new = NULL;
      guint64 offset = 0;
      gint n;

      /* get new offset in bytes */
      for (n = 0; n < GST_EVENT_DISCONT_OFFSET_LEN (event); n++) {
        GstFormatValue *value = &GST_EVENT_DISCONT_OFFSET (event, n);
        if (value->format == GST_FORMAT_BYTES) {
          offset = value->value;
          break;
        }
      }

      if (GST_EVENT_DISCONT_NEW_MEDIA (event) ||
	  offset < id3->id3_tag_size) {
        gst_id3types_stop (id3);
        gst_id3types_start (id3);
        new = gst_event_new_discontinuous (GST_EVENT_DISCONT_NEW_MEDIA (event),
					   GST_FORMAT_BYTES, 0, 0);
      } else {
        gst_id3types_cont (id3);
        new = gst_event_new_discontinuous (FALSE, GST_FORMAT_BYTES,
					   offset - id3->id3_tag_size, 0);
      }
      gst_pad_event_default (pad, new);
      break;
    }
    case GST_EVENT_SEEK:
      /* comes from the src pad... */
      switch (GST_EVENT_SEEK_FORMAT (event)) {
        case GST_FORMAT_BYTES: {
          gint64 offset = GST_EVENT_SEEK_OFFSET (event);
          GstEvent *new = gst_event_new_seek (GST_SEEK_METHOD_SET | GST_FORMAT_BYTES,
					      offset + id3->id3_tag_size);

          gst_pad_send_event (GST_PAD_PEER (id3->sinkpad), new);
          break;
        }
        default:
          g_warning ("Non-bytes seek event in id3types");
          res = FALSE;
          break;
      }
      break;
    default:
      res = FALSE;
      break;
  }

  gst_event_unref (event);

  return res;
}

static const GstQueryType *
gst_id3types_get_query_types (GstPad *pad)
{
  static const GstQueryType types[] = {
    GST_QUERY_TOTAL,
    GST_QUERY_POSITION,
    0
  };

  return types;
}

static gboolean
gst_id3types_handle_query (GstPad       *pad,
			   GstQueryType  type, 
			   GstFormat    *format,
			   gint64       *value)
{
  GstID3Types *id3 = GST_ID3TYPES (gst_pad_get_parent (pad));

  g_return_val_if_fail (type == GST_QUERY_TOTAL ||
			type == GST_QUERY_POSITION, FALSE);

  if (*format == GST_FORMAT_BYTES) {
    gst_pad_query (id3->sinkpad, type, format, value);

    if (*value >= id3->id3_tag_size)
      *value -= id3->id3_tag_size;
    else
      *value = 0;

    return TRUE;
  }

  return FALSE;
}

static void
gst_id3types_chain (GstPad    *pad,
		    GstData *_data)
{
  GstBuffer *buf = GST_BUFFER (_data);
  GstID3Types *id3 = GST_ID3TYPES (gst_pad_get_parent (pad));

  if (GST_IS_EVENT (buf)) {
    gst_id3types_event (id3->sinkpad, GST_EVENT (buf));
  } else {
    /* passthrough mode */
    gst_pad_push (id3->srcpad, GST_DATA (buf));
  }
}

#ifdef HAVE_MAD
/* gracefuly ripped from madplay */
static GstCaps *
id3_to_caps (struct id3_tag const *tag)
{
  unsigned int i;
  struct id3_frame const *frame;
  id3_ucs4_t const *ucs4;
  id3_utf8_t *utf8;
  GstProps *props;
  GstPropsEntry *entry;
  GstCaps *caps;
  GList *values;

  struct {
    char const *id;
    char const *name;
  } const info[] = {
    { ID3_FRAME_TITLE,   "Title"        },
    { "TIT3",            "Subtitle"     },
    { "TCOP",            "Copyright"    },
    { "TPRO",            "Produced"     },
    { "TCOM",            "Composer"     },
    { ID3_FRAME_ARTIST,  "Artist"       },
    { "TPE2",            "Orchestra"    },
    { "TPE3",            "Conductor"    },
    { "TEXT",            "Lyricist"     },
    { ID3_FRAME_ALBUM,   "Album"        },
    { ID3_FRAME_YEAR,    "Year"         },
    { ID3_FRAME_TRACK,   "Track"        },
    { "TPUB",            "Publisher"    },
    { ID3_FRAME_GENRE,   "Genre"        },
    { "TRSN",            "Station"      },
    { "TENC",            "Encoder"      },
  };

  /* text information */
  props = gst_props_empty_new ();

  for (i = 0; i < sizeof(info) / sizeof(info[0]); ++i) {
    union id3_field const *field;
    unsigned int nstrings, namelen, j;
    char const *name;

    frame = id3_tag_findframe(tag, info[i].id, 0);
    if (frame == 0)
      continue;

    field    = &frame->fields[1];
    nstrings = id3_field_getnstrings(field);

    name = info[i].name;

    if (name) {
      namelen = name ? strlen(name) : 0;

      values = NULL;
      for (j = 0; j < nstrings; ++j) {
        ucs4 = id3_field_getstrings(field, j);
        g_assert(ucs4);

        if (strcmp(info[i].id, ID3_FRAME_GENRE) == 0)
	  ucs4 = id3_genre_name(ucs4);

        utf8 = id3_ucs4_utf8duplicate(ucs4);
        if (utf8 == 0)
	  goto fail;

        entry = gst_props_entry_new (name, GST_PROPS_STRING_TYPE, utf8);
	values = g_list_prepend (values, entry);
        free(utf8);
      }
      if (values) {
        values = g_list_reverse (values);

        if (g_list_length (values) == 1) {
          gst_props_add_entry (props, (GstPropsEntry *) values->data);
        }
        else {
          entry = gst_props_entry_new(name, GST_PROPS_GLIST_TYPE, values);
          gst_props_add_entry (props, (GstPropsEntry *) entry);
        }
        g_list_free (values);
      }
    }
  }

  values = NULL;
  i = 0;
  while ((frame = id3_tag_findframe(tag, ID3_FRAME_COMMENT, i++))) {
    ucs4 = id3_field_getstring(&frame->fields[2]);
    g_assert(ucs4);

    if (*ucs4)
      continue;

    ucs4 = id3_field_getfullstring(&frame->fields[3]);
    g_assert(ucs4);

    utf8 = id3_ucs4_utf8duplicate(ucs4);
    if (utf8 == 0)
      goto fail;

    entry = gst_props_entry_new ("Comment", GST_PROPS_STRING_TYPE, utf8);
    values = g_list_prepend (values, entry);
    free(utf8);
  }
  if (values) {
    values = g_list_reverse (values);

    if (g_list_length (values) == 1) {
      gst_props_add_entry (props, (GstPropsEntry *) values->data);
    }
    else {
      entry = gst_props_entry_new("Comment", GST_PROPS_GLIST_TYPE, values);
      gst_props_add_entry (props, (GstPropsEntry *) entry);
    }
    g_list_free (values);
  }

  gst_props_debug (props);

  caps = gst_caps_new ("mad_metadata",
		       "application/x-gst-metadata",
		       props);
  if (0) {
fail:
    g_warning ("mad: could not parse ID3 tag");

    return NULL;
  }

  return caps;
}
#endif

static void
gst_id3types_loop (GstElement *element)
{
  const GList *type_list = gst_type_get_list ();
  GstID3Types *id3;
  GstByteStream *bs;
  GstBuffer *buf = NULL;
  guint size = 0, version = 0;
#ifdef HAVE_MAD
  struct id3_tag *tag = NULL;
#endif

  g_return_if_fail (GST_IS_ID3TYPES (element));
  id3 = GST_ID3TYPES (element);

  if (id3->pass_through) {
    gst_id3types_chain (id3->sinkpad, gst_pad_pull (id3->sinkpad));
    return;
  }

  bs = gst_bytestream_new (id3->sinkpad);

  /* first loop... */
  if (!gst_id3types_detect (bs, &version, &size)) {
    gst_element_error (element,
		       "No ID3 tag detected");
    goto error;
  }

  /* OK: so we've now got a ID3 tag of that size, let's read it */
  if (gst_bytestream_peek (bs, &buf, size) != size) {
    gst_element_error (element,
		       "Failed to read ID3 tag - hard disk on fire?");
    goto error;
  }
  gst_bytestream_flush (bs, size);

#ifdef HAVE_MAD
  /* here, we should parse the ID3 header and notify the app */
  if (!(tag = id3_tag_parse (GST_BUFFER_DATA (buf), size))) {
    gst_element_error (element,
		       "Failed to parse ID3v1/2 tags");
    goto error;
  } else {
    gst_caps_replace_sink (&id3->metadata, id3_to_caps (tag));
    id3_tag_delete (tag);
    g_object_notify (G_OBJECT (id3), "metadata");
  }
#endif

  /* done */
  id3->id3_tag_size = size;

  /* this will help us detecting the media stream type after
   * this id3 thingy... Please note that this is a cruel hack
   * for as long as spider doesn't support multi-type-finding.
   */
  while (type_list) {
    GSList *factories;
    GstType *type = (GstType *) type_list->data;

    factories = type->factories;

    while (factories) {
      GstTypeFactory *factory = GST_TYPE_FACTORY (factories->data);
      GstTypeFindFunc typefindfunc = (GstTypeFindFunc) factory->typefindfunc;
      GstCaps *caps;

      if (typefindfunc && (caps = typefindfunc (bs, factory))) {
        GST_DEBUG ("found type: %d \"%s\" \"%s\"\n",
		   caps->id, type->mime, gst_caps_get_name (caps));

        id3->srcpad = gst_pad_new_from_template (GST_PAD_TEMPLATE_GET (src_templ),
						 "src");
        gst_pad_set_event_function (id3->srcpad, gst_id3types_event);
        gst_pad_set_event_mask_function (id3->srcpad, gst_id3types_event_mask);
        gst_pad_set_query_type_function (id3->srcpad, gst_id3types_get_query_types);
        gst_pad_set_query_function (id3->srcpad, gst_id3types_handle_query);
        gst_pad_try_set_caps (id3->srcpad, caps);
        gst_element_add_pad (GST_ELEMENT (id3), id3->srcpad);
        goto tf_done;
      }
      factories = g_slist_next (factories);
    }
    type_list = g_list_next (type_list);
  }
  gst_element_error (element,
		     "Media stream type not detected after id3v1/2 header");
  goto error;

tf_done:
  if (bs->listavail) {
    gst_bytestream_read (bs, &buf, bs->listavail);
    gst_pad_push (id3->srcpad, GST_DATA (buf));
  }
  gst_id3types_cont (id3);
error:
#ifdef HAVE_MAD
  if (tag) {
    id3_tag_delete (tag);
  }
#endif
  gst_bytestream_destroy (bs);
}

static void
gst_id3types_get_property (GObject    *object,
			   guint       prop_id,
			   GValue     *value,
			   GParamSpec *pspec)
{
  GstID3Types *id3;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_ID3TYPES (object));
  id3 = GST_ID3TYPES (object);

  switch (prop_id) {
    case ARG_METADATA:
      g_value_set_boxed (value, id3->metadata);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstElementStateReturn 
gst_id3types_change_state (GstElement *element) 
{
  GstID3Types *id3;

  g_return_val_if_fail (GST_IS_ID3TYPES (element), GST_STATE_FAILURE);
  id3 = GST_ID3TYPES (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_READY_TO_PAUSED:
      gst_id3types_start (id3);
      break;
    case GST_STATE_PAUSED_TO_READY:
      gst_id3types_stop (id3);
      break;
    default:
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstTypeFactory *type;
  GstElementFactory *factory;

  /* create an elementfactory for the id3types element */
  factory = gst_element_factory_new ("id3types",
		                     GST_TYPE_ID3TYPES,
                                     &id3types_details);
  g_return_val_if_fail (factory != NULL, FALSE);
  gst_element_factory_set_rank (factory, GST_ELEMENT_RANK_PRIMARY);

  gst_element_factory_add_pad_template (factory,
	GST_PAD_TEMPLATE_GET (sink_templ));
  gst_element_factory_add_pad_template (factory,
	GST_PAD_TEMPLATE_GET (src_templ));

  gst_plugin_add_feature (plugin, GST_PLUGIN_FEATURE (factory));

  /* and typefinding */
  type = gst_type_factory_new (&id3type_definition);
  gst_plugin_add_feature (plugin, GST_PLUGIN_FEATURE (type));

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "id3types",
  plugin_init
};
