/* GStreamer - Matroska I/O wrapper
 * Copyright (C) <2003> Ronald Bultje <rbultje@ronald.bitfreak.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <gst/gst.h>

#include <typeinfo>

#include "gstmatroska-io.h"

using namespace std;
using namespace LIBEBML_NAMESPACE;

GstMatroskaIO::GstMatroskaIO (void)
{
  this->pad = NULL;
  this->bs = NULL;
}

/* usage: "gst-matroska-io://%p", where %p is pointer to pad */

GstMatroskaIO::GstMatroskaIO (const char      *path,
			      const open_mode  mode)
{
  g_assert (strncmp(path, "gst-matroska-io://", 18) == 0);
  g_assert (sscanf (path+18, "%p", &this->pad) == 1);

  switch (mode) {
    case MODE_READ:
      g_assert (GST_PAD_IS_SINK (this->pad));
      this->bs = gst_bytestream_new (this->pad);
      break;
    case MODE_WRITE:
    case MODE_CREATE:
      g_assert (GST_PAD_IS_SRC (this->pad));
      /* a pad is enough */
      break;
    default:
      g_assert (0);
  }

  this->eos = FALSE;
}

GstMatroskaIO::~GstMatroskaIO (void)
{
  if (this->pad != NULL) {
    this->close();
  }
}

uint64
GstMatroskaIO::getFilePointer (void)
{
  g_assert (this->pad != NULL);

  if (this->bs == NULL) {
    /* write */
    gint64 ret;
    GstFormat fmt = GST_FORMAT_BYTES;
    gst_pad_query (this->pad,
		   GST_QUERY_POSITION,
		   &fmt, &ret);
    return (guint64) ret;
  } else {
    /* read */
    return gst_bytestream_tell (this->bs);
  }
}

void
GstMatroskaIO::setFilePointer (int64     offset,
			       seek_mode mode)
{
  GstSeekType seek_type = (GstSeekType) 0;

  g_assert (this->pad != NULL);

  switch (mode) {
    case seek_beginning:
      seek_type = GST_SEEK_METHOD_SET;
      break;
    case seek_current:
      seek_type = GST_SEEK_METHOD_CUR;
      break;
    case seek_end:
      seek_type = GST_SEEK_METHOD_END;
      break;
    default:
      g_assert (0);
      break;
  }

  if (this->bs == NULL) {
    /* write */
    GstEvent *event = gst_event_new_seek (seek_type, offset);
    gst_pad_push (this->pad, GST_DATA (event));
  } else {
    /* read */
    gst_bytestream_seek (this->bs, offset, seek_type);
    this->eos = FALSE;
  }
}

uint32
GstMatroskaIO::read (void   *buffer,
                     size_t  size)
{
  guint32 total = 0, read;
  guint8 *data;

  g_assert (this->pad != NULL);
  g_assert (this->bs != NULL);

  while (!this->eos && size > total) {
    read = gst_bytestream_peek_bytes (this->bs, &data, size - total);
    memcpy (((guint8 *) buffer) + total, data, read);
    gst_bytestream_flush (this->bs, read);
    total += read;
 
    if (total < size) {
      GstEvent *event = NULL;
      guint32 remaining;

      gst_bytestream_get_status (this->bs, &remaining, &event);

      if (!event) {
        g_warning ("gstffmpegprotocol: no bytestream event");
        return total;
      }

      switch (GST_EVENT_TYPE (event)) {
        case GST_EVENT_DISCONTINUOUS:
          gst_bytestream_flush_fast (this->bs, remaining);
          total = 0; /* re-read */
          break;
        case GST_EVENT_EOS:
	  this->eos = TRUE;
	  break;
        default:
          break;
      }

      gst_event_unref (event);
    }
  }
  
  return total;
}

size_t
GstMatroskaIO::write (const void *buffer,
                      size_t      size)
{
  GstBuffer *outbuf;

  g_assert (this->pad != NULL);
  g_assert (this->bs == NULL);

  /* create buffer and push data further */
  outbuf = gst_buffer_new_and_alloc (size);
  GST_BUFFER_SIZE (outbuf) = size;
  memcpy (GST_BUFFER_DATA (outbuf), buffer, size);

  gst_pad_push (this->pad, GST_DATA(outbuf));

  return size;
}

void
GstMatroskaIO::close (void)
{
  g_assert (this->pad != NULL);

  if (this->bs != NULL) {
    gst_bytestream_destroy (this->bs);
    this->bs = NULL;
  }

  this->pad = NULL;
}

bool
GstMatroskaIO::eof (void)
{
  g_assert (this->pad != NULL);

  return (this->eos == TRUE);
}

char *
GstMatroskaIO::gets (char   *buffer,
                     size_t  max_size)
{
  g_assert (0);
  return NULL;
}

string
GstMatroskaIO::getline (void)
{
  g_assert (0);
  return NULL;
}

size_t
GstMatroskaIO::puts_unl (const char *s)
{
  g_assert (0);
  return 0;
}
