/* baumbrl.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "baumbrl.h"
#include "sercomm.h"

/* BAUM DEVICE MAPPINGS */

/* 1. KEY MAPPING */

/*		VARIO80 */
/* 			bit0-6 display keys (top), bit7-17 front keys, bit18-28 back keys, bit19-30 chord keys */
		
/* 		DM80P */
/* 			bit0-6 display keys */

/* 2. SWITCH MAPPING */

/*		DM80P */
/*			bit0-6 */

/*		INKA */
/*			bit0-4 */

/* 3. SENSOR BANK MAPPING (all devices) */

/* horizontal mechnical sensors above main display   - bank 0 */
/* horizontal optical sensors above main display     - bank 1 */
/* vertical optical sensors left                     - bank 2 */
/* vertical optical sensors right                    - bank 3 */
/* horizontal mechnical sensors above status display - bank 4 */
/* horizontal optical sensors above status display   - bank 5 */


/* Definitions */

/* Baum Devices */
typedef enum {baumdt_generic, baumdt_vario40, baumdt_vario20, baumdt_vario80, baumdt_dm80p, baumdt_inka} BAUM_DEV_TYPE;

/* Device data */

typedef struct
{
	unsigned char Version;
	unsigned short CRC;
	unsigned char Volume;
	unsigned char UARTData;
	unsigned char SpeechData[256];	/* OK 256 ? */
	unsigned char SpeechInfo;
	unsigned char HOSLength;
	unsigned char HOSState[11];
	unsigned char VOSState[8];
	unsigned char HMSLength;
	unsigned char HMSState[11];
	
	unsigned char SwitchState;
	
	unsigned char KeyState;
	unsigned char FrontKeyState;			/* VARIO80 */
	unsigned char BackKeyState;				/* VARIO80 */
	unsigned char ChordKeyState;			/* VARIO80 */
	unsigned short FrontKeyState2;		/* VARIO80 10 front keys model */
	unsigned short BackKeyState2;			/* VARIO80 10 front keys model */
	unsigned long CumulatedKeyState;	/* all above key states */
	
	unsigned char HOSValue;	
	unsigned char VOSValueLeft;
	unsigned char VOSValueRight;
	/* unsigned short VOSValue; */
	unsigned char HMSValue;

	char KeyCodes[256];								/* device specific key codes (DK - display, FK - front, BK - back, CK - chord) */
	char SwitchCodes[256];						/* device specific switch codes (SWxx) */
	char SensorCodes[128];						/* device specific sensor codes (HMS - horizontal mechanical, HOS - horizontal optical sensor, LOS - vertical optical sensor left, ROS - vertical optical sensor right) */
	
	unsigned char LastError;
} BAUM_DEVICE_DATA;

/* Baum Input Parser States */
typedef enum
{	
				bips_idle = 0, bips_exp_id, bips_exp_version, bips_exp_version_2,
				bips_exp_volume, bips_exp_UART_data, bips_exp_speech_data, bips_exp_speech_info,
				bips_exp_HOS_data, bips_exp_VOS_data, bips_exp_HMS_data,
				bips_exp_switches, bips_exp_keys, bips_exp_front_keys, bips_exp_back_keys,
				bips_exp_chord_keys, bips_exp_front_keys_2, bips_exp_back_keys_2,
				bips_exp_HOS_value, bips_exp_VOS_values, bips_exp_HMS_value,
				bips_exp_error_code, bips_exp_test_1, bips_exp_test_2, bips_exp_test_3
				
} BIP_STATES;


/* Globals */

static BRL_DEV_CALLBACK			ClientCallback = NULL;
static BAUM_DEV_TYPE				BaumDevType = baumdt_generic;

static unsigned char 	Mask8[] =	{0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80 };
/*Defined but not used
static unsigned short 	Mask16[] =	{0x0001, 0x0002, 0x0004, 0x0008, 0x0010, 0x0020, 0x0040, 0x0080,
					 0x0100, 0x0200, 0x0400, 0x0800, 0x1000, 0x2000, 0x4000, 0x8000 };								
*/
static unsigned long 	Mask32[] = {
	0x00000001,0x00000002,0x00000004,0x00000008,0x00000010,0x00000020,0x00000040,0x00000080,
	0x00000100,0x00000200,0x00000400,0x00000800,0x00001000,0x00002000,0x00004000,0x00008000,	
	0x00010000,0x00020000,0x00040000,0x00080000,0x00100000,0x00200000,0x00400000,0x00800000,
	0x01000000,0x02000000,0x04000000,0x08000000,0x10000000,0x20000000,0x40000000,0x80000000
};


static BIP_STATES						BIPState = bips_idle;
static short								BIPIndex = 0;

static BAUM_DEVICE_DATA			baumdd;

/* Functions */

/* ... */
void clear_device_data (BAUM_DEVICE_DATA *bdd)
{
	memset (bdd, 0, sizeof(BAUM_DEVICE_DATA));
}

/* ... */
void reset_bip()
{
	BIPIndex = 0;
	BIPState = bips_idle;
}

/* ... */
short check_escape (unsigned char NewVal)
{
	short rv = 0;
	static short esc = 0;
	
	if (esc)
	{
		/* waiting for the second escape	 */
		if (NewVal == 0x1B)
		{
			rv = 1; /* do it */
		}
		else
		{
			reset_bip();		/* reset to idle		 */
		}
		esc = 0;
	}
	else
	{
		/* not waiting for the second escape */
		if (NewVal == 0x1B)
		{
			esc = 1;
		}
		else {
			rv = 1;	/* do it */
			esc = 0;
		}
	}

	return rv;
}

short get_no_from_bitmask (unsigned char *Buff, short MaxLen)
{
	short rv = 0;
	short i, j;

	/* fprintf (stderr, "BITMASK: (len:%d)", MaxLen); */
	
	for (i = 0; i < MaxLen; ++i)
	{
		
		/* fprintf (stderr, "%02x ", Buff[i]); */
	
		if (Buff[i])
		{
			for (j = 0; j < 8; ++j)
			 {
				if ( Buff[i] & Mask8[j])
				{
					break;
				}
			}
			rv = 8 * i + j + 1;
			break;
		}
		
	}

	return rv;
}

/* XXXCodes Events */
/* ... */
void OnKeysChanged ()
{
	/* NOTE: called for any display, front, back, chord key changes */
	/* fires when all keys are released */
	
	int i;
	int kcix;
	
	BRAILLE_EVENT_CODE		bec;
	BRAILLE_EVENT_DATA   	bed;

	static unsigned long pressed_keys = 0;
	
	/* fprintf (stderr, "KEY:%02x\n", baumdd.CumulatedKeyState); */
	
	if (baumdd.CumulatedKeyState)
	{
		/* some keys pressed, add the new pressed keys */
		pressed_keys |= baumdd.CumulatedKeyState;	
	}
	else
	{
		/* all keys released */
		
		/* fill in key key codes */
				
		kcix = 0;
/*		kcix += sprintf (&baumdd.KeyCodes[kcix], "");*/
		for (i = 0; i < 32; ++i)
		{
			
			if (pressed_keys & Mask32[i])
			{
				/* key pressed */
				if ( 0 <= i && i <= 6)
				{
					/* display keys */
					kcix += sprintf (&baumdd.KeyCodes[kcix], "DK%02d", i);
				}
				
				if ( 7 <= i && i <= 17)
				{
					/* front keys */
					kcix += sprintf (&baumdd.KeyCodes[kcix], "FK%02d", i);
				}
				
				if ( 18 <= i && i <= 28)
				{
					/* back keys */
					kcix += sprintf (&baumdd.KeyCodes[kcix], "BK%02d", i);
				}
				
				if ( 19 <= i && i <= 30)
				{
					/* chord keys */
					kcix += sprintf (&baumdd.KeyCodes[kcix], "CK%02d", i);
				}
				
			}
		
		}
						
		/* fire the key code event		 */
		bec = bec_key_codes;
		bed.KeyCodes = baumdd.KeyCodes;						
		ClientCallback (bec, &bed);
		
		/* clear presed_keys */
		pressed_keys = 0;
	}
	
		
}

/* ... */

void OnSensorsChanged (BRAILLE_EVENT_DATA* bed)
{

	BRAILLE_EVENT_CODE		bec;
	/* BRAILLE_EVENT_DATA   	bed; */
	int rise_event = 1;
	
	/* horizontal mechnical sensors above main display   - bank 0 */
	/* horizontal optical sensors above main display     - bank 1 */
	/* vertical optical sensors left                     - bank 2 */
	/* vertical optical sensors right                    - bank 3 */
	/* horizontal mechnical sensors above status display - bank 4 */
	/* horizontal optical sensors above status display   - bank 5 */
	
	/* build the sensor code */
	
	/* fprintf (stderr, "SV:%d\n", bed->Sensor.Value); */
	
	if (bed->Sensor.Value < 0)
	{
		/* all sensor released */
/*		sprintf (&baumdd.SensorCodes[0], "");*/
		baumdd.SensorCodes[0] = '\0';
	}
	else
	{
	
  	switch (bed->Sensor.Technology)
  	{
 			/* one of the sensors is pressed */
 			case st_optical:
 			
 				switch (bed->Sensor.Bank)
 				{
 					case 1:	/* horizontal optical sensors above main display */
 						sprintf (&baumdd.SensorCodes[0], "HOS%02d", bed->Sensor.Value);
 					break;
 				
 					case 2:	/* vertical optical sensors left */
 						sprintf (&baumdd.SensorCodes[0], "LOS%02d", bed->Sensor.Value);
 					break;
 				
 					case 3:	/* vertical optical sensors right */
 						sprintf (&baumdd.SensorCodes[0], "ROS%02d", bed->Sensor.Value);
 					break;
 				
 					default:
 						rise_event = 0;
 					break;
 				}
 			break;
 		
 			case st_mechanical:
 				sprintf (&baumdd.SensorCodes[0], "HMS%02d", bed->Sensor.Value);
 			break;
 		
 			default:
 				rise_event = 0;
 			break;
  	}
	
	}

	/* add the switch codes and fire the sensor event		 */
	if (rise_event)
	{
		bec = bec_sensor;	
		bed->Sensor.SensorCodes = baumdd.SensorCodes;						
		ClientCallback (bec, bed);
	}
		
}

/* ... */

void OnSwitchPadChanged (BRAILLE_EVENT_DATA* bed)
{
	
	int i;
	int scix;
	BRAILLE_EVENT_CODE		bec;
	/* BRAILLE_EVENT_DATA   	bed; */
	
	/* build the switch_pad codes */
	scix = 0;
/*	scix += sprintf (&baumdd.SwitchCodes[scix], "");*/
	for (i = 0; i < 32; ++i)
	{
		if (baumdd.SwitchState & Mask32[i])
		{
			scix += sprintf (&baumdd.SwitchCodes[scix], "SW%02d", i);
		}		
	}
	
	
	/* add the switch codes and fire switch_pad event		 */
	/* NOTE: the entire switchpad is sent on every change */

	bec = bec_switch_pad;	/* NOTE: includes switch bits and codes */
	bed->SwitchPad.SwitchCodes = baumdd.SwitchCodes;						
	ClientCallback (bec, bed);
	
	
}

/* ... */
short baum_brl_input_parser (int NewVal)
{

	static unsigned char 		BIPBuff[256];
	static BRAILLE_EVENT_CODE	bec;
	static BRAILLE_EVENT_DATA   	bed;

	unsigned char tNewVal = 0;
	static unsigned char inka_key_to_std_key_bitmap [] = {2, 1, 0, 5, 4, 3, 6, 7};
	short i;

	if (!ClientCallback) return 0;	/* if we don't have a client callback doesn't make sense to go further */

	/* !!! TBI !!! */
	/* fprintf (stderr, "%02x ", NewVal & 0xFF); */

	bec = bec_raw_byte;
	bed.RawByte = NewVal;
	ClientCallback (bec, &bed);

	switch (BIPState)
	{
	
	case bips_idle:
		
		if (NewVal == 0x1b)
		{
			/* info-block start */
			BIPState = bips_exp_id;
			BIPIndex = 0;
		}
		break;

	case bips_exp_id:
		
		switch (NewVal)
		{		
		
		case 0x05:
			BIPState = bips_exp_version;
			break;
		
		case 0x06:
			BIPState = bips_exp_version_2;
			break;
		
		case 0x09:
			BIPState = bips_exp_volume;
			break;
		
		case 0x0a:
			BIPState = bips_exp_UART_data;
			break;
		
		case 0x0b:
			BIPState = bips_exp_speech_data;
			break;
		
		case 0x14:
			BIPState = bips_exp_speech_info;
			break;

		case 0x1B:	/* double escape, framing error, stay idle */
			BIPState = bips_idle;
			break;

		case 0x20:
			BIPState = bips_exp_HOS_data;
			break;
		
		case 0x21:
			BIPState = bips_exp_VOS_data;
			break;
		
		case 0x22:
		
			if (BaumDevType == baumdt_inka)
			{
				/* SPECIAL CASE INKA */
				BIPState = bips_exp_switches;
			}
			else
			{
				BIPState = bips_exp_HMS_data;
			}
			break;
		
		case 0x23:
			BIPState = bips_exp_switches;
			break;
		
		case 0x24:
			BIPState = bips_exp_keys;
			break;
		
		case 0x25:
			BIPState = bips_exp_HOS_value;
			break;
		
		case 0x26:
			BIPState = bips_exp_VOS_values;
			break;
		
		case 0x27:
			BIPState = bips_exp_HMS_value;
			break;

		case 0x28:
			BIPState = bips_exp_front_keys;
			break;

		case 0x29:
			BIPState = bips_exp_back_keys;
			break;

		case 0x2B:
			BIPState = bips_exp_chord_keys;
			break;

		case 0x2C:
			BIPState = bips_exp_front_keys_2;
			break;

		case 0x2D:
			BIPState = bips_exp_back_keys_2;
			break;

		case 0x40:
			BIPState = bips_exp_error_code;
			break;
		
		case 0x62:
			BIPState = bips_exp_test_1;
			break;
		
		case 0x63:
			BIPState = bips_exp_test_2;
			break;
		
		case 0x64:
			BIPState = bips_exp_test_3;
			break;
		
		default:
			/* unknown packet ID */
			reset_bip();
			break;
		}
		break;

		/* -------------------------------- */

		case bips_exp_version:
			if (check_escape (NewVal))
			{
				baumdd.Version = NewVal;
				reset_bip();
			}
			break;
		
		case bips_exp_version_2:
			if (check_escape (NewVal))
			{
				if (BIPIndex  < sizeof(BIPBuff)) {
					BIPBuff[BIPIndex++] = NewVal;
					if (BIPIndex >= 3)
					{
						baumdd.Version = BIPBuff[0];
						baumdd.CRC = BIPBuff[2] + ( ((short)BIPBuff[1]) << 8);			/* !!! TBR !!! */
						reset_bip();
					}
				}
				else
				{
					/* buffer overflow */
					reset_bip();
				}
			}
			break;
		
		case bips_exp_volume:
			if (check_escape (NewVal))
			 {
				baumdd.Volume = NewVal;
				reset_bip();
			}
			break;
		
		case bips_exp_UART_data:
			if (check_escape (NewVal))
			{
				baumdd.UARTData = NewVal;
				reset_bip();
			}
			break;

		case bips_exp_speech_data:
			
			if (check_escape (NewVal))
			{
				
				if (BIPIndex  < sizeof(BIPBuff))
				{
					BIPBuff[BIPIndex++] = NewVal;
					if (NewVal == '\r')
					{
						memcpy (&baumdd.SpeechData[0], &BIPBuff[0], BIPIndex);
						/* no braille event here */
						reset_bip();
					}				
				}
				else
				{
					/* buffer overflow, reset parser */
					reset_bip();
				}
				
			}
			break;

		case bips_exp_speech_info:

			if (check_escape (NewVal))
			{
				baumdd.SpeechInfo = NewVal;
				/* no braille event here */
				reset_bip();
			}
			break;

		case bips_exp_HOS_data:
			
			if (check_escape (NewVal))
			{
				if (BIPIndex  < sizeof(BIPBuff))
				{			
					BIPBuff[BIPIndex++] = NewVal;
					if (BIPIndex >= baumdd.HOSLength)
					{
						/* copy to device data */
						memcpy (&baumdd.HOSState[0], &BIPBuff[0], baumdd.HOSLength);
						
						/* compute sensor number */
						baumdd.HOSValue = get_no_from_bitmask (&baumdd.HOSState[0], baumdd.HOSLength);

						bec = bec_sensor;				

						bed.Sensor.Bank = 1;											/* horizontal optical sensors mapped as bank 1 */
						bed.Sensor.Value = baumdd.HOSValue - 1;   /* 0 based for C ... */
						bed.Sensor.AssociatedDisplay = 0;             	/* associated to main display */
						bed.Sensor.Technology = st_optical;
				
						/* ClientCallback (bec, &bed); */
						
						OnSensorsChanged (&bed);

						reset_bip();
					}
				}
				else {
					/* buffer overflow */
					reset_bip();
				}
			}
			break;

		case bips_exp_VOS_data:
			
			if (check_escape (NewVal))
			{
				if (BIPIndex  < sizeof(BIPBuff))
				{							
					BIPBuff[BIPIndex++] = NewVal;
					
					if (BIPIndex >= 8)
					{
						
						/* copy to device data */
						memcpy (&baumdd.VOSState[0], &BIPBuff[0], 8);

						/* compute the VOS values */
						baumdd.VOSValueLeft = get_no_from_bitmask (&baumdd.VOSState[0], 4);
						baumdd.VOSValueRight = get_no_from_bitmask (&baumdd.VOSState[4], 4);						

			                       /* common for both banks */
						bec = bec_sensor;				
						bed.Sensor.AssociatedDisplay = -1;             						/* no associated display */
						bed.Sensor.Technology = st_optical;

						/* left sensor bank */
						bed.Sensor.Bank = 2;														/* horizontal mechnaical sensors mapped as bank 0 */
						bed.Sensor.Value = baumdd.VOSValueLeft - 1;   		/* 0 based for C ...				 */
						/* ClientCallback (bec, &bed); */
						OnSensorsChanged (&bed);

						/* right sensor bank */
						bed.Sensor.Bank = 3;													/* horizontal mechnaical sensors mapped as bank 0 */
						bed.Sensor.Value = baumdd.VOSValueRight - 1;   /* 0 based for C ... */
						/* ClientCallback (bec, &bed); */
						OnSensorsChanged (&bed);
						
						reset_bip();
					}

				}
				else
				{
					reset_bip();
				}
			}
			break;

		case bips_exp_HMS_data:
			
			/* fprintf (stderr, "HMS DATA\n"); */
			
			if (check_escape (NewVal))
			{
				if (BIPIndex  < sizeof(BIPBuff))
				{			
					BIPBuff[BIPIndex++] = NewVal;
					if (BIPIndex >= baumdd.HMSLength)
					{
						
						/* copy to device data */
						memcpy (&baumdd.HMSState[0], &BIPBuff[0], baumdd.HMSLength);
						
						/* compute the value */
						baumdd.HMSValue = get_no_from_bitmask (&baumdd.HMSState[0], baumdd.HMSLength);										

						bec = bec_sensor;				

						bed.Sensor.Bank = 0;											/* horizontal mechanical sensors mapped as bank 0 */
						bed.Sensor.Value = baumdd.HMSValue - 1;   /* 0 based for C ... */
						bed.Sensor.AssociatedDisplay = 0;             	/* associated to main display */
						bed.Sensor.Technology = st_mechanical;
				
						/* ClientCallback (bec, &bed); */
						OnSensorsChanged (&bed);
						
						reset_bip();
					}
				}
				else
				{
					reset_bip();
				}
			}
			break;

		case bips_exp_switches:				
			
			if (check_escape (NewVal))
			{	

				/* special case INKA */
				if (BaumDevType == baumdt_inka)
				{
					NewVal &= 0x0F;	/* only 4 switches */
				}
				
				/* make xor with old switches to detect the changed bits */
				/* BYTE ts = baumdd.SwitchState ^ NewVal; */

				/* store the new switch state */
				baumdd.SwitchState = NewVal;
				
				
				/* bec = bec_switch_bits; */
				bed.SwitchPad.SwitchBits = baumdd.SwitchState;						
				/* ClientCallback (bec, &bed); */
				OnSwitchPadChanged (&bed);
				
				/* reset the state machine */
				reset_bip();
				
			}	
			break;

		case bips_exp_keys:
			
			if (check_escape (NewVal))
			{
					
				switch (BaumDevType)
				{
					case baumdt_dm80p:										
											
						/* direct mapping, device has no other keys */
						baumdd.CumulatedKeyState = baumdd.KeyState= ~NewVal & 0x7F;	/* invert the status bit, 1 means make, 0 means break */
						break;
					
					case baumdt_vario40:
					case baumdt_vario20:
						/* direct mapping, device has no other keys */
						baumdd.CumulatedKeyState = baumdd.KeyState = NewVal & 0x3F;	
					break;
					
					case baumdt_vario80:
												
						/* baumdd.KeyState = ~NewVal;	 invert the status bit, 1 means make, 0 means break; */
						baumdd.KeyState = NewVal  & 0x3F;	
						
						/* map the actual keys togheter with front-, back- and chord keys */
						baumdd.CumulatedKeyState = ((unsigned long) baumdd.KeyState & 0x0000003F)+
																			(((unsigned long ) baumdd.FrontKeyState) << 6   & 0x0000FFC0)+
																			(((unsigned long ) baumdd.BackKeyState) << 16  & 0x03FF0000)+
																			(((unsigned long ) baumdd.ChordKeyState) << 26 & 0x7C000000)	;

						break;

					case baumdt_inka:

						/* special case INKA - different layout */

						NewVal ^= 0x3F;
						tNewVal = 0;
						
						for (i = 0; i < 8; ++i)
						{
						
							if (NewVal & Mask8[i])
							{
								tNewVal |= Mask8[inka_key_to_std_key_bitmap[i]];
							}
						}

						baumdd.KeyState = ~tNewVal;	/* invert the status bit, 1 means make, 0 means break; */
						
						/* direct mapping, device has no other keys */
						baumdd.CumulatedKeyState = baumdd.KeyState;

						break;

					case baumdt_generic:
					default:
						break;
				}

				/* fire the key bits				 */
				bec = bec_key_bits;
				bed.KeyBits = baumdd.CumulatedKeyState;						
				ClientCallback (bec, &bed);
								
				/* call for further processing				 */
				OnKeysChanged ();
				
				/* reset the state machine */
				reset_bip();
			}
			break;

		case bips_exp_front_keys:
			
			if (check_escape (NewVal))
			{

				/* make xor with old keys to detect the changed bits */
				/* BYTE tk = baumdd.FrontKeyState ^ NewVal;	 */

				/* store the state */
				baumdd.FrontKeyState = NewVal;	

				/* !!! TBI !!! add to cumulated key state */
				
				/* fire the raw event */
				/* fire the cooked event(s) */

				fprintf (stderr, "\nFRONT_KEY: %02x", baumdd.FrontKeyState);						
				
				/* reset the state machine */
				reset_bip();
			
			}
			break;

		case bips_exp_back_keys:
			
			if (check_escape (NewVal))
			{

				/* make xor with old keys to detect the changed bits */
				/* BYTE tk = baumdd.BackKeyState ^ NewVal;	 */

				/* store the state */
				baumdd.BackKeyState = NewVal;	
				
				/* !!! TBI !!! add to cumulated key state */

				/* fire the raw event				 */
				/* fire the cooked event(s) */
				
				fprintf (stderr, "\nBACK_KEY: %02x", baumdd.BackKeyState);										
				
				/* reset the state machine */
				reset_bip();
			
			}
			break;

		case bips_exp_chord_keys:
			if (check_escape (NewVal))
			{
				
				/*                         BIT: 0  1  2  3  4  5  6  7 */
				/* static short chord_key_map[] = {3, 2, 1, 7, 4, 5, 6, 0};	 <- Key No */

				/* make xor with old keys to detect the changed bits */
				/* BYTE tk = baumdd.ChordKeyState ^ NewVal;	 */

				/* store the state */
				baumdd.ChordKeyState = NewVal;	

				/* !!! TBI !!! add to cumulated key state */
				
				/* fire the raw event				 */
				/* fire the cooked event(s) */

				fprintf (stderr, "\nCHORD_KEY: %02x", baumdd.ChordKeyState);										
				
				/* reset the state machine */
				reset_bip();
			
			}
			break;

		case bips_exp_front_keys_2:
			
			if (check_escape (NewVal))
			{
				if (BIPIndex  < 2) {
					/* !!! TBI !!! */
					BIPBuff[BIPIndex++] = NewVal;					
					if (BIPIndex >= 2)
					{
						
						/* WORD nv = MAKEWORD(IPBuff[1], IPBuff[0]); */

						/* make xor with old keys to detect the changed bits */
						/* WORD tk = baumdd.FrontKeyState2 ^ nv;	 */
						
						/* store the cuurent state */
						/* baumdd.FrontKeyState2 = nv; */

						/* fire the raw event */
						
						/* fire the cooked event(s) */

						fprintf (stderr, "\nFRONT_KEY2");

						reset_bip();
					}
				}
				else {
					/* buffer overflow */
					reset_bip();
				}
			}
			break;

		case bips_exp_back_keys_2:
			
			if (check_escape (NewVal))
			{
				/* !!! TBI !!! */
				if (BIPIndex  < 2)
				{
					BIPBuff[BIPIndex++] = NewVal;
					if (BIPIndex >= 2)
					{
						/* WORD nv = MAKEWORD(IPBuff[1], IPBuff[0]); */

						/* make xor with old keys to detect the changed bits */
						/* WORD tk = baumdd.BackKeyState2 ^ nv;	  !!! TBR !!! */
						
						/* store the current state */
						/* baumdd.BackKeyState2 = nv; */
						
						/* fire the raw event						 */
						/* fire the cooked event(s) */
                   fprintf (stderr, "\nBACK_KEY2");
						reset_bip();
					}
				}
				else {
					/* buffer overflow */
					reset_bip();
				}
			}
			break;

		case bips_exp_HOS_value:
			
			if (check_escape (NewVal))
			{
				
				baumdd.HOSValue = NewVal;				

				/* fprintf (stderr, "\nHOS_VAL2: %d", baumdd.HOSValue);				 */
				bec = bec_sensor;				

				bed.Sensor.Bank = 1;											/* horizontal optical sensors mapped as bank 1 */
				bed.Sensor.Value = baumdd.HOSValue - 1;   /* 0 based for C ... */
				bed.Sensor.AssociatedDisplay = 0;             	/* associated to main display */
				bed.Sensor.Technology = st_optical;
				
				/* ClientCallback (bec, &bed); */
				OnSensorsChanged (&bed);

				reset_bip();
			}
			break;

		case bips_exp_VOS_values:

			if (check_escape (NewVal))
			{
				
				BIPBuff[BIPIndex++] = NewVal;
				
				switch (BaumDevType)
				{
				
				
				case baumdt_inka:
					/* special case INKA  !!! TBR !!! */

					bed.Sensor.AssociatedDisplay = -1;             							/* no associated display */
					bed.Sensor.Technology = st_optical;
										
					if (BIPBuff[0] >= 65)
					{
						baumdd.VOSValueRight = BIPBuff[0] - 65;

						bed.Sensor.Bank = 3;																	/* horizontal mechnaical sensors mapped as bank 0 */
						bed.Sensor.Value = baumdd.VOSValueRight - 1;		/* 0 based for C ... */
				
						/* ClientCallback (bec, &bed); */
						OnSensorsChanged (&bed);

					}
					else
					{
						baumdd.VOSValueLeft = BIPBuff[0];

						bed.Sensor.Bank = 2;																/* horizontal mechnaical sensors mapped as bank 0 */
						bed.Sensor.Value = baumdd.VOSValueLeft - 1;   /* 0 based for C ... */
					
						/* ClientCallback (bec, &bed); */
						OnSensorsChanged (&bed);

					}
										
					reset_bip();					
					break;

				default:
					/* DM80P and all others */
					if (BIPIndex >= 2)
					{
			
						baumdd.VOSValueLeft = BIPBuff[0];
						baumdd.VOSValueRight = BIPBuff[1];

            /* common for both banks */
						bec = bec_sensor;				
						bed.Sensor.AssociatedDisplay = -1;             						/* no associated display */
						bed.Sensor.Technology = st_optical;

						/* left sensor bank */
						bed.Sensor.Bank = 2;																/* horizontal mechnaical sensors mapped as bank 0 */
						bed.Sensor.Value = baumdd.VOSValueLeft - 1;   /* 0 based for C ...				 */
						/* ClientCallback (bec, &bed); */
						OnSensorsChanged (&bed);

						/* right sensor bank */
						bed.Sensor.Bank = 3;													/* horizontal mechnaical sensors mapped as bank 0 */
						bed.Sensor.Value = baumdd.VOSValueRight - 1;   /* 0 based for C ... */
						/* ClientCallback (bec, &bed); */
						OnSensorsChanged (&bed);
						
						reset_bip();
					}
				
					break;
				}
			}
			break;
		
		case bips_exp_HMS_value:
			
			/* fprintf (stderr, "HMS VALUE\n"); */
			if (check_escape (NewVal))
			{
				
				baumdd.HMSValue = NewVal;				
								
				bec = bec_sensor;				

				bed.Sensor.Bank = 0;											/* horizontal mechnaical sensors mapped as bank 0 */
				bed.Sensor.Value = baumdd.HMSValue - 1;   /* 0 based for C ... */
				bed.Sensor.AssociatedDisplay = 0;         /* associated to main display */
				bed.Sensor.Technology = st_mechanical;
				
				/* ClientCallback (bec, &bed); */
				OnSensorsChanged (&bed);

				reset_bip();
			}
			break;

		case bips_exp_error_code:
			{
				baumdd.LastError = NewVal;			
				fprintf (stderr, "\nBAUM DEVICE ERROR : %d", baumdd.LastError);												
				reset_bip();
			}
			break;

		default:
			/* internal error, invalid parser state */
			reset_bip();
			/* assert (TRUE); */
			break;
	}

	return 0; /* 0-give next byte, 1-repeat last byte */
}

/* ... */
int baum_brl_send_dots (unsigned char *Dots, short Count, short Blocking)
{
	int rv = 0;
	int i, realcnt = 0;
	unsigned char sendbuff[256];
			
	if (BaumDevType == baumdt_inka)
	{
		sendbuff[0] = 0x1b;
		sendbuff[1] = 0x01;
		sendbuff[2] = 0x00;	/* special case: INKA needs a pos byte here */
		realcnt = 3;
	}
	else
	{
		sendbuff[0] = 0x1b;
		sendbuff[1] = 0x01;
		realcnt = 2;	
	}

	/* go byte by byte, take care of the "ESC doubling" */
	
	for (i = 0; i < Count; ++i) {
		
		if (Dots[i] == 0x1B)
		{
			sendbuff [realcnt++] = Dots[i];
		}
		sendbuff [realcnt++] = Dots[i];
		
	}

	rv = brl_ser_send_data ((char*)sendbuff, realcnt, Blocking);

	return rv;
}

/* ... */
void baum_brl_close_device ()
{
	/* close serial communication */
	brl_ser_set_callback (NULL);				
	brl_ser_stop_timer ();	
	brl_ser_close_port ();
}

/* ... */
int baum_brl_open_device (char* DeviceName, short Port, BRL_DEV_CALLBACK DeviceCallback, BRL_DEVICE *Device)
{

	int rv = 0;

	clear_device_data (&baumdd);

	if (
			(strcmp ("VARIO", DeviceName) == 0) ||
			(strcmp ("VARIO40", DeviceName) == 0)
	)
	{
		Device->CellCount = 40;
		Device->DisplayCount = 1;
		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 40;
		Device->Displays[0].Type = bdt_main;
		Device->InputType = bit_bits;
		Device->KeyCount = 6;

		/* !!! MORE !!! */
		baumdd.HMSLength = 5;
		BaumDevType = baumdt_vario40;
		rv = 1;
	}

	else if (strcmp ("VARIO20", DeviceName) == 0) 	
	{
		Device->CellCount = 20;
		Device->DisplayCount = 1;
		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 20;
		Device->Displays[0].Type = bdt_main;
		Device->InputType = bit_bits;
		Device->KeyCount = 6;

		/* !!! MORE !!! */
		baumdd.HMSLength = 5;	/* ??? */
		BaumDevType = baumdt_vario20;

		rv = 1;
	}

	else if (strcmp ("VARIO80", DeviceName) == 0) 	
	{		
		Device->CellCount = 84;
		Device->DisplayCount = 2;
		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 80;
		Device->Displays[0].Type = bdt_main;
		Device->Displays[1].StartCell = 80;
		Device->Displays[1].Width = 4;
		Device->Displays[1].Type = bdt_status;
		Device->InputType = bit_bits;
		Device->KeyCount = 31;
		Device->SwitchCount = 0;
		Device->SensorBankCount = 2;

		/* !!! MORE !!! */
		baumdd.HMSLength = 11;
		BaumDevType = baumdt_vario80;
		rv = 1;
	}

	else if (strcmp ("DM80P", DeviceName) == 0) 	
	{	
		Device->CellCount = 84;
		Device->DisplayCount = 2;
		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 80;
		Device->Displays[0].Type = bdt_main;
		Device->Displays[1].StartCell = 80;
		Device->Displays[1].Width = 4;
		Device->Displays[1].Type = bdt_status;
		Device->InputType = bit_bits;
		Device->KeyCount = 7;
		Device->SwitchCount = 6;
		Device->SensorBankCount = 5;

		BaumDevType = baumdt_dm80p;

		rv = 1;
	}

	else if (strcmp ("INKA", DeviceName) == 0) 	
	{
		/* status main aux*/
		/*56 = 4 + 40+ 12*/
		Device->CellCount = 44;
		Device->DisplayCount = 2;
		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 40;
		Device->Displays[0].Type = bdt_main;
		Device->Displays[1].StartCell = 40;
		Device->Displays[1].Width = 4;
		Device->Displays[1].Type = bdt_status;

		Device->InputType = bit_bits;
		Device->KeyCount = 6;

		/* !!! MORE !!! */
		baumdd.HMSLength = 5;	/* ??? */
		BaumDevType = baumdt_inka;

		rv = 1;
	}

	else
	{
		/* unknown device */
	}

	if (rv)
	{
		/* fill device functions		 */
		Device->close_device = baum_brl_close_device;
		Device->send_dots = baum_brl_send_dots;

		/* open serial communication */
		if (brl_ser_open_port (Port))
		{
			brl_ser_set_callback (baum_brl_input_parser);				
			rv = brl_ser_set_comm_param (19200, 'N', 1, 'N');		
			/* brl_ser_start_timer (10);	 ASYNC CALLBACK */
			rv &= brl_ser_init_glib_poll ();		/* SYNC CALLBACK */
			ClientCallback = DeviceCallback;
			
			/* set sensors to compressed mode (HOS/HMS value)			 */
			/* rv = brl_ser_send_data ("\x1B\x12\x08\x00", 4, 1);  ??? HOW ??? */
			
			/* ask for all input to get the initial switch data */
			rv &= brl_ser_send_data ("\x1B\x08", 2, 1);
		}	
		else
		{
			/* err */
			rv = 0;
		}
	}

    return rv;
}

