/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/* gnome-vfs-mime-info.c - GNOME mime-information implementation.

   Copyright (C) 1998 Miguel de Icaza
   All rights reserved.

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <config.h>
#include "gnome-vfs-mime-info.h"

#include "gnome-vfs-mime.h"
#include "gnome-vfs-mime-private.h"

#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <dirent.h>
#ifdef NEED_GNOMESUPPORT_H
#include "gnomesupport.h"
#endif

#if !defined getc_unlocked && !defined HAVE_GETC_UNLOCKED
# define getc_unlocked(fp) getc (fp)
#endif

typedef struct {
	char       *mime_type;
	GHashTable *keys;
} GnomeMimeContext;

/* Describes the directories we scan for information */
typedef struct {
	char *dirname;
	struct stat s;
	unsigned int valid : 1;
	unsigned int system_dir : 1;
} mime_dir_source_t;

/* These ones are used to automatically reload mime info on demand */
static mime_dir_source_t gnome_mime_dir, user_mime_dir;
static time_t last_checked;

/* To initialize the module automatically */
static gboolean gnome_vfs_mime_inited = FALSE;


static GList *current_lang = NULL;
/* we want to replace the previous key if the current key has a higher
   language level */
static char *previous_key = NULL;
static int previous_key_lang_level = -1;


/*
 * A hash table containing all of the Mime records for specific
 * mime types (full description, like image/png)
 */
static GHashTable *specific_types;

/*
 * A hash table containing all of the Mime records for non-specific
 * mime types (like image/\*)
 */
static GHashTable *generic_types;

/*
 * A hash table containing all of the Mime records for all registered
 * mime types
 */
static GHashTable *registered_types;

/* Prototypes */
static void write_registered_mime_data (void);


static GnomeMimeContext *
context_new (GString *str)
{
	GnomeMimeContext *context;
	GHashTable *table;
	char *mime_type, *p;

	mime_type = g_strdup (str->str);
	
	if ((p = strstr (mime_type, "/*")) == NULL){
		table = specific_types;
	} else {
		*(p+1) = 0;
		table = generic_types;
	}
	
	context = g_hash_table_lookup (table, mime_type);

	if (context) {
		g_free (mime_type);
		return context;
	}
	
	context = g_new (GnomeMimeContext, 1);
	context->mime_type = mime_type;
	context->keys = g_hash_table_new (g_str_hash, g_str_equal);

	g_hash_table_insert (table, context->mime_type, context);
	return context;
}

static GnomeMimeContext *
list_context_new (GString *str)
{
	GnomeMimeContext *context;
	char *mime_type;
		
	mime_type = g_strdup (str->str);
		
	context = g_hash_table_lookup (registered_types, mime_type);

	if (context) {
		g_free (mime_type);
		return context;
	}
	
	context = g_new (GnomeMimeContext, 1);
	context->mime_type = mime_type;

	context->keys = g_hash_table_new (g_str_hash, g_str_equal);

	g_hash_table_insert (registered_types, context->mime_type, context);
	return context;
}

static gboolean
release_key_and_value (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	g_free (value);

	return TRUE;
}

static void
context_destroy (GnomeMimeContext *context)
{
	/*
	 * Destroy it
	 */
	g_hash_table_foreach_remove (context->keys, release_key_and_value, NULL);
	g_hash_table_destroy (context->keys);
	g_free (context->mime_type);
	g_free (context);
}

static void
context_destroy_and_unlink (GnomeMimeContext *context)
{
	/*
	 * Remove the context from our hash tables, we dont know
	 * where it is: so just remove it from both (it can
	 * only be in one).
	 */
	g_hash_table_remove (specific_types, context->mime_type);
	g_hash_table_remove (generic_types, context->mime_type);
	g_hash_table_remove (registered_types, context->mime_type);

	context_destroy (context);
}

/* this gives us a number of the language in the current language list,
   the higher the number the "better" the translation */
static int
language_level (const char *lang)
{
	int i;
	GList *li;

	if (lang == NULL)
		return 0;

	for (i = 1, li = current_lang; li != NULL; i++, li = g_list_next (li)) {
		if (strcmp ((const char *) li->data, lang) == 0)
			return i;
	}

	return -1;
}


static void
context_add_key (GnomeMimeContext *context, char *key, char *lang, char *value)
{
	char *v;
	char *orig_key;
	int lang_level;

	lang_level = language_level(lang);
	/* wrong language completely */
	if (lang_level<0)
		return;

	/* if we have some language defined and
	   if there was a previous_key */
	if (lang_level > 0 &&
	    previous_key) {
		/* if our new key has a better lang_level then remove the
		   previous key */
		if (previous_key_lang_level <= lang_level) {
			if (g_hash_table_lookup_extended (context->keys,
							  previous_key,
							  (gpointer *)&orig_key,
							  (gpointer *)&v)) {
				g_hash_table_remove (context->keys, orig_key);
				g_free(orig_key);
				g_free(v);
			}
		/* else, our language level really sucks and the previous
		   translation was of better language quality so just
		   ignore us */
		} else
			return;
	}

	if (g_hash_table_lookup_extended (context->keys, key,
					  (gpointer *)&orig_key,
					  (gpointer *)&v)) {
		/* if we found it in the database already, just replace it
		   here */
		g_free (v);
		g_hash_table_insert (context->keys, orig_key,
				     g_strdup (value));
	} else {
		g_hash_table_insert (context->keys, g_strdup(key),
				     g_strdup (value));
	}
	/* set this as the previous key */
	g_free(previous_key);
	previous_key = g_strdup(key);
	previous_key_lang_level = lang_level;
}

typedef enum {
	STATE_NONE,
	STATE_LANG,
	STATE_LOOKING_FOR_KEY,
	STATE_ON_MIME_TYPE,
	STATE_ON_KEY,
	STATE_ON_VALUE
} ParserState;

static void
load_mime_type_info_from (char *filename)
{
	FILE *mime_file;
	gboolean in_comment, context_used;
	GString *line;
	int column, c;
	ParserState state;
	GnomeMimeContext *context;
	char *key;
	char *lang;
	
	mime_file = fopen (filename, "r");
	if (mime_file == NULL)
		return;

	in_comment = FALSE;
	context_used = FALSE;
	column = -1;
	context = NULL;
	key = NULL;
	lang = NULL;
	line = g_string_sized_new (120);
	state = STATE_NONE;
	
	while ((c = getc_unlocked (mime_file)) != EOF){
		column++;
		if (c == '\r')
			continue;

		if (c == '#' && column == 0){		
			in_comment = TRUE;
			continue;
		}
		
		if (c == '\n'){
			in_comment = FALSE;
			column = 0;
			if (state == STATE_ON_MIME_TYPE){

				/* set previous key to nothing
				   for this mime type */
				g_free(previous_key);
				previous_key = NULL;
				previous_key_lang_level = -1;

				context = context_new (line);
				context_used = FALSE;
				g_string_assign (line, "");
				state = STATE_LOOKING_FOR_KEY;
				continue;
			}
			if (state == STATE_ON_VALUE){
				context_used = TRUE;
				context_add_key (context, key, lang, line->str);
				g_string_assign (line, "");
				g_free (key);
				key = NULL;
				g_free (lang);
				lang = NULL;
				state = STATE_LOOKING_FOR_KEY;
				continue;
			}
			continue;
		}

		if (in_comment)
			continue;

		switch (state){
		case STATE_NONE:
			if (c != ' ' && c != '\t')
				state = STATE_ON_MIME_TYPE;
			else
				break;
			/* fall down */
			
		case STATE_ON_MIME_TYPE:
			if (c == ':'){
				in_comment = TRUE;
				break;
			}
			g_string_append_c (line, c);
			break;

		case STATE_LOOKING_FOR_KEY:
			if (c == '\t' || c == ' ')
				break;

			if (c == '['){
				state = STATE_LANG;
				break;
			}

			if (column == 1){
				state = STATE_ON_MIME_TYPE;
				g_string_append_c (line, c);
				break;
			}
			state = STATE_ON_KEY;
			/* falldown */

		case STATE_ON_KEY:
			if (c == '\\'){
				c = getc (mime_file);
				if (c == EOF)
					break;
			}
			if (c == '='){
				key = g_strdup (line->str);
				g_string_assign (line, "");
				state = STATE_ON_VALUE;
				break;
			}
			g_string_append_c (line, c);
			break;

		case STATE_ON_VALUE:
			g_string_append_c (line, c);
			break;
			
		case STATE_LANG:
			if (c == ']'){
				state = STATE_ON_KEY;      
				if (line->str [0]){
					g_free(lang);
					lang = g_strdup(line->str);
				} else {
					in_comment = TRUE;
					state = STATE_LOOKING_FOR_KEY;
				}
				g_string_assign (line, "");
				break;
			}
			g_string_append_c (line, c);
			break;
		}
	}

	if (context){
		if (key && line->str [0])
			context_add_key (context, key, lang, line->str);
		else
			if (!context_used)
				context_destroy_and_unlink (context);
	}

	g_string_free (line, TRUE);
	g_free (key);
	g_free (lang);

	/* free the previous_key stuff */
	g_free(previous_key);
	previous_key = NULL;
	previous_key_lang_level = -1;

	fclose (mime_file);
}

/*
 *  load_mime_list_info_from
 *
 *  Why this special function when a similar one is already in the code?
 *  Because we need to handle the case where ':' is used to delimit
 *  the start of a key in a .mime file instead of '=' as is used in
 *  the .key file.  Why is this done?  Why are there two mime database 
 *  files with differing standards?  We may never know.  
 *  Until we have a better solution, this will suffice.	
 *  
 *  Both ':' and '=' are used to delimit the start of a key.
 */
 
static void
load_mime_list_info_from (char *filename)
{
	FILE *mime_file;
	gboolean in_comment, context_used;
	GString *line;
	int column, c;
	ParserState state;
	GnomeMimeContext *context;
	char *key;
	char *lang;
	
	mime_file = fopen (filename, "r");
	if (mime_file == NULL)
		return;

	in_comment = FALSE;
	context_used = FALSE;
	column = -1;
	context = NULL;
	key = NULL;
	lang = NULL;
	line = g_string_sized_new (120);
	state = STATE_NONE;
	
	while ((c = getc_unlocked (mime_file)) != EOF){
		column++;
		if (c == '\r')
			continue;

		if (c == '#' && column == 0){
			in_comment = TRUE;
			continue;
		}
		
		if (c == '\n'){
			in_comment = FALSE;
			column = 0;
			if (state == STATE_ON_MIME_TYPE){

				/* set previous key to nothing
				   for this mime type */
				g_free(previous_key);
				previous_key = NULL;
				previous_key_lang_level = -1;

				context = list_context_new (line);
				context_used = FALSE;
				g_string_assign (line, "");
				state = STATE_LOOKING_FOR_KEY;
				continue;
			}
			if (state == STATE_ON_VALUE){
				context_used = TRUE;
				context_add_key (context, key, lang, line->str);
				g_string_assign (line, "");
				g_free (key);
				key = NULL;
				g_free (lang);
				lang = NULL;
				state = STATE_LOOKING_FOR_KEY;
				continue;
			}
			continue;
		}

		if (in_comment)
			continue;

		switch (state){
		case STATE_NONE:
			if (c != ' ' && c != '\t')
				state = STATE_ON_MIME_TYPE;
			else
				break;
			/* fall down */
			
		case STATE_ON_MIME_TYPE:
			g_string_append_c (line, c);
			break;

		case STATE_LOOKING_FOR_KEY:
			if (c == '\t' || c == ' ')
				break;

			if (c == '['){
				state = STATE_LANG;
				break;
			}

			if (column == 1){
				state = STATE_ON_MIME_TYPE;
				g_string_append_c (line, c);
				break;
			}
			state = STATE_ON_KEY;
			/* falldown */

		case STATE_ON_KEY:
			if (c == '\\'){
				c = getc (mime_file);
				if (c == EOF)
					break;
			}			
			if (c == '=') {
				key = g_strdup (line->str);				
				g_string_assign (line, "");				
				state = STATE_ON_VALUE;
				break;
			}

			if (c == ':') {
				key = g_strdup (line->str);				
				g_string_assign (line, "");

				/* Skip space after colon.  There should be one
				 * there.  That is how the file is defined. */
				c = getc_unlocked (mime_file);
				if (c != ' ') {
					/* Revert seek */
					ungetc (c, mime_file);
				} else {
					column++;
				}
				
				state = STATE_ON_VALUE;
				break;
			}

			g_string_append_c (line, c);
			break;

		case STATE_ON_VALUE:
			g_string_append_c (line, c);
			break;
			
		case STATE_LANG:
			if (c == ']') {
				state = STATE_ON_KEY;      
				if (line->str [0]){
					g_free(lang);
					lang = g_strdup(line->str);
				} else {
					in_comment = TRUE;
					state = STATE_LOOKING_FOR_KEY;
				}
				g_string_assign (line, "");
				break;
			}
			g_string_append_c (line, c);
			break;
		}
	}

	if (context){
		if (key && line->str [0])
			context_add_key (context, key, lang, line->str);
		else
			if (!context_used)
				context_destroy_and_unlink (context);
	}

	g_string_free (line, TRUE);
	g_free (key);
	g_free (lang);

	/* free the previous_key stuff */
	g_free(previous_key);
	previous_key = NULL;
	previous_key_lang_level = -1;

	fclose (mime_file);
}

static void
mime_info_load (mime_dir_source_t *source)
{
	DIR *dir;
	struct dirent *dent;
	const int extlen = sizeof (".keys") - 1;
	char *filename;
	
	if (stat (source->dirname, &source->s) != -1)
		source->valid = TRUE;
	else
		source->valid = FALSE;
	
	dir = opendir (source->dirname);
	if (!dir){
		source->valid = FALSE;
		return;
	}
	if (source->system_dir){
		filename = g_concat_dir_and_file (source->dirname, "gnome-vfs.keys");
		load_mime_type_info_from (filename);
		g_free (filename);
	}

	while ((dent = readdir (dir)) != NULL){
		
		int len = strlen (dent->d_name);

		if (len <= extlen)
			continue;
		if (strcmp (dent->d_name + len - extlen, ".keys"))
			continue;
		if (source->system_dir && !strcmp (dent->d_name, "gnome-vfs.keys"))
			continue;

		if (source->system_dir && !strcmp (dent->d_name, "gnome.keys")) {
			/* Ignore the obsolete "official" one so it doesn't override
			 * the new official one.
			 */
			continue;
		}
			
		if (!source->system_dir && !strcmp (dent->d_name, "user.keys"))
			continue;

		filename = g_concat_dir_and_file (source->dirname, dent->d_name);
		load_mime_type_info_from (filename);
		g_free (filename);
	}
	if (!source->system_dir) {
		filename = g_concat_dir_and_file (source->dirname, "user.keys");
		load_mime_type_info_from (filename);
		g_free (filename);
	}
	closedir (dir);
}

static void
mime_list_load (mime_dir_source_t *source)
{
	DIR *dir;
	struct dirent *dent;
	const int extlen = sizeof (".mime") - 1;
	char *filename;
	
	if (stat (source->dirname, &source->s) != -1)
		source->valid = TRUE;
	else
		source->valid = FALSE;
	
	dir = opendir (source->dirname);
	if (!dir){
		source->valid = FALSE;
		return;
	}
	if (source->system_dir){
		filename = g_concat_dir_and_file (source->dirname, "gnome-vfs.mime");
		load_mime_list_info_from (filename);
		g_free (filename);
	}

	while ((dent = readdir (dir)) != NULL){
		
		int len = strlen (dent->d_name);

		if (len <= extlen)
			continue;
		if (strcmp (dent->d_name + len - extlen, ".mime"))
			continue;
		if (source->system_dir && !strcmp (dent->d_name, "gnome-vfs.mime"))
			continue;

		if (source->system_dir && !strcmp (dent->d_name, "gnome.mime")) {
			/* Ignore the obsolete "official" one so it doesn't override
			 * the new official one.
			 */
			continue;
		}
			
		if (!source->system_dir && !strcmp (dent->d_name, "user.mime"))
			continue;

		filename = g_concat_dir_and_file (source->dirname, dent->d_name);
		load_mime_list_info_from (filename);
		g_free (filename);
	}
	if (!source->system_dir) {
		filename = g_concat_dir_and_file (source->dirname, "user.mime");
		load_mime_list_info_from (filename);
		g_free (filename);
	}
	closedir (dir);
}

static void
load_mime_type_info (void)
{
	mime_info_load (&gnome_mime_dir);
	mime_info_load (&user_mime_dir);
	mime_list_load (&gnome_mime_dir);
}

static void
gnome_vfs_mime_init ()
{
	/*
	 * The hash tables that store the mime keys.
	 */
	specific_types = g_hash_table_new (g_str_hash, g_str_equal);
	generic_types  = g_hash_table_new (g_str_hash, g_str_equal);
	registered_types  = g_hash_table_new (g_str_hash, g_str_equal);
	
	current_lang = gnome_i18n_get_language_list ("LC_MESSAGES");
	if(current_lang)
		current_lang = g_list_reverse(current_lang);

	/*
	 * Setup the descriptors for the information loading
	 */

	/* FIXME bugzilla.eazel.com 796: Looks in gnome-libs prefix instead of gnome-vfs prefix. */
	gnome_mime_dir.dirname = gnome_unconditional_datadir_file ("mime-info");
	gnome_mime_dir.system_dir = TRUE;
	
	/* FIXME bugzilla.eazel.com 796: Looks in gnome-libs prefix instead of gnome-vfs prefix. */
	user_mime_dir.dirname  = gnome_util_home_file ("mime-info");
	user_mime_dir.system_dir = FALSE;

	/*
	 * Load
	 */
	load_mime_type_info ();

	last_checked = time (NULL);
	gnome_vfs_mime_inited = TRUE;
}

static gboolean
remove_keys (gpointer key, gpointer value, gpointer user_data)
{
	GnomeMimeContext *context = value;

	context_destroy (context);
	
	return TRUE;
}

static void
maybe_reload (void)
{
	time_t now = time (NULL);
	gboolean need_reload = FALSE;
	struct stat s;
	
	if (last_checked + 5 >= now)
		return;

	if (stat (gnome_mime_dir.dirname, &s) != -1)
		if (s.st_mtime != gnome_mime_dir.s.st_mtime)
			need_reload = TRUE;

	if (stat (user_mime_dir.dirname, &s) != -1)
		if (s.st_mtime != user_mime_dir.s.st_mtime)
			need_reload = TRUE;

	last_checked = now;
	
	if (need_reload) {
	        gnome_vfs_mime_info_reload ();
	}
}

static void
gnome_vfs_mime_info_clear (void)
{
	if (specific_types != NULL) {
		g_hash_table_foreach_remove (specific_types, remove_keys, NULL);
	}
	if (generic_types != NULL) {
		g_hash_table_foreach_remove (generic_types, remove_keys, NULL);
	}
	if (registered_types != NULL) {
		g_hash_table_foreach_remove (registered_types, remove_keys, NULL);
	}
}

void
gnome_vfs_mime_info_shutdown (void)
{
	gnome_vfs_mime_info_clear ();

	if (specific_types != NULL) {
		g_hash_table_destroy (specific_types);
	}
	if (generic_types != NULL) {
		g_hash_table_destroy (generic_types);
	}
	if (registered_types != NULL) {
		g_hash_table_destroy (registered_types);
	}
}

void
gnome_vfs_mime_info_reload (void)
{
	/* 1. Clean */
	gnome_vfs_mime_info_clear ();
	
	/* 2. Reload */
	load_mime_type_info ();
}


/**
 * gnome_vfs_mime_context_get_value:
 * @mime_type: a mime type.
 * @key: A key to lookup for the given mime-type
 *
 * This function retrieves the value associated with @key in 
 * the given GnomeMimeContext.  The string is private, you
 * should not free the result.
 */
const char *
gnome_vfs_mime_get_value (const char *mime_type, const char *key)
{
	char *value, *generic_type, *p;
	GnomeMimeContext *context;
	
	g_return_val_if_fail (key != NULL, NULL);

	if (mime_type == NULL) {
		return NULL;
	}

	if (!gnome_vfs_mime_inited)
		gnome_vfs_mime_init ();

	maybe_reload ();
	
	context = g_hash_table_lookup (specific_types, mime_type);
	if (context){
		value = g_hash_table_lookup (context->keys, key);

		if (value)
			return value;
	}

	generic_type = g_strdup (mime_type);
	p = strchr (generic_type, '/');
	if (p)
		*(p+1) = 0;
	
	context = g_hash_table_lookup (generic_types, generic_type);
	g_free (generic_type);
	
	if (context){
		value = g_hash_table_lookup (context->keys, key);
		if (value)
			return value;
	}

	return NULL;
}

static void
assemble_list (gpointer key, gpointer value, gpointer user_data)
{
	GList **listp = user_data;

	(*listp) = g_list_prepend ((*listp), key);
}

/**
 * gnome_vfs_mime_get_keys:
 * @mime_type: the mime type to lookup.
 *
 * Returns a GList that contains private strings with all of the keys
 * associated with the @mime_type.  
 */
GList *
gnome_vfs_mime_get_keys (const char *mime_type)
{
	char *p, *generic_type;
	GnomeMimeContext *context;
	GList *list = NULL, *l;
	
	if (mime_type == NULL) {
		return NULL;
	}

	if (!gnome_vfs_mime_inited)
		gnome_vfs_mime_init ();

	maybe_reload ();
	
	generic_type = g_strdup (mime_type);
	p = strchr (generic_type, '/');
	if (p)
		*(p+1) = 0;
	
	context = g_hash_table_lookup (generic_types, generic_type);
	g_free (generic_type);
	if (context){
		g_hash_table_foreach (
			context->keys, assemble_list, &list);
	}

	context = g_hash_table_lookup (specific_types, mime_type);
	if (context){
		g_hash_table_foreach (
			context->keys, assemble_list, &list);
	}

	for (l = list; l;){
		if (l->next){
			void *this = l->data;
			GList *m;

			for (m = l->next; m; m = m->next){
				if (strcmp ((char*) this, (char*) m->data) != 0)
					continue;
				list = g_list_remove (list, m->data);
				break;
			}
		}
		l = l->next;
	}
	return list;
}

/**
 * gnome_vfs_mime_program:
 * @mime_type: the mime_type 
 *
 * Returns the program intended to be loaded for this given mime-type
 */
const char *
gnome_vfs_mime_program (const char *mime_type)
{
	return gnome_vfs_mime_get_value (mime_type, "open");
}

/**
 * gnome_vfs_mime_program_name:
 * @prog_name: the program associated with a given mime-type
 *
 * Returns the program name for the program string returned by gnome_vfs_mime_program.
 * This memory must be freed when done.
 */
gchar *
gnome_vfs_mime_program_name (const char *prog_name)
{
    guint i = 0;
	
	while (prog_name[i] != (char) NULL && prog_name[i] != (char) 32) i++;
	return g_strndup (prog_name, i);
}

/**
 * gnome_vfs_mime_description:
 * @mime_type: the mime type
 *
 * Returns the description for this mime-type
 */
const char *
gnome_vfs_mime_description (const char *mime_type)
{
	return gnome_vfs_mime_get_value (mime_type, "description");
}

/**
 * gnome_vfs_mime_extensions:
 * @mime_type: the mime type
 *
 * Returns the description for this mime-type
 */
GList *
gnome_vfs_mime_get_extensions (const char *mime_type)
{
	GList *list;
	const char *extensions;
	gchar **elements;
	int index;
	GnomeMimeContext *context;

	list = NULL;
	
	if (mime_type == NULL) {
		return NULL;
	}

	if (!gnome_vfs_mime_inited) {
		gnome_vfs_mime_init ();
	}

	maybe_reload ();
	
	context = g_hash_table_lookup (registered_types, mime_type);
	if (context) {
		extensions = g_hash_table_lookup (context->keys, "ext");		
	}

	if (extensions != NULL) {
		/* Parse the extensions and add to list */
		elements = g_strsplit (extensions, " ", 0);	
		if (elements != NULL) {
			index = 0;
			
			while (elements[index] != NULL) {
				if (strcmp (elements[index], "") != 0) {
					list = g_list_append (list, g_strdup (elements[index]));
				}
				index++;
			}			
			g_strfreev (elements);
		}		
	}		
	return list;
}

/**
 * gnome_vfs_mime_extension_list_free:
 * @list: the extensions list
 *
 * Call this function on the list returned by gnome_vfs_mime_extensions
 * to free the list and all of its elements.
 */
void
gnome_vfs_mime_extension_list_free (GList *list)
{
	g_list_foreach (list, (GFunc) g_free, NULL);
	g_list_free (list);
}

/**
 * gnome_vfs_mime_test:
 * @mime_type: the mime type
 *
 * Returns the command to be executed on the file before considering
 * the file to match this mime_type.
 */
const char *
gnome_vfs_mime_test (const char *mime_type)
{
	return gnome_vfs_mime_get_value (mime_type, "test");
}

/**
 * gnome_vfs_mime_composetyped:
 * @mime_type: the mime type
 *
 * Returns the command to be executed to compose a message of
 * the given mime_type
 */
const char *
gnome_vfs_mime_composetyped (const char *mime_type)
{
	return gnome_vfs_mime_get_value (mime_type, "compose");
}

static gboolean
gnome_vfs_mime_flag (const char *mime_type, gchar *key, gchar *flag)
{
	const char *str;
	
	str = gnome_vfs_mime_get_value (mime_type, key);
	if (str){
		if (strstr (str, flag) != NULL)
			return TRUE;
	}
	return FALSE;
}

/**
 * gnome_vfs_mime_copiousoutput:
 * @mime_type: the mime type
 * @key: the key which stores the flags for a command
 *
 * Returns a boolean value, whether the mime_type open
 * command will produce lots of output
 */
gboolean 
gnome_vfs_mime_copiousoutput (const char *mime_type, gchar *key)
{
	return gnome_vfs_mime_flag (mime_type, key, "copiousoutput");
}

/**
 * gnome_vfs_mime_needsterminal
 * @mime_type: the mime type
 * @key: the key which stores the flags for a command
 *
 * Returns a boolean value, whether the mime_type open
 * command will required a terminal.
 */
gboolean
gnome_vfs_mime_needsterminal (const char *mime_type, gchar *key)
{
	return gnome_vfs_mime_flag (mime_type, key, "needsterminal");
}

static gint
mime_list_sort (gconstpointer a, gconstpointer b)
{
	return (strcmp (a, b));
}

/*
 *  get_key_name
 *
 *  Hash table function that adds the name of the mime type
 *  to the supplied GList.
 *  
 */
static void 
get_key_name (gpointer key, gpointer value, gpointer user_data)
{
	GnomeMimeContext *context;
	char *name;
	GList **list = user_data;
	
	/* Get context and exit if NULL */	
	context = (GnomeMimeContext *) value;
	if (context == NULL) {
		return;
	}

	/* Get name from key and exit if key is NULL or string is empty */	
	name = (char *)key;
	if (key == NULL || strlen (name) == 0) {
		return;
	}
		
	(*list) = g_list_insert_sorted ((*list), g_strdup(context->mime_type), mime_list_sort);
}

/*
 * gnome_vfs_get_registered_mime_types
 *
 *  Return the list containing the name of all 
 *  registrered mime types.
 */
GList *
gnome_vfs_get_registered_mime_types (void)
{
	GList *type_list = NULL;
	
	if (!gnome_vfs_mime_inited) {
		gnome_vfs_mime_init ();
	}

	maybe_reload ();

	/* Extract mime type names */
	g_hash_table_foreach (registered_types, get_key_name, &type_list);

	return type_list;
}

/**
 * gnome_vfs_mime_commit_registered_types:
 *
 * This function commits the mime info in the registered type
 * hash table to disk.
 */

void
gnome_vfs_mime_commit_registered_types (void)
{
	write_registered_mime_data ();
}

/**
 * gnome_vfs_mime_registered_mime_type_list_free:
 * @list: the extensions list
 *
 * Call this function on the list returned by gnome_vfs_get_registered_mime_types
 * to free the list and all of its elements.
 */
void
gnome_vfs_mime_registered_mime_type_list_free (GList *list)
{
	g_list_foreach (list, (GFunc) g_free, NULL);
	g_list_free (list);
}

/**
 * gnome_vfs_mime_set_registered_type_key:
 * @mime_type: 	Mime type to set key for
 * @key: 	The key to set
 * @data: 	The data to set for the key
 * 
 * This functions sets the key data for the registered mime
 * type's hash table.
 */
void
gnome_vfs_mime_set_registered_type_key (const char *mime_type, gpointer key, gpointer data)
{
	GnomeMimeContext *context;
	
	context = g_hash_table_lookup (registered_types, mime_type);
	if (context != NULL) {
		g_hash_table_insert (context->keys, key, data);
	}
}

static void
write_mime_data_foreach (gpointer key, gpointer value, gpointer data)
{
	FILE *file;
	char *key_string;
	char *key_data;

	key_string = key;
	key_data = value;
	file = data;

	fwrite ("\n\t", 1, 2, file);
	fwrite (key_string, 1, strlen (key_string), file);
	fwrite (": ", 1, 2, file);
	fwrite (key_data, 1, strlen (key_data), file);
	fwrite ("\n", 1, 1, file);
}

static void
write_registered_mime_data (void)
{
	DIR *dir;
	FILE *file;
	char *filename;
	GList *list, *element;
	GnomeMimeContext *context;
	
	if (stat (gnome_mime_dir.dirname, &gnome_mime_dir.s) != -1)
		gnome_mime_dir.valid = TRUE;
	else
		gnome_mime_dir.valid = FALSE;
	
	dir = opendir (gnome_mime_dir.dirname);
	if (!dir){
		gnome_mime_dir.valid = FALSE;
		return;
	}
	
	if (gnome_mime_dir.system_dir){
		list = gnome_vfs_get_registered_mime_types ();
		if (list == NULL) {
			return;
		}
		
		filename = g_concat_dir_and_file (gnome_mime_dir.dirname, "gnome-vfs.mime");

        	remove (filename);
		file = fopen (filename, "w");
		if (file == NULL) {
			return;
		}

		/* Get data in list */
		for (element = list; element != NULL; element = element->next) {
			context = g_hash_table_lookup (registered_types, element->data);
			if (context != NULL) {
				fwrite (context->mime_type, 1, strlen (context->mime_type), file);				
				g_hash_table_foreach (context->keys, write_mime_data_foreach, file);
				fwrite ("\n", 1, 1, file);
			}		
		}

		/* Cleanup file */
		fclose (file);
		g_free (filename);

		gnome_vfs_mime_registered_mime_type_list_free (list);
	}
}


