/* rio500-method.h - VFS modules for Rio500
 *
 *  Copyright (C) 2001,2002 Bastien Nocera
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Author: Bastien Nocera <hadess@hadess.net>
 * Inspired by Alex Larsson's neato SMB vfs method
 */

#include <string.h>
#include <glib.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

#include <libgnomevfs/gnome-vfs-method.h>
#include <libgnomevfs/gnome-vfs-module.h>
#include <libgnomevfs/gnome-vfs-module-shared.h>
#include <libgnomevfs/gnome-vfs-module-callback-module-api.h>
#include <libgnomevfs/gnome-vfs-standard-callbacks.h>

#include <rio500/librio500_api.h>

typedef struct _RioTreeContent RioTreeContent;

struct _RioTreeContent {
	Rio500 *rio_hw;
	gboolean have_flash;
	GList *int_content, *ext_content, *virt_content;
	gulong int_mem_total, ext_mem_total;
	gulong int_mem_free, ext_mem_free;
	gboolean dirty;
};

/* Global vars: */
static GMutex *rio_lock = NULL;
static RioTreeContent *rio = NULL;

#define DEBUG_RIO_ENABLE
#define DEBUG_RIO_LOCKS

#ifdef DEBUG_RIO_ENABLE
#define DEBUG_RIO(x) g_print x
#else
#define DEBUG_RIO(x) 
#endif

#ifdef DEBUG_RIO_LOCKS
#define LOCK_RIO() 	{g_mutex_lock (rio_lock); g_print ("LOCK %s\n", G_GNUC_PRETTY_FUNCTION);}
#define UNLOCK_RIO() 	{g_print ("UNLOCK %s\n", G_GNUC_PRETTY_FUNCTION); g_mutex_unlock (rio_lock);}
#else
#define LOCK_RIO() 	g_mutex_lock (rio_lock)
#define UNLOCK_RIO() 	g_mutex_unlock (rio_lock)
#endif

static void update_tree_content ();
static void destroy_rio ();
static const char *gnome_vfs_uri_get_basename (const GnomeVFSURI *uri);

const char *
gnome_vfs_uri_get_basename (const GnomeVFSURI *uri)
{
	char *p;

	g_return_val_if_fail (uri != NULL, NULL);

	if (uri->text == NULL) {
		return NULL;
	}

	p = strrchr (uri->text, GNOME_VFS_URI_PATH_CHR);
	if (p == NULL) {
		return NULL;
	}

	p++;
	if (*p == '\0') {
		return NULL;
	}

	return p;
}

static GList *
rio_get_content_from_card (int card)
{
	if (card == 0)
		return rio->int_content;

	if ((card == 1) && (rio->have_flash == TRUE))
		return rio->ext_content;

	return NULL;
}

static GnomeVFSResult
lookup_uri(GnomeVFSURI *uri, gint *card, gint *folder, gint *song)
{
	gchar *cardname;
	const gchar *path;
	GList *content, *list;
	gchar **str;
	RioFolderEntry *folder_entry = NULL;
	RioSongEntry *song_entry;

	*card = *folder = *song = -1;

	DEBUG_RIO(("lookup_uri: %s\n", gnome_vfs_uri_to_string(uri, 0)));

	update_tree_content();

	LOCK_RIO();
	if (rio == NULL)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}

	cardname = (gchar *) gnome_vfs_uri_get_host_name (uri);
	if (cardname != NULL)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_INVALID_URI;
	}

	/* Is this root? */
	if (gnome_vfs_uri_has_parent (uri) == 0)
	{
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	{
		const gchar *tmp;

		tmp = gnome_vfs_uri_get_path(uri);
		path = gnome_vfs_unescape_string(tmp, "/");
		if (path == NULL)
			path = tmp;

		DEBUG_RIO(("lookup_uri: path %s\n", path));
	}

	str = g_strsplit(path, "/", 0);
	/* this is pretty safe i think */
	cardname = g_strdup(str[1]);

	DEBUG_RIO(("lookup_uri: cardname %s\n", cardname));

	/* now we are in a flash card... maybe */
	if (g_strcasecmp(cardname, "Internal") == 0)
	{
		/* internal card */
		DEBUG_RIO(("lookup_uri: internal card detected\n"));
		*card = 0;
		content = rio->int_content;
	} else {
		if (!g_strcasecmp(cardname, "External") &&
				rio->have_flash == TRUE)
		{
			DEBUG_RIO(("lookup_uri: external card detected\n"));
			/* external card */
			*card = 1;
			content = rio->ext_content;
		} else {
			DEBUG_RIO(("lookup_uri: nonexistant card detected\n"));
			/* none existant directory */
			g_free(cardname);
			g_strfreev(str);
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_NOT_FOUND;
		}
	}
	g_free(cardname);

	if (str[2] == NULL)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	/* look for the folder name */
	DEBUG_RIO(("lookup_uri: looking up folder '%s'\n", str[2]));
	for (list = content; list != NULL; list = list->next)
	{
		folder_entry = (RioFolderEntry *) list->data;

		if (folder_entry
				&& (strcmp(str[2], folder_entry->name) == 0))
		{
			DEBUG_RIO(("lookup_uri: found folder %d\n", folder_entry->folder_num));
			/* match the folder */
			*folder = folder_entry->folder_num;
			break;
		}
	}

	/* if the folder didn't match anything on the rio, go "404" */
	if (*folder == -1)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_FOUND;
	}

	/* is it a directory ? */
	if (str[3] == NULL)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	/* look for the song */
	for (list = folder_entry->songs; list != NULL; list = list->next)
	{
		song_entry = (RioSongEntry *) list->data;
		if (song_entry
				&& (strcmp(str[3], song_entry->name) == 0))
		{
			/* match the song */
			*song = song_entry->song_num;
			break;
		}
	}
	/* if the song didn't match anything on the rio, go "404" */
	if (*song == -1)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_FOUND;
	}

	/* oh my ! too much information there */
	if (str[4] != NULL)
	{
		g_strfreev(str);
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_INVALID_URI;
	}

	g_strfreev(str);
	UNLOCK_RIO();
	return GNOME_VFS_OK;
}

static void
update_tree_content()
{
	RioFolderEntry *folder_entry = NULL;
	gint card_num, major, minor;

	DEBUG_RIO(("update_tree_content start\n"));

	if (rio != NULL)
	{
		DEBUG_RIO(("update_tree_content: RioTreeContent is not NULL\n"));
		if (rio->dirty == TRUE)
		{
			DEBUG_RIO(("update_tree_content: RioTreeContent is dirty\n"));
			destroy_rio(rio);
		} else {
			DEBUG_RIO(("update_tree_content: RioTreeContent is not dirty\n"));
			return;
		}
	}

	DEBUG_RIO(("update_tree_content: creating new RioTreeContent\n"));

	LOCK_RIO();

	rio = g_new(RioTreeContent, 1);
	rio->rio_hw = NULL;
	rio->int_content = NULL;
	rio->ext_content = NULL;
	rio->virt_content = NULL;
	rio->int_mem_total = rio->ext_mem_total = rio->int_mem_free =
		rio->ext_mem_free = 0;
	rio->dirty = FALSE;

	rio->rio_hw = rio_new();
	if (rio_check (rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("update_tree_content: check of rio is FALSE, destroying\n"));
		destroy_rio (rio);
		rio = NULL;
		UNLOCK_RIO();
		return;
	}

	rio_get_revision (rio->rio_hw, &major, &minor);

	DEBUG_RIO(("update_tree_content: version %d.%d\n", major, minor));
	
	card_num = rio_get_card_number (rio->rio_hw);
	if (card_num >= 2)
		rio->have_flash = TRUE;

	rio_set_card (rio->rio_hw, 0);
	rio->int_mem_free = rio_memory_left (rio->rio_hw);
	rio->int_mem_total = rio_get_mem_total (rio->rio_hw);
	rio->int_content = rio_get_content (rio->rio_hw);

	DEBUG_RIO(("update_tree_content: got internal card info\n"));

	folder_entry = g_new(RioFolderEntry, 1);
	folder_entry->name = g_strdup("Internal");
	folder_entry->folder_num = 0;
	folder_entry->songs = NULL;
	rio->virt_content = g_list_append(rio->virt_content,
			(gpointer) folder_entry);

	DEBUG_RIO(("update_tree_content: created dummy folder for internal\n"));

	if (rio->have_flash == TRUE)
	{
		rio_set_card(rio->rio_hw, 1);
		rio->ext_mem_free = rio_memory_left(rio->rio_hw);
		rio->ext_mem_total = rio_get_mem_total(rio->rio_hw);

		DEBUG_RIO(("update_tree_content: getting external card info\n"));
		
		rio->ext_content = rio_get_content(rio->rio_hw);

		DEBUG_RIO(("update_tree_content: got external card info\n"));

		folder_entry = g_new(RioFolderEntry, 1);
		folder_entry->name = g_strdup("External");
		folder_entry->folder_num = 1;
		folder_entry->songs = NULL;
		rio->virt_content = g_list_append(rio->virt_content,
				(gpointer) folder_entry);

		DEBUG_RIO(("update_tree_content: created dummy folder for external\n"));
	}
	rio_delete(rio->rio_hw);
	UNLOCK_RIO();

	return;
}

static void
destroy_rio()
{
	g_return_if_fail(rio != NULL);

	DEBUG_RIO(("destroy_rio: start\n"));
	if (rio->int_content != NULL)
		rio_destroy_content(rio->int_content);
	DEBUG_RIO(("destroy_rio: destroyed int_content\n"));
	if (rio->have_flash == TRUE
			&& rio->ext_content != NULL)
		rio_destroy_content(rio->ext_content);
	DEBUG_RIO(("destroy_rio: destroyed ext_content\n"));
	if (rio->virt_content != NULL)
		rio_destroy_content(rio->virt_content);
	DEBUG_RIO(("destroy_rio: destroyed virt_content\n"));

	g_free(rio);
	rio = NULL;

	DEBUG_RIO(("destroy_rio: end\n"));
}

static GnomeVFSResult
rio_to_vfs_error(gint error)
{
	switch (error)
	{
	case RIO_FILEERR:
		return GNOME_VFS_ERROR_IO;
	case RIO_FORMAT:
		return GNOME_VFS_ERROR_GENERIC;
	case RIO_ENDCOMM:
		return GNOME_VFS_ERROR_INTERNAL;
	case RIO_INITCOMM:
		return GNOME_VFS_ERROR_INTERNAL;
	case RIO_NODIR:
		return GNOME_VFS_ERROR_NOT_FOUND;
	case RIO_NOMEM:
		return GNOME_VFS_ERROR_NO_SPACE;
	case PC_MEMERR:
		return GNOME_VFS_ERROR_NO_MEMORY;
	case RIO_SUCCESS:
		return GNOME_VFS_OK;
	}

	g_warning("Rio generated error %d not found\n", error);
	return GNOME_VFS_ERROR_GENERIC;
}

typedef struct {
	/* is it a read or a write to the rio500 */
	gboolean is_read;
	/* handle for the temp file */
	GnomeVFSHandle *handle;
	/* For writing */
	gchar *filename;
	gint folder;
	gint space_left;
} FileHandle;

static GnomeVFSResult
do_open (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle **method_handle,
	 GnomeVFSURI *uri,
	 GnomeVFSOpenMode mode,
	 GnomeVFSContext *context)
{
	FileHandle *handle = NULL;
	GnomeVFSResult res;
	gint card, folder, song;
	gboolean is_read = FALSE;

	DEBUG_RIO(("do_open() %s mode %d\n",
				gnome_vfs_uri_to_string (uri, 0), mode));

	if (g_file_test ("/tmp/rio500_tmp_file.mp3", G_FILE_TEST_EXISTS)
			== TRUE)
		return GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES;

	if (mode & GNOME_VFS_OPEN_READ)
		is_read = TRUE;
	else if (mode & GNOME_VFS_OPEN_WRITE)
		is_read = FALSE;
	else
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_open(): lookup %d %d %d\n",
				card, folder, song));

	LOCK_RIO();

	if ((res != GNOME_VFS_OK) && (is_read == TRUE))
	{
		UNLOCK_RIO();
		DEBUG_RIO(("do_open(): file not found on read open, result\n"));
		return res;
	}

	if ((folder != -1) && (song == -1) && (is_read == TRUE))
	{
		UNLOCK_RIO();
		DEBUG_RIO(("do_open(): file is a directory on a read open\n"));
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	if ((song == -1) && (is_read == TRUE))
	{
		UNLOCK_RIO();
		DEBUG_RIO(("do_open(): file not found on a read open\n"));
		return GNOME_VFS_ERROR_NOT_FOUND;
	}

	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_open: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, card);

	if (is_read == TRUE)
	{
		GnomeVFSHandle *local_handle;
		gint res;


		DEBUG_RIO(("do_open: opening read-only\n"));

		rio->rio_hw = rio_new();
		res = rio_get_song(rio->rio_hw, "/tmp/rio500_tmp_file.mp3",
				folder, song);
		rio_delete (rio->rio_hw);

		if (res != RIO_SUCCESS)
		{
			unlink ("/tmp/rio500_tmp_file.mp3");
			UNLOCK_RIO();
			DEBUG_RIO(("do_open(): error getting song from the rio\n"));
			return GNOME_VFS_ERROR_IO;
		}

		DEBUG_RIO(("do_open: copied to /tmp/rio500_tmp_file.mp3\n"));

		if (gnome_vfs_open (&local_handle, "/tmp/rio500_tmp_file.mp3",
				GNOME_VFS_OPEN_READ) != GNOME_VFS_OK)
		{
			unlink ("/tmp/rio500_tmp_file.mp3");
			UNLOCK_RIO();
			DEBUG_RIO(("do_open(): error opening the tmp file\n"));
			return GNOME_VFS_ERROR_IO;
		}

		DEBUG_RIO(("do_open: opened local file\n"));

		/* Create the handle */
		handle = g_new(FileHandle, 1);
		/* global */
		handle->is_read = is_read;
		handle->handle = local_handle;
		handle->filename = NULL;
	} else {
		GnomeVFSHandle *local_handle;
		gchar *basename, *local;

		basename = g_strdup (gnome_vfs_uri_get_basename (uri));
		local = g_strdup_printf ("file://%s/%s", g_get_tmp_dir(), basename);
		if (gnome_vfs_create (&local_handle, local,
		    GNOME_VFS_OPEN_WRITE, FALSE, 0600) != GNOME_VFS_OK)
		{
			UNLOCK_RIO();
			DEBUG_RIO(("do_open(): error opening the temp file for writing %s\n", local));
			g_free (local);
			return GNOME_VFS_ERROR_IO;
		}

		/* Create the handle */
		handle = g_new(FileHandle, 1);
		handle->is_read = is_read;
		handle->handle = local_handle;
		handle->folder = folder;
		handle->filename = gnome_vfs_get_local_path_from_uri (local);
		handle->space_left = (card == 0 ? rio->int_mem_free :
				rio->ext_mem_free);

		g_free (local);
	}

	*method_handle = (GnomeVFSMethodHandle *)handle;
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  GnomeVFSContext *context)

{
	FileHandle *handle = (FileHandle *)method_handle;
	GnomeVFSResult res = GNOME_VFS_OK;

	DEBUG_RIO(("do_close(): is_read: %d\n", handle->is_read));

	if (handle->is_read == TRUE)
	{
		gint tmp;
		
		DEBUG_RIO(("do_close(): unlinking temp file\n"));
		tmp = unlink ("/tmp/rio500_tmp_file.mp3");
		if (tmp != 0)
			res = GNOME_VFS_ERROR_IO;
		else
			res = GNOME_VFS_OK;
	} else if (handle->space_left > 0) {
		gint tmp;

		DEBUG_RIO(("do_close(): uploading song %s to folder %d\n",
					handle->filename, handle->folder));

		rio->rio_hw = rio_new();
		tmp = rio_add_song (rio->rio_hw, handle->folder,
				handle->filename);
		rio_delete(rio->rio_hw);

		rio->dirty = TRUE;

		DEBUG_RIO(("do_close(): uploading result: %d\n", res));

		unlink (handle->filename);
		res = rio_to_vfs_error(tmp);
	} else {
		DEBUG_RIO(("do_close(): not uploading, no memory\n"));
		unlink (handle->filename);
		res = GNOME_VFS_ERROR_NO_SPACE;
	}

	/* delete the handle's content */
	gnome_vfs_close (handle->handle);
	g_free (handle->filename);
	g_free (handle);

	UNLOCK_RIO();

	return res;
}

static GnomeVFSResult
do_read (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 gpointer buffer,
	 GnomeVFSFileSize num_bytes,
	 GnomeVFSFileSize *bytes_read,
	 GnomeVFSContext *context)
{
	FileHandle *handle = (FileHandle *)method_handle;

	DEBUG_RIO(("do_read() %Lu bytes\n", num_bytes));

	return gnome_vfs_read (handle->handle, buffer, num_bytes, bytes_read);
}

static GnomeVFSResult
do_write (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  gconstpointer buffer,
	  GnomeVFSFileSize num_bytes,
	  GnomeVFSFileSize *bytes_written,
	  GnomeVFSContext *context)


{
	GnomeVFSResult result;
	FileHandle *handle = (FileHandle *)method_handle;

	DEBUG_RIO (("do_write() %p\n", method_handle));

	/* Check if we have enough space left for that much on the rio */
	DEBUG_RIO (("do_write() space left: %d / bytes: %d / diff: %d\n",
				handle->space_left, num_bytes,
				handle->space_left - num_bytes));

	if ((handle->space_left - num_bytes) <= 0)
	{
		/* Set it so that it's negative */
		handle->space_left -= num_bytes;
		return GNOME_VFS_ERROR_NO_SPACE;
	}

	result = gnome_vfs_write (handle->handle, buffer, num_bytes,
			bytes_written);

	handle->space_left -= *bytes_written;
	return result;
}

static GnomeVFSResult
do_create (GnomeVFSMethod *method,
	   GnomeVFSMethodHandle **method_handle,
	   GnomeVFSURI *uri,
	   GnomeVFSOpenMode mode,
	   gboolean exclusive,
	   guint perm,
	   GnomeVFSContext *context)

{
	DEBUG_RIO (("do_create() %s mode %d\n",
				gnome_vfs_uri_to_string (uri, 0), mode));


	return do_open (method, method_handle, uri, GNOME_VFS_OPEN_WRITE,
			context);
}

static GnomeVFSResult
do_get_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  GnomeVFSFileInfo *file_info,
		  GnomeVFSFileInfoOptions options,
		  GnomeVFSContext *context)

{
	GnomeVFSResult res;
	gint card, folder, song;

	DEBUG_RIO (("do_get_file_info() %s\n",
				gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_get_file_info(): lookup %d %d %d\n",
				card, folder, song));

	LOCK_RIO();
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}

	file_info->valid_fields =
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

	/* Is this root? */
	if (gnome_vfs_uri_has_parent (uri) == 0) {
		file_info->name = g_strdup ("/");
		file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->mime_type = g_strdup ("x-directory/normal");
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	/* is it a folder */
	if (song == -1)
	{
		/* is it not inside a card ? */
		if (folder == -1)
		{
			file_info->name = g_strdup ((card == 0) ? "Internal" :
					"External");
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup ("x-directory/normal");
		} else {
			file_info->name = g_strdup
				(gnome_vfs_uri_extract_short_name (uri));
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup ("x-directory/normal");
		}
	} else {
		file_info->name = g_strdup
			(gnome_vfs_uri_extract_short_name (uri));
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->mime_type = g_strdup ("audio/x-mp3");
	}

	UNLOCK_RIO();

	return GNOME_VFS_OK;
}

static gboolean
do_is_local (GnomeVFSMethod *method,
	     const GnomeVFSURI *uri)
{
	DEBUG_RIO (("do_is_local()\n"));

	return FALSE;
}

typedef struct {
	GList *list;
	gchar *dirname;
	guint pos;
	gboolean list_type_is_folder;
} DirectoryHandle;

static GnomeVFSResult
do_open_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle **method_handle,
		   GnomeVFSURI *uri,
		   GnomeVFSFileInfoOptions options,
		   GnomeVFSContext *context)

{
	GnomeVFSResult res;
	DirectoryHandle *directory_handle;
	GList *list = NULL;
	gint card, folder, song;
	const gchar *path; //FIXME
	gboolean list_type_is_folder = TRUE;
	
	DEBUG_RIO(("do_open_directory() %s\n",
		gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_open_directory(): lookup %d %d %d\n",
				card, folder, song));
	LOCK_RIO();
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}
	if (song != -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	}

	if (folder == -1)
	{
		/* can be rio500: or rio500://internal/ */
		if (card == -1)
		{
			list = rio->virt_content;
			path = "/";
		} else {
			if (card == 1 && rio->have_flash == FALSE)
			{
				UNLOCK_RIO();
				return GNOME_VFS_ERROR_NOT_FOUND;
			}
			if (card == 0)
			{
				list = rio->int_content;
				path = "Internal";
			} else {
				list = rio->ext_content;
				path = "External";
			}
		}
	} else {
		GList *content = NULL;
		RioFolderEntry *entry = NULL;

		list_type_is_folder = FALSE;

		content = rio_get_content_from_card (card);

		entry = (RioFolderEntry *) g_list_nth_data(content, folder);
		path = entry->name;
		list = entry->songs;
	}

	/* Construct the handle */
	directory_handle = g_new0 (DirectoryHandle, 1);
	directory_handle->dirname = g_strdup (path);
	directory_handle->pos = 0;
	directory_handle->list = list;
	directory_handle->list_type_is_folder = list_type_is_folder;
	*method_handle = (GnomeVFSMethodHandle *) directory_handle;

	UNLOCK_RIO();

	DEBUG_RIO(("do_open_directory end\n"));
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close_directory (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSContext *context)
{
	DirectoryHandle *directory_handle = (DirectoryHandle *) method_handle;

	DEBUG_RIO(("do_close_directory: %p\n", directory_handle));

	if (directory_handle == NULL)
		return GNOME_VFS_OK;

	//FIXME
	g_free (directory_handle->dirname);
	g_free (directory_handle);
	DEBUG_RIO(("do_close_directory: end\n"));

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_read_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle *method_handle,
		   GnomeVFSFileInfo *file_info,
		   GnomeVFSContext *context)
{
	DirectoryHandle *dh = (DirectoryHandle *) method_handle;
	GList *list;
	RioFolderEntry *folder_entry;
	RioSongEntry *song_entry;

	DEBUG_RIO (("do_read_directory()\n"));

	LOCK_RIO();
	if (dh->pos > (g_list_length(dh->list)+2) )
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_EOF;
	}

	/* . or .. directories */
	if (dh->pos == 0 || dh->pos == 1)
	{
		file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_NONE |
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->name = g_strdup ((dh->pos == 0) ? "." : "..");
		file_info->mime_type = g_strdup ("x-directory/normal");
		DEBUG_RIO (("do_read_directory(): read folder %s\n", file_info->name));
		dh->pos++;
		UNLOCK_RIO();
		return GNOME_VFS_OK;
	}

	list = g_list_nth(dh->list, dh->pos - 2);
	if (list == NULL)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_EOF;
	}

	GNOME_VFS_FILE_INFO_SET_LOCAL (file_info, FALSE);

	/* Tell gnome-vfs which fields will be valid */
	file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_NONE |
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

	/* Fill the fields */
	if (dh->list_type_is_folder == TRUE)
	{
		folder_entry = (RioFolderEntry *)(list->data);
		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->name = g_strdup (folder_entry->name);
		file_info->mime_type = g_strdup ("x-directory/normal");
		DEBUG_RIO (("do_read_directory(): read folder %s\n", file_info->name));
	} else {
		file_info->valid_fields |= GNOME_VFS_FILE_INFO_FIELDS_SIZE;
		song_entry = (RioSongEntry *)(list->data);
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->name = g_strdup(song_entry->name);
		file_info->size = song_entry->size;
		file_info->mime_type = g_strdup("audio/x-mp3");
		DEBUG_RIO (("do_read_directory(): read file %s\n", file_info->name));
	}

	dh->pos++;

	DEBUG_RIO (("do_read_directory(): position %d\n", dh->pos));
	UNLOCK_RIO();
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_unlink (GnomeVFSMethod *method,
	   GnomeVFSURI *uri,
	   GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	gint card, folder, song;

	DEBUG_RIO (("do_unlink() %s\n",
				gnome_vfs_uri_to_string (uri, 0)));

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_unlink(): lookup %d %d %d\n",
				card, folder, song));

	LOCK_RIO();
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}

	if (folder == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	}

	if (song == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_unlink: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, card);
	res = rio_to_vfs_error(rio_del_song(rio->rio_hw,
				folder, song));
	rio_delete(rio->rio_hw);

	if (res == GNOME_VFS_OK)
		rio->dirty = TRUE;

	UNLOCK_RIO();

	return res;
}

static GnomeVFSResult
do_check_same_fs (GnomeVFSMethod *method,
		  GnomeVFSURI *a,
		  GnomeVFSURI *b,
		  gboolean *same_fs_return,
		  GnomeVFSContext *context)
{
	DEBUG_RIO (("do_check_same_fs()\n"));
	
	*same_fs_return = FALSE;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_move (GnomeVFSMethod *method,
	 GnomeVFSURI *old_uri,
	 GnomeVFSURI *new_uri,
	 gboolean force_replace,
	 GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	gint old_card, old_folder, old_song;
	gint new_card, new_folder, new_song;
	gboolean move_folder = FALSE;
	const gchar *new_name;

	g_warning("This function of the rio500 method is not properly supported\nPlease mail hadess@hadess.net with information about your program\n");

	DEBUG_RIO (("do_move() %s %s\n",
		    gnome_vfs_uri_to_string (old_uri, 0),
		    gnome_vfs_uri_to_string (new_uri, 0)));
	LOCK_RIO();

	res = lookup_uri(old_uri, &old_card, &old_folder, &old_song);
	DEBUG_RIO(("do_move(): lookup (src) %d %d %d\n",
				old_card, old_folder, old_song));
	/* if the original file isn't found, bail */
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}

	res = lookup_uri(new_uri, &new_card, &new_folder, &new_song);
	DEBUG_RIO(("do_move(): lookup (dest) %d %d %d\n",
				new_card, new_folder, new_song));

	/* if the dest file is found and force_replace is false, bail */
	if (res == GNOME_VFS_OK && force_replace == FALSE)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_FILE_EXISTS;
	} else {
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	/* if the dest uri is invalid, bail */
	if (res == GNOME_VFS_ERROR_INVALID_URI)
	{
		UNLOCK_RIO();
		return res;
	}

	/* if the move crosses disks, or tries to the rename the cards, bail */
	if (new_card != old_card || new_card == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	/* Moving songs */
	if (old_song != -1)
	{
		/* are they in the same folder ? */
		if (old_folder != new_folder)
		{
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_NOT_SUPPORTED;
		}
		move_folder = FALSE;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_move: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, old_card);

	/* Getting the name of the file/folder */
	{
		const gchar *tmp;

		tmp = gnome_vfs_uri_get_basename(new_uri);
		if (tmp == NULL)
			new_name = tmp;
		else
			new_name = gnome_vfs_unescape_string(tmp, "/");
	}

	if (move_folder == TRUE)
	{
		res = rio_to_vfs_error(rio_rename_folder(rio->rio_hw,
					old_folder, (gchar *)new_name));
	} else {
		res = rio_to_vfs_error(rio_rename_song(rio->rio_hw,
					old_folder, old_song,
					(gchar *)new_name));
	}
	rio_delete(rio->rio_hw);

	if (res == GNOME_VFS_OK)
		rio->dirty = TRUE;

	UNLOCK_RIO();

	return res;
}


static GnomeVFSResult
do_truncate_handle (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSFileSize where,
		    GnomeVFSContext *context)

{
	DEBUG_RIO(("do_truncate_handle\n"));
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_make_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *uri,
		   guint perm,
		   GnomeVFSContext *context)
{
	GnomeVFSResult res;
	gint card, folder, song;
	const gchar *folder_name = NULL;

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_make_directory(): lookup %d %d %d\n",
				card, folder, song));
	LOCK_RIO();
	if (res == GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_FILE_EXISTS;
	}
	if (res == GNOME_VFS_ERROR_IO)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}

	if (res == GNOME_VFS_ERROR_INVALID_URI
			|| card == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	}

	if (folder != -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_make_directory: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, card);
	{
		const gchar *tmp;

		tmp = gnome_vfs_uri_get_basename(uri);
		if (tmp == NULL)
			folder_name = tmp;
		else
			folder_name = gnome_vfs_unescape_string(tmp, "/");
	}
	res = rio_to_vfs_error(rio_add_folder(rio->rio_hw,
			(gchar *)folder_name));
	rio_delete(rio->rio_hw);

	if (res == GNOME_VFS_OK)
		rio->dirty = TRUE;

	UNLOCK_RIO();

	return res;
}

static GnomeVFSResult
do_remove_directory (GnomeVFSMethod *method,
		     GnomeVFSURI *uri,
		     GnomeVFSContext *context)
{
	GnomeVFSResult res;
	gint card, folder, song;
	GList *list;

	res = lookup_uri(uri, &card, &folder, &song);
	DEBUG_RIO(("do_remove_directory(): lookup %d %d %d\n",
				card, folder, song));

	LOCK_RIO();
	if (res != GNOME_VFS_OK)
	{
		UNLOCK_RIO();
		return res;
	}

	if (folder == -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	if (song != -1)
	{
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	}

	rio->rio_hw = rio_new();
	if (rio_check(rio->rio_hw) == FALSE)
	{
		DEBUG_RIO(("do_remove_directory: check of rio is FALSE, destroying\n"));
		UNLOCK_RIO();
		return GNOME_VFS_ERROR_IO;
	}
	rio_set_card(rio->rio_hw, card);
	list = rio_get_content_from_card (card);
	/* Format the card when deleting the last directory */
	if (g_list_length (list) == 1)
		res  = rio_to_vfs_error(rio_format(rio->rio_hw));
	else
		res = rio_to_vfs_error(rio_del_folder(rio->rio_hw, folder));
	rio_delete(rio->rio_hw);

	if (res == GNOME_VFS_OK)
		rio->dirty = TRUE;

	UNLOCK_RIO();

	return res;
}

static GnomeVFSResult
do_set_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  const GnomeVFSFileInfo *info,
		  GnomeVFSSetFileInfoMask mask,
		  GnomeVFSContext *context)
{
	GnomeVFSResult res = GNOME_VFS_OK;
	gint card, folder, song;
	gboolean rename_folder = FALSE;
	const gchar *new_name;

	DEBUG_RIO (("do_set_file_info: mask %d\n", mask));

	if (mask &
		(GNOME_VFS_SET_FILE_INFO_PERMISSIONS |
		GNOME_VFS_SET_FILE_INFO_OWNER |
		GNOME_VFS_SET_FILE_INFO_TIME))
			return GNOME_VFS_ERROR_NOT_SUPPORTED;

	if (mask & GNOME_VFS_SET_FILE_INFO_NAME) {
		new_name = gnome_vfs_unescape_string(info->name, "/");
		DEBUG_RIO (("set_info: set new name: %s\n", new_name));

		res = lookup_uri(uri, &card, &folder, &song);
		DEBUG_RIO(("do_set_file_info(): lookup %d %d %d\n",
					card, folder, song));

		LOCK_RIO();
		if (res != GNOME_VFS_OK)
		{
			UNLOCK_RIO();
			return res;
		}

		if (folder == -1)
		{
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_ACCESS_DENIED;
		}

		if (song == -1)
			rename_folder = TRUE;

		{
			GnomeVFSURI *new_uri;
			gchar *tmp, *new_filename;
			gint new_card, new_folder, new_song;

			tmp = gnome_vfs_uri_extract_dirname(uri);
			DEBUG_RIO(("do_set_file_info(): dirname %s\n",
						tmp));
			new_filename = g_strdup_printf("%s%s",
					tmp, new_name);
			DEBUG_RIO(("do_set_file_info(): filename %s\n",
						new_filename));
			new_uri = gnome_vfs_uri_new(new_filename);
			
			UNLOCK_RIO();
			res = lookup_uri(new_uri, &new_card, &new_folder,
					&new_song);
			LOCK_RIO();

			g_free(new_filename);
			g_free(tmp);
			gnome_vfs_uri_unref(new_uri);
			
			if (res != GNOME_VFS_ERROR_NOT_FOUND)
			{
				UNLOCK_RIO();
				return GNOME_VFS_ERROR_FILE_EXISTS;
			}
		}

		rio->rio_hw = rio_new();
		if (rio_check(rio->rio_hw) == FALSE)
		{
			DEBUG_RIO(("do_set_file_info: check of rio is FALSE, destroying\n"));
			UNLOCK_RIO();
			return GNOME_VFS_ERROR_IO;
		}
		rio_set_card(rio->rio_hw, card);

		if (rename_folder == TRUE)
		{
			res = rio_to_vfs_error(rio_rename_folder(rio->rio_hw,
						folder, (gchar *)new_name));
		} else {
			res = rio_to_vfs_error(rio_rename_song(rio->rio_hw,
						folder, song,
						(gchar *)new_name));
		}
		rio_delete(rio->rio_hw);

		if (res == GNOME_VFS_OK)
			rio->dirty = TRUE;

		UNLOCK_RIO();
	}

	return res;
}

static GnomeVFSMethod method = {
	sizeof (GnomeVFSMethod),
	do_open,
	do_create,
	do_close,
	do_read,
	do_write,
	NULL, /* do_seek */
	NULL, /* do_tell */
	do_truncate_handle,
	do_open_directory,
	do_close_directory,
	do_read_directory,
	do_get_file_info,
	NULL, /* do_get_file_info_from_handle */
	do_is_local,
	do_make_directory,
	do_remove_directory,
	do_move,
	do_unlink,
	do_check_same_fs,
	do_set_file_info,
	NULL, /* do_truncate */
	NULL, /* do_find_directory */
	NULL  /* do_create_symbolic_link */
};

GnomeVFSMethod *
vfs_module_init (const char *method_name, const char *args)
{
	rio_lock = g_mutex_new();

	DEBUG_RIO (("<-- rio500 module init called -->\n"));
	
	/* Make sure we don't have a dangling temp file */
	unlink ("/tmp/rio500_tmp_file.mp3");
	
	return &method;
}

void
vfs_module_shutdown (GnomeVFSMethod *method)
{
	DEBUG_RIO (("<-- rio500 module shutdown called -->\n"));

	/* Destroy the temp file */
	unlink ("/tmp/rio500_tmp_file.mp3");
	destroy_rio(rio);
	g_mutex_free(rio_lock);
}

