/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* user-settings.c: this file is part of users-admin, a ximian-setup-tool frontend 
 * for user administration.
 * 
 * Copyright (C) 2000-2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Carlos Garnacho Parro <garparr@teleline.es>.
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gst.h"

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <utmp.h>
#include <ctype.h>

#include "users-table.h"
#include "table.h"
#include "callbacks.h"
#include "user-settings.h"
#include "privileges-table.h"
#include "groups-table.h"
#include "test-battery.h"

extern GstTool *tool;

static gboolean
check_user_delete (OobsUser *user)
{
	GtkWidget *dialog;
	gint response;

	if (oobs_user_get_uid (user) == 0) {
		dialog = gtk_message_dialog_new (GTK_WINDOW (tool->main_dialog),
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("Administrator account can not be deleted"));

		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
							  _("This would leave the system unusable."));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

		return FALSE;
	}

	/* FIXME: could check that user is logged in */

	dialog = gtk_message_dialog_new (GTK_WINDOW (tool->main_dialog),
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_WARNING,
					 GTK_BUTTONS_NONE,
					 _("Are you sure you want to delete account \"%s\"?"),
					 oobs_user_get_login_name (user));
	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("This will disable this user's access to the system "
						    "without deleting the user's home directory."));
	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_DELETE, GTK_RESPONSE_ACCEPT,
				NULL);

	response = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);

	return (response == GTK_RESPONSE_ACCEPT);
}

gboolean
user_delete (GtkTreeModel *model, GtkTreePath *path)
{
	GtkTreeIter iter;
	OobsUsersConfig *config;
	OobsUser *user;
	OobsList *users_list;
	OobsListIter *list_iter;

	if (!gtk_tree_model_get_iter (model, &iter, path))
		return FALSE;

	gtk_tree_model_get (model, &iter,
			    COL_USER_OBJECT, &user,
			    COL_USER_ITER, &list_iter,
			    -1);

	if (check_user_delete (user)) {
		config = OOBS_USERS_CONFIG (GST_USERS_TOOL (tool)->users_config);
		users_list = oobs_users_config_get_users (config);
		oobs_list_remove (users_list, list_iter);

		gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

		return TRUE;
	}

	return FALSE;
}

static void
setup_groups_combo (GtkWidget *widget)
{
	GtkWidget *table = gst_dialog_get_widget (tool->main_dialog, "groups_table");
	GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (table));

	gtk_combo_box_set_model (GTK_COMBO_BOX (widget), model);
}

static void
set_entry_text (GtkWidget *entry, const gchar *text)
{
	gtk_entry_set_text (GTK_ENTRY (entry), (text) ? text : "");
}

static void
set_main_group (OobsUser *user)
{
	GtkWidget *combo;
	GtkTreeModel *model;
	GtkTreeIter iter;
	OobsGroup *main_group, *group;
	gboolean valid;

	main_group = oobs_user_get_main_group (user);

	if (!main_group)
		main_group = oobs_users_config_get_default_group (GST_USERS_TOOL (tool)->users_config);

	combo = gst_dialog_get_widget (tool->main_dialog, "user_settings_group");
	model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
	valid = gtk_tree_model_get_iter_first (model, &iter);

	while (valid) {
		gtk_tree_model_get (model, &iter,
				    COL_GROUP_OBJECT, &group,
				    -1);

		if (main_group == group) {
			gtk_combo_box_set_active_iter (GTK_COMBO_BOX (combo), &iter);
			break;
		}
		
		valid = gtk_tree_model_iter_next (model, &iter);
	}
}

static OobsGroup*
get_main_group (const gchar *name)
{
	GtkWidget *combo;
	GtkTreeModel *model;
	GtkTreeIter iter;
	OobsGroup *group;

	combo = gst_dialog_get_widget (tool->main_dialog, "user_settings_group");
	model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));

	if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo), &iter)) {
		/* create new group for the user */
		OobsGroupsConfig *config;
		OobsList *groups_list;
		OobsListIter list_iter;

		group = oobs_group_new (name);
		oobs_group_set_gid (group, group_settings_find_new_gid ());

		/* FIXME: this should be in a generic function */
		config = OOBS_GROUPS_CONFIG (GST_USERS_TOOL (tool)->groups_config);
		groups_list = oobs_groups_config_get_groups (config);
		oobs_list_append (groups_list, &list_iter);
		oobs_list_set (groups_list, &list_iter, group);

		groups_table_add_group (group, &list_iter);
		oobs_object_commit (OOBS_OBJECT (config));

		return group;
	}

	gtk_tree_model_get (model, &iter,
			    COL_GROUP_OBJECT, &group,
			    -1);

	return group;
}

static uid_t
find_new_uid (void)
{
	OobsUsersConfig *config;
	OobsList *list;
	OobsListIter list_iter;
	GObject *user;
	gboolean valid;
	uid_t new_uid, uid, uid_min, uid_max;

	config = OOBS_USERS_CONFIG (GST_USERS_TOOL (tool)->users_config);
	list = oobs_users_config_get_users (config);
	valid = oobs_list_get_iter_first (list, &list_iter);
	uid_min = GST_USERS_TOOL (tool)->minimum_uid;
	uid_max = GST_USERS_TOOL (tool)->maximum_uid;

	new_uid = uid_min - 1;

	while (valid) {
		user = oobs_list_get (list, &list_iter);
		uid = oobs_user_get_uid (OOBS_USER (user));
		g_object_unref (user);

		if (uid <= uid_max && uid >= uid_min && new_uid < uid)
			new_uid = uid;

		valid = oobs_list_iter_next (list, &list_iter);
	}

	new_uid++;

	return new_uid;
}

static void
set_login_length (GtkWidget *entry)
{
	gint max_len;
#ifdef __FreeBSD__
	max_len = UT_NAMESIZE;
#else
	struct utmp ut;

	max_len = sizeof (ut.ut_user);
#endif

	gtk_entry_set_max_length (GTK_ENTRY (entry), max_len);
}

GtkWidget *
user_settings_dialog_new (OobsUser *user)
{
	OobsUsersConfig *config;
	GtkWidget *dialog, *widget;
	const gchar *login;
	gchar *title;

	dialog = gst_dialog_get_widget (tool->main_dialog, "user_settings_dialog");
	login = (gchar*) oobs_user_get_login_name (user);

	if (!login) {
		g_object_set_data (G_OBJECT (dialog), "is_new", GINT_TO_POINTER (TRUE));
		gtk_window_set_title (GTK_WINDOW (dialog), _("New user account"));

		config = OOBS_USERS_CONFIG (GST_USERS_TOOL (tool)->users_config);
		widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_shell");
		set_entry_text (GTK_BIN (widget)->child,
				oobs_users_config_get_default_shell (config));

		widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_uid");
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget), find_new_uid ());
	} else {
		g_object_set_data (G_OBJECT (dialog), "is_new", GINT_TO_POINTER (FALSE));

		title = g_strdup_printf (_("Account '%s' Properties"), login);
		gtk_window_set_title (GTK_WINDOW (dialog), title);
		g_free (title);

		widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_shell");
		set_entry_text (GTK_BIN (widget)->child, oobs_user_get_shell (user));

		widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_home");
		set_entry_text (widget, oobs_user_get_home_directory (user));

		widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_uid");
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget), oobs_user_get_uid (user));
	}

	privileges_table_set_from_user (user);
	set_main_group (user);

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_name");
	set_entry_text (widget, login);
	set_login_length (widget);

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_real_name");
	set_entry_text (widget, oobs_user_get_full_name (user));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_room_number");
	set_entry_text (widget, oobs_user_get_room_number (user));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_wphone");
	set_entry_text (widget, oobs_user_get_work_phone_number (user));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_hphone");
	set_entry_text (widget, oobs_user_get_home_phone_number (user));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_passwd1");
	set_entry_text (widget, (login) ? "password" : NULL);
	g_object_set_data (G_OBJECT (widget), "changed", GINT_TO_POINTER (FALSE));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_passwd2");
	set_entry_text (widget, (login) ? "password" : NULL);

	/* set always the first page */
	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_notebook");
	gtk_notebook_set_current_page (GTK_NOTEBOOK (widget), 0);

	/* set manual password */
	widget = gst_dialog_get_widget (tool->main_dialog, "user_passwd_manual");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);

	return dialog;
}

gboolean
user_settings_dialog_user_is_new (void)
{
	GtkWidget *dialog;
	gboolean is_new;

	dialog = gst_dialog_get_widget (tool->main_dialog, "user_settings_dialog");
	is_new = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (dialog), "is_new"));

	return is_new;
}

static gboolean
is_user_root (OobsUser *user)
{
	const gchar *login = oobs_user_get_login_name (user);

	if (!login)
		return FALSE;

	return (strcmp (login, "root") == 0);
}

static gboolean
login_exists (const gchar *login)
{
	OobsUsersConfig *config;
	OobsList *users_list;
	OobsListIter iter;
	GObject *user;
	gboolean valid;
	const gchar *user_login;

	config = OOBS_USERS_CONFIG (GST_USERS_TOOL (tool)->users_config);
	users_list = oobs_users_config_get_users (config);
	valid = oobs_list_get_iter_first (users_list, &iter);

	while (valid) {
		user = oobs_list_get (users_list, &iter);
		user_login = oobs_user_get_login_name (OOBS_USER (user));
		g_object_unref (user);

		if (user_login && strcmp (login, user_login) == 0)
			return TRUE;

		valid = oobs_list_iter_next (users_list, &iter);
	}

	return FALSE;
}

/* FIXME: this function is duplicated in group-settings.c */
static gboolean
is_valid_name (const gchar *name)
{
	/*
	 * User/group names must start with a letter, and may not
	 * contain colons, commas, newlines (used in passwd/group
	 * files...) or any non-printable characters.
	 */
        if (!*name || !isalpha(*name))
                return FALSE;

        while (*name) {
		if (!isdigit (*name) && !islower (*name) && *name != '-')
                        return FALSE;
                name++;
        }

        return TRUE;
}

static void
check_login (gchar **primary_text, gchar **secondary_text, gpointer data)
{
	OobsUser *user = OOBS_USER (data);
	GtkWidget *widget;
	const gchar *login;

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_name");
	login = gtk_entry_get_text (GTK_ENTRY (widget));

	if (strlen (login) < 1) {
		*primary_text = g_strdup (_("User name is empty"));
		*secondary_text = g_strdup (_("A user name must be specified."));
	} else if (is_user_root (user) && strcmp (login, "root") != 0) {
		*primary_text = g_strdup (_("User name of the administrator account should not be modified"));
	        *secondary_text = g_strdup (_("This would leave the system unusable."));
	} else if (!is_valid_name (login)) {
		*primary_text = g_strdup (_("User name has invalid characters"));
		*secondary_text = g_strdup (_("Please set a valid user name consisting of "
					      "a lower case letter followed by lower case "
					      "letters and numbers."));
	} else if (user_settings_dialog_user_is_new () && login_exists (login)) {
		*primary_text = g_strdup_printf (_("User name \"%s\" already exists"), login);
		*secondary_text = g_strdup (_("Please select a different user name."));
	}
}

static void
check_comments (gchar **primary_text, gchar **secondary_text, gpointer data)
{
	OobsUser *user = OOBS_USER (data);
	GtkWidget *name, *room_number, *wphone, *hphone;
	gchar *comment, *ch;

	name = gst_dialog_get_widget (tool->main_dialog, "user_settings_real_name");
	room_number = gst_dialog_get_widget (tool->main_dialog, "user_settings_room_number");
	wphone = gst_dialog_get_widget (tool->main_dialog, "user_settings_wphone");
	hphone = gst_dialog_get_widget (tool->main_dialog, "user_settings_hphone");

	comment = g_strjoin (" ",
			     gtk_entry_get_text (GTK_ENTRY (name)),
			     gtk_entry_get_text (GTK_ENTRY (room_number)),
			     gtk_entry_get_text (GTK_ENTRY (wphone)),
			     gtk_entry_get_text (GTK_ENTRY (hphone)),
			     NULL);

	if ((ch = g_utf8_strchr (comment, -1, ',')) ||
	    (ch = g_utf8_strchr (comment, -1, '=')) ||
	    (ch = g_utf8_strchr (comment, -1, ':'))) {
		*primary_text   = g_strdup_printf (_("Invalid character \"%c\" in comment"), *ch);
		*secondary_text = g_strdup (_("Check that this character is not used."));
	}

	g_free (comment);
}

static void
check_home (gchar **primary_text, gchar **secondary_text, gpointer data)
{
	OobsUser *user = OOBS_USER (data);
	GtkWidget *widget;
	const gchar *home;

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_home");
	home = gtk_entry_get_text (GTK_ENTRY (widget));

	if (strlen (home) < 1 || !g_path_is_absolute (home)) {
		*primary_text   = g_strdup (N_("Incomplete path in home directory"));
		*secondary_text = g_strdup (N_("Please enter full path for home directory\n"
					       "<span size=\"smaller\">i.e.: /home/john</span>."));
	}
}

static void
check_uid (gchar **primary_text, gchar **secondary_text, gpointer data)
{
	OobsUser *user = OOBS_USER (data);
	GtkWidget *widget;
	gint uid;

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_uid");
	uid = gtk_spin_button_get_value (GTK_SPIN_BUTTON (widget));

	if (is_user_root (user) && uid != 0) {
		*primary_text   = g_strdup (_("User ID of the administrator account should not be modified"));
		*secondary_text = g_strdup (_("This would leave the system unusable."));
	}
}

static void
check_shell (gchar **primary_text, gchar **secondary_text, gpointer data)
{
	OobsUser *user = OOBS_USER (data);
	GtkWidget *widget;
	const gchar *path;

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_shell");
	path = gtk_entry_get_text (GTK_ENTRY (GTK_BIN (widget)->child));

	if (strlen (path) < 1 || !g_path_is_absolute (path)) {
		*primary_text = g_strdup (_("Incomplete path in shell"));
		*secondary_text = g_strdup (_("Please enter full path for shell\n"
					      "<span size=\"smaller\">i.e.: /bin/bash</span>."));
	}
}

static void
check_password (gchar **primary_text, gchar **secondary_text, gpointer data)
{
	OobsUser *user = OOBS_USER (data);
	GtkWidget *widget;
	const gchar *password, *confirmation;

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_passwd1");
	password = gtk_entry_get_text (GTK_ENTRY (widget));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_passwd2");
	confirmation = gtk_entry_get_text (GTK_ENTRY (widget));

	if (strlen (password) < 1) {
		*primary_text = g_strdup (_("Password should not be empty"));
		*secondary_text = g_strdup (_("A password must be provided."));
	} else if (strcmp (password, confirmation) != 0) {
		*primary_text = g_strdup (_("Password confirmation isn't correct"));
		*secondary_text = g_strdup (_("Check that you have provided the same password in both text fields."));
	}
}

gint
user_settings_dialog_run (GtkWidget *dialog, OobsUser *user)
{
	gint response;
	gboolean valid;
	TestBattery battery[] = {
		check_login,
		check_comments,
		check_home,
		check_uid,
		check_shell,
		check_password,
		NULL
	};

	do {
		response = gtk_dialog_run (GTK_DIALOG (dialog));

		valid = (response == GTK_RESPONSE_OK) ?
			test_battery_run (battery, GTK_WINDOW (dialog), user) : TRUE;
	} while (!valid);

	gtk_widget_hide (dialog);
	return response;
}

void
user_settings_dialog_get_data (OobsUser *user)
{
	GtkWidget *widget;
	OobsGroup *group;
	gchar *str;
	gboolean password_changed;
	
	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_name");
	oobs_user_set_login_name (user, gtk_entry_get_text (GTK_ENTRY (widget)));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_real_name");
	oobs_user_set_full_name (user, gtk_entry_get_text (GTK_ENTRY (widget)));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_room_number");
	oobs_user_set_room_number (user, gtk_entry_get_text (GTK_ENTRY (widget)));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_wphone");
	oobs_user_set_work_phone_number (user, gtk_entry_get_text (GTK_ENTRY (widget)));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_hphone");
	oobs_user_set_home_phone_number (user, gtk_entry_get_text (GTK_ENTRY (widget)));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_shell");
	oobs_user_set_shell (user, gtk_entry_get_text (GTK_ENTRY (GTK_BIN (widget)->child)));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_home");
	oobs_user_set_home_directory (user, gtk_entry_get_text (GTK_ENTRY (widget)));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_uid");
	oobs_user_set_uid (user, gtk_spin_button_get_value (GTK_SPIN_BUTTON (widget)));

	widget = gst_dialog_get_widget (tool->main_dialog, "user_passwd_manual");

	/* manual password? */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget))) {
		widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_passwd1");
		password_changed = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (widget), "changed"));

		if (password_changed)
			oobs_user_set_password (user, gtk_entry_get_text (GTK_ENTRY (widget)));
	} else {
		widget = gst_dialog_get_widget (tool->main_dialog, "user_settings_random_passwd");
		oobs_user_set_password (user, gtk_entry_get_text (GTK_ENTRY (widget)));
	}

	group = get_main_group (oobs_user_get_login_name (user));
	oobs_user_set_main_group (user, group);

	privileges_table_save (user);
}
