/* GNOME Scan - Scan as easy as you print
 * Copyright © 2006-2008  Étienne Bersac <bersace@gnome.org>
 *
 * GNOME Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * GNOME Scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with GNOME Scan. If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

#include <gnome-scan-common.h>
#include <glib/gi18n-lib.h>
#include <string.h>




enum  {
	GNOME_SCAN_FORMAT_DUMMY_PROPERTY
};
static char** _vala_array_dup1 (char** self, int length);
static char** _vala_array_dup2 (char** self, int length);
static gpointer gnome_scan_format_parent_class = NULL;
static void gnome_scan_format_finalize (GObject* obj);
#define GNOME_SCAN_MM_PER_INCH 25.4
static void gnome_scan_warn_unsupported_unit (GnomeScanUnit unit);
static void gnome_scan_rectangle_instance_init (GnomeScanRectangle * self);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);




GType gnome_scan_unit_get_type (void) {
	static GType gnome_scan_unit_type_id = 0;
	if (G_UNLIKELY (gnome_scan_unit_type_id == 0)) {
		static const GEnumValue values[] = {{GNOME_SCAN_UNIT_NONE, "GNOME_SCAN_UNIT_NONE", "none"}, {GNOME_SCAN_UNIT_PIXEL, "GNOME_SCAN_UNIT_PIXEL", "pixel"}, {GNOME_SCAN_UNIT_POINTS, "GNOME_SCAN_UNIT_POINTS", "points"}, {GNOME_SCAN_UNIT_INCH, "GNOME_SCAN_UNIT_INCH", "inch"}, {GNOME_SCAN_UNIT_MM, "GNOME_SCAN_UNIT_MM", "mm"}, {GNOME_SCAN_UNIT_BIT, "GNOME_SCAN_UNIT_BIT", "bit"}, {GNOME_SCAN_UNIT_DPI, "GNOME_SCAN_UNIT_DPI", "dpi"}, {GNOME_SCAN_UNIT_PERCENT, "GNOME_SCAN_UNIT_PERCENT", "percent"}, {GNOME_SCAN_UNIT_MICROSECOND, "GNOME_SCAN_UNIT_MICROSECOND", "microsecond"}, {0, NULL, NULL}};
		gnome_scan_unit_type_id = g_enum_register_static ("GnomeScanUnit", values);
	}
	return gnome_scan_unit_type_id;
}


char* gnome_scan_unit_to_string (GnomeScanUnit unit) {
	switch (unit) {
		case GNOME_SCAN_UNIT_PIXEL:
		{
			const char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = _ ("px"), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
		}
		case GNOME_SCAN_UNIT_POINTS:
		{
			const char* _tmp2;
			_tmp2 = NULL;
			return (_tmp2 = _ ("pt"), (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
		}
		case GNOME_SCAN_UNIT_MM:
		{
			const char* _tmp4;
			_tmp4 = NULL;
			return (_tmp4 = _ ("mm"), (_tmp4 == NULL) ? NULL : g_strdup (_tmp4));
		}
		case GNOME_SCAN_UNIT_BIT:
		{
			const char* _tmp6;
			_tmp6 = NULL;
			return (_tmp6 = _ ("bit"), (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
		}
		case GNOME_SCAN_UNIT_DPI:
		{
			const char* _tmp8;
			_tmp8 = NULL;
			return (_tmp8 = _ ("dpi"), (_tmp8 == NULL) ? NULL : g_strdup (_tmp8));
		}
		case GNOME_SCAN_UNIT_PERCENT:
		{
			return g_strdup ("%");
		}
		case GNOME_SCAN_UNIT_MICROSECOND:
		{
			const char* _tmp11;
			_tmp11 = NULL;
			return (_tmp11 = _ ("µs"), (_tmp11 == NULL) ? NULL : g_strdup (_tmp11));
		}
		default:
		{
			return g_strdup ("");
		}
	}
}



GType gnome_scan_status_get_type (void) {
	static GType gnome_scan_status_type_id = 0;
	if (G_UNLIKELY (gnome_scan_status_type_id == 0)) {
		static const GEnumValue values[] = {{GNOME_SCAN_STATUS_UNKNOWN, "GNOME_SCAN_STATUS_UNKNOWN", "unknown"}, {GNOME_SCAN_STATUS_FAILED, "GNOME_SCAN_STATUS_FAILED", "failed"}, {GNOME_SCAN_STATUS_INITIALIZING, "GNOME_SCAN_STATUS_INITIALIZING", "initializing"}, {GNOME_SCAN_STATUS_UNCONFIGURED, "GNOME_SCAN_STATUS_UNCONFIGURED", "unconfigured"}, {GNOME_SCAN_STATUS_READY, "GNOME_SCAN_STATUS_READY", "ready"}, {GNOME_SCAN_STATUS_PROCESSING, "GNOME_SCAN_STATUS_PROCESSING", "processing"}, {GNOME_SCAN_STATUS_DONE, "GNOME_SCAN_STATUS_DONE", "done"}, {0, NULL, NULL}};
		gnome_scan_status_type_id = g_enum_register_static ("GnomeScanStatus", values);
	}
	return gnome_scan_status_type_id;
}


GnomeScanPoint* gnome_scan_point_dup (const GnomeScanPoint* self) {
	GnomeScanPoint* dup;
	dup = g_new0 (GnomeScanPoint, 1);
	memcpy (dup, self, sizeof (GnomeScanPoint));
	return dup;
}


void gnome_scan_point_free (GnomeScanPoint* self) {
	g_free (self);
}


GType gnome_scan_point_get_type (void) {
	static GType gnome_scan_point_type_id = 0;
	if (gnome_scan_point_type_id == 0) {
		gnome_scan_point_type_id = g_boxed_type_register_static ("GnomeScanPoint", (GBoxedCopyFunc) gnome_scan_point_dup, (GBoxedFreeFunc) gnome_scan_point_free);
	}
	return gnome_scan_point_type_id;
}


GnomeScanRange* gnome_scan_range_dup (const GnomeScanRange* self) {
	GnomeScanRange* dup;
	dup = g_new0 (GnomeScanRange, 1);
	memcpy (dup, self, sizeof (GnomeScanRange));
	return dup;
}


void gnome_scan_range_free (GnomeScanRange* self) {
	g_free (self);
}


GType gnome_scan_range_get_type (void) {
	static GType gnome_scan_range_type_id = 0;
	if (gnome_scan_range_type_id == 0) {
		gnome_scan_range_type_id = g_boxed_type_register_static ("GnomeScanRange", (GBoxedCopyFunc) gnome_scan_range_dup, (GBoxedFreeFunc) gnome_scan_range_free);
	}
	return gnome_scan_range_type_id;
}


static char** _vala_array_dup1 (char** self, int length) {
	char** result;
	int i;
	const char* _tmp0;
	result = g_new0 (char*, length);
	for (i = 0; i < length; i++) {
		result[i] = (_tmp0 = self[i], (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	}
	return result;
}


static char** _vala_array_dup2 (char** self, int length) {
	char** result;
	int i;
	const char* _tmp3;
	result = g_new0 (char*, length);
	for (i = 0; i < length; i++) {
		result[i] = (_tmp3 = self[i], (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
	}
	return result;
}


GnomeScanFormat* gnome_scan_format_construct (GType object_type, const char* name, const char* desc, const char* domain, const char* icon_name, char** mime_types, int mime_types_length1, char** suffixes, int suffixes_length1) {
	GnomeScanFormat * self;
	char** _tmp2;
	char** _tmp1;
	const char* _tmp0;
	char** _tmp5;
	char** _tmp4;
	const char* _tmp3;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (desc != NULL, NULL);
	g_return_val_if_fail (domain != NULL, NULL);
	g_return_val_if_fail (icon_name != NULL, NULL);
	self = g_object_newv (object_type, 0, NULL);
	self->name = name;
	self->desc = desc;
	self->domain = domain;
	self->icon_name = icon_name;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->mime_types = (_tmp2 = (_tmp1 = mime_types, (_tmp1 == NULL) ? ((gpointer) _tmp1) : _vala_array_dup1 (_tmp1, mime_types_length1)), self->mime_types = (_vala_array_free (self->mime_types, self->mime_types_length1, (GDestroyNotify) g_free), NULL), self->mime_types_length1 = mime_types_length1, _tmp2);
	_tmp5 = NULL;
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->suffixes = (_tmp5 = (_tmp4 = suffixes, (_tmp4 == NULL) ? ((gpointer) _tmp4) : _vala_array_dup2 (_tmp4, suffixes_length1)), self->suffixes = (_vala_array_free (self->suffixes, self->suffixes_length1, (GDestroyNotify) g_free), NULL), self->suffixes_length1 = suffixes_length1, _tmp5);
	return self;
}


GnomeScanFormat* gnome_scan_format_new (const char* name, const char* desc, const char* domain, const char* icon_name, char** mime_types, int mime_types_length1, char** suffixes, int suffixes_length1) {
	return gnome_scan_format_construct (GNOME_SCAN_TYPE_FORMAT, name, desc, domain, icon_name, mime_types, mime_types_length1, suffixes, suffixes_length1);
}


static void gnome_scan_format_class_init (GnomeScanFormatClass * klass) {
	gnome_scan_format_parent_class = g_type_class_peek_parent (klass);
	G_OBJECT_CLASS (klass)->finalize = gnome_scan_format_finalize;
}


static void gnome_scan_format_instance_init (GnomeScanFormat * self) {
}


static void gnome_scan_format_finalize (GObject* obj) {
	GnomeScanFormat * self;
	self = GNOME_SCAN_FORMAT (obj);
	self->mime_types = (_vala_array_free (self->mime_types, self->mime_types_length1, (GDestroyNotify) g_free), NULL);
	self->suffixes = (_vala_array_free (self->suffixes, self->suffixes_length1, (GDestroyNotify) g_free), NULL);
	G_OBJECT_CLASS (gnome_scan_format_parent_class)->finalize (obj);
}


GType gnome_scan_format_get_type (void) {
	static GType gnome_scan_format_type_id = 0;
	if (gnome_scan_format_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GnomeScanFormatClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gnome_scan_format_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GnomeScanFormat), 0, (GInstanceInitFunc) gnome_scan_format_instance_init, NULL };
		gnome_scan_format_type_id = g_type_register_static (G_TYPE_OBJECT, "GnomeScanFormat", &g_define_type_info, 0);
	}
	return gnome_scan_format_type_id;
}


void gnome_scan_enum_value_init (GnomeScanEnumValue *self, const GValue* value, const char* label, const char* domain) {
	GValue _tmp1 = {0};
	GValue _tmp0 = {0};
	char* _tmp3;
	const char* _tmp2;
	char* _tmp5;
	const char* _tmp4;
	g_return_if_fail (label != NULL);
	memset (self, 0, sizeof (GnomeScanEnumValue));
	(*self).value = (_tmp1 = G_IS_VALUE (&(*value)) ? (g_value_init (&_tmp0, G_VALUE_TYPE (&(*value))), g_value_copy (&(*value), &_tmp0), _tmp0) : (*value), G_IS_VALUE (&(*self).value) ? (g_value_unset (&(*self).value), NULL) : NULL, _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	(*self).label = (_tmp3 = (_tmp2 = label, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), (*self).label = (g_free ((*self).label), NULL), _tmp3);
	_tmp5 = NULL;
	_tmp4 = NULL;
	(*self).domain = (_tmp5 = (_tmp4 = domain, (_tmp4 == NULL) ? NULL : g_strdup (_tmp4)), (*self).domain = (g_free ((*self).domain), NULL), _tmp5);
}


void gnome_scan_enum_value_copy (const GnomeScanEnumValue* self, GnomeScanEnumValue* dest) {
	const char* _tmp2;
	const char* _tmp1;
	GValue _tmp0 = {0};
	_tmp2 = NULL;
	_tmp1 = NULL;
	dest->value = G_IS_VALUE (&self->value) ? (g_value_init (&_tmp0, G_VALUE_TYPE (&self->value)), g_value_copy (&self->value, &_tmp0), _tmp0) : self->value;
	dest->label = (_tmp1 = self->label, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
	dest->domain = (_tmp2 = self->domain, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


void gnome_scan_enum_value_destroy (GnomeScanEnumValue* self) {
	G_IS_VALUE (&self->value) ? (g_value_unset (&self->value), NULL) : NULL;
	self->label = (g_free (self->label), NULL);
	self->domain = (g_free (self->domain), NULL);
}


GnomeScanEnumValue* gnome_scan_enum_value_dup (const GnomeScanEnumValue* self) {
	GnomeScanEnumValue* dup;
	dup = g_new0 (GnomeScanEnumValue, 1);
	gnome_scan_enum_value_copy (self, dup);
	return dup;
}


void gnome_scan_enum_value_free (GnomeScanEnumValue* self) {
	gnome_scan_enum_value_destroy (self);
	g_free (self);
}


GType gnome_scan_enum_value_get_type (void) {
	static GType gnome_scan_enum_value_type_id = 0;
	if (gnome_scan_enum_value_type_id == 0) {
		gnome_scan_enum_value_type_id = g_boxed_type_register_static ("GnomeScanEnumValue", (GBoxedCopyFunc) gnome_scan_enum_value_dup, (GBoxedFreeFunc) gnome_scan_enum_value_free);
	}
	return gnome_scan_enum_value_type_id;
}


/* debugging facility, return the nickk of value.*/
char* gnome_scan_enum_get_nick (GType enum_type, gint value) {
	GEnumClass* klass;
	GEnumValue* enum_value;
	klass = (GEnumClass*) g_type_class_ref (enum_type);
	enum_value = g_enum_get_value (klass, value);
	if (enum_value == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = g_strdup_printf ("%i", value), (klass == NULL) ? NULL : (klass = (g_type_class_unref (klass), NULL)), _tmp0);
	} else {
		const char* _tmp1;
		char* _tmp2;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = enum_value->value_nick, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), (klass == NULL) ? NULL : (klass = (g_type_class_unref (klass), NULL)), _tmp2);
	}
	(klass == NULL) ? NULL : (klass = (g_type_class_unref (klass), NULL));
}


/* debugging facility. return the list of flags in value as "{nick,nick}"*/
char* gnome_scan_flags_get_nicks (GType flags_type, gint value) {
	GFlagsClass* klass;
	GFlagsValue* flags_value;
	char* nicks;
	char* _tmp3;
	klass = (GFlagsClass*) g_type_class_ref (flags_type);
	flags_value = NULL;
	nicks = g_strdup ("{");
	for (; TRUE; value = value ^ flags_value->value) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (value > 0) {
			_tmp0 = ((flags_value = g_flags_get_first_value (klass, (guint) value)) != NULL);
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		if (flags_value != NULL) {
			char* _tmp2;
			if (g_utf8_strlen (nicks, -1) > 1) {
				char* _tmp1;
				_tmp1 = NULL;
				nicks = (_tmp1 = g_strconcat (nicks, ",", NULL), nicks = (g_free (nicks), NULL), _tmp1);
			}
			_tmp2 = NULL;
			nicks = (_tmp2 = g_strconcat (nicks, flags_value->value_nick, NULL), nicks = (g_free (nicks), NULL), _tmp2);
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = g_strconcat (nicks, "}", NULL), (klass == NULL) ? NULL : (klass = (g_type_class_unref (klass), NULL)), nicks = (g_free (nicks), NULL), _tmp3);
}


static void gnome_scan_warn_unsupported_unit (GnomeScanUnit unit) {
	char* _tmp0;
	_tmp0 = NULL;
	g_warning ("gnome-scan-common.vala:152: Unit %s conversion not supported.", _tmp0 = gnome_scan_enum_get_nick (GNOME_SCAN_TYPE_UNIT, (gint) unit));
	_tmp0 = (g_free (_tmp0), NULL);
}


/**
 * gnome_scan_convert:
 * @val:the value to convert
 * @from:the current @val unit
 * @to:the target unit fro @val
 * @res:the resolution in dpi
 *
 * Convert @val from @from unit to @to at @res resolution. Useful for
 * preview area, #GnomeScanScanner, etc.
 **/
double gnome_scan_convert (double val, GnomeScanUnit from, GnomeScanUnit to, double res) {
	if (from == to) {
		return val;
	}
	switch (from) {
		case GNOME_SCAN_UNIT_NONE:
		case GNOME_SCAN_UNIT_BIT:
		case GNOME_SCAN_UNIT_PERCENT:
		case GNOME_SCAN_UNIT_MICROSECOND:
		case GNOME_SCAN_UNIT_DPI:
		{
			gnome_scan_warn_unsupported_unit (from);
			return val;
		}
		default:
		{
			switch (to) {
				case GNOME_SCAN_UNIT_NONE:
				case GNOME_SCAN_UNIT_BIT:
				case GNOME_SCAN_UNIT_PERCENT:
				case GNOME_SCAN_UNIT_MICROSECOND:
				case GNOME_SCAN_UNIT_DPI:
				{
					gnome_scan_warn_unsupported_unit (to);
					return val;
				}
				default:
				{
					return gnome_scan_convert_from_mm (gnome_scan_convert_to_mm (val, from, res), to, res);
				}
			}
		}
	}
}


double gnome_scan_convert_from_mm (double val, GnomeScanUnit to, double res) {
	switch (to) {
		case GNOME_SCAN_UNIT_NONE:
		case GNOME_SCAN_UNIT_BIT:
		case GNOME_SCAN_UNIT_PERCENT:
		case GNOME_SCAN_UNIT_MICROSECOND:
		case GNOME_SCAN_UNIT_DPI:
		{
			gnome_scan_warn_unsupported_unit (to);
			return val;
		}
		case GNOME_SCAN_UNIT_MM:
		{
			return val;
		}
		case GNOME_SCAN_UNIT_PIXEL:
		{
			return val * (GNOME_SCAN_MM_PER_INCH / res);
		}
	}
	return (double) 0;
}


double gnome_scan_convert_to_mm (double val, GnomeScanUnit from, double res) {
	switch (from) {
		case GNOME_SCAN_UNIT_NONE:
		case GNOME_SCAN_UNIT_BIT:
		case GNOME_SCAN_UNIT_PERCENT:
		case GNOME_SCAN_UNIT_MICROSECOND:
		case GNOME_SCAN_UNIT_DPI:
		{
			gnome_scan_warn_unsupported_unit (from);
			return val;
		}
		case GNOME_SCAN_UNIT_MM:
		{
			return val;
		}
		case GNOME_SCAN_UNIT_PIXEL:
		{
			return val / (GNOME_SCAN_MM_PER_INCH / res);
		}
	}
	return (double) 0;
}


void gnome_scan_rectangle_rotate (GnomeScanRectangle* self, GnomeScanRectangle* area, gint angle) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (area != NULL);
	angle = angle % (360);
	switch (angle) {
		case 0:
		{
			break;
		}
		case 270:
		{
			self->width = self->height;
			self->height = self->width;
			self->x = self->y;
			self->y = MAX ((area->width - self->x) - self->width, area->x);
			break;
		}
		case 180:
		{
			self->x = MAX ((area->width - self->x) - self->width, area->x);
			self->y = MAX ((area->height - self->y) - self->height, area->y);
			break;
		}
		case 90:
		{
			self->width = self->height;
			self->height = self->width;
			self->y = self->x;
			self->x = MAX ((area->height - self->y) - self->height, area->y);
			break;
		}
		default:
		{
			g_warning ("gnome-scan-common.vala:271: %i degree rotation is not supported", angle);
			break;
		}
	}
}


void gnome_scan_rectangle_convert (GnomeScanRectangle* self, GnomeScanUnit from, GnomeScanUnit to, double res) {
	g_return_if_fail (self != NULL);
	gnome_scan_rectangle_convert_to_mm (self, from, res);
	gnome_scan_rectangle_convert_from_mm (self, to, res);
}


void gnome_scan_rectangle_convert_to_mm (GnomeScanRectangle* self, GnomeScanUnit from, double res) {
	g_return_if_fail (self != NULL);
	self->x = gnome_scan_convert_to_mm (self->x, from, res);
	self->y = gnome_scan_convert_to_mm (self->y, from, res);
	self->width = gnome_scan_convert_to_mm (self->width, from, res);
	self->height = gnome_scan_convert_to_mm (self->height, from, res);
}


void gnome_scan_rectangle_convert_from_mm (GnomeScanRectangle* self, GnomeScanUnit to, double res) {
	g_return_if_fail (self != NULL);
	self->x = gnome_scan_convert_from_mm (self->x, to, res);
	self->y = gnome_scan_convert_from_mm (self->y, to, res);
	self->width = gnome_scan_convert_from_mm (self->width, to, res);
	self->height = gnome_scan_convert_from_mm (self->height, to, res);
}


GnomeScanRectangle* gnome_scan_rectangle_new (void) {
	GnomeScanRectangle* self;
	self = g_slice_new0 (GnomeScanRectangle);
	gnome_scan_rectangle_instance_init (self);
	return self;
}


static void gnome_scan_rectangle_instance_init (GnomeScanRectangle * self) {
}


void gnome_scan_rectangle_free (GnomeScanRectangle* self) {
	g_slice_free (GnomeScanRectangle, self);
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}




