/* GNOME Scan - Scan as easy as you print
 * Copyright © 2006-2008  Étienne Bersac <bersace@gnome.org>
 *
 * GNOME Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * GNOME Scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with GNOME Scan. If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */
 
using GLib;

namespace Gnome {
    namespace Scan {
		
		/**
		 * SECTION: gnome-scan-module-manager
		 * @short_description: Finding installed modules
		 *
		 * The #GnomeScanModuleManager search in a list of path shared object,
		 * and instanciate a #GnomeScanModule for each shared object. Upon
		 * initialization, the #GnomeScanModule will register new GTypes. This
		 * implementation is meant to be portable, but never tested on other
		 * plateform that GNU/Linux.
		 **/
		public class ModuleManager : Object {
			/**
			 * GnomeScanModuleManager:path:
			 *
			 * The search path where are installed #GnomeScanModule shared
			 * object. A search path is a string formed by a list of absolute
			 * path to directory separated by #G_SEARCHPATH_SEPARATOR.
			 **/
			[Description(nick="Path", blurb="Module path")]
			public string path { get; construct set; }

			private SList<TypeModule> modules;
			
			/**
			 * gnome_scan_module_manager_new:
			 * @path: Search path.
			 * 
			 * Create a new #GnomeScanModuleManager which will handle @path.
			 * 
			 * Returns: a new #GnomeScanModuleManager
			 **/
			public ModuleManager (string path) {
				this.path = path;
			}

			/**
			 * gnome_scan_module_manager_query_modules:
			 * @self: a #GnomeScanModuleManager
			 * 
			 * Search for shared objects in path, instanciate and initialize a
			 * #GnomeScanModule for each shared object. Note that it won't search
			 * in subdirectories.
			 **/
			public void query_modules () {
				Module module;
				Dir dir = null;
				string[] paths = path.split (GLib.Path.SEARCHPATH_SEPARATOR_S);
				string name;
				string filename;
	
				foreach (string path in paths) {
					try {
						dir = Dir.open (path, 0);
					}
					catch (GLib.Error error) {
						/* Show warning only for absolute path, else should be devel path… */
						if (path[0] == '/')
							warning("%s", error.message);
					}

					if (dir == null)
						continue;
				  
					/* Scan for lib*.{so,la} */
					while ((name = dir.read_name ()) != null) {
						if (is_valid_module_name (name)) {
							filename = Path.build_filename (path, name);
							module = new Module (filename);
							
							if (!module.use ()) {
								warning ("%s", GLib.Module.error());
								// throw exception ?
							}
							modules.append (module);
						}
					}
				}
			}
			
			/**
			 * gnome_scan_module_manager_unload_modules:
			 * @self: a #GnomeScanModuleManager
			 * 
			 * Search for shared objects in path, instanciate and initialize a
			 * #GnomeScanModule for each shared object. Note that it won't search
			 * in subdirectories.
			 **/
			public void unload_modules () {
				foreach (TypeModule module in modules)
				module.unuse ();
			}

			private static bool is_valid_module_name (string name)
			{
				return name.has_prefix ("lib")
				&& (name.has_suffix (GLib.Module.SUFFIX)
					|| name.has_suffix (".la"));
			}
		}
    }
}
