/* GNOME Scan - Scan as easy as you print
 * Copyright © 2006-2008  Étienne Bersac <bersace@gnome.org>
 *
 * GNOME Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * GNOME Scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with GNOME Scan. If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

#include <gnome-scan-dialog.h>
#include <gdk/gdk.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gi18n-lib.h>
#include <gnome-scan-acquisition-widget.h>
#include <gnome-scan-node.h>
#include <gnome-scan-scanner-selector.h>
#include <gnome-scan-option-page.h>
#include <gnome-scan-common.h>
#include <gnome-scan-option.h>
#include <gnome-scan-scanner.h>
#include <gnome-scan-sink.h>


#define GNOME_SCAN_TYPE_PAGES (gnome_scan_pages_get_type ())

typedef enum  {
	GNOME_SCAN_PAGES_GENERAL,
	GNOME_SCAN_PAGES_FRONT_SCANNER,
	GNOME_SCAN_PAGES_FRONT_SINK,
	GNOME_SCAN_PAGES_PREVIEW,
	GNOME_SCAN_PAGES_ADVANCED,
	GNOME_SCAN_PAGES_PROCESSING,
	GNOME_SCAN_PAGES_OUTPUT,
	GNOME_SCAN_PAGES_ACQUISITION,
	GNOME_SCAN_PAGES_LAST
} GnomeScanPages;



GType gnome_scan_pages_get_type (void);
struct _GnomeScanDialogPrivate {
	GnomeScanJob* _job;
	GtkNotebook* notebook;
	GtkBox* acquisition_page;
	GnomeScanAcquisitionWidget* acquisitor;
	GtkButton* bcancel;
	GtkButton* bclose;
	GtkButton* bscan;
	GtkButton* bnext;
	GtkButton* bback;
	GtkWidget** pages;
	gint pages_length1;
	gint pages_size;
	GnomeScanNode* scanner;
	GnomeScanScannerSelector* selector;
};

#define GNOME_SCAN_DIALOG_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_SCAN_TYPE_DIALOG, GnomeScanDialogPrivate))
enum  {
	GNOME_SCAN_DIALOG_DUMMY_PROPERTY,
	GNOME_SCAN_DIALOG_JOB
};
static void _gnome_scan_dialog_on_button_close_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self);
static void _gnome_scan_dialog_on_button_scan_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self);
static void _gnome_scan_dialog_on_button_back_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self);
static void _gnome_scan_dialog_on_button_cancel_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self);
static void _gnome_scan_dialog_on_button_next_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self);
static void gnome_scan_dialog_add_buttons (GnomeScanDialog* self, GtkBox* container);
static GtkButton* gnome_scan_dialog_add_button (GnomeScanDialog* self, GtkBox* button_box, const char* stock_id);
static GtkWidget* gnome_scan_dialog_build_page (GnomeScanDialog* self, GnomeScanPages page_id);
static void _gnome_scan_dialog_on_probe_done_gnome_scan_scanner_selector_probe_done (GnomeScanScannerSelector* _sender, gpointer self);
static void gnome_scan_dialog_build_general_tab (GnomeScanDialog* self);
static void gnome_scan_dialog_build_scanner_page (GnomeScanDialog* self);
static void gnome_scan_dialog_build_sink_page (GnomeScanDialog* self);
static void gnome_scan_dialog_build_acquisition_page (GnomeScanDialog* self);
static void gnome_scan_dialog_build_option_page (GnomeScanDialog* self, GnomeScanPages page_id);
static void gnome_scan_dialog_append_page (GnomeScanDialog* self, gint tab, const char* label, GtkContainer* page);
static void gnome_scan_dialog_build_node_ui (GnomeScanDialog* self, GnomeScanNode* node);
static void gnome_scan_dialog_destroy_node_ui (GnomeScanDialog* self, GnomeScanNode* node);
static GnomeScanPages gnome_scan_dialog_get_page_id_for_option (GnomeScanDialog* self, GnomeScanNode* node, GnomeScanOption* option);
static void gnome_scan_dialog_init_buttons (GnomeScanDialog* self);
static gboolean _gnome_scan_acquisition_widget_monitor_job_gsource_func (gpointer self);
static void* _gnome_scan_dialog_job_run_thread_gthread_func (gpointer self);
static void gnome_scan_dialog_scan (GnomeScanDialog* self);
static void gnome_scan_dialog_back (GnomeScanDialog* self);
static void gnome_scan_dialog_close (GnomeScanDialog* self);
static void* gnome_scan_dialog_job_run_thread (GnomeScanDialog* self);
static void gnome_scan_dialog_on_probe_done (GnomeScanDialog* self, GnomeScanScannerSelector* selector);
static void gnome_scan_dialog_on_scanner_selected (GnomeScanDialog* self, GnomeScanJob* job, GParamSpec* pspec);
static void gnome_scan_dialog_on_status_changed (GnomeScanDialog* self, GnomeScanJob* job, GParamSpec* pspec);
static void gnome_scan_dialog_on_button_scan_clicked (GnomeScanDialog* self, GtkButton* button);
static void gnome_scan_dialog_on_button_next_clicked (GnomeScanDialog* self, GtkButton* button);
static void gnome_scan_dialog_on_button_cancel_clicked (GnomeScanDialog* self, GtkButton* button);
static void gnome_scan_dialog_on_button_back_clicked (GnomeScanDialog* self, GtkButton* button);
static void gnome_scan_dialog_on_button_close_clicked (GnomeScanDialog* self, GtkButton* button);
static gboolean gnome_scan_dialog_on_window_delete_event (GnomeScanDialog* self, GnomeScanDialog* window, const GdkEvent* evt);
static gboolean _gnome_scan_dialog_on_window_delete_event_gtk_widget_delete_event (GnomeScanDialog* _sender, const GdkEvent* event, gpointer self);
static void _gnome_scan_dialog_on_scanner_selected_g_object_notify (GnomeScanJob* _sender, GParamSpec* pspec, gpointer self);
static void _gnome_scan_dialog_on_status_changed_g_object_notify (GnomeScanJob* _sender, GParamSpec* pspec, gpointer self);
static GObject * gnome_scan_dialog_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer gnome_scan_dialog_parent_class = NULL;
static void gnome_scan_dialog_finalize (GObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);




GType gnome_scan_pages_get_type (void) {
	static GType gnome_scan_pages_type_id = 0;
	if (G_UNLIKELY (gnome_scan_pages_type_id == 0)) {
		static const GEnumValue values[] = {{GNOME_SCAN_PAGES_GENERAL, "GNOME_SCAN_PAGES_GENERAL", "general"}, {GNOME_SCAN_PAGES_FRONT_SCANNER, "GNOME_SCAN_PAGES_FRONT_SCANNER", "front-scanner"}, {GNOME_SCAN_PAGES_FRONT_SINK, "GNOME_SCAN_PAGES_FRONT_SINK", "front-sink"}, {GNOME_SCAN_PAGES_PREVIEW, "GNOME_SCAN_PAGES_PREVIEW", "preview"}, {GNOME_SCAN_PAGES_ADVANCED, "GNOME_SCAN_PAGES_ADVANCED", "advanced"}, {GNOME_SCAN_PAGES_PROCESSING, "GNOME_SCAN_PAGES_PROCESSING", "processing"}, {GNOME_SCAN_PAGES_OUTPUT, "GNOME_SCAN_PAGES_OUTPUT", "output"}, {GNOME_SCAN_PAGES_ACQUISITION, "GNOME_SCAN_PAGES_ACQUISITION", "acquisition"}, {GNOME_SCAN_PAGES_LAST, "GNOME_SCAN_PAGES_LAST", "last"}, {0, NULL, NULL}};
		gnome_scan_pages_type_id = g_enum_register_static ("GnomeScanPages", values);
	}
	return gnome_scan_pages_type_id;
}


GnomeScanDialog* gnome_scan_dialog_construct (GType object_type, GtkWindow* parent, GnomeScanJob* job) {
	GParameter * __params;
	GParameter * __params_it;
	GnomeScanDialog * self;
	g_return_val_if_fail (job != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "job";
	g_value_init (&__params_it->value, GNOME_SCAN_TYPE_JOB);
	g_value_set_object (&__params_it->value, job);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	if (parent != NULL) {
		gtk_window_set_transient_for ((GtkWindow*) self, parent);
		g_object_set ((GtkWindow*) self, "window-position", GTK_WIN_POS_CENTER_ON_PARENT, NULL);
	}
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


GnomeScanDialog* gnome_scan_dialog_new (GtkWindow* parent, GnomeScanJob* job) {
	return gnome_scan_dialog_construct (GNOME_SCAN_TYPE_DIALOG, parent, job);
}


void gnome_scan_dialog_run (GnomeScanDialog* self) {
	GdkCursor* _tmp0;
	g_return_if_fail (self != NULL);
	gnome_scan_scanner_selector_probe_scanners (self->priv->selector);
	gnome_scan_dialog_init_buttons (self);
	gtk_widget_show_all ((GtkWidget*) self);
	_tmp0 = NULL;
	gdk_window_set_cursor (((GtkWidget*) self)->window, _tmp0 = gdk_cursor_new (GDK_WATCH));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (gdk_cursor_unref (_tmp0), NULL));
	gtk_main ();
}


static void _gnome_scan_dialog_on_button_close_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self) {
	gnome_scan_dialog_on_button_close_clicked (self, _sender);
}


static void _gnome_scan_dialog_on_button_scan_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self) {
	gnome_scan_dialog_on_button_scan_clicked (self, _sender);
}


static void _gnome_scan_dialog_on_button_back_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self) {
	gnome_scan_dialog_on_button_back_clicked (self, _sender);
}


static void _gnome_scan_dialog_on_button_cancel_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self) {
	gnome_scan_dialog_on_button_cancel_clicked (self, _sender);
}


static void _gnome_scan_dialog_on_button_next_clicked_gtk_button_clicked (GtkButton* _sender, gpointer self) {
	gnome_scan_dialog_on_button_next_clicked (self, _sender);
}


/* INTERNAL*/
static void gnome_scan_dialog_add_buttons (GnomeScanDialog* self, GtkBox* container) {
	GtkButtonBox* button_box;
	GtkButtonBox* _tmp0;
	GtkButton* _tmp1;
	GtkButton* _tmp2;
	GtkButton* _tmp3;
	GtkButton* _tmp4;
	GtkButton* _tmp5;
	g_return_if_fail (self != NULL);
	g_return_if_fail (container != NULL);
	button_box = NULL;
	_tmp0 = NULL;
	button_box = (_tmp0 = (GtkButtonBox*) g_object_ref_sink ((GtkHButtonBox*) gtk_hbutton_box_new ()), (button_box == NULL) ? NULL : (button_box = (g_object_unref (button_box), NULL)), _tmp0);
	gtk_box_set_spacing ((GtkBox*) button_box, 6);
	g_object_set (button_box, "layout-style", GTK_BUTTONBOX_END, NULL);
	gtk_box_pack_start (container, (GtkWidget*) button_box, FALSE, TRUE, (guint) 0);
	_tmp1 = NULL;
	self->priv->bclose = (_tmp1 = gnome_scan_dialog_add_button (self, (GtkBox*) button_box, GTK_STOCK_CLOSE), (self->priv->bclose == NULL) ? NULL : (self->priv->bclose = (g_object_unref (self->priv->bclose), NULL)), _tmp1);
	g_signal_connect_object (self->priv->bclose, "clicked", (GCallback) _gnome_scan_dialog_on_button_close_clicked_gtk_button_clicked, self, 0);
	/* TODO: register stock item "scan".*/
	_tmp2 = NULL;
	self->priv->bscan = (_tmp2 = gnome_scan_dialog_add_button (self, (GtkBox*) button_box, GTK_STOCK_APPLY), (self->priv->bscan == NULL) ? NULL : (self->priv->bscan = (g_object_unref (self->priv->bscan), NULL)), _tmp2);
	g_signal_connect_object (self->priv->bscan, "clicked", (GCallback) _gnome_scan_dialog_on_button_scan_clicked_gtk_button_clicked, self, 0);
	_tmp3 = NULL;
	self->priv->bback = (_tmp3 = gnome_scan_dialog_add_button (self, (GtkBox*) button_box, GTK_STOCK_EDIT), (self->priv->bback == NULL) ? NULL : (self->priv->bback = (g_object_unref (self->priv->bback), NULL)), _tmp3);
	g_signal_connect_object (self->priv->bback, "clicked", (GCallback) _gnome_scan_dialog_on_button_back_clicked_gtk_button_clicked, self, 0);
	_tmp4 = NULL;
	self->priv->bcancel = (_tmp4 = gnome_scan_dialog_add_button (self, (GtkBox*) button_box, GTK_STOCK_CANCEL), (self->priv->bcancel == NULL) ? NULL : (self->priv->bcancel = (g_object_unref (self->priv->bcancel), NULL)), _tmp4);
	g_signal_connect_object (self->priv->bcancel, "clicked", (GCallback) _gnome_scan_dialog_on_button_cancel_clicked_gtk_button_clicked, self, 0);
	_tmp5 = NULL;
	self->priv->bnext = (_tmp5 = gnome_scan_dialog_add_button (self, (GtkBox*) button_box, GTK_STOCK_GO_FORWARD), (self->priv->bnext == NULL) ? NULL : (self->priv->bnext = (g_object_unref (self->priv->bnext), NULL)), _tmp5);
	g_signal_connect_object (self->priv->bnext, "clicked", (GCallback) _gnome_scan_dialog_on_button_next_clicked_gtk_button_clicked, self, 0);
	(button_box == NULL) ? NULL : (button_box = (g_object_unref (button_box), NULL));
}


static GtkButton* gnome_scan_dialog_add_button (GnomeScanDialog* self, GtkBox* button_box, const char* stock_id) {
	GtkButton* button;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (button_box != NULL, NULL);
	g_return_val_if_fail (stock_id != NULL, NULL);
	button = g_object_ref_sink ((GtkButton*) gtk_button_new_from_stock (stock_id));
	g_object_set ((GtkWidget*) button, "sensitive", FALSE, NULL);
	gtk_box_pack_start (button_box, (GtkWidget*) button, FALSE, FALSE, (guint) 0);
	return button;
}


static GtkWidget* gnome_scan_dialog_build_page (GnomeScanDialog* self, GnomeScanPages page_id) {
	GtkWidget* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	if (GTK_IS_WIDGET (self->priv->pages[page_id])) {
		GtkWidget* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->pages[page_id], (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	}
	switch (page_id) {
		case GNOME_SCAN_PAGES_GENERAL:
		{
			gnome_scan_dialog_build_general_tab (self);
			break;
		}
		case GNOME_SCAN_PAGES_FRONT_SCANNER:
		{
			gnome_scan_dialog_build_scanner_page (self);
			break;
		}
		case GNOME_SCAN_PAGES_FRONT_SINK:
		{
			gnome_scan_dialog_build_sink_page (self);
			break;
		}
		case GNOME_SCAN_PAGES_ADVANCED:
		case GNOME_SCAN_PAGES_PROCESSING:
		case GNOME_SCAN_PAGES_OUTPUT:
		{
			gnome_scan_dialog_build_option_page (self, page_id);
			break;
		}
		case GNOME_SCAN_PAGES_ACQUISITION:
		{
			gnome_scan_dialog_build_acquisition_page (self);
			break;
		}
		default:
		{
			char* _tmp2;
			_tmp2 = NULL;
			g_warning ("gnome-scan-dialog.vala:181: Don't know how to build %s page", _tmp2 = gnome_scan_enum_get_nick (GNOME_SCAN_TYPE_PAGES, (gint) page_id));
			_tmp2 = (g_free (_tmp2), NULL);
			break;
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->pages[page_id], (_tmp3 == NULL) ? NULL : g_object_ref (_tmp3));
}


static void _gnome_scan_dialog_on_probe_done_gnome_scan_scanner_selector_probe_done (GnomeScanScannerSelector* _sender, gpointer self) {
	gnome_scan_dialog_on_probe_done (self, _sender);
}


static void gnome_scan_dialog_build_general_tab (GnomeScanDialog* self) {
	GtkVPaned* paned;
	GnomeScanScannerSelector* _tmp0;
	GtkVBox* box;
	g_return_if_fail (self != NULL);
	paned = g_object_ref_sink ((GtkVPaned*) gtk_vpaned_new ());
	_tmp0 = NULL;
	self->priv->selector = (_tmp0 = g_object_ref_sink (gnome_scan_scanner_selector_new (self->priv->_job)), (self->priv->selector == NULL) ? NULL : (self->priv->selector = (g_object_unref (self->priv->selector), NULL)), _tmp0);
	gtk_container_set_border_width ((GtkContainer*) self->priv->selector, (guint) 6);
	g_signal_connect_object (self->priv->selector, "probe-done", (GCallback) _gnome_scan_dialog_on_probe_done_gnome_scan_scanner_selector_probe_done, self, 0);
	gtk_paned_pack1 ((GtkPaned*) paned, (GtkWidget*) self->priv->selector, TRUE, FALSE);
	gtk_container_set_border_width ((GtkContainer*) paned, (guint) 6);
	box = g_object_ref_sink ((GtkVBox*) gtk_vbox_new (FALSE, 6));
	gtk_container_set_border_width ((GtkContainer*) box, (guint) 6);
	gtk_paned_pack2 ((GtkPaned*) paned, (GtkWidget*) box, FALSE, FALSE);
	gnome_scan_dialog_append_page (self, (gint) GNOME_SCAN_PAGES_GENERAL, _ ("_General"), (GtkContainer*) paned);
	(paned == NULL) ? NULL : (paned = (g_object_unref (paned), NULL));
	(box == NULL) ? NULL : (box = (g_object_unref (box), NULL));
}


static void gnome_scan_dialog_build_scanner_page (GnomeScanDialog* self) {
	GnomeScanOptionPage* opage;
	GtkPaned* _tmp1;
	GtkWidget* _tmp0;
	GtkPaned* paned;
	GtkBox* _tmp3;
	GtkWidget* _tmp2;
	GtkBox* box;
	g_return_if_fail (self != NULL);
	opage = (GnomeScanOptionPage*) g_object_ref_sink (gnome_scan_option_hpage_new ());
	_tmp1 = NULL;
	_tmp0 = NULL;
	paned = (_tmp1 = (_tmp0 = self->priv->pages[GNOME_SCAN_PAGES_GENERAL], GTK_IS_PANED (_tmp0) ? ((GtkPaned*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	box = (_tmp3 = (_tmp2 = gtk_paned_get_child2 (paned), GTK_IS_BOX (_tmp2) ? ((GtkBox*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : g_object_ref (_tmp3));
	gtk_box_pack_start (box, (GtkWidget*) opage, FALSE, TRUE, (guint) 0);
	gnome_scan_dialog_append_page (self, (gint) GNOME_SCAN_PAGES_FRONT_SCANNER, NULL, (GtkContainer*) opage);
	(opage == NULL) ? NULL : (opage = (g_object_unref (opage), NULL));
	(paned == NULL) ? NULL : (paned = (g_object_unref (paned), NULL));
	(box == NULL) ? NULL : (box = (g_object_unref (box), NULL));
}


static void gnome_scan_dialog_build_sink_page (GnomeScanDialog* self) {
	GnomeScanOptionPage* opage;
	GtkWidget* _tmp0;
	GtkPaned* paned;
	GtkWidget* _tmp1;
	GtkBox* box;
	g_return_if_fail (self != NULL);
	opage = (GnomeScanOptionPage*) g_object_ref_sink (gnome_scan_option_hpage_new ());
	_tmp0 = NULL;
	paned = (_tmp0 = self->priv->pages[GNOME_SCAN_PAGES_GENERAL], GTK_IS_PANED (_tmp0) ? ((GtkPaned*) _tmp0) : NULL);
	_tmp1 = NULL;
	box = (_tmp1 = gtk_paned_get_child2 (paned), GTK_IS_BOX (_tmp1) ? ((GtkBox*) _tmp1) : NULL);
	gtk_box_pack_start (box, (GtkWidget*) opage, FALSE, TRUE, (guint) 0);
	gnome_scan_dialog_append_page (self, (gint) GNOME_SCAN_PAGES_FRONT_SINK, NULL, (GtkContainer*) opage);
	(opage == NULL) ? NULL : (opage = (g_object_unref (opage), NULL));
}


static void gnome_scan_dialog_build_acquisition_page (GnomeScanDialog* self) {
	GnomeScanAcquisitionWidget* _tmp0;
	GnomeScanOptionVPage* _tmp1;
	GtkContainer* page;
	GtkWidget* _tmp3;
	GtkWidget* _tmp2;
	g_return_if_fail (self != NULL);
	_tmp0 = NULL;
	self->priv->acquisitor = (_tmp0 = g_object_ref_sink (gnome_scan_acquisition_widget_new (self->priv->_job, _ ("Acquisition"), _ ("The software now acquires and processes images according to settings."), "scanner")), (self->priv->acquisitor == NULL) ? NULL : (self->priv->acquisitor = (g_object_unref (self->priv->acquisitor), NULL)), _tmp0);
	gtk_box_pack_start (self->priv->acquisition_page, (GtkWidget*) self->priv->acquisitor, TRUE, TRUE, (guint) 0);
	_tmp1 = NULL;
	page = (_tmp1 = g_object_ref_sink (gnome_scan_option_vpage_new (FALSE)), GTK_IS_CONTAINER (_tmp1) ? ((GtkContainer*) _tmp1) : NULL);
	gtk_container_set_border_width (page, (guint) 12);
	gtk_box_pack_start (self->priv->acquisition_page, (GtkWidget*) page, FALSE, TRUE, (guint) 0);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->pages[GNOME_SCAN_PAGES_ACQUISITION] = (_tmp3 = (_tmp2 = (GtkWidget*) page, (_tmp2 == NULL) ? NULL : g_object_ref (_tmp2)), (self->priv->pages[GNOME_SCAN_PAGES_ACQUISITION] == NULL) ? NULL : (self->priv->pages[GNOME_SCAN_PAGES_ACQUISITION] = (g_object_unref (self->priv->pages[GNOME_SCAN_PAGES_ACQUISITION]), NULL)), _tmp3);
	(page == NULL) ? NULL : (page = (g_object_unref (page), NULL));
}


static void gnome_scan_dialog_build_option_page (GnomeScanDialog* self, GnomeScanPages page_id) {
	char* label;
	GnomeScanOptionVPage* _tmp6;
	g_return_if_fail (self != NULL);
	label = NULL;
	switch (page_id) {
		case GNOME_SCAN_PAGES_ADVANCED:
		{
			char* _tmp1;
			const char* _tmp0;
			_tmp1 = NULL;
			_tmp0 = NULL;
			label = (_tmp1 = (_tmp0 = _ ("_Advanced"), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), label = (g_free (label), NULL), _tmp1);
			break;
		}
		case GNOME_SCAN_PAGES_PROCESSING:
		{
			char* _tmp3;
			const char* _tmp2;
			_tmp3 = NULL;
			_tmp2 = NULL;
			label = (_tmp3 = (_tmp2 = _ ("_Processing"), (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), label = (g_free (label), NULL), _tmp3);
			break;
		}
		case GNOME_SCAN_PAGES_OUTPUT:
		{
			char* _tmp5;
			const char* _tmp4;
			_tmp5 = NULL;
			_tmp4 = NULL;
			label = (_tmp5 = (_tmp4 = _ ("_Output"), (_tmp4 == NULL) ? NULL : g_strdup (_tmp4)), label = (g_free (label), NULL), _tmp5);
			break;
		}
		default:
		{
			label = (g_free (label), NULL);
			return;
		}
	}
	_tmp6 = NULL;
	gnome_scan_dialog_append_page (self, (gint) page_id, label, (GtkContainer*) (_tmp6 = g_object_ref_sink (gnome_scan_option_vpage_new (TRUE))));
	(_tmp6 == NULL) ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL));
	label = (g_free (label), NULL);
}


static void gnome_scan_dialog_append_page (GnomeScanDialog* self, gint tab, const char* label, GtkContainer* page) {
	GtkWidget* _tmp1;
	GtkWidget* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (page != NULL);
	if (self->priv->pages[tab] != NULL) {
		return;
	}
	if (label != NULL) {
		GtkWidget* tab_label;
		tab_label = (GtkWidget*) g_object_ref_sink ((GtkLabel*) gtk_label_new_with_mnemonic (label));
		if (gtk_container_get_border_width (page) == 0) {
			gtk_container_set_border_width (page, (guint) 12);
		}
		gtk_notebook_append_page (self->priv->notebook, (GtkWidget*) page, tab_label);
		(tab_label == NULL) ? NULL : (tab_label = (g_object_unref (tab_label), NULL));
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->pages[tab] = (_tmp1 = (_tmp0 = (GtkWidget*) page, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->pages[tab] == NULL) ? NULL : (self->priv->pages[tab] = (g_object_unref (self->priv->pages[tab]), NULL)), _tmp1);
	gtk_widget_show_all ((GtkWidget*) page);
}


static void gnome_scan_dialog_build_node_ui (GnomeScanDialog* self, GnomeScanNode* node) {
	GnomeScanPages page_id;
	GnomeScanOptionPage* page;
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	page_id = 0;
	page = NULL;
	{
		GSList* option_collection;
		GSList* option_it;
		option_collection = gnome_scan_node_get_options (node);
		for (option_it = option_collection; option_it != NULL; option_it = option_it->next) {
			GnomeScanOption* option;
			option = (GnomeScanOption*) option_it->data;
			{
				GnomeScanOptionPage* _tmp1;
				GtkWidget* _tmp0;
				if (gnome_scan_option_get_hint (option) == GNOME_SCAN_OPTION_HINT_HIDDEN) {
					continue;
				}
				page_id = gnome_scan_dialog_get_page_id_for_option (self, node, option);
				_tmp1 = NULL;
				_tmp0 = NULL;
				page = (_tmp1 = (_tmp0 = gnome_scan_dialog_build_page (self, page_id), GNOME_SCAN_IS_OPTION_PAGE (_tmp0) ? ((GnomeScanOptionPage*) _tmp0) : NULL), (page == NULL) ? NULL : (page = (g_object_unref (page), NULL)), _tmp1);
				if (!(GNOME_SCAN_IS_OPTION_PAGE (page))) {
					continue;
				}
				gnome_scan_option_page_pack_option (page, option);
				gtk_widget_show_all ((GtkWidget*) page);
			}
		}
	}
	(page == NULL) ? NULL : (page = (g_object_unref (page), NULL));
}


static void gnome_scan_dialog_destroy_node_ui (GnomeScanDialog* self, GnomeScanNode* node) {
	GnomeScanPages page_id;
	GnomeScanOptionPage* page;
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	page_id = GNOME_SCAN_PAGES_ADVANCED;
	page = NULL;
	{
		GSList* option_collection;
		GSList* option_it;
		option_collection = gnome_scan_node_get_options (node);
		for (option_it = option_collection; option_it != NULL; option_it = option_it->next) {
			GnomeScanOption* option;
			option = (GnomeScanOption*) option_it->data;
			{
				GtkWidget* _tmp0;
				if (gnome_scan_option_get_hint (option) == GNOME_SCAN_OPTION_HINT_HIDDEN) {
					continue;
				}
				page_id = gnome_scan_dialog_get_page_id_for_option (self, node, option);
				_tmp0 = NULL;
				page = (_tmp0 = self->priv->pages[page_id], GNOME_SCAN_IS_OPTION_PAGE (_tmp0) ? ((GnomeScanOptionPage*) _tmp0) : NULL);
				gnome_scan_option_page_unpack_option (page, option);
				gtk_widget_show_all ((GtkWidget*) page);
			}
		}
	}
}


static GnomeScanPages gnome_scan_dialog_get_page_id_for_option (GnomeScanDialog* self, GnomeScanNode* node, GnomeScanOption* option) {
	g_return_val_if_fail (self != NULL, 0);
	g_return_val_if_fail (node != NULL, 0);
	g_return_val_if_fail (option != NULL, 0);
	switch (gnome_scan_option_get_hint (option)) {
		case GNOME_SCAN_OPTION_HINT_HIDDEN:
		{
			g_warning ("gnome-scan-dialog.vala:311: Option %s of node %s is hidden.", g_type_name (G_TYPE_FROM_INSTANCE ((GObject*) node)), gnome_scan_option_get_name (option));
			return GNOME_SCAN_PAGES_LAST;
		}
		case GNOME_SCAN_OPTION_HINT_PRIMARY:
		case GNOME_SCAN_OPTION_HINT_REPEAT:
		{
			if (GNOME_SCAN_IS_SCANNER (node)) {
				return GNOME_SCAN_PAGES_FRONT_SCANNER;
			} else {
				if (GNOME_SCAN_IS_SINK (node)) {
					return GNOME_SCAN_PAGES_FRONT_SINK;
				} else {
					return GNOME_SCAN_PAGES_ADVANCED;
				}
			}
		}
		case GNOME_SCAN_OPTION_HINT_SECONDARY:
		{
			return GNOME_SCAN_PAGES_ADVANCED;
		}
		case GNOME_SCAN_OPTION_HINT_PREVIEW:
		{
			return GNOME_SCAN_PAGES_PREVIEW;
		}
	}
	return GNOME_SCAN_PAGES_ADVANCED;
}


static void gnome_scan_dialog_init_buttons (GnomeScanDialog* self) {
	g_return_if_fail (self != NULL);
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bclose, FALSE);
	gtk_widget_show ((GtkWidget*) self->priv->bclose);
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bscan, FALSE);
	gtk_widget_show ((GtkWidget*) self->priv->bscan);
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bback, TRUE);
	gtk_widget_hide ((GtkWidget*) self->priv->bback);
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bcancel, TRUE);
	gtk_widget_hide ((GtkWidget*) self->priv->bcancel);
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bnext, TRUE);
	gtk_widget_hide ((GtkWidget*) self->priv->bnext);
}


static gboolean _gnome_scan_acquisition_widget_monitor_job_gsource_func (gpointer self) {
	return gnome_scan_acquisition_widget_monitor_job (self);
}


static void* _gnome_scan_dialog_job_run_thread_gthread_func (gpointer self) {
	return gnome_scan_dialog_job_run_thread (self);
}


static void gnome_scan_dialog_scan (GnomeScanDialog* self) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	{
		g_timeout_add ((guint) 42, _gnome_scan_acquisition_widget_monitor_job_gsource_func, self->priv->acquisitor);
		g_thread_create (_gnome_scan_dialog_job_run_thread_gthread_func, self, FALSE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_THREAD_ERROR) {
				goto __catch2_g_thread_error;
			}
			goto __finally2;
		}
	}
	goto __finally2;
	__catch2_g_thread_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			/* run in same thread ?*/
			g_warning ("gnome-scan-dialog.vala:352: Unable to create thread for acquiring");
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally2:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
}


static void gnome_scan_dialog_back (GnomeScanDialog* self) {
	g_return_if_fail (self != NULL);
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->notebook, FALSE);
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->acquisition_page, TRUE);
	gtk_widget_hide ((GtkWidget*) self->priv->acquisition_page);
	gnome_scan_dialog_init_buttons (self);
	gtk_widget_show_all ((GtkWidget*) self->priv->notebook);
}


static void gnome_scan_dialog_close (GnomeScanDialog* self) {
	g_return_if_fail (self != NULL);
	gtk_main_quit ();
}


static void* gnome_scan_dialog_job_run_thread (GnomeScanDialog* self) {
	g_return_val_if_fail (self != NULL, NULL);
	gnome_scan_job_run (self->priv->_job);
	return NULL;
}


/* CALLBACKS*/
static void gnome_scan_dialog_on_probe_done (GnomeScanDialog* self, GnomeScanScannerSelector* selector) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (selector != NULL);
	gdk_window_set_cursor (((GtkWidget*) self)->window, NULL);
}


static void gnome_scan_dialog_on_scanner_selected (GnomeScanDialog* self, GnomeScanJob* job, GParamSpec* pspec) {
	GnomeScanNode* _tmp1;
	GnomeScanNode* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (job != NULL);
	g_return_if_fail (pspec != NULL);
	if (GNOME_SCAN_NODE (gnome_scan_job_get_scanner (job)) == self->priv->scanner) {
		return;
	}
	if (self->priv->scanner != NULL) {
		gnome_scan_dialog_destroy_node_ui (self, self->priv->scanner);
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->scanner = (_tmp1 = (_tmp0 = (GnomeScanNode*) gnome_scan_job_get_scanner (job), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->scanner == NULL) ? NULL : (self->priv->scanner = (g_object_unref (self->priv->scanner), NULL)), _tmp1);
	if (self->priv->scanner != NULL) {
		gnome_scan_dialog_build_node_ui (self, self->priv->scanner);
	}
}


static void gnome_scan_dialog_on_status_changed (GnomeScanDialog* self, GnomeScanJob* job, GParamSpec* pspec) {
	char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (job != NULL);
	g_return_if_fail (pspec != NULL);
	_tmp0 = NULL;
	g_debug ("gnome-scan-dialog.vala:400: job status updated to %s", _tmp0 = gnome_scan_enum_get_nick (GNOME_SCAN_TYPE_STATUS, (gint) gnome_scan_job_get_status (job)));
	_tmp0 = (g_free (_tmp0), NULL);
	switch (gnome_scan_job_get_status (job)) {
		case GNOME_SCAN_STATUS_UNCONFIGURED:
		{
			g_object_set ((GtkWidget*) self->priv->bclose, "sensitive", TRUE, NULL);
			g_object_set ((GtkWidget*) self->priv->bscan, "sensitive", FALSE, NULL);
			break;
		}
		case GNOME_SCAN_STATUS_READY:
		{
			g_object_set ((GtkWidget*) self->priv->bclose, "sensitive", TRUE, NULL);
			g_object_set ((GtkWidget*) self->priv->bscan, "sensitive", TRUE, NULL);
			break;
		}
		case GNOME_SCAN_STATUS_PROCESSING:
		{
			g_object_set ((GtkWidget*) self->priv->bclose, "sensitive", FALSE, NULL);
			g_object_set ((GtkWidget*) self->priv->bscan, "sensitive", FALSE, NULL);
			gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bscan, TRUE);
			gtk_widget_hide ((GtkWidget*) self->priv->bscan);
			gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bcancel, FALSE);
			g_object_set ((GtkWidget*) self->priv->bcancel, "sensitive", TRUE, NULL);
			gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bnext, FALSE);
			g_object_set ((GtkWidget*) self->priv->bnext, "sensitive", FALSE, NULL);
			gtk_widget_set_no_show_all ((GtkWidget*) self->priv->bback, FALSE);
			g_object_set ((GtkWidget*) self->priv->bback, "sensitive", FALSE, NULL);
			break;
		}
		case GNOME_SCAN_STATUS_DONE:
		{
			g_object_set ((GtkWidget*) self->priv->bclose, "sensitive", TRUE, NULL);
			g_object_set ((GtkWidget*) self->priv->bback, "sensitive", TRUE, NULL);
			g_object_set ((GtkWidget*) self->priv->bcancel, "sensitive", FALSE, NULL);
			g_object_set ((GtkWidget*) self->priv->bnext, "sensitive", TRUE, NULL);
			g_object_set ((GtkWidget*) self->priv->bscan, "sensitive", TRUE, NULL);
			break;
		}
		case GNOME_SCAN_STATUS_FAILED:
		{
			break;
		}
		default:
		{
			break;
		}
	}
	gtk_widget_show_all ((GtkWidget*) self);
}


static void gnome_scan_dialog_on_button_scan_clicked (GnomeScanDialog* self, GtkButton* button) {
	GtkWidget* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (button != NULL);
	_tmp0 = NULL;
	_tmp0 = gnome_scan_dialog_build_page (self, GNOME_SCAN_PAGES_ACQUISITION);
	(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->notebook, TRUE);
	gtk_widget_hide ((GtkWidget*) self->priv->notebook);
	gtk_widget_set_no_show_all ((GtkWidget*) self->priv->acquisition_page, FALSE);
	gtk_widget_show_all ((GtkWidget*) self);
	gnome_scan_dialog_scan (self);
}


static void gnome_scan_dialog_on_button_next_clicked (GnomeScanDialog* self, GtkButton* button) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (button != NULL);
	gnome_scan_dialog_scan (self);
}


static void gnome_scan_dialog_on_button_cancel_clicked (GnomeScanDialog* self, GtkButton* button) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (button != NULL);
	gnome_scan_job_cancel (self->priv->_job);
	gnome_scan_dialog_back (self);
}


static void gnome_scan_dialog_on_button_back_clicked (GnomeScanDialog* self, GtkButton* button) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (button != NULL);
	gnome_scan_job_end (self->priv->_job);
	gnome_scan_dialog_back (self);
}


static void gnome_scan_dialog_on_button_close_clicked (GnomeScanDialog* self, GtkButton* button) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (button != NULL);
	gnome_scan_job_end (self->priv->_job);
	gnome_scan_dialog_close (self);
}


static gboolean gnome_scan_dialog_on_window_delete_event (GnomeScanDialog* self, GnomeScanDialog* window, const GdkEvent* evt) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (window != NULL, FALSE);
	gnome_scan_dialog_close (self);
	return FALSE;
}


GnomeScanJob* gnome_scan_dialog_get_job (GnomeScanDialog* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_job;
}


void gnome_scan_dialog_set_job (GnomeScanDialog* self, GnomeScanJob* value) {
	GnomeScanJob* _tmp2;
	GnomeScanJob* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_job = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1)), (self->priv->_job == NULL) ? NULL : (self->priv->_job = (g_object_unref (self->priv->_job), NULL)), _tmp2);
	g_object_notify ((GObject *) self, "job");
}


static gboolean _gnome_scan_dialog_on_window_delete_event_gtk_widget_delete_event (GnomeScanDialog* _sender, const GdkEvent* event, gpointer self) {
	return gnome_scan_dialog_on_window_delete_event (self, _sender, event);
}


static void _gnome_scan_dialog_on_scanner_selected_g_object_notify (GnomeScanJob* _sender, GParamSpec* pspec, gpointer self) {
	gnome_scan_dialog_on_scanner_selected (self, _sender, pspec);
}


static void _gnome_scan_dialog_on_status_changed_g_object_notify (GnomeScanJob* _sender, GParamSpec* pspec, gpointer self) {
	gnome_scan_dialog_on_status_changed (self, _sender, pspec);
}


static GObject * gnome_scan_dialog_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	GnomeScanDialogClass * klass;
	GObjectClass * parent_class;
	GnomeScanDialog * self;
	klass = GNOME_SCAN_DIALOG_CLASS (g_type_class_peek (GNOME_SCAN_TYPE_DIALOG));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GNOME_SCAN_DIALOG (obj);
	{
		GtkBox* box;
		GtkBox* _tmp0;
		GtkNotebook* _tmp1;
		GtkBox* _tmp2;
		GtkWidget** _tmp3;
		box = NULL;
		gtk_container_set_border_width ((GtkContainer*) self, (guint) 6);
		g_object_set ((GtkWindow*) self, "default-width", 320, NULL);
		g_object_set ((GtkWindow*) self, "default-height", 440, NULL);
		gtk_window_set_icon_name ((GtkWindow*) self, "scanner");
		gtk_window_set_title ((GtkWindow*) self, _ ("Scan"));
		g_object_set ((GtkWindow*) self, "window-position", GTK_WIN_POS_CENTER, NULL);
		g_signal_connect_object ((GtkWidget*) self, "delete-event", (GCallback) _gnome_scan_dialog_on_window_delete_event_gtk_widget_delete_event, self, 0);
		gtk_window_set_type_hint ((GtkWindow*) self, GDK_WINDOW_TYPE_HINT_DIALOG);
		g_signal_connect_object ((GObject*) self->priv->_job, "notify::scanner", (GCallback) _gnome_scan_dialog_on_scanner_selected_g_object_notify, self, 0);
		g_signal_connect_object ((GObject*) self->priv->_job, "notify::status", (GCallback) _gnome_scan_dialog_on_status_changed_g_object_notify, self, 0);
		_tmp0 = NULL;
		box = (_tmp0 = (GtkBox*) g_object_ref_sink ((GtkVBox*) gtk_vbox_new (FALSE, 6)), (box == NULL) ? NULL : (box = (g_object_unref (box), NULL)), _tmp0);
		gtk_container_set_border_width ((GtkContainer*) box, (guint) 6);
		gtk_container_add ((GtkContainer*) self, (GtkWidget*) box);
		/* notebook*/
		_tmp1 = NULL;
		self->priv->notebook = (_tmp1 = g_object_ref_sink ((GtkNotebook*) gtk_notebook_new ()), (self->priv->notebook == NULL) ? NULL : (self->priv->notebook = (g_object_unref (self->priv->notebook), NULL)), _tmp1);
		gtk_container_set_border_width ((GtkContainer*) self->priv->notebook, (guint) 6);
		gtk_box_pack_start (box, (GtkWidget*) self->priv->notebook, TRUE, TRUE, (guint) 0);
		_tmp2 = NULL;
		self->priv->acquisition_page = (_tmp2 = (GtkBox*) g_object_ref_sink ((GtkVBox*) gtk_vbox_new (FALSE, 6)), (self->priv->acquisition_page == NULL) ? NULL : (self->priv->acquisition_page = (g_object_unref (self->priv->acquisition_page), NULL)), _tmp2);
		gtk_widget_set_no_show_all ((GtkWidget*) self->priv->acquisition_page, TRUE);
		gtk_box_pack_start (box, (GtkWidget*) self->priv->acquisition_page, TRUE, TRUE, (guint) 0);
		/* build basic UI*/
		_tmp3 = NULL;
		self->priv->pages = (_tmp3 = g_new0 (GtkWidget*, GNOME_SCAN_PAGES_LAST + 1), self->priv->pages = (_vala_array_free (self->priv->pages, self->priv->pages_length1, (GDestroyNotify) g_object_unref), NULL), self->priv->pages_length1 = GNOME_SCAN_PAGES_LAST, self->priv->pages_size = self->priv->pages_length1, _tmp3);
		gnome_scan_dialog_build_general_tab (self);
		gnome_scan_dialog_build_scanner_page (self);
		gnome_scan_dialog_build_node_ui (self, (GnomeScanNode*) gnome_scan_job_get_sink (self->priv->_job));
		/* buttons*/
		gnome_scan_dialog_add_buttons (self, box);
		(box == NULL) ? NULL : (box = (g_object_unref (box), NULL));
	}
	return obj;
}


static void gnome_scan_dialog_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GnomeScanDialog * self;
	gpointer boxed;
	self = GNOME_SCAN_DIALOG (object);
	switch (property_id) {
		case GNOME_SCAN_DIALOG_JOB:
		g_value_set_object (value, gnome_scan_dialog_get_job (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gnome_scan_dialog_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GnomeScanDialog * self;
	self = GNOME_SCAN_DIALOG (object);
	switch (property_id) {
		case GNOME_SCAN_DIALOG_JOB:
		gnome_scan_dialog_set_job (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gnome_scan_dialog_class_init (GnomeScanDialogClass * klass) {
	gnome_scan_dialog_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GnomeScanDialogPrivate));
	G_OBJECT_CLASS (klass)->get_property = gnome_scan_dialog_get_property;
	G_OBJECT_CLASS (klass)->set_property = gnome_scan_dialog_set_property;
	G_OBJECT_CLASS (klass)->constructor = gnome_scan_dialog_constructor;
	G_OBJECT_CLASS (klass)->finalize = gnome_scan_dialog_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_DIALOG_JOB, g_param_spec_object ("job", "job", "job", GNOME_SCAN_TYPE_JOB, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}


static void gnome_scan_dialog_instance_init (GnomeScanDialog * self) {
	self->priv = GNOME_SCAN_DIALOG_GET_PRIVATE (self);
	self->priv->scanner = NULL;
}


static void gnome_scan_dialog_finalize (GObject* obj) {
	GnomeScanDialog * self;
	self = GNOME_SCAN_DIALOG (obj);
	(self->priv->_job == NULL) ? NULL : (self->priv->_job = (g_object_unref (self->priv->_job), NULL));
	(self->priv->notebook == NULL) ? NULL : (self->priv->notebook = (g_object_unref (self->priv->notebook), NULL));
	(self->priv->acquisition_page == NULL) ? NULL : (self->priv->acquisition_page = (g_object_unref (self->priv->acquisition_page), NULL));
	(self->priv->acquisitor == NULL) ? NULL : (self->priv->acquisitor = (g_object_unref (self->priv->acquisitor), NULL));
	(self->priv->bcancel == NULL) ? NULL : (self->priv->bcancel = (g_object_unref (self->priv->bcancel), NULL));
	(self->priv->bclose == NULL) ? NULL : (self->priv->bclose = (g_object_unref (self->priv->bclose), NULL));
	(self->priv->bscan == NULL) ? NULL : (self->priv->bscan = (g_object_unref (self->priv->bscan), NULL));
	(self->priv->bnext == NULL) ? NULL : (self->priv->bnext = (g_object_unref (self->priv->bnext), NULL));
	(self->priv->bback == NULL) ? NULL : (self->priv->bback = (g_object_unref (self->priv->bback), NULL));
	self->priv->pages = (_vala_array_free (self->priv->pages, self->priv->pages_length1, (GDestroyNotify) g_object_unref), NULL);
	(self->priv->scanner == NULL) ? NULL : (self->priv->scanner = (g_object_unref (self->priv->scanner), NULL));
	(self->priv->selector == NULL) ? NULL : (self->priv->selector = (g_object_unref (self->priv->selector), NULL));
	G_OBJECT_CLASS (gnome_scan_dialog_parent_class)->finalize (obj);
}


GType gnome_scan_dialog_get_type (void) {
	static GType gnome_scan_dialog_type_id = 0;
	if (gnome_scan_dialog_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GnomeScanDialogClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gnome_scan_dialog_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GnomeScanDialog), 0, (GInstanceInitFunc) gnome_scan_dialog_instance_init, NULL };
		gnome_scan_dialog_type_id = g_type_register_static (GTK_TYPE_WINDOW, "GnomeScanDialog", &g_define_type_info, 0);
	}
	return gnome_scan_dialog_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}




