/* Flegita - Scanner Utility
 *
 * flegitaactionwidget.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n.h>
#include "flegitaactionwidget.h"
#include <cairo.h>
#include <cairo-pdf.h>
#include "flegitatypes.h"


#define FLEGITA_ACTION_WIDGET_ERROR			(g_type_qname (FLEGITA_TYPE_ACTION_WIDGET))
#define	FLEGITA_ACTION_WIDGET_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)			        (G_TYPE_INSTANCE_GET_PRIVATE ((obj), FLEGITA_TYPE_ACTION_WIDGET, FlegitaActionWidgetPrivate))

typedef struct _FlegitaActionWidgetPrivate		FlegitaActionWidgetPrivate;
typedef struct _FlegitaActionInfo				FlegitaActionInfo;

struct _FlegitaActionWidgetPrivate {
  gboolean		dispose_has_run;

  GtkWidget		*action;	/* save, mail, print, fax */
  GtkWidget		*dir;		/* dir */
  GtkWidget		*contact;	/* contact */
  GtkWidget		*file;		/* filename + format */

  FlegitaActionInfo		*info;
};

struct _FlegitaActionInfo {
  FlegitaAction		action;
  gchar*		dir;
  const gchar* 		filename;
  FlegitaFormat		format;
  gint			resolution;

  gboolean		new_image; 	/* wether an incoming image is the
					   first of an mass acquisition */
  cairo_t		*cairo;
  cairo_surface_t	*surface;
  gchar			*realpath;
};

enum Column {
  COLUMN_ICON,
  COLUMN_LABEL,
  N_COLUMNS
};

/********************************
 * 	    CALLBACKS		*
 ********************************/

void				faw_file_selected				(GtkComboBox *combo,
										 FlegitaActionWidget *faw);

void				faw_action_selected				(GtkComboBox *combo,
										 FlegitaActionWidget *faw);

void				faw_format_selected				(GtkComboBox *combo,
										 FlegitaActionWidget *faw);

void				faw_acquisition_started				(GnomeScanContext* context,
										 GnomeScanForecast *forecast,
										 FlegitaActionWidget *faw);

void				faw_image_acquired				(GnomeScanContext *context,
										 GnomeScanResult *acquisition,
										 FlegitaActionWidget *faw);

void				faw_acquisition_terminated			(GnomeScanContext *context,
										 FlegitaActionWidget *faw);

void				faw_dir_selection_changed			(GtkFileChooser *chooser,
										 FlegitaActionWidget *faw);

void				faw_file_changed				(GtkCellEditable *editable,
										 FlegitaActionWidget *faw);

/********************************
 * 	      OTHER		*
 ********************************/

GtkWidget*			faw_create_action_selector			(FlegitaActionWidget *faw);

GtkWidget*			faw_create_contact_selector			(FlegitaActionWidget *faw);

GtkWidget*			faw_create_dir_selector				(FlegitaActionWidget *faw);

GtkWidget*			faw_create_file_selector			(FlegitaActionWidget *faw);

void				faw_file_selector_fill				(FlegitaFormat format,
										 GtkListStore *store);

void				faw_action_save					(GnomeScanResult *acquisition,
										 FlegitaActionInfo *info);

void				faw_write_image					(GdkPixbuf *pixbuf,
										 gchar *filename,
										 FlegitaFormat format);

void				faw_write_pdf					(FlegitaActionInfo *info,
										 GdkPixbuf *pixbuf,
										 gchar *filename);


/********************************
 * 	      GOBJECT		*
 ********************************/

void				flegita_action_widget_set_property		(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void				flegita_action_widget_get_property		(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

void				flegita_action_widget_finalize			(GObject *obj);

void				flegita_action_widget_dispose			(GObject *obj);


enum {
  PROP_0,
  PROP_CONTEXT
};

enum {
  N_SIGNALS
};

static guint signals[N_SIGNALS];


G_DEFINE_TYPE (FlegitaActionWidget, flegita_action_widget, GTK_TYPE_HBOX);


void
flegita_action_widget_class_init (FlegitaActionWidgetClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property = flegita_action_widget_set_property;
  gobject_class->get_property = flegita_action_widget_get_property;
  gobject_class->dispose 	= flegita_action_widget_dispose;
  /*   gobject_class->finalize 	= flegita_action_widget_finalize; */

  g_type_class_add_private (gobject_class,
			    sizeof (FlegitaActionWidgetPrivate));

  /* Properties */
  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

  /* Signals */
}

void
flegita_action_widget_init (FlegitaActionWidget *widget)
{
  FlegitaActionWidgetPrivate *priv = GET_PRIVATE (widget);

  widget->context = NULL;

  priv->dispose_has_run = FALSE;
  priv->action		= NULL;
  priv->dir		= NULL;
  priv->contact		= NULL;
  priv->file		= NULL;
  priv->info 		= g_new0 (FlegitaActionInfo, 1);
}

void
flegita_action_widget_dispose (GObject *obj)
{
  FlegitaActionWidget *widget = FLEGITA_ACTION_WIDGET (obj);
  FlegitaActionWidgetPrivate *priv = GET_PRIVATE (widget);
  FlegitaActionWidgetClass *b_klass = FLEGITA_ACTION_WIDGET_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref devices */
  g_object_unref (widget->context);
  g_free (priv->info);

  priv->dispose_has_run = TRUE;

  /* chain */
  /*   FLEGITA_ACTION_WIDGET_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
flegita_action_widget_finalize (GObject *obj)
{
  FlegitaActionWidget *widget = FLEGITA_ACTION_WIDGET (obj);
  FlegitaActionWidgetClass *b_klass = FLEGITA_ACTION_WIDGET_GET_CLASS (obj);

  /*   FLEGITA_ACTION_WIDGET_PARENT_CLASS (b_klass)->finalize (obj); */
}

void
flegita_action_widget_set_property (GObject *obj,
				    guint property_id,
				    const GValue *value,
				    GParamSpec *pspec)
{
  FlegitaActionWidget *widget = FLEGITA_ACTION_WIDGET (obj);
  FlegitaActionWidgetPrivate *priv = GET_PRIVATE (widget);

  switch (property_id) {
  case PROP_CONTEXT:
    widget->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}

void
flegita_action_widget_get_property (GObject *obj,
				    guint property_id,
				    GValue *value,
				    GParamSpec *pspec)
{
  FlegitaActionWidget *widget = FLEGITA_ACTION_WIDGET (obj);
  FlegitaActionWidgetPrivate *priv = GET_PRIVATE (widget);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, widget->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}




/********************************
 * 	      METHODS		*
 ********************************/

/**
 * flegita_action_widget_new:
 * @context: a #GnomeScanContext
 * 
 * Create a new #FlegitaActionWidget connected to @context.
 * 
 * Return value: a new #FlegitaActionWidget
 **/
GtkWidget*
flegita_action_widget_new (GnomeScanContext *context)
{
  FlegitaActionWidget *faw;
  FlegitaActionWidgetPrivate *priv;
  GtkWidget *widget;
  gchar *dirname;

  widget = GTK_WIDGET (g_object_new (FLEGITA_TYPE_ACTION_WIDGET,
				     "context", context,
				     NULL));
  
  faw = FLEGITA_ACTION_WIDGET (widget);
  priv = GET_PRIVATE (faw);

  gtk_box_set_spacing (GTK_BOX (widget), 6);

  /* action */
  priv->action = faw_create_action_selector (faw);
  gtk_box_pack_start (GTK_BOX (widget), priv->action, FALSE, FALSE, 0);

  /* dir */
  priv->dir = faw_create_dir_selector (faw);
  gtk_box_pack_start (GTK_BOX (widget), priv->dir, FALSE, FALSE, 0);
  gtk_widget_set_no_show_all (priv->dir, TRUE);

  /* contact */
  /*   priv->contact = faw_create_contact_selector (faw); */
  /*   gtk_box_pack_start (GTK_BOX (widget), priv->contact, FALSE, FALSE, 0); */
  /*   gtk_widget_set_no_show_all (priv->contact, TRUE); */

  /* file */
  priv->file = faw_create_file_selector (faw);
  gtk_box_pack_start (GTK_BOX (widget), priv->file, TRUE, TRUE, 0);
  gtk_widget_set_no_show_all (priv->file, TRUE);

  g_signal_connect (context,
		    "image-received",
		    (GCallback) faw_image_acquired,
		    faw);

  g_signal_connect (context,
		    "acquisition-started",
		    (GCallback) faw_acquisition_started,
		    faw);

  g_signal_connect (context,
		    "acquisition-terminated",
		    (GCallback) faw_acquisition_terminated,
		    faw);

  /* preselect */
  gtk_combo_box_set_active (GTK_COMBO_BOX (priv->action), 0);

  return widget;
}

/********************************
 * 	    CALLBACKS		*
 ********************************/

void
faw_action_selected (GtkComboBox *combo,
		     FlegitaActionWidget *faw)
{
  FlegitaActionWidgetPrivate *priv = GET_PRIVATE (faw);
  gint active = gtk_combo_box_get_active (combo);

  switch (active) {
  case ACTION_SAVE:
    gtk_widget_set_no_show_all (priv->dir, FALSE); 
    gtk_widget_set_no_show_all (priv->file, FALSE); 

    gtk_widget_show_all (priv->dir);
    gtk_widget_show_all (priv->file);
    /*     gtk_widget_hide (priv->contact); */
    break;
  case ACTION_MAIL:
    gtk_widget_set_no_show_all (priv->contact, FALSE); 
    gtk_widget_set_no_show_all (priv->file, FALSE); 

    gtk_widget_hide (priv->dir);
    gtk_widget_show_all (priv->contact);
    gtk_widget_show_all (priv->file);
    break;
  }

  priv->info->action = active;
}

void
faw_dir_selection_changed (GtkFileChooser *chooser,
			   FlegitaActionWidget *faw)
{
  GET_PRIVATE (faw)->info->dir = gtk_file_chooser_get_filename (chooser);
}

void
faw_file_changed (GtkCellEditable *editable,
		  FlegitaActionWidget *faw)
{
  GET_PRIVATE (faw)->info->filename = gtk_entry_get_text (GTK_ENTRY (editable));
}

void
faw_format_selected (GtkComboBox *combo,
		     FlegitaActionWidget *faw)
{
  FlegitaFormat format = gtk_combo_box_get_active (combo);
  FlegitaActionInfo *info =  GET_PRIVATE (faw)->info;
  info->format = format;

  switch (format) {
  case FORMAT_PNG:
  case FORMAT_JPEG:
  case FORMAT_TIFF:
/*   case FORMAT_PDF:		/\* For now, we create PDF for screen, not print *\/ */
    info->resolution = gdk_screen_get_resolution (gdk_screen_get_default ());
    break;
  case FORMAT_PDF:
    /* TODO: check correct resolution for PDF (ready to print with nice result) */
    info->resolution = 300;
    break;
  }

  gnome_scan_context_set_resolutions (faw->context,
				      info->resolution, info->resolution);
}

void
faw_acquisition_started (GnomeScanContext *context,
			 GnomeScanForecast *forecast,
			 FlegitaActionWidget *faw)
{
  FlegitaActionWidgetPrivate *priv = GET_PRIVATE (faw);
  priv->info->new_image = TRUE;
}

void
faw_image_acquired (GnomeScanContext *context,
		    GnomeScanResult *acquisition,
		    FlegitaActionWidget *faw)
{
  FlegitaActionWidgetPrivate *priv = GET_PRIVATE (faw);

  switch (priv->info->action) {
  case ACTION_SAVE:
    faw_action_save (acquisition, priv->info);
    break;
  }
}

void
faw_acquisition_terminated (GnomeScanContext *context,
			    FlegitaActionWidget *faw)
{
  FlegitaActionWidgetPrivate *priv = GET_PRIVATE (faw);

  if (priv->info->cairo) {
    cairo_destroy (priv->info->cairo);
    priv->info->cairo = NULL;
    cairo_surface_destroy (priv->info->surface);
    priv->info->surface = NULL;
    g_free (priv->info->realpath);
    priv->info->realpath = NULL;
  }
}



/********************************
 * 	      OTHER		*
 ********************************/

GtkWidget*
faw_create_action_selector (FlegitaActionWidget *faw)
{
  GtkWidget *widget;
  GtkListStore *store;
  GtkTreeIter iter;
  GtkCellRenderer *renderer;

  store = gtk_list_store_new (N_COLUMNS,
			      G_TYPE_STRING,
			      G_TYPE_STRING);

  gtk_list_store_append (store, &iter);
  gtk_list_store_set (store, &iter, COLUMN_ICON, GTK_STOCK_SAVE_AS, COLUMN_LABEL, _("Save"), -1);

  /*   gtk_list_store_append (store, &iter); */
  /*   gtk_list_store_set (store, &iter, COLUMN_ICON, GNOME_STOCK_MAIL, COLUMN_LABEL, _("Mail"), -1); */

  /*   gtk_list_store_append (store, &iter); */
  /*   gtk_list_store_set (store, &iter, COLUMN_ICON, GTK_STOCK_PRINT, COLUMN_LABEL, _("Print"), -1); */

  /* create the combo */
  widget = gtk_combo_box_new_with_model (GTK_TREE_MODEL (store));

  /* icon */
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_cell_renderer_set_fixed_size (GTK_CELL_RENDERER (renderer), 16, 16);
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (widget), renderer, FALSE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (widget), renderer, "stock-id", COLUMN_ICON);

  /* label */
  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (widget), renderer, TRUE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (widget), renderer, "markup", COLUMN_LABEL);

  g_signal_connect (GTK_COMBO_BOX (widget),
		    "changed",
		    (GCallback) faw_action_selected,
		    faw);

  return widget;
}

GtkWidget*
faw_create_contact_selector (FlegitaActionWidget *faw)
{
  GtkWidget *box, *widget;

  box = gtk_hbox_new (FALSE, 6);

  /* TODO: Choose attached filename and format  */

  return box;
}

GtkWidget*
faw_create_dir_selector (FlegitaActionWidget *faw)
{
  GtkWidget *box, *widget;

  box = gtk_hbox_new (FALSE, 6);
  /* translators, label "in" before [dir] selector */
  widget = gtk_label_new_with_mnemonic (_("in"));
  gtk_box_pack_start (GTK_BOX (box), widget, FALSE, TRUE, 0);

  /* translators,  select folder dialog title */
  widget = gtk_file_chooser_button_new (_("Save In..."), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
  gtk_box_pack_start (GTK_BOX (box), widget, FALSE, TRUE, 0);

  g_signal_connect (GTK_FILE_CHOOSER (widget),
		    "selection-changed",
		    (GCallback) faw_dir_selection_changed,
		    faw);

  /* default value */
  /* TODO: store last dir */
  gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (widget), g_get_home_dir ());

  return box;
}

GtkWidget*
faw_create_file_selector (FlegitaActionWidget *faw)
{
  GtkWidget *box, *widget;
  GSList *formats;
  GtkFileFilter *filter;
  GtkListStore *store;
  GtkTreeIter iter;
  GtkCellRenderer *renderer;
  gchar* dir;
  gint i;

  box = gtk_hbox_new (FALSE, 0);

  /* NAME */
  widget = gtk_entry_new ();
  gtk_entry_set_width_chars (GTK_ENTRY (widget), 8);

  g_signal_connect (GTK_CELL_EDITABLE (widget),
		    "changed",
		    (GCallback) faw_file_changed,
		    faw);

  /* TODO: store laste filename */
  /* translators, default proposition for filename. */
  gtk_entry_set_text (GTK_ENTRY (widget), _("Scanned picture"));

  gtk_box_pack_start (GTK_BOX (box), widget, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (box), gtk_label_new_with_mnemonic ("."), FALSE, TRUE, 0);

  /* FORMAT */
  store = gtk_list_store_new (N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING);
  gs_enum_foreach (FLEGITA_TYPE_FORMAT, (GSEFunc) faw_file_selector_fill, store);

  /* create the combo */
  widget = gtk_combo_box_new_with_model (GTK_TREE_MODEL (store));

  /* icon */
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (widget), renderer, FALSE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (widget), renderer, "icon-name", COLUMN_ICON);
  gtk_cell_renderer_set_fixed_size (GTK_CELL_RENDERER (renderer), 16, 16);

  /* label */
  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (widget), renderer, TRUE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (widget), renderer, "markup", COLUMN_LABEL);

  g_signal_connect (GTK_COMBO_BOX (widget),
  		    "changed",
  		    (GCallback) faw_format_selected,
  		    faw);

  gtk_combo_box_set_active (GTK_COMBO_BOX (widget), FORMAT_PNG);

  gtk_box_pack_start (GTK_BOX (box), widget, FALSE, TRUE, 0);

  return box;
}

void
faw_file_selector_fill (FlegitaFormat format,
			GtkListStore *store)
{
  static const gchar *icons[] = {
    "gnome-mime-image",
    "gnome-mime-image",
    "gnome-mime-image",
    "stock_save-pdf"
  };

  GtkTreeIter iter;
  gtk_list_store_append (store, &iter);
  gtk_list_store_set (store, &iter,
		      COLUMN_ICON, icons[format],
		      COLUMN_LABEL, gs_enum_get_nick (FLEGITA_TYPE_FORMAT, format),
		      -1);
}

void
faw_action_save (GnomeScanResult *acquisition,
		 FlegitaActionInfo *info)
{
  GError *error = NULL;
  gchar *basename = NULL, *filename = NULL, *number = NULL;
  gint n;

  /* build filename */
  n = 0;
  number = "";
  do {
    number = n ? g_strdup_printf ("-%i", n) : "";

    if (basename)
      g_free (basename);
    basename = g_strconcat (info->filename, number, ".", gs_enum_get_nick (FLEGITA_TYPE_FORMAT, info->format), NULL);

    if (filename)
      g_free (filename);

    filename = g_build_filename (info->dir, basename, NULL);

    n++;
  } while (g_file_test (filename, G_FILE_TEST_EXISTS));

  switch (info->format) {
  case FORMAT_PNG:
  case FORMAT_JPEG:
  case FORMAT_TIFF:
    faw_write_image (acquisition->image, filename, info->format);
    g_free (filename);
    break;
  case FORMAT_PDF:
    info->realpath = info->realpath ? info->realpath : filename;
    faw_write_pdf (info, acquisition->image, info->realpath);
    break;
  }
}

/* maybe the simplest backend for so many format :) */
void
faw_write_image (GdkPixbuf *pixbuf,
		 gchar *filename,
		 FlegitaFormat format)
{
  /* WRITE PIXBUF TO IMAGE USING GDK */
  gdk_pixbuf_save (pixbuf, filename, gs_enum_get_nick (FLEGITA_TYPE_FORMAT, format), NULL, NULL);
}

void
faw_write_pdf (FlegitaActionInfo *info,
	       GdkPixbuf *pixbuf,
	       gchar *filename)
{
  cairo_t *cr;
  cairo_surface_t *cs;

  g_debug ("Writing to %s", info->realpath);
  if (info->new_image) {
    info->surface = cs = cairo_pdf_surface_create (info->realpath,
						   /* dpi / 72 = dot per point */
						   gdk_pixbuf_get_width (pixbuf) / (info->resolution / 72),
						   gdk_pixbuf_get_height (pixbuf) / (info->resolution / 72));
    cairo_surface_set_fallback_resolution (cs,(gint) info->resolution, (gint) info->resolution);
    info->cairo = cr = cairo_create (cs);
    info->new_image = FALSE;
  }
  else {
    cr = info->cairo;
    cs = info->surface;
  }

  /* paint the image in a page */
  gdk_cairo_set_source_pixbuf (cr, pixbuf, 0, 0);
  cairo_paint (cr);
  cairo_show_page (cr);
/*   cairo_surface_flush (cs); */
}
