/* -*- Mode: C; indent-tabs-mode: t; tab-width: 8; c-basic-offset: 8 -*- */

/* gnome-remote-shell - Wrapper for opening remote shells
 * Copyright (C) 2002 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <netdb.h>
#include <gconf/gconf-client.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtkspinbutton.h>
#include <glade/glade.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-ui-init.h>

#define SSH_PORT 22
#define TELNET_PORT 23

static GConfClient *conf_client;
static GtkWidget *dialog;
static GtkWidget *ssh_method;
static GtkWidget *telnet_method;
static GtkWidget *host_entry;
static GtkWidget *user_entry;
static GtkWidget *port_default;
static GtkWidget *port_entry;
static GtkWidget *window_width;
static GtkWidget *window_height;

void	dialog_destroyed_cb (GtkWidget *widget, gpointer user_data);
void	dialog_response_cb (GtkDialog *dialog, gint response_id, gpointer user_data);
void	entry_activate_cb (GtkWidget *widget, gpointer user_data);
void	port_default_toggled_cb (GtkToggleButton *button, gpointer user_data);
void	radio_button_toggled_cb (GtkToggleButton *button, gpointer user_data);

static gboolean check_network_status (const gchar *host, gint port);
static void	activate_shell (void);
static void	set_spin_from_config (GtkWidget *spin, const gchar *key, gint default_value);
static gboolean	validate_host (const gchar *host);

void
dialog_destroyed_cb (GtkWidget *widget, gpointer user_data)
{
	gtk_main_quit ();
}

void
dialog_response_cb (GtkDialog *dialog, gint response_id, gpointer user_data)
{
	if (response_id == GTK_RESPONSE_OK)
		activate_shell ();
	else 
		gtk_widget_destroy (GTK_WIDGET (dialog));
}

void
radio_button_toggled_cb (GtkToggleButton *button, gpointer user_data)
{
	gint port = -1;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ssh_method))) {
		gtk_widget_set_sensitive (user_entry, TRUE);
	} else {
		gtk_widget_set_sensitive (user_entry, FALSE);
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (port_default))) {
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ssh_method)))
			port = SSH_PORT;
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (telnet_method)))
			port = TELNET_PORT;
		if (port > 0)
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (port_entry), port);
	}
}

void
entry_activate_cb (GtkWidget *widget, gpointer user_data)
{
	activate_shell ();
}

void
port_default_toggled_cb (GtkToggleButton *button, gpointer user_data)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button))) {
		gtk_widget_set_sensitive (port_entry, FALSE);
	} else {
		gtk_widget_set_sensitive (port_entry, TRUE);
	}
}

static gboolean
check_network_status (const gchar *host, gint port)
{
	struct sockaddr_in *addr;
	struct hostent *hostname;
	gint sd;
	gchar *msgerror;

	hostname = gethostbyname (host);

	addr = (struct sockaddr_in *) g_malloc (sizeof (struct sockaddr_in));

	sd = socket (AF_INET, SOCK_STREAM, 0);
	addr->sin_family = AF_INET;
	addr->sin_port = g_htons (port);
	addr->sin_addr = *(struct in_addr *) hostname->h_addr;

	errno = 0;
	if (connect (sd, (struct sockaddr *) addr, sizeof (struct sockaddr_in)) != 0)
	{
		msgerror = (gchar *) strerror (errno);
		msgerror = g_locale_to_utf8 (msgerror, strlen (msgerror), NULL, NULL, NULL);
		gnome_error_dialog_parented (_(msgerror), GTK_WINDOW (dialog));
		g_free (addr);

		return FALSE;
	}
	shutdown (sd, 2);

	g_free (addr);

	return TRUE;
}

static void
activate_shell (void)
{
	GConfValue *conf_value;
	GError *err = NULL;
	gchar *cmd;
	const gchar *host = NULL, *user = NULL;
	gchar *term_app, *parm_exec, *parm_title;
	gint port;
	gchar *geometry;

	/*
	 * Get the port.
	 * Use the default port if the checkbutton is active
	 * (22 for secure shell, 23 for telnet)
	 */
	port = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (port_entry));

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (port_default))) {
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ssh_method)))
			port = SSH_PORT;
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (telnet_method)))
			port = TELNET_PORT;
	}

	/*
	 * Get the host and do a quick lookup.
	 */
	host = gtk_entry_get_text (GTK_ENTRY (gnome_entry_gtk_entry (GNOME_ENTRY (host_entry))));
	if (!host || !strlen (host)) {
		gnome_error_dialog_parented (_("No host specified."), GTK_WINDOW (dialog));
		return;
	}
					   
	if (validate_host(host) == FALSE)
		return;

	if (check_network_status (host, port) == FALSE)
		return;

	/*
	 * Get the terminal preferences from GConf.
	 */
	conf_value = gconf_client_get (conf_client, "/desktop/gnome/applications/terminal/exec", NULL);
	if (conf_value != NULL) {
		term_app = g_strdup (gconf_value_get_string (conf_value));
		gconf_value_free (conf_value);
	} else {
		term_app = g_strdup ("gnome-terminal");
	}

	conf_value = gconf_client_get (conf_client, "/desktop/gnome/applications/terminal/exec_arg", NULL);
	if (conf_value != NULL) {
		parm_exec = g_strdup (gconf_value_get_string (conf_value));
		gconf_value_free (conf_value);
	} else {
		parm_exec = g_strdup ("-x");
	}

	/*
	 * Do some intelligent guessing from the terminal exec line.
	 */
	if (strlen(term_app) >= 14 && !strncmp(term_app, "gnome-terminal", 14)) {
		geometry = g_strdup_printf (
			"--geometry=%dx%d",
			gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (window_width)),
			gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (window_height)));
		parm_title = g_strdup_printf ("-t");
	} else {
		geometry = g_strdup_printf (
			"-geometry %dx%d",
			gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (window_width)),
			gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (window_height)));
		parm_title = g_strdup_printf ("-T");
	}

		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ssh_method))) {
			user = gtk_entry_get_text (GTK_ENTRY (gnome_entry_gtk_entry (GNOME_ENTRY (user_entry))));
			if (!user || !strlen (user)) {
				gnome_error_dialog_parented (_("A user name is required for SSH"),
							     GTK_WINDOW (dialog));
				return;
			}

			if (port > 0) {
				cmd = g_strdup_printf (
				"%s %s %s \"%s@%s - Secure shell\" %s \"ssh -p %d -X %s@%s\"",
				term_app, geometry, parm_title, user, host, parm_exec, port, user, host);
			} else {
				cmd = g_strdup_printf (
				"%s %s %s \"%s@%s - Secure shell\" %s \"ssh -X %s@%s\"",
				term_app, geometry, parm_title, user, host, parm_exec, user, host);
			}
		} else {
			if (port > 0) {
				cmd = g_strdup_printf (
				"%s %s %s \"%s - Remote shell\" %s \"telnet %s %d\"",
				term_app, geometry, parm_title, host, parm_exec, host, port);
			} else {
				cmd = g_strdup_printf (
				"%s %s %s \"%s - Remote shell\" %s \"telnet %s\"",
				term_app, geometry, parm_title, host, parm_exec, host);
			}
		}

		if (!g_spawn_command_line_async (cmd, &err)) {
			gnome_error_dialog_parented (err->message, GTK_WINDOW (dialog));
			g_error_free (err);
		}

		g_free (cmd);
		g_free (geometry);
	g_free (parm_exec);
	g_free (parm_title);
	g_free (term_app);

	/* Save the dialog state */
	gconf_client_set_int (conf_client, "/apps/gnome-remote-shell/TerminalWidth",
			      (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (window_width)),
			      NULL);
	gconf_client_set_int (conf_client, "/apps/gnome-remote-shell/TerminalHeight",
			      (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (window_height)),
			      NULL);
	
	if (host != NULL)
		gnome_entry_prepend_history (GNOME_ENTRY (host_entry), TRUE, host);
	if (user != NULL)
		gnome_entry_prepend_history (GNOME_ENTRY (user_entry), TRUE, user);

	gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
set_spin_from_config (GtkWidget *spin, const gchar *key, gint default_value)
{
	GConfValue *conf_value;

	conf_value = gconf_client_get (conf_client, key, NULL);
	if (conf_value) {
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin),
					   (gdouble) gconf_value_get_int (conf_value));
		gconf_value_free (conf_value);
	} else {
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), (gdouble) default_value);
	}
}

static gboolean
validate_host (const gchar *host)
{
	struct hostent *hostname;

	hostname = gethostbyname(host);
	if (hostname == NULL) {
		gnome_error_dialog_parented (_("The host cannot be found."), GTK_WINDOW (dialog));
		return FALSE;
	}
	
	return TRUE;
}

int
main (int argc, char **argv)
{
	const gchar *xml_file = DATADIR "gnome-remote-shell.glade";
	GladeXML *xml;
	gchar *icon_path;

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	gnome_program_init ("gnome-remote-shell", VERSION,
			    LIBGNOMEUI_MODULE, argc, argv, NULL);

	icon_path = g_build_filename (GNOME_ICONDIR, "gnome-remote-shell.png", NULL);
	gnome_window_icon_set_default_from_file (icon_path);
	g_free (icon_path);

	conf_client = gconf_client_get_default ();

	/* create the main dialog */
	xml = glade_xml_new (xml_file, NULL, NULL);

	dialog = glade_xml_get_widget (xml, "gnome_remote_shell");
	ssh_method = glade_xml_get_widget (xml, "ssh_method");
	telnet_method = glade_xml_get_widget (xml, "telnet_method");
	host_entry = glade_xml_get_widget (xml, "host_entry");
	user_entry = glade_xml_get_widget (xml, "user_entry");
	port_entry = glade_xml_get_widget (xml, "port_entry");
	port_default = glade_xml_get_widget (xml, "port_default");
	window_width = glade_xml_get_widget (xml, "window_width");
	window_height = glade_xml_get_widget (xml, "window_height");

	/* set settings from gconf */
	set_spin_from_config (window_width, "/apps/gnome-remote-shell/TerminalWidth", 80);
	set_spin_from_config (window_height, "/apps/gnome-remote-shell/TerminalHeight", 25);	

	glade_xml_signal_autoconnect (xml);

	gtk_main ();

	g_object_unref (G_OBJECT (conf_client));

	return 0;
}
