/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <time.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>


#define DVB_TYPE_EVENT_ELEMENT (dvb_event_element_get_type ())
#define DVB_EVENT_ELEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EVENT_ELEMENT, DVBEventElement))
#define DVB_EVENT_ELEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EVENT_ELEMENT, DVBEventElementClass))
#define DVB_IS_EVENT_ELEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EVENT_ELEMENT))
#define DVB_IS_EVENT_ELEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EVENT_ELEMENT))
#define DVB_EVENT_ELEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EVENT_ELEMENT, DVBEventElementClass))

typedef struct _DVBEventElement DVBEventElement;
typedef struct _DVBEventElementClass DVBEventElementClass;
typedef struct _DVBEventElementPrivate DVBEventElementPrivate;

#define DVB_TYPE_ID_BUS_SCHEDULE (dvb_id_bus_schedule_get_type ())
#define DVB_ID_BUS_SCHEDULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_ID_BUS_SCHEDULE, DVBIDBusSchedule))
#define DVB_IS_ID_BUS_SCHEDULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_ID_BUS_SCHEDULE))
#define DVB_ID_BUS_SCHEDULE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_ID_BUS_SCHEDULE, DVBIDBusScheduleIface))

typedef struct _DVBIDBusSchedule DVBIDBusSchedule;
typedef struct _DVBIDBusScheduleIface DVBIDBusScheduleIface;

#define DVB_TYPE_EVENT_INFO (dvb_event_info_get_type ())
typedef struct _DVBEventInfo DVBEventInfo;

#define DVB_TYPE_SCHEDULE (dvb_schedule_get_type ())
#define DVB_SCHEDULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SCHEDULE, DVBSchedule))
#define DVB_SCHEDULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SCHEDULE, DVBScheduleClass))
#define DVB_IS_SCHEDULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SCHEDULE))
#define DVB_IS_SCHEDULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SCHEDULE))
#define DVB_SCHEDULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SCHEDULE, DVBScheduleClass))

typedef struct _DVBSchedule DVBSchedule;
typedef struct _DVBScheduleClass DVBScheduleClass;
typedef struct _DVBSchedulePrivate DVBSchedulePrivate;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_EPG_STORE (dvb_epg_store_get_type ())
#define DVB_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EPG_STORE, DVBEPGStore))
#define DVB_IS_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EPG_STORE))
#define DVB_EPG_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_EPG_STORE, DVBEPGStoreIface))

typedef struct _DVBEPGStore DVBEPGStore;
typedef struct _DVBEPGStoreIface DVBEPGStoreIface;

#define DVB_TYPE_EVENT (dvb_event_get_type ())
#define DVB_EVENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EVENT, DVBEvent))
#define DVB_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EVENT, DVBEventClass))
#define DVB_IS_EVENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EVENT))
#define DVB_IS_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EVENT))
#define DVB_EVENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EVENT, DVBEventClass))

typedef struct _DVBEvent DVBEvent;
typedef struct _DVBEventClass DVBEventClass;
typedef struct _DVBEventPrivate DVBEventPrivate;

#define DVB_EVENT_TYPE_AUDIO_COMPONENT (dvb_event_audio_component_get_type ())
#define DVB_EVENT_AUDIO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponent))
#define DVB_EVENT_AUDIO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponentClass))
#define DVB_EVENT_IS_AUDIO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT))
#define DVB_EVENT_IS_AUDIO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_AUDIO_COMPONENT))
#define DVB_EVENT_AUDIO_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponentClass))

typedef struct _DVBEventAudioComponent DVBEventAudioComponent;
typedef struct _DVBEventAudioComponentClass DVBEventAudioComponentClass;

#define DVB_EVENT_TYPE_VIDEO_COMPONENT (dvb_event_video_component_get_type ())
#define DVB_EVENT_VIDEO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponent))
#define DVB_EVENT_VIDEO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponentClass))
#define DVB_EVENT_IS_VIDEO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT))
#define DVB_EVENT_IS_VIDEO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_VIDEO_COMPONENT))
#define DVB_EVENT_VIDEO_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponentClass))

typedef struct _DVBEventVideoComponent DVBEventVideoComponent;
typedef struct _DVBEventVideoComponentClass DVBEventVideoComponentClass;

#define DVB_EVENT_TYPE_TELETEXT_COMPONENT (dvb_event_teletext_component_get_type ())
#define DVB_EVENT_TELETEXT_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponent))
#define DVB_EVENT_TELETEXT_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponentClass))
#define DVB_EVENT_IS_TELETEXT_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT))
#define DVB_EVENT_IS_TELETEXT_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_TELETEXT_COMPONENT))
#define DVB_EVENT_TELETEXT_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponentClass))

typedef struct _DVBEventTeletextComponent DVBEventTeletextComponent;
typedef struct _DVBEventTeletextComponentClass DVBEventTeletextComponentClass;
typedef struct _DBusObjectVTable _DBusObjectVTable;

/**
     * We don't want to hold the complete information about
     * every event in memory. Just remember id and starttime
     * so we can have a sorted list.
     */
struct _DVBEventElement {
	GObject parent_instance;
	DVBEventElementPrivate * priv;
	guint id;
	gint64 starttime;
};

struct _DVBEventElementClass {
	GObjectClass parent_class;
};

struct _DVBEventInfo {
	guint32 id;
	guint32 next;
	char* name;
	guint duration;
	char* short_description;
};

/* public uint[] local_start; */
struct _DVBIDBusScheduleIface {
	GTypeInterface parent_iface;
	guint32* (*GetAllEvents) (DVBIDBusSchedule* self, int* result_length1);
	DVBEventInfo* (*GetAllEventInfos) (DVBIDBusSchedule* self, int* result_length1);
	DVBEventInfo (*GetInformations) (DVBIDBusSchedule* self, guint32 event_id);
	guint32 (*NowPlaying) (DVBIDBusSchedule* self);
	guint32 (*Next) (DVBIDBusSchedule* self, guint32 event_id);
	char* (*GetName) (DVBIDBusSchedule* self, guint32 event_id);
	char* (*GetShortDescription) (DVBIDBusSchedule* self, guint32 event_id);
	char* (*GetExtendedDescription) (DVBIDBusSchedule* self, guint32 event_id);
	guint (*GetDuration) (DVBIDBusSchedule* self, guint32 event_id);
	guint* (*GetLocalStartTime) (DVBIDBusSchedule* self, guint32 event_id, int* result_length1);
	gint64 (*GetLocalStartTimestamp) (DVBIDBusSchedule* self, guint32 event_id);
	gboolean (*IsRunning) (DVBIDBusSchedule* self, guint32 event_id);
	gboolean (*IsScrambled) (DVBIDBusSchedule* self, guint32 event_id);
};

/**
     * Represents a series of events of a channel
     */
struct _DVBSchedule {
	GObject parent_instance;
	DVBSchedulePrivate * priv;
};

struct _DVBScheduleClass {
	GObjectClass parent_class;
};

struct _DVBEPGStoreIface {
	GTypeInterface parent_iface;
	gboolean (*add_or_update_event) (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
	DVBEvent* (*get_event) (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
	gboolean (*remove_event) (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
	gboolean (*contains_event) (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
	GeeList* (*get_events) (DVBEPGStore* self, guint channel_sid, guint group_id);
	gboolean (*remove_events_of_group) (DVBEPGStore* self, guint group_id);
};

struct _DVBSchedulePrivate {
	DVBChannel* _channel;
	GSequence* events;
	GStaticRecMutex __lock_events;
	GeeMap* event_id_map;
	DVBEPGStore* epgstore;
};

/**
     * Represents an EPG event (i.e. a show with all its information)
     */
struct _DVBEvent {
	GTypeInstance parent_instance;
	volatile int ref_count;
	DVBEventPrivate * priv;
	guint id;
	guint year;
	guint month;
	guint hour;
	guint day;
	guint minute;
	guint second;
	guint duration;
	guint running_status;
	gboolean free_ca_mode;
	char* name;
	char* description;
	char* extended_description;
	GSList* audio_components;
	GSList* video_components;
	GSList* teletext_components;
};

struct _DVBEventClass {
	GTypeClass parent_class;
	void (*finalize) (DVBEvent *self);
};

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};


static gpointer dvb_event_element_parent_class = NULL;
static gpointer dvb_schedule_parent_class = NULL;
static DVBIDBusScheduleIface* dvb_schedule_dvb_id_bus_schedule_parent_iface = NULL;

GType dvb_event_element_get_type (void);
enum  {
	DVB_EVENT_ELEMENT_DUMMY_PROPERTY
};
gint dvb_event_element_compare (DVBEventElement* event1, DVBEventElement* event2);
void dvb_event_element_destroy (void* data);
DVBEventElement* dvb_event_element_new (void);
DVBEventElement* dvb_event_element_construct (GType object_type);
static void dvb_event_element_finalize (GObject* obj);
GType dvb_event_info_get_type (void);
DVBEventInfo* dvb_event_info_dup (const DVBEventInfo* self);
void dvb_event_info_free (DVBEventInfo* self);
void dvb_event_info_copy (const DVBEventInfo* self, DVBEventInfo* dest);
void dvb_event_info_destroy (DVBEventInfo* self);
GType dvb_id_bus_schedule_get_type (void);
GType dvb_schedule_get_type (void);
GType dvb_channel_get_type (void);
gpointer dvb_event_ref (gpointer instance);
void dvb_event_unref (gpointer instance);
GParamSpec* dvb_param_spec_event (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_value_set_event (GValue* value, gpointer v_object);
gpointer dvb_value_get_event (const GValue* value);
GType dvb_event_get_type (void);
GType dvb_epg_store_get_type (void);
#define DVB_SCHEDULE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_SCHEDULE, DVBSchedulePrivate))
enum  {
	DVB_SCHEDULE_DUMMY_PROPERTY,
	DVB_SCHEDULE_CHANNEL
};
static void dvb_schedule_set_channel (DVBSchedule* self, DVBChannel* value);
DVBSchedule* dvb_schedule_new (DVBChannel* channel);
DVBSchedule* dvb_schedule_construct (GType object_type, DVBChannel* channel);
DVBEvent* dvb_schedule_get_event (DVBSchedule* self, guint event_id);
gboolean dvb_event_has_expired (DVBEvent* self);
gboolean dvb_epg_store_remove_event (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
DVBChannel* dvb_schedule_get_channel (DVBSchedule* self);
guint dvb_channel_get_Sid (DVBChannel* self);
guint dvb_channel_get_GroupId (DVBChannel* self);
void dvb_schedule_remove_expired_events (DVBSchedule* self);
DVBEvent* dvb_epg_store_get_event (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
gpointer dvb_event_audio_component_ref (gpointer instance);
void dvb_event_audio_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_audio_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_audio_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_audio_component (const GValue* value);
GType dvb_event_audio_component_get_type (void);
gpointer dvb_event_video_component_ref (gpointer instance);
void dvb_event_video_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_video_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_video_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_video_component (const GValue* value);
GType dvb_event_video_component_get_type (void);
gpointer dvb_event_teletext_component_ref (gpointer instance);
void dvb_event_teletext_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_teletext_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_teletext_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_teletext_component (const GValue* value);
GType dvb_event_teletext_component_get_type (void);
static void dvb_schedule_create_and_add_event_element (DVBSchedule* self, DVBEvent* event);
gboolean dvb_epg_store_add_or_update_event (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
void dvb_schedule_add (DVBSchedule* self, DVBEvent* event);
struct tm dvb_event_get_utc_start_time (DVBEvent* self);
static gint _dvb_event_element_compare_gcompare_data_func (void* a, void* b, gpointer self);
gboolean dvb_schedule_contains (DVBSchedule* self, guint event_id);
gboolean dvb_event_is_running (DVBEvent* self);
DVBEvent* dvb_schedule_get_running_event (DVBSchedule* self);
static guint32* dvb_schedule_real_GetAllEvents (DVBIDBusSchedule* base, int* result_length1);
static void _vala_DVBEventInfo_array_free (DVBEventInfo* array, gint array_length);
static DVBEventInfo* dvb_schedule_real_GetAllEventInfos (DVBIDBusSchedule* base, int* result_length1);
static DVBEventInfo dvb_schedule_real_GetInformations (DVBIDBusSchedule* base, guint32 event_id);
static guint32 dvb_schedule_real_NowPlaying (DVBIDBusSchedule* base);
static guint32 dvb_schedule_real_Next (DVBIDBusSchedule* base, guint32 event_id);
static char* dvb_schedule_real_GetName (DVBIDBusSchedule* base, guint32 event_id);
static char* dvb_schedule_real_GetShortDescription (DVBIDBusSchedule* base, guint32 event_id);
static char* dvb_schedule_real_GetExtendedDescription (DVBIDBusSchedule* base, guint32 event_id);
static guint dvb_schedule_real_GetDuration (DVBIDBusSchedule* base, guint32 event_id);
struct tm dvb_event_get_local_start_time (DVBEvent* self);
static guint* dvb_schedule_to_time_array (const struct tm* local_time, int* result_length1);
static guint* dvb_schedule_real_GetLocalStartTime (DVBIDBusSchedule* base, guint32 event_id, int* result_length1);
static gint64 dvb_schedule_real_GetLocalStartTimestamp (DVBIDBusSchedule* base, guint32 event_id);
static gboolean dvb_schedule_real_IsRunning (DVBIDBusSchedule* base, guint32 event_id);
static gboolean dvb_schedule_real_IsScrambled (DVBIDBusSchedule* base, guint32 event_id);
static void _dvb_event_element_destroy_gdestroy_notify (void* data);
DVBEPGStore* dvb_factory_get_epg_store (void);
GeeList* dvb_epg_store_get_events (DVBEPGStore* self, guint channel_sid, guint group_id);
static GObject * dvb_schedule_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
void dvb_schedule_dbus_register_object (DBusConnection* connection, const char* path, void* object);
void _dvb_schedule_dbus_unregister (DBusConnection* connection, void* user_data);
DBusHandlerResult dvb_schedule_dbus_message (DBusConnection* connection, DBusMessage* message, void* object);
static DBusMessage* _dbus_dvb_schedule_introspect (DVBSchedule* self, DBusConnection* connection, DBusMessage* message);
static void dvb_schedule_finalize (GObject* obj);
static void dvb_schedule_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_schedule_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);
static void _vala_dbus_unregister_object (gpointer connection, GObject* object);

static const DBusObjectPathVTable _dvb_schedule_dbus_path_vtable = {_dvb_schedule_dbus_unregister, dvb_schedule_dbus_message};
static const _DBusObjectVTable _dvb_schedule_dbus_vtable = {dvb_schedule_dbus_register_object};


gint dvb_event_element_compare (DVBEventElement* event1, DVBEventElement* event2) {
	gint result;
	gboolean _tmp0_;
	_tmp0_ = FALSE;
	if (event1 == NULL) {
		_tmp0_ = event2 == NULL;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		result = 0;
		return result;
	} else {
		gboolean _tmp1_;
		_tmp1_ = FALSE;
		if (event1 == NULL) {
			_tmp1_ = event2 != NULL;
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			result = +1;
			return result;
		} else {
			gboolean _tmp2_;
			_tmp2_ = FALSE;
			if (event1 != NULL) {
				_tmp2_ = event2 == NULL;
			} else {
				_tmp2_ = FALSE;
			}
			if (_tmp2_) {
				result = -1;
				return result;
			}
		}
	}
	if (event1->starttime < event2->starttime) {
		result = -1;
		return result;
	} else {
		if (event1->starttime > event2->starttime) {
			result = +1;
			return result;
		} else {
			result = 0;
			return result;
		}
	}
}


void dvb_event_element_destroy (void* data) {
	DVBEventElement* _tmp0_;
	DVBEventElement* e;
	_tmp0_ = NULL;
	e = (_tmp0_ = DVB_EVENT_ELEMENT (data), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
	g_object_unref (e);
	(e == NULL) ? NULL : (e = (g_object_unref (e), NULL));
}


/**
     * We don't want to hold the complete information about
     * every event in memory. Just remember id and starttime
     * so we can have a sorted list.
     */
DVBEventElement* dvb_event_element_construct (GType object_type) {
	DVBEventElement * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBEventElement* dvb_event_element_new (void) {
	return dvb_event_element_construct (DVB_TYPE_EVENT_ELEMENT);
}


static void dvb_event_element_class_init (DVBEventElementClass * klass) {
	dvb_event_element_parent_class = g_type_class_peek_parent (klass);
	G_OBJECT_CLASS (klass)->finalize = dvb_event_element_finalize;
}


static void dvb_event_element_instance_init (DVBEventElement * self) {
}


static void dvb_event_element_finalize (GObject* obj) {
	DVBEventElement * self;
	self = DVB_EVENT_ELEMENT (obj);
	G_OBJECT_CLASS (dvb_event_element_parent_class)->finalize (obj);
}


GType dvb_event_element_get_type (void) {
	static GType dvb_event_element_type_id = 0;
	if (dvb_event_element_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBEventElementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_event_element_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBEventElement), 0, (GInstanceInitFunc) dvb_event_element_instance_init, NULL };
		dvb_event_element_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBEventElement", &g_define_type_info, 0);
	}
	return dvb_event_element_type_id;
}


DVBSchedule* dvb_schedule_construct (GType object_type, DVBChannel* channel) {
	GParameter * __params;
	GParameter * __params_it;
	DVBSchedule * self;
	g_return_val_if_fail (channel != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "channel";
	g_value_init (&__params_it->value, DVB_TYPE_CHANNEL);
	g_value_set_object (&__params_it->value, channel);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBSchedule* dvb_schedule_new (DVBChannel* channel) {
	return dvb_schedule_construct (DVB_TYPE_SCHEDULE, channel);
}


void dvb_schedule_remove_expired_events (DVBSchedule* self) {
	GSList* expired_events;
	g_return_if_fail (self != NULL);
	expired_events = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		{
			gint i;
			i = 0;
			{
				gboolean _tmp0_;
				_tmp0_ = TRUE;
				while (TRUE) {
					GSequenceIter* iter;
					DVBEventElement* _tmp1_;
					DVBEventElement* element;
					DVBEvent* e;
					gboolean _tmp2_;
					if (!_tmp0_) {
						i++;
					}
					_tmp0_ = FALSE;
					if (!(i < g_sequence_get_length (self->priv->events))) {
						break;
					}
					iter = g_sequence_get_iter_at_pos (self->priv->events, i);
					_tmp1_ = NULL;
					element = (_tmp1_ = (DVBEventElement*) g_sequence_get (iter), (_tmp1_ == NULL) ? NULL : g_object_ref (_tmp1_));
					e = dvb_schedule_get_event (self, element->id);
					_tmp2_ = FALSE;
					if (e != NULL) {
						_tmp2_ = dvb_event_has_expired (e);
					} else {
						_tmp2_ = FALSE;
					}
					if (_tmp2_) {
						expired_events = g_slist_prepend (expired_events, iter);
					} else {
						/* events are sorted, all other events didn't expire, too*/
						(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
						(e == NULL) ? NULL : (e = (dvb_event_unref (e), NULL));
						break;
					}
					(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
					(e == NULL) ? NULL : (e = (dvb_event_unref (e), NULL));
				}
			}
		}
		{
			GSList* iter_collection;
			GSList* iter_it;
			iter_collection = expired_events;
			for (iter_it = iter_collection; iter_it != NULL; iter_it = iter_it->next) {
				GSequenceIter* iter;
				iter = (GSequenceIter*) iter_it->data;
				{
					DVBEventElement* _tmp3_;
					DVBEventElement* element;
					g_debug ("Schedule.vala:104: Removing expired event");
					_tmp3_ = NULL;
					element = (_tmp3_ = (DVBEventElement*) g_sequence_get (iter), (_tmp3_ == NULL) ? NULL : g_object_ref (_tmp3_));
					gee_map_remove (self->priv->event_id_map, GUINT_TO_POINTER (element->id));
					g_sequence_remove (iter);
					dvb_epg_store_remove_event (self->priv->epgstore, element->id, dvb_channel_get_Sid (self->priv->_channel), dvb_channel_get_GroupId (self->priv->_channel));
					(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
				}
			}
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	(expired_events == NULL) ? NULL : (expired_events = (g_slist_free (expired_events), NULL));
}


DVBEvent* dvb_schedule_get_event (DVBSchedule* self, guint event_id) {
	DVBEvent* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = dvb_epg_store_get_event (self->priv->epgstore, event_id, dvb_channel_get_Sid (self->priv->_channel), dvb_channel_get_GroupId (self->priv->_channel));
	return result;
}


/**
         * When an event with the same id already exists, it's replaced
         */
void dvb_schedule_add (DVBSchedule* self, DVBEvent* event) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (event != NULL);
	if (dvb_event_has_expired (event)) {
		return;
	}
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (!gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER (event->id))) {
			dvb_schedule_create_and_add_event_element (self, event);
		}
		dvb_epg_store_add_or_update_event (self->priv->epgstore, event, dvb_channel_get_Sid (self->priv->_channel), dvb_channel_get_GroupId (self->priv->_channel));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
}


static gint _dvb_event_element_compare_gcompare_data_func (void* a, void* b, gpointer self) {
	return dvb_event_element_compare (a, b);
}


/**
         * Create event element from @event and add it to list of events
         */
static void dvb_schedule_create_and_add_event_element (DVBSchedule* self, DVBEvent* event) {
	DVBEventElement* element;
	struct tm utc_starttime;
	DVBEventElement* _tmp0_;
	GSequenceIter* iter;
	g_return_if_fail (self != NULL);
	g_return_if_fail (event != NULL);
	element = dvb_event_element_new ();
	element->id = event->id;
	utc_starttime = dvb_event_get_utc_start_time (event);
	element->starttime = (gint64) mktime (&utc_starttime);
	_tmp0_ = NULL;
	iter = g_sequence_insert_sorted (self->priv->events, (_tmp0_ = element, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), _dvb_event_element_compare_gcompare_data_func, NULL);
	gee_map_set (self->priv->event_id_map, GUINT_TO_POINTER (event->id), iter);
	g_assert (g_sequence_get_length (self->priv->events) == gee_map_get_size (self->priv->event_id_map));
	(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
}


gboolean dvb_schedule_contains (DVBSchedule* self, guint event_id) {
	gboolean result;
	gboolean val;
	g_return_val_if_fail (self != NULL, FALSE);
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		val = gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER (event_id));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = val;
	return result;
}


DVBEvent* dvb_schedule_get_running_event (DVBSchedule* self) {
	DVBEvent* result;
	DVBEvent* running_event;
	g_return_val_if_fail (self != NULL, NULL);
	running_event = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		{
			gint i;
			i = 0;
			{
				gboolean _tmp0_;
				_tmp0_ = TRUE;
				while (TRUE) {
					GSequenceIter* iter;
					DVBEventElement* _tmp1_;
					DVBEventElement* element;
					DVBEvent* event;
					gboolean _tmp2_;
					if (!_tmp0_) {
						i++;
					}
					_tmp0_ = FALSE;
					if (!(i < g_sequence_get_length (self->priv->events))) {
						break;
					}
					iter = g_sequence_get_iter_at_pos (self->priv->events, i);
					_tmp1_ = NULL;
					element = (_tmp1_ = (DVBEventElement*) g_sequence_get (iter), (_tmp1_ == NULL) ? NULL : g_object_ref (_tmp1_));
					event = dvb_schedule_get_event (self, element->id);
					_tmp2_ = FALSE;
					if (event != NULL) {
						_tmp2_ = dvb_event_is_running (event);
					} else {
						_tmp2_ = FALSE;
					}
					if (_tmp2_) {
						DVBEvent* _tmp4_;
						DVBEvent* _tmp3_;
						_tmp4_ = NULL;
						_tmp3_ = NULL;
						running_event = (_tmp4_ = (_tmp3_ = event, (_tmp3_ == NULL) ? NULL : dvb_event_ref (_tmp3_)), (running_event == NULL) ? NULL : (running_event = (dvb_event_unref (running_event), NULL)), _tmp4_);
						(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
						(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
						break;
					}
					(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
					(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
				}
			}
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = running_event;
	return result;
}


/*
        public weak Event get_event_around (Time time) {
            return new Event ();
        }*/
static guint32* dvb_schedule_real_GetAllEvents (DVBIDBusSchedule* base, int* result_length1) {
	DVBSchedule * self;
	guint32* result;
	guint32* _tmp1_;
	gint event_ids_size;
	gint event_ids_length1;
	gint _tmp0_;
	guint32* event_ids;
	guint32* _tmp4_;
	self = (DVBSchedule*) base;
	_tmp1_ = NULL;
	event_ids = (_tmp1_ = g_new0 (guint32, _tmp0_ = g_sequence_get_length (self->priv->events)), event_ids_length1 = _tmp0_, event_ids_size = event_ids_length1, _tmp1_);
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		{
			gint i;
			i = 0;
			{
				gboolean _tmp2_;
				_tmp2_ = TRUE;
				while (TRUE) {
					GSequenceIter* iter;
					DVBEventElement* _tmp3_;
					DVBEventElement* element;
					if (!_tmp2_) {
						i++;
					}
					_tmp2_ = FALSE;
					if (!(i < g_sequence_get_length (self->priv->events))) {
						break;
					}
					iter = g_sequence_get_iter_at_pos (self->priv->events, i);
					_tmp3_ = NULL;
					element = (_tmp3_ = (DVBEventElement*) g_sequence_get (iter), (_tmp3_ == NULL) ? NULL : g_object_ref (_tmp3_));
					event_ids[i] = (guint32) element->id;
					(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
				}
			}
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	_tmp4_ = NULL;
	result = (_tmp4_ = event_ids, *result_length1 = event_ids_length1, _tmp4_);
	return result;
	event_ids = (g_free (event_ids), NULL);
}


static void _vala_DVBEventInfo_array_free (DVBEventInfo* array, gint array_length) {
	if (array != NULL) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			dvb_event_info_destroy (&array[i]);
		}
	}
	g_free (array);
}


static DVBEventInfo* dvb_schedule_real_GetAllEventInfos (DVBIDBusSchedule* base, int* result_length1) {
	DVBSchedule * self;
	DVBEventInfo* result;
	DVBEventInfo* _tmp1_;
	gint events_size;
	gint events_length1;
	gint _tmp0_;
	DVBEventInfo* events;
	DVBEventInfo* _tmp12_;
	self = (DVBSchedule*) base;
	_tmp1_ = NULL;
	events = (_tmp1_ = g_new0 (DVBEventInfo, _tmp0_ = g_sequence_get_length (self->priv->events)), events_length1 = _tmp0_, events_size = events_length1, _tmp1_);
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		GSequenceIter* iter;
		iter = g_sequence_get_begin_iter (self->priv->events);
		if (!g_sequence_iter_is_end (iter)) {
			DVBEventElement* _tmp2_;
			DVBEventElement* element;
			gint i;
			_tmp2_ = NULL;
			element = (_tmp2_ = (DVBEventElement*) g_sequence_get (iter), (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_));
			i = 0;
			while (TRUE) {
				DVBEventInfo _tmp3_ = {0};
				DVBEventInfo event_info;
				DVBEvent* event;
				char* _tmp5_;
				const char* _tmp4_;
				char* _tmp7_;
				const char* _tmp6_;
				DVBEventInfo _tmp11_ = {0};
				DVBEventInfo _tmp10_ = {0};
				if (!(!g_sequence_iter_is_end (iter))) {
					break;
				}
				event_info = (memset (&_tmp3_, 0, sizeof (DVBEventInfo)), _tmp3_);
				event = dvb_schedule_get_event (self, element->id);
				event_info.id = (guint32) element->id;
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				event_info.name = (_tmp5_ = (_tmp4_ = event->name, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)), event_info.name = (g_free (event_info.name), NULL), _tmp5_);
				event_info.duration = event->duration;
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				event_info.short_description = (_tmp7_ = (_tmp6_ = event->description, (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_)), event_info.short_description = (g_free (event_info.short_description), NULL), _tmp7_);
				/*
				                        Time local_time = event.get_local_start_time ();
				                        event_info.local_start = to_time_array (local_time);
				                        */
				iter = g_sequence_iter_next (iter);
				if (g_sequence_iter_is_end (iter)) {
					event_info.next = (guint32) 0;
				} else {
					DVBEventElement* _tmp9_;
					DVBEventElement* _tmp8_;
					_tmp9_ = NULL;
					_tmp8_ = NULL;
					element = (_tmp9_ = (_tmp8_ = (DVBEventElement*) g_sequence_get (iter), (_tmp8_ == NULL) ? NULL : g_object_ref (_tmp8_)), (element == NULL) ? NULL : (element = (g_object_unref (element), NULL)), _tmp9_);
					event_info.next = (guint32) element->id;
				}
				events[i] = (_tmp11_ = (dvb_event_info_copy (&event_info, &_tmp10_), _tmp10_), dvb_event_info_destroy (&events[i]), _tmp11_);
				i++;
				dvb_event_info_destroy (&event_info);
				(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
			}
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	_tmp12_ = NULL;
	result = (_tmp12_ = events, *result_length1 = events_length1, _tmp12_);
	return result;
	events = (_vala_DVBEventInfo_array_free (events, events_length1), NULL);
}


static DVBEventInfo dvb_schedule_real_GetInformations (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	DVBEventInfo result;
	DVBEventInfo _tmp0_ = {0};
	DVBEventInfo event_info;
	self = (DVBSchedule*) base;
	event_info = (memset (&_tmp0_, 0, sizeof (DVBEventInfo)), _tmp0_);
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp1_;
			DVBEventElement* element;
			DVBEvent* event;
			char* _tmp3_;
			const char* _tmp2_;
			char* _tmp5_;
			const char* _tmp4_;
			DVBEventElement* _tmp7_;
			DVBEventElement* _tmp6_;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp1_ = NULL;
			element = (_tmp1_ = (DVBEventElement*) g_sequence_get (iter), (_tmp1_ == NULL) ? NULL : g_object_ref (_tmp1_));
			event = dvb_schedule_get_event (self, element->id);
			event_info.id = (guint32) element->id;
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			event_info.name = (_tmp3_ = (_tmp2_ = event->name, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), event_info.name = (g_free (event_info.name), NULL), _tmp3_);
			event_info.duration = event->duration;
			_tmp5_ = NULL;
			_tmp4_ = NULL;
			event_info.short_description = (_tmp5_ = (_tmp4_ = event->description, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)), event_info.short_description = (g_free (event_info.short_description), NULL), _tmp5_);
			/*
			                    Time local_time = event.get_local_start_time ();
			                    event_info.local_start = to_time_array (local_time);
			                    */
			iter = g_sequence_iter_next (iter);
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			element = (_tmp7_ = (_tmp6_ = (DVBEventElement*) g_sequence_get (iter), (_tmp6_ == NULL) ? NULL : g_object_ref (_tmp6_)), (element == NULL) ? NULL : (element = (g_object_unref (element), NULL)), _tmp7_);
			if (g_sequence_iter_is_end (iter)) {
				event_info.next = (guint32) 0;
			} else {
				event_info.next = (guint32) element->id;
			}
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = event_info;
	return result;
}


static guint32 dvb_schedule_real_NowPlaying (DVBIDBusSchedule* base) {
	DVBSchedule * self;
	guint32 result;
	DVBEvent* event;
	guint _tmp0_;
	self = (DVBSchedule*) base;
	event = dvb_schedule_get_running_event (self);
	_tmp0_ = 0U;
	if (event == NULL) {
		_tmp0_ = (guint) 0;
	} else {
		_tmp0_ = event->id;
	}
	result = (guint32) _tmp0_;
	(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
	return result;
}


static guint32 dvb_schedule_real_Next (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	guint32 result;
	guint32 next_event;
	self = (DVBSchedule*) base;
	next_event = (guint32) 0;
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			GSequenceIter* next_iter;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			next_iter = g_sequence_iter_next (iter);
			/* Check if a new event follows*/
			if (!g_sequence_iter_is_end (next_iter)) {
				DVBEventElement* _tmp0_;
				DVBEventElement* element;
				_tmp0_ = NULL;
				element = (_tmp0_ = (DVBEventElement*) g_sequence_get (next_iter), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
				next_event = (guint32) element->id;
				(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			}
		} else {
			g_debug ("Schedule.vala:280: No event with id %u", (guint) event_id);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = next_event;
	return result;
}


static char* dvb_schedule_real_GetName (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	char* result;
	char* name;
	self = (DVBSchedule*) base;
	name = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp0_;
			DVBEventElement* element;
			DVBEvent* event;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp0_ = NULL;
			element = (_tmp0_ = (DVBEventElement*) g_sequence_get (iter), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			event = dvb_schedule_get_event (self, element->id);
			if (event->name != NULL) {
				char* _tmp2_;
				const char* _tmp1_;
				_tmp2_ = NULL;
				_tmp1_ = NULL;
				name = (_tmp2_ = (_tmp1_ = event->name, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), name = (g_free (name), NULL), _tmp2_);
			}
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		} else {
			g_debug ("Schedule.vala:298: No event with id %u", (guint) event_id);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = name;
	return result;
}


static char* dvb_schedule_real_GetShortDescription (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	char* result;
	char* desc;
	self = (DVBSchedule*) base;
	desc = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp0_;
			DVBEventElement* element;
			DVBEvent* event;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp0_ = NULL;
			element = (_tmp0_ = (DVBEventElement*) g_sequence_get (iter), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			event = dvb_schedule_get_event (self, element->id);
			if (event->description != NULL) {
				char* _tmp2_;
				const char* _tmp1_;
				_tmp2_ = NULL;
				_tmp1_ = NULL;
				desc = (_tmp2_ = (_tmp1_ = event->description, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), desc = (g_free (desc), NULL), _tmp2_);
			}
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		} else {
			g_debug ("Schedule.vala:316: No event with id %u", (guint) event_id);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = desc;
	return result;
}


static char* dvb_schedule_real_GetExtendedDescription (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	char* result;
	char* desc;
	self = (DVBSchedule*) base;
	desc = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp0_;
			DVBEventElement* element;
			DVBEvent* event;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp0_ = NULL;
			element = (_tmp0_ = (DVBEventElement*) g_sequence_get (iter), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			event = dvb_schedule_get_event (self, element->id);
			if (event->extended_description != NULL) {
				char* _tmp2_;
				const char* _tmp1_;
				_tmp2_ = NULL;
				_tmp1_ = NULL;
				desc = (_tmp2_ = (_tmp1_ = event->extended_description, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), desc = (g_free (desc), NULL), _tmp2_);
			}
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		} else {
			g_debug ("Schedule.vala:334: No event with id %u", (guint) event_id);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = desc;
	return result;
}


static guint dvb_schedule_real_GetDuration (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	guint result;
	guint duration;
	self = (DVBSchedule*) base;
	duration = (guint) 0;
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp0_;
			DVBEventElement* element;
			DVBEvent* event;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp0_ = NULL;
			element = (_tmp0_ = (DVBEventElement*) g_sequence_get (iter), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			event = dvb_schedule_get_event (self, element->id);
			duration = event->duration;
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		} else {
			g_debug ("Schedule.vala:351: No event with id %u", (guint) event_id);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = duration;
	return result;
}


static guint* dvb_schedule_real_GetLocalStartTime (DVBIDBusSchedule* base, guint32 event_id, int* result_length1) {
	DVBSchedule * self;
	guint* result;
	guint* _tmp1_;
	gint start_size;
	gint start_length1;
	guint* _tmp0_;
	guint* start;
	guint* _tmp5_;
	self = (DVBSchedule*) base;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	start = (_tmp1_ = (_tmp0_ = g_new0 (guint, 0), _tmp0_), start_length1 = 0, start_size = start_length1, _tmp1_);
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp2_;
			DVBEventElement* element;
			DVBEvent* event;
			struct tm local_time;
			guint* _tmp4_;
			gint _tmp3_;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp2_ = NULL;
			element = (_tmp2_ = (DVBEventElement*) g_sequence_get (iter), (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_));
			event = dvb_schedule_get_event (self, element->id);
			local_time = dvb_event_get_local_start_time (event);
			_tmp4_ = NULL;
			start = (_tmp4_ = dvb_schedule_to_time_array (&local_time, &_tmp3_), start = (g_free (start), NULL), start_length1 = _tmp3_, start_size = start_length1, _tmp4_);
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		} else {
			g_debug ("Schedule.vala:369: No event with id %u", (guint) event_id);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	_tmp5_ = NULL;
	result = (_tmp5_ = start, *result_length1 = start_length1, _tmp5_);
	return result;
	start = (g_free (start), NULL);
}


static gint64 dvb_schedule_real_GetLocalStartTimestamp (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	gint64 result;
	gint64 ret;
	self = (DVBSchedule*) base;
	ret = (gint64) 0;
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp0_;
			DVBEventElement* element;
			DVBEvent* event;
			struct tm local_time;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp0_ = NULL;
			element = (_tmp0_ = (DVBEventElement*) g_sequence_get (iter), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			event = dvb_schedule_get_event (self, element->id);
			local_time = dvb_event_get_local_start_time (event);
			ret = (gint64) mktime (&local_time);
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = ret;
	return result;
}


static gboolean dvb_schedule_real_IsRunning (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	gboolean result;
	gboolean val;
	self = (DVBSchedule*) base;
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp0_;
			DVBEventElement* element;
			DVBEvent* event;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp0_ = NULL;
			element = (_tmp0_ = (DVBEventElement*) g_sequence_get (iter), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			event = dvb_schedule_get_event (self, element->id);
			val = dvb_event_is_running (event);
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		} else {
			g_debug ("Schedule.vala:400: No event with id %u", (guint) event_id);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = val;
	return result;
}


static gboolean dvb_schedule_real_IsScrambled (DVBIDBusSchedule* base, guint32 event_id) {
	DVBSchedule * self;
	gboolean result;
	gboolean val;
	self = (DVBSchedule*) base;
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_events);
	{
		if (gee_map_contains (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id))) {
			GSequenceIter* iter;
			DVBEventElement* _tmp0_;
			DVBEventElement* element;
			DVBEvent* event;
			iter = (GSequenceIter*) gee_map_get (self->priv->event_id_map, GUINT_TO_POINTER ((guint) event_id));
			_tmp0_ = NULL;
			element = (_tmp0_ = (DVBEventElement*) g_sequence_get (iter), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			event = dvb_schedule_get_event (self, element->id);
			val = !event->free_ca_mode;
			(element == NULL) ? NULL : (element = (g_object_unref (element), NULL));
			(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
		} else {
			g_debug ("Schedule.vala:417: No event with id %u", (guint) event_id);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_events);
	result = val;
	return result;
}


static guint* dvb_schedule_to_time_array (const struct tm* local_time, int* result_length1) {
	guint* result;
	guint* _tmp0_;
	gint start_size;
	gint start_length1;
	guint* start;
	guint* _tmp1_;
	_tmp0_ = NULL;
	start = (_tmp0_ = g_new0 (guint, 6), start_length1 = 6, start_size = start_length1, _tmp0_);
	start[0] = (guint) ((*local_time).tm_year + 1900);
	start[1] = (guint) ((*local_time).tm_mon + 1);
	start[2] = (guint) (*local_time).tm_mday;
	start[3] = (guint) (*local_time).tm_hour;
	start[4] = (guint) (*local_time).tm_min;
	start[5] = (guint) (*local_time).tm_sec;
	_tmp1_ = NULL;
	result = (_tmp1_ = start, *result_length1 = start_length1, _tmp1_);
	return result;
	start = (g_free (start), NULL);
}


DVBChannel* dvb_schedule_get_channel (DVBSchedule* self) {
	DVBChannel* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_channel;
	return result;
}


static void dvb_schedule_set_channel (DVBSchedule* self, DVBChannel* value) {
	g_return_if_fail (self != NULL);
	self->priv->_channel = value;
	g_object_notify ((GObject *) self, "channel");
}


static void _dvb_event_element_destroy_gdestroy_notify (void* data) {
	dvb_event_element_destroy (data);
}


static GObject * dvb_schedule_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBScheduleClass * klass;
	GObjectClass * parent_class;
	DVBSchedule * self;
	klass = DVB_SCHEDULE_CLASS (g_type_class_peek (DVB_TYPE_SCHEDULE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_SCHEDULE (obj);
	{
		GSequence* _tmp0_;
		GeeMap* _tmp1_;
		GeeList* events;
		_tmp0_ = NULL;
		self->priv->events = (_tmp0_ = g_sequence_new (_dvb_event_element_destroy_gdestroy_notify), (self->priv->events == NULL) ? NULL : (self->priv->events = (g_sequence_free (self->priv->events), NULL)), _tmp0_);
		_tmp1_ = NULL;
		self->priv->event_id_map = (_tmp1_ = (GeeMap*) gee_hash_map_new (G_TYPE_UINT, NULL, NULL, G_TYPE_POINTER, NULL, NULL, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->event_id_map == NULL) ? NULL : (self->priv->event_id_map = (g_object_unref (self->priv->event_id_map), NULL)), _tmp1_);
		self->priv->epgstore = dvb_factory_get_epg_store ();
		events = dvb_epg_store_get_events (self->priv->epgstore, dvb_channel_get_Sid (self->priv->_channel), dvb_channel_get_GroupId (self->priv->_channel));
		{
			GeeIterator* _event_it;
			_event_it = gee_iterable_iterator ((GeeIterable*) events);
			while (TRUE) {
				DVBEvent* event;
				if (!gee_iterator_next (_event_it)) {
					break;
				}
				event = (DVBEvent*) gee_iterator_get (_event_it);
				if (dvb_event_has_expired (event)) {
					dvb_epg_store_remove_event (self->priv->epgstore, event->id, dvb_channel_get_Sid (self->priv->_channel), dvb_channel_get_GroupId (self->priv->_channel));
				} else {
					dvb_schedule_create_and_add_event_element (self, event);
				}
				(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
			}
			(_event_it == NULL) ? NULL : (_event_it = (g_object_unref (_event_it), NULL));
		}
		(events == NULL) ? NULL : (events = (g_object_unref (events), NULL));
	}
	return obj;
}


void _dvb_schedule_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_schedule_introspect (DVBSchedule* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.Schedule\">\n  <method name=\"GetAllEvents\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetAllEventInfos\">\n    <arg name=\"result\" type=\"a(uusus)\" direction=\"out\"/>\n  </method>\n  <method name=\"GetInformations\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"(uusus)\" direction=\"out\"/>\n  </method>\n  <method name=\"NowPlaying\">\n    <arg name=\"result\" type=\"u\" direction=\"out\"/>\n  </method>\n  <method name=\"Next\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"u\" direction=\"out\"/>\n  </method>\n  <method name=\"GetName\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetShortDescription\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetExtendedDescription\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDuration\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"u\" direction=\"out\"/>\n  </method>\n  <method name=\"GetLocalStartTime\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetLocalStartTimestamp\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"x\" direction=\"out\"/>\n  </method>\n  <method name=\"IsRunning\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"IsScrambled\">\n    <arg name=\"event_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	dbus_free_string_array (children);
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


DBusHandlerResult dvb_schedule_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_schedule_introspect (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if (dvb_id_bus_schedule_dbus_message (connection, message, object) == DBUS_HANDLER_RESULT_HANDLED) {
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


void dvb_schedule_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_schedule_dbus_path_vtable, object);
		g_object_weak_ref (object, _vala_dbus_unregister_object, connection);
	}
	dvb_id_bus_schedule_dbus_register_object (connection, path, object);
}


static void dvb_schedule_class_init (DVBScheduleClass * klass) {
	dvb_schedule_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBSchedulePrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_schedule_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_schedule_set_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_schedule_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_schedule_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_SCHEDULE_CHANNEL, g_param_spec_object ("channel", "channel", "channel", DVB_TYPE_CHANNEL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_type_set_qdata (DVB_TYPE_SCHEDULE, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_schedule_dbus_vtable));
}


static void dvb_schedule_dvb_id_bus_schedule_interface_init (DVBIDBusScheduleIface * iface) {
	dvb_schedule_dvb_id_bus_schedule_parent_iface = g_type_interface_peek_parent (iface);
	iface->GetAllEvents = dvb_schedule_real_GetAllEvents;
	iface->GetAllEventInfos = dvb_schedule_real_GetAllEventInfos;
	iface->GetInformations = dvb_schedule_real_GetInformations;
	iface->NowPlaying = dvb_schedule_real_NowPlaying;
	iface->Next = dvb_schedule_real_Next;
	iface->GetName = dvb_schedule_real_GetName;
	iface->GetShortDescription = dvb_schedule_real_GetShortDescription;
	iface->GetExtendedDescription = dvb_schedule_real_GetExtendedDescription;
	iface->GetDuration = dvb_schedule_real_GetDuration;
	iface->GetLocalStartTime = dvb_schedule_real_GetLocalStartTime;
	iface->GetLocalStartTimestamp = dvb_schedule_real_GetLocalStartTimestamp;
	iface->IsRunning = dvb_schedule_real_IsRunning;
	iface->IsScrambled = dvb_schedule_real_IsScrambled;
}


static void dvb_schedule_instance_init (DVBSchedule * self) {
	self->priv = DVB_SCHEDULE_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_events);
}


static void dvb_schedule_finalize (GObject* obj) {
	DVBSchedule * self;
	self = DVB_SCHEDULE (obj);
	g_static_rec_mutex_free (&self->priv->__lock_events);
	(self->priv->events == NULL) ? NULL : (self->priv->events = (g_sequence_free (self->priv->events), NULL));
	(self->priv->event_id_map == NULL) ? NULL : (self->priv->event_id_map = (g_object_unref (self->priv->event_id_map), NULL));
	G_OBJECT_CLASS (dvb_schedule_parent_class)->finalize (obj);
}


GType dvb_schedule_get_type (void) {
	static GType dvb_schedule_type_id = 0;
	if (dvb_schedule_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBScheduleClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_schedule_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBSchedule), 0, (GInstanceInitFunc) dvb_schedule_instance_init, NULL };
		static const GInterfaceInfo dvb_id_bus_schedule_info = { (GInterfaceInitFunc) dvb_schedule_dvb_id_bus_schedule_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_schedule_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBSchedule", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_schedule_type_id, DVB_TYPE_ID_BUS_SCHEDULE, &dvb_id_bus_schedule_info);
	}
	return dvb_schedule_type_id;
}


static void dvb_schedule_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBSchedule * self;
	gpointer boxed;
	self = DVB_SCHEDULE (object);
	switch (property_id) {
		case DVB_SCHEDULE_CHANNEL:
		g_value_set_object (value, dvb_schedule_get_channel (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_schedule_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBSchedule * self;
	self = DVB_SCHEDULE (object);
	switch (property_id) {
		case DVB_SCHEDULE_CHANNEL:
		dvb_schedule_set_channel (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}


static void _vala_dbus_unregister_object (gpointer connection, GObject* object) {
	char* path;
	path = g_object_steal_data ((GObject*) object, "dbus_object_path");
	dbus_connection_unregister_object_path (connection, path);
	g_free (path);
}




