/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <sqlite3.h>
#include <gio/gio.h>
#include <stdlib.h>
#include <string.h>


#define DVB_TYPE_CONFIG_STORE (dvb_config_store_get_type ())
#define DVB_CONFIG_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CONFIG_STORE, DVBConfigStore))
#define DVB_IS_CONFIG_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CONFIG_STORE))
#define DVB_CONFIG_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_CONFIG_STORE, DVBConfigStoreIface))

typedef struct _DVBConfigStore DVBConfigStore;
typedef struct _DVBConfigStoreIface DVBConfigStoreIface;

#define DVB_TYPE_DEVICE_GROUP (dvb_device_group_get_type ())
#define DVB_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroup))
#define DVB_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))
#define DVB_IS_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE_GROUP))
#define DVB_IS_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE_GROUP))
#define DVB_DEVICE_GROUP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))

typedef struct _DVBDeviceGroup DVBDeviceGroup;
typedef struct _DVBDeviceGroupClass DVBDeviceGroupClass;

#define DVB_TYPE_DEVICE (dvb_device_get_type ())
#define DVB_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE, DVBDevice))
#define DVB_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE, DVBDeviceClass))
#define DVB_IS_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE))
#define DVB_IS_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE))
#define DVB_DEVICE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE, DVBDeviceClass))

typedef struct _DVBDevice DVBDevice;
typedef struct _DVBDeviceClass DVBDeviceClass;

#define DVB_TYPE_TIMERS_STORE (dvb_timers_store_get_type ())
#define DVB_TIMERS_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TIMERS_STORE, DVBTimersStore))
#define DVB_IS_TIMERS_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TIMERS_STORE))
#define DVB_TIMERS_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_TIMERS_STORE, DVBTimersStoreIface))

typedef struct _DVBTimersStore DVBTimersStore;
typedef struct _DVBTimersStoreIface DVBTimersStoreIface;

#define DVB_TYPE_TIMER (dvb_timer_get_type ())
#define DVB_TIMER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TIMER, DVBTimer))
#define DVB_TIMER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_TIMER, DVBTimerClass))
#define DVB_IS_TIMER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TIMER))
#define DVB_IS_TIMER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_TIMER))
#define DVB_TIMER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_TIMER, DVBTimerClass))

typedef struct _DVBTimer DVBTimer;
typedef struct _DVBTimerClass DVBTimerClass;

#define DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE (dvb_sqlite_config_timers_store_get_type ())
#define DVB_SQLITE_CONFIG_TIMERS_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE, DVBSqliteConfigTimersStore))
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE, DVBSqliteConfigTimersStoreClass))
#define DVB_IS_SQLITE_CONFIG_TIMERS_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE))
#define DVB_IS_SQLITE_CONFIG_TIMERS_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE))
#define DVB_SQLITE_CONFIG_TIMERS_STORE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE, DVBSqliteConfigTimersStoreClass))

typedef struct _DVBSqliteConfigTimersStore DVBSqliteConfigTimersStore;
typedef struct _DVBSqliteConfigTimersStoreClass DVBSqliteConfigTimersStoreClass;
typedef struct _DVBSqliteConfigTimersStorePrivate DVBSqliteConfigTimersStorePrivate;

#define DVB_TYPE_ADAPTER_TYPE (dvb_adapter_type_get_type ())

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

struct _DVBConfigStoreIface {
	GTypeInterface parent_iface;
	GeeList* (*get_all_device_groups) (DVBConfigStore* self);
	gboolean (*add_device_group) (DVBConfigStore* self, DVBDeviceGroup* dev_group);
	gboolean (*remove_device_group) (DVBConfigStore* self, DVBDeviceGroup* devgroup);
	gboolean (*add_device_to_group) (DVBConfigStore* self, DVBDevice* dev, DVBDeviceGroup* devgroup);
	gboolean (*remove_device_from_group) (DVBConfigStore* self, DVBDevice* dev, DVBDeviceGroup* devgroup);
	gboolean (*update_from_group) (DVBConfigStore* self, DVBDeviceGroup* devgroup);
};

struct _DVBTimersStoreIface {
	GTypeInterface parent_iface;
	GeeList* (*get_all_timers_of_device_group) (DVBTimersStore* self, DVBDeviceGroup* dev);
	gboolean (*add_timer_to_device_group) (DVBTimersStore* self, DVBTimer* timer, DVBDeviceGroup* dev);
	gboolean (*remove_timer_from_device_group) (DVBTimersStore* self, guint timer_id, DVBDeviceGroup* dev);
};

struct _DVBSqliteConfigTimersStore {
	GObject parent_instance;
	DVBSqliteConfigTimersStorePrivate * priv;
};

struct _DVBSqliteConfigTimersStoreClass {
	GObjectClass parent_class;
};

struct _DVBSqliteConfigTimersStorePrivate {
	sqlite3_stmt* select_devices_statement;
	sqlite3_stmt* delete_group_statement;
	sqlite3_stmt* insert_group_statement;
	sqlite3_stmt* update_group_statement;
	sqlite3_stmt* delete_device_statement;
	sqlite3_stmt* delete_group_devices_statement;
	sqlite3_stmt* insert_device_statement;
	sqlite3_stmt* select_timers_statement;
	sqlite3_stmt* delete_timer_statement;
	sqlite3_stmt* insert_timer_statement;
	sqlite3_stmt* contains_group_statement;
	sqlite3_stmt* contains_timer_statement;
	sqlite3* db;
};

typedef enum  {
	DVB_ADAPTER_TYPE_UNKNOWN,
	DVB_ADAPTER_TYPE_DVB_T,
	DVB_ADAPTER_TYPE_DVB_S,
	DVB_ADAPTER_TYPE_DVB_C
} DVBAdapterType;


static gpointer dvb_sqlite_config_timers_store_parent_class = NULL;
static DVBConfigStoreIface* dvb_sqlite_config_timers_store_dvb_config_store_parent_iface = NULL;
static DVBTimersStoreIface* dvb_sqlite_config_timers_store_dvb_timers_store_parent_iface = NULL;

GType dvb_device_group_get_type (void);
GType dvb_device_get_type (void);
GType dvb_config_store_get_type (void);
GType dvb_timer_get_type (void);
GType dvb_timers_store_get_type (void);
GType dvb_sqlite_config_timers_store_get_type (void);
#define DVB_SQLITE_CONFIG_TIMERS_STORE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE, DVBSqliteConfigTimersStorePrivate))
enum  {
	DVB_SQLITE_CONFIG_TIMERS_STORE_DUMMY_PROPERTY
};
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_DEVICE_GROUPS "CREATE TABLE device_groups (\n        group_id INTEGER,\n        adapter_type INTEGER(1),\n        channels_file VARCHAR(255),\n        recordings_dir VARCHAR(255),\n        name VARCHAR(255),\n        PRIMARY KEY(group_id))"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_DEVICES "CREATE TABLE devices (\n        group_id INTEGER,\n        adapter INTEGER,\n        frontend INTEGER,\n        PRIMARY KEY(adapter, frontend))"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_TIMERS "CREATE TABLE timers (\n        timer_id INTEGER,\n        group_id INTEGER,\n        channel_sid INTEGER,\n        year INTEGER,\n        month INTEGER,\n        day INTEGER,\n        hour INTEGER,\n        minute INTEGER,\n        duration INTEGER,\n        PRIMARY KEY(timer_id))"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_ALL_GROUPS "SELECT * FROM device_groups"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_DEVICES "SELECT * FROM devices WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_GROUP "DELETE FROM device_groups WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_GROUP "INSERT INTO device_groups VALUES (?, ?, ?, ?, ?)"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CONTAINS_GROUP "SELECT COUNT(*) FROM device_groups WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_UPDATE_GROUP "UPDATE device_groups SET adapter_type=?, channels_file=?, recordings_dir=?, name=? WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_DEVICE "DELETE FROM devices WHERE adapter=? AND frontend=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_GROUP_DEVICES "DELETE FROM devices WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_DEVICE "INSERT INTO devices VALUES (?, ?, ?)"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_TIMERS "SELECT * FROM timers WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_TIMER "DELETE FROM timers WHERE timer_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_TIMER "INSERT INTO timers VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CONTAINS_TIMER "SELECT COUNT(*) FROM timers WHERE timer_id=?"
static void dvb_sqlite_config_timers_store_print_last_error (DVBSqliteConfigTimersStore* self);
DVBDevice* dvb_device_new (guint adapter, guint frontend, gboolean get_type_and_name);
DVBDevice* dvb_device_construct (GType object_type, guint adapter, guint frontend, gboolean get_type_and_name);
GType dvb_adapter_type_get_type (void);
GType dvb_channel_list_get_type (void);
DVBChannelList* dvb_channel_list_restore_from_file (GFile* channelsfile, DVBAdapterType type, guint group_id, GError** error);
void dvb_device_set_Channels (DVBDevice* self, DVBChannelList* value);
void dvb_device_set_RecordingsDirectory (DVBDevice* self, GFile* value);
gboolean main_get_disable_epg_scanner (void);
DVBDeviceGroup* dvb_device_group_new (guint id, DVBDevice* reference_device, gboolean with_epg_scanner);
DVBDeviceGroup* dvb_device_group_construct (GType object_type, guint id, DVBDevice* reference_device, gboolean with_epg_scanner);
void dvb_device_group_set_Name (DVBDeviceGroup* self, const char* value);
gboolean dvb_device_group_add (DVBDeviceGroup* self, DVBDevice* device);
static GeeList* dvb_sqlite_config_timers_store_real_get_all_device_groups (DVBConfigStore* base);
gboolean dvb_sqlite_config_timers_store_contains_group (DVBSqliteConfigTimersStore* self, guint group_id);
guint dvb_device_group_get_Id (DVBDeviceGroup* self);
DVBChannelList* dvb_device_group_get_Channels (DVBDeviceGroup* self);
GFile* dvb_channel_list_get_channels_file (DVBChannelList* self);
GFile* dvb_device_group_get_RecordingsDirectory (DVBDeviceGroup* self);
DVBAdapterType dvb_device_group_get_Type (DVBDeviceGroup* self);
const char* dvb_device_group_get_Name (DVBDeviceGroup* self);
gboolean dvb_config_store_add_device_to_group (DVBConfigStore* self, DVBDevice* dev, DVBDeviceGroup* devgroup);
static gboolean dvb_sqlite_config_timers_store_real_add_device_group (DVBConfigStore* base, DVBDeviceGroup* dev_group);
static gboolean dvb_sqlite_config_timers_store_real_remove_device_group (DVBConfigStore* base, DVBDeviceGroup* devgroup);
guint dvb_device_get_Adapter (DVBDevice* self);
guint dvb_device_get_Frontend (DVBDevice* self);
static gboolean dvb_sqlite_config_timers_store_real_add_device_to_group (DVBConfigStore* base, DVBDevice* dev, DVBDeviceGroup* devgroup);
static gboolean dvb_sqlite_config_timers_store_real_remove_device_from_group (DVBConfigStore* base, DVBDevice* dev, DVBDeviceGroup* devgroup);
GType dvb_channel_get_type (void);
DVBChannel* dvb_channel_list_get_channel (DVBChannelList* self, guint sid);
DVBTimer* dvb_timer_new (guint32 id, DVBChannel* channel, gint year, gint month, gint day, gint hour, gint minute, guint duration, const char* name);
DVBTimer* dvb_timer_construct (GType object_type, guint32 id, DVBChannel* channel, gint year, gint month, gint day, gint hour, gint minute, guint duration, const char* name);
static GeeList* dvb_sqlite_config_timers_store_real_get_all_timers_of_device_group (DVBTimersStore* base, DVBDeviceGroup* dev);
gboolean dvb_sqlite_config_timers_store_contains_timer (DVBSqliteConfigTimersStore* self, guint timer_id);
guint32 dvb_timer_get_Id (DVBTimer* self);
DVBChannel* dvb_timer_get_Channel (DVBTimer* self);
guint dvb_channel_get_Sid (DVBChannel* self);
guint dvb_timer_get_Year (DVBTimer* self);
guint dvb_timer_get_Month (DVBTimer* self);
guint dvb_timer_get_Day (DVBTimer* self);
guint dvb_timer_get_Hour (DVBTimer* self);
guint dvb_timer_get_Minute (DVBTimer* self);
guint dvb_timer_get_Duration (DVBTimer* self);
static gboolean dvb_sqlite_config_timers_store_real_add_timer_to_device_group (DVBTimersStore* base, DVBTimer* timer, DVBDeviceGroup* dev);
static gboolean dvb_sqlite_config_timers_store_real_remove_timer_from_device_group (DVBTimersStore* base, guint timer_id, DVBDeviceGroup* dev);
static gboolean dvb_sqlite_config_timers_store_real_update_from_group (DVBConfigStore* base, DVBDeviceGroup* devgroup);
void dvb_utils_mkdirs (GFile* directory, GError** error);
static sqlite3* dvb_sqlite_config_timers_store_get_db_handler (void);
DVBSqliteConfigTimersStore* dvb_sqlite_config_timers_store_new (void);
DVBSqliteConfigTimersStore* dvb_sqlite_config_timers_store_construct (GType object_type);
static GObject * dvb_sqlite_config_timers_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static void dvb_sqlite_config_timers_store_finalize (GObject* obj);



static GeeList* dvb_sqlite_config_timers_store_real_get_all_device_groups (DVBConfigStore* base) {
	DVBSqliteConfigTimersStore * self;
	GeeList* result;
	GError * _inner_error_;
	GeeList* groups;
	sqlite3_stmt* statement;
	sqlite3_stmt* _tmp2_;
	gint _tmp1_;
	sqlite3_stmt* _tmp0_;
	self = (DVBSqliteConfigTimersStore*) base;
	_inner_error_ = NULL;
	groups = (GeeList*) gee_array_list_new (DVB_TYPE_DEVICE_GROUP, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	statement = NULL;
	_tmp2_ = NULL;
	_tmp0_ = NULL;
	if ((_tmp1_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_ALL_GROUPS, -1, &_tmp0_, NULL), statement = (_tmp2_ = _tmp0_, (statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL)), _tmp2_), _tmp1_) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = groups;
		(statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL));
		return result;
	}
	while (TRUE) {
		gint group_id;
		GeeList* devs;
		gint group_type;
		DVBAdapterType type;
		GFile* channels_file;
		DVBChannelList* channels;
		GFile* rec_dir;
		DVBDevice* ref_dev;
		DVBDeviceGroup* group;
		if (!(sqlite3_step (statement) == SQLITE_ROW)) {
			break;
		}
		group_id = sqlite3_column_int (statement, 0);
		sqlite3_reset (self->priv->select_devices_statement);
		if (sqlite3_bind_int (self->priv->select_devices_statement, 1, group_id) != SQLITE_OK) {
			dvb_sqlite_config_timers_store_print_last_error (self);
			continue;
		}
		/* Get devices of group*/
		devs = (GeeList*) gee_array_list_new (DVB_TYPE_DEVICE, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
		while (TRUE) {
			guint adapter;
			guint frontend;
			DVBDevice* _tmp3_;
			if (!(sqlite3_step (self->priv->select_devices_statement) == SQLITE_ROW)) {
				break;
			}
			adapter = (guint) sqlite3_column_int (self->priv->select_devices_statement, 1);
			frontend = (guint) sqlite3_column_int (self->priv->select_devices_statement, 2);
			/* Create new device*/
			_tmp3_ = NULL;
			gee_collection_add ((GeeCollection*) devs, _tmp3_ = dvb_device_new (adapter, frontend, TRUE));
			(_tmp3_ == NULL) ? NULL : (_tmp3_ = (g_object_unref (_tmp3_), NULL));
		}
		/* No devices for this group*/
		if (gee_collection_get_size ((GeeCollection*) devs) == 0) {
			g_debug ("SqliteConfigTimersStore.vala:174: Group %d has no devices", group_id);
			(devs == NULL) ? NULL : (devs = (g_object_unref (devs), NULL));
			continue;
		}
		/* Get adapter type*/
		group_type = sqlite3_column_int (statement, 1);
		type = 0;
		switch (group_type) {
			case 1:
			{
				type = DVB_ADAPTER_TYPE_DVB_T;
				break;
			}
			case 2:
			{
				type = DVB_ADAPTER_TYPE_DVB_S;
				break;
			}
			case 3:
			{
				type = DVB_ADAPTER_TYPE_DVB_C;
				break;
			}
			default:
			{
				g_debug ("SqliteConfigTimersStore.vala:186: Group %d has unknown type %d", group_id, group_type);
				continue;
			}
		}
		/* Get channel list*/
		channels_file = g_file_new_for_path (sqlite3_column_text (statement, 2));
		channels = NULL;
		{
			DVBChannelList* _tmp4_;
			DVBChannelList* _tmp5_;
			_tmp4_ = dvb_channel_list_restore_from_file (channels_file, type, (guint) group_id, &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch33_g_error;
				goto __finally33;
			}
			_tmp5_ = NULL;
			channels = (_tmp5_ = _tmp4_, (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp5_);
		}
		goto __finally33;
		__catch33_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				g_warning ("SqliteConfigTimersStore.vala:199: Could not read channels: %s", e->message);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				(devs == NULL) ? NULL : (devs = (g_object_unref (devs), NULL));
				(channels_file == NULL) ? NULL : (channels_file = (g_object_unref (channels_file), NULL));
				(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
				continue;
			}
		}
		__finally33:
		if (_inner_error_ != NULL) {
			(devs == NULL) ? NULL : (devs = (g_object_unref (devs), NULL));
			(channels_file == NULL) ? NULL : (channels_file = (g_object_unref (channels_file), NULL));
			(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
			(groups == NULL) ? NULL : (groups = (g_object_unref (groups), NULL));
			(statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return NULL;
		}
		rec_dir = g_file_new_for_path (sqlite3_column_text (statement, 3));
		/* Set reference device*/
		ref_dev = (DVBDevice*) gee_list_get (devs, 0);
		dvb_device_set_Channels (ref_dev, channels);
		dvb_device_set_RecordingsDirectory (ref_dev, rec_dir);
		/* Create device group*/
		group = dvb_device_group_new ((guint) group_id, ref_dev, !main_get_disable_epg_scanner ());
		dvb_device_group_set_Name (group, sqlite3_column_text (statement, 4));
		{
			gint i;
			i = 1;
			{
				gboolean _tmp6_;
				_tmp6_ = TRUE;
				while (TRUE) {
					DVBDevice* _tmp7_;
					if (!_tmp6_) {
						i++;
					}
					_tmp6_ = FALSE;
					if (!(i < gee_collection_get_size ((GeeCollection*) devs))) {
						break;
					}
					_tmp7_ = NULL;
					dvb_device_group_add (group, _tmp7_ = (DVBDevice*) gee_list_get (devs, i));
					(_tmp7_ == NULL) ? NULL : (_tmp7_ = (g_object_unref (_tmp7_), NULL));
				}
			}
		}
		gee_collection_add ((GeeCollection*) groups, group);
		(devs == NULL) ? NULL : (devs = (g_object_unref (devs), NULL));
		(channels_file == NULL) ? NULL : (channels_file = (g_object_unref (channels_file), NULL));
		(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
		(rec_dir == NULL) ? NULL : (rec_dir = (g_object_unref (rec_dir), NULL));
		(ref_dev == NULL) ? NULL : (ref_dev = (g_object_unref (ref_dev), NULL));
		(group == NULL) ? NULL : (group = (g_object_unref (group), NULL));
	}
	result = groups;
	(statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL));
	return result;
}


static gboolean dvb_sqlite_config_timers_store_real_add_device_group (DVBConfigStore* base, DVBDeviceGroup* dev_group) {
	DVBSqliteConfigTimersStore * self;
	gboolean result;
	char* channels;
	char* recdir;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	gboolean _tmp3_;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev_group != NULL, FALSE);
	if (dvb_sqlite_config_timers_store_contains_group (self, dvb_device_group_get_Id (dev_group))) {
		result = FALSE;
		return result;
	}
	channels = g_file_get_path (dvb_channel_list_get_channels_file (dvb_device_group_get_Channels (dev_group)));
	recdir = g_file_get_path (dvb_device_group_get_RecordingsDirectory (dev_group));
	sqlite3_reset (self->priv->insert_group_statement);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	if (sqlite3_bind_int (self->priv->insert_group_statement, 1, (gint) dvb_device_group_get_Id (dev_group)) != SQLITE_OK) {
		_tmp3_ = TRUE;
	} else {
		_tmp3_ = sqlite3_bind_int (self->priv->insert_group_statement, 2, (gint) dvb_device_group_get_Type (dev_group)) != SQLITE_OK;
	}
	if (_tmp3_) {
		_tmp2_ = TRUE;
	} else {
		const char* _tmp4_;
		_tmp4_ = NULL;
		_tmp2_ = sqlite3_bind_text (self->priv->insert_group_statement, 3, (_tmp4_ = channels, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)), -1, g_free) != SQLITE_OK;
	}
	if (_tmp2_) {
		_tmp1_ = TRUE;
	} else {
		const char* _tmp5_;
		_tmp5_ = NULL;
		_tmp1_ = sqlite3_bind_text (self->priv->insert_group_statement, 4, (_tmp5_ = recdir, (_tmp5_ == NULL) ? NULL : g_strdup (_tmp5_)), -1, g_free) != SQLITE_OK;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		const char* _tmp6_;
		_tmp6_ = NULL;
		_tmp0_ = sqlite3_bind_text (self->priv->insert_group_statement, 5, (_tmp6_ = dvb_device_group_get_Name (dev_group), (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_)), -1, g_free) != SQLITE_OK;
	}
	if (_tmp0_) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		channels = (g_free (channels), NULL);
		recdir = (g_free (recdir), NULL);
		return result;
	}
	if (sqlite3_step (self->priv->insert_group_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		channels = (g_free (channels), NULL);
		recdir = (g_free (recdir), NULL);
		return result;
	}
	{
		GeeIterator* _dev_it;
		_dev_it = gee_iterable_iterator ((GeeIterable*) dev_group);
		while (TRUE) {
			DVBDevice* dev;
			if (!gee_iterator_next (_dev_it)) {
				break;
			}
			dev = (DVBDevice*) gee_iterator_get (_dev_it);
			dvb_config_store_add_device_to_group ((DVBConfigStore*) self, dev, dev_group);
			(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
		}
		(_dev_it == NULL) ? NULL : (_dev_it = (g_object_unref (_dev_it), NULL));
	}
	result = TRUE;
	channels = (g_free (channels), NULL);
	recdir = (g_free (recdir), NULL);
	return result;
}


static gboolean dvb_sqlite_config_timers_store_real_remove_device_group (DVBConfigStore* base, DVBDeviceGroup* devgroup) {
	DVBSqliteConfigTimersStore * self;
	gboolean result;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (devgroup != NULL, FALSE);
	sqlite3_reset (self->priv->delete_group_statement);
	if (sqlite3_bind_int (self->priv->delete_group_statement, 1, (gint) dvb_device_group_get_Id (devgroup)) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->delete_group_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	sqlite3_reset (self->priv->delete_group_devices_statement);
	if (sqlite3_bind_int (self->priv->delete_group_devices_statement, 1, (gint) dvb_device_group_get_Id (devgroup)) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->delete_group_devices_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	result = TRUE;
	return result;
}


gboolean dvb_sqlite_config_timers_store_contains_group (DVBSqliteConfigTimersStore* self, guint group_id) {
	gboolean result;
	gint c;
	g_return_val_if_fail (self != NULL, FALSE);
	sqlite3_reset (self->priv->contains_group_statement);
	if (sqlite3_bind_int (self->priv->contains_group_statement, 1, (gint) group_id) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	c = 0;
	while (TRUE) {
		if (!(sqlite3_step (self->priv->contains_group_statement) == SQLITE_ROW)) {
			break;
		}
		c = sqlite3_column_int (self->priv->contains_group_statement, 0);
	}
	result = c > 0;
	return result;
}


static gboolean dvb_sqlite_config_timers_store_real_add_device_to_group (DVBConfigStore* base, DVBDevice* dev, DVBDeviceGroup* devgroup) {
	DVBSqliteConfigTimersStore * self;
	gboolean result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev != NULL, FALSE);
	g_return_val_if_fail (devgroup != NULL, FALSE);
	sqlite3_reset (self->priv->insert_device_statement);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (sqlite3_bind_int (self->priv->insert_device_statement, 1, (gint) dvb_device_group_get_Id (devgroup)) != SQLITE_OK) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = sqlite3_bind_int (self->priv->insert_device_statement, 2, (gint) dvb_device_get_Adapter (dev)) != SQLITE_OK;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = sqlite3_bind_int (self->priv->insert_device_statement, 3, (gint) dvb_device_get_Frontend (dev)) != SQLITE_OK;
	}
	if (_tmp0_) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->insert_device_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	result = TRUE;
	return result;
}


static gboolean dvb_sqlite_config_timers_store_real_remove_device_from_group (DVBConfigStore* base, DVBDevice* dev, DVBDeviceGroup* devgroup) {
	DVBSqliteConfigTimersStore * self;
	gboolean result;
	gboolean _tmp0_;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev != NULL, FALSE);
	g_return_val_if_fail (devgroup != NULL, FALSE);
	sqlite3_reset (self->priv->delete_device_statement);
	_tmp0_ = FALSE;
	if (sqlite3_bind_int (self->priv->delete_device_statement, 1, (gint) dvb_device_get_Adapter (dev)) != SQLITE_OK) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = sqlite3_bind_int (self->priv->delete_device_statement, 2, (gint) dvb_device_get_Frontend (dev)) != SQLITE_OK;
	}
	if (_tmp0_) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->delete_device_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	result = TRUE;
	return result;
}


static GeeList* dvb_sqlite_config_timers_store_real_get_all_timers_of_device_group (DVBTimersStore* base, DVBDeviceGroup* dev) {
	DVBSqliteConfigTimersStore * self;
	GeeList* result;
	GeeList* timers;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev != NULL, NULL);
	timers = (GeeList*) gee_array_list_new (DVB_TYPE_TIMER, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	sqlite3_reset (self->priv->select_timers_statement);
	if (sqlite3_bind_int (self->priv->select_timers_statement, 1, (gint) dvb_device_group_get_Id (dev)) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = timers;
		return result;
	}
	while (TRUE) {
		guint tid;
		guint sid;
		guint duration;
		gint year;
		gint month;
		gint day;
		gint hour;
		gint minute;
		DVBChannel* channel;
		DVBTimer* _tmp0_;
		if (!(sqlite3_step (self->priv->select_timers_statement) == SQLITE_ROW)) {
			break;
		}
		tid = 0U;
		sid = 0U;
		duration = 0U;
		year = 0;
		month = 0;
		day = 0;
		hour = 0;
		minute = 0;
		tid = (guint) sqlite3_column_int (self->priv->select_timers_statement, 0);
		sid = (guint) sqlite3_column_int (self->priv->select_timers_statement, 2);
		year = sqlite3_column_int (self->priv->select_timers_statement, 3);
		month = sqlite3_column_int (self->priv->select_timers_statement, 4);
		day = sqlite3_column_int (self->priv->select_timers_statement, 5);
		hour = sqlite3_column_int (self->priv->select_timers_statement, 6);
		minute = sqlite3_column_int (self->priv->select_timers_statement, 7);
		duration = (guint) sqlite3_column_int (self->priv->select_timers_statement, 8);
		channel = dvb_channel_list_get_channel (dvb_device_group_get_Channels (dev), sid);
		_tmp0_ = NULL;
		gee_collection_add ((GeeCollection*) timers, _tmp0_ = dvb_timer_new ((guint32) tid, channel, year, month, day, hour, minute, duration, NULL));
		(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
	}
	result = timers;
	return result;
}


static gboolean dvb_sqlite_config_timers_store_real_add_timer_to_device_group (DVBTimersStore* base, DVBTimer* timer, DVBDeviceGroup* dev) {
	DVBSqliteConfigTimersStore * self;
	gboolean result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	gboolean _tmp3_;
	gboolean _tmp4_;
	gboolean _tmp5_;
	gboolean _tmp6_;
	gboolean _tmp7_;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (timer != NULL, FALSE);
	g_return_val_if_fail (dev != NULL, FALSE);
	if (dvb_sqlite_config_timers_store_contains_timer (self, (guint) dvb_timer_get_Id (timer))) {
		result = FALSE;
		return result;
	}
	sqlite3_reset (self->priv->insert_timer_statement);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	_tmp4_ = FALSE;
	_tmp5_ = FALSE;
	_tmp6_ = FALSE;
	_tmp7_ = FALSE;
	if (sqlite3_bind_int (self->priv->insert_timer_statement, 1, (gint) dvb_timer_get_Id (timer)) != SQLITE_OK) {
		_tmp7_ = TRUE;
	} else {
		_tmp7_ = sqlite3_bind_int (self->priv->insert_timer_statement, 2, (gint) dvb_device_group_get_Id (dev)) != SQLITE_OK;
	}
	if (_tmp7_) {
		_tmp6_ = TRUE;
	} else {
		_tmp6_ = sqlite3_bind_int (self->priv->insert_timer_statement, 3, (gint) dvb_channel_get_Sid (dvb_timer_get_Channel (timer))) != SQLITE_OK;
	}
	if (_tmp6_) {
		_tmp5_ = TRUE;
	} else {
		_tmp5_ = sqlite3_bind_int (self->priv->insert_timer_statement, 4, (gint) dvb_timer_get_Year (timer)) != SQLITE_OK;
	}
	if (_tmp5_) {
		_tmp4_ = TRUE;
	} else {
		_tmp4_ = sqlite3_bind_int (self->priv->insert_timer_statement, 5, (gint) dvb_timer_get_Month (timer)) != SQLITE_OK;
	}
	if (_tmp4_) {
		_tmp3_ = TRUE;
	} else {
		_tmp3_ = sqlite3_bind_int (self->priv->insert_timer_statement, 6, (gint) dvb_timer_get_Day (timer)) != SQLITE_OK;
	}
	if (_tmp3_) {
		_tmp2_ = TRUE;
	} else {
		_tmp2_ = sqlite3_bind_int (self->priv->insert_timer_statement, 7, (gint) dvb_timer_get_Hour (timer)) != SQLITE_OK;
	}
	if (_tmp2_) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = sqlite3_bind_int (self->priv->insert_timer_statement, 8, (gint) dvb_timer_get_Minute (timer)) != SQLITE_OK;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = sqlite3_bind_int (self->priv->insert_timer_statement, 9, (gint) dvb_timer_get_Duration (timer)) != SQLITE_OK;
	}
	if (_tmp0_) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->insert_timer_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	result = TRUE;
	return result;
}


static gboolean dvb_sqlite_config_timers_store_real_remove_timer_from_device_group (DVBTimersStore* base, guint timer_id, DVBDeviceGroup* dev) {
	DVBSqliteConfigTimersStore * self;
	gboolean result;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev != NULL, FALSE);
	sqlite3_reset (self->priv->delete_timer_statement);
	if (sqlite3_bind_int (self->priv->delete_timer_statement, 1, (gint) timer_id) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->delete_timer_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	result = TRUE;
	return result;
}


gboolean dvb_sqlite_config_timers_store_contains_timer (DVBSqliteConfigTimersStore* self, guint timer_id) {
	gboolean result;
	gint c;
	g_return_val_if_fail (self != NULL, FALSE);
	sqlite3_reset (self->priv->contains_timer_statement);
	if (sqlite3_bind_int (self->priv->contains_timer_statement, 1, (gint) timer_id) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	c = 0;
	while (TRUE) {
		if (!(sqlite3_step (self->priv->contains_timer_statement) == SQLITE_ROW)) {
			break;
		}
		c = sqlite3_column_int (self->priv->contains_timer_statement, 0);
	}
	result = c > 0;
	return result;
}


static gboolean dvb_sqlite_config_timers_store_real_update_from_group (DVBConfigStore* base, DVBDeviceGroup* devgroup) {
	DVBSqliteConfigTimersStore * self;
	gboolean result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	gboolean _tmp3_;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (devgroup != NULL, FALSE);
	sqlite3_reset (self->priv->update_group_statement);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	if (sqlite3_bind_int (self->priv->update_group_statement, 1, (gint) dvb_device_group_get_Type (devgroup)) != SQLITE_OK) {
		_tmp3_ = TRUE;
	} else {
		_tmp3_ = sqlite3_bind_text (self->priv->update_group_statement, 2, g_file_get_path (dvb_channel_list_get_channels_file (dvb_device_group_get_Channels (devgroup))), -1, g_free) != SQLITE_OK;
	}
	if (_tmp3_) {
		_tmp2_ = TRUE;
	} else {
		_tmp2_ = sqlite3_bind_text (self->priv->update_group_statement, 3, g_file_get_path (dvb_device_group_get_RecordingsDirectory (devgroup)), -1, g_free) != SQLITE_OK;
	}
	if (_tmp2_) {
		_tmp1_ = TRUE;
	} else {
		const char* _tmp4_;
		_tmp4_ = NULL;
		_tmp1_ = sqlite3_bind_text (self->priv->update_group_statement, 4, (_tmp4_ = dvb_device_group_get_Name (devgroup), (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)), -1, g_free) != SQLITE_OK;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = sqlite3_bind_int (self->priv->update_group_statement, 5, (gint) dvb_device_group_get_Id (devgroup)) != SQLITE_OK;
	}
	if (_tmp0_) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->update_group_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	result = TRUE;
	return result;
}


static void dvb_sqlite_config_timers_store_print_last_error (DVBSqliteConfigTimersStore* self) {
	g_return_if_fail (self != NULL);
	g_critical ("SqliteConfigTimersStore.vala:434: SQLite error: %d, %s", sqlite3_errcode (self->priv->db), sqlite3_errmsg (self->priv->db));
}


static sqlite3* dvb_sqlite_config_timers_store_get_db_handler (void) {
	sqlite3* result;
	GError * _inner_error_;
	GFile* config_dir;
	GFile* config_cache;
	GFile* dbfile;
	gboolean create_tables;
	sqlite3* db;
	sqlite3* _tmp3_;
	gint _tmp2_;
	sqlite3* _tmp1_;
	char* _tmp0_;
	_inner_error_ = NULL;
	config_dir = g_file_new_for_path (g_get_user_config_dir ());
	config_cache = g_file_get_child (config_dir, "gnome-dvb-daemon");
	dbfile = g_file_get_child (config_cache, "configtimers.sqlite3");
	if (!g_file_query_exists (config_cache, NULL)) {
		{
			dvb_utils_mkdirs (config_cache, &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch34_g_error;
				goto __finally34;
			}
		}
		goto __finally34;
		__catch34_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				g_critical ("SqliteConfigTimersStore.vala:448: Could not create directory: %s", e->message);
				result = NULL;
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				(config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL));
				(config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL));
				(dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL));
				return result;
			}
		}
		__finally34:
		if (_inner_error_ != NULL) {
			(config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL));
			(config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL));
			(dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return NULL;
		}
	}
	create_tables = !g_file_query_exists (dbfile, NULL);
	db = NULL;
	_tmp3_ = NULL;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = sqlite3_open (_tmp0_ = g_file_get_path (dbfile), &_tmp1_);
	db = (_tmp3_ = _tmp1_, (db == NULL) ? NULL : (db = (sqlite3_close (db), NULL)), _tmp3_);
	_tmp2_;
	_tmp0_ = (g_free (_tmp0_), NULL);
	if (create_tables) {
		char* errormsg;
		char* _tmp6_;
		gint _tmp5_;
		char* _tmp4_;
		gint val;
		char* _tmp9_;
		gint _tmp8_;
		char* _tmp7_;
		char* _tmp12_;
		gint _tmp11_;
		char* _tmp10_;
		errormsg = NULL;
		_tmp6_ = NULL;
		_tmp4_ = NULL;
		val = (_tmp5_ = sqlite3_exec (db, DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_DEVICE_GROUPS, NULL, NULL, &_tmp4_), errormsg = (_tmp6_ = _tmp4_, errormsg = (g_free (errormsg), NULL), _tmp6_), _tmp5_);
		if (val != SQLITE_OK) {
			g_critical ("SqliteConfigTimersStore.vala:461: SQLite error: %s", errormsg);
			result = NULL;
			errormsg = (g_free (errormsg), NULL);
			(config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL));
			(config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL));
			(dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL));
			(db == NULL) ? NULL : (db = (sqlite3_close (db), NULL));
			return result;
		}
		_tmp9_ = NULL;
		_tmp7_ = NULL;
		val = (_tmp8_ = sqlite3_exec (db, DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_DEVICES, NULL, NULL, &_tmp7_), errormsg = (_tmp9_ = _tmp7_, errormsg = (g_free (errormsg), NULL), _tmp9_), _tmp8_);
		if (val != SQLITE_OK) {
			g_critical ("SqliteConfigTimersStore.vala:467: SQLite error: %s", errormsg);
			result = NULL;
			errormsg = (g_free (errormsg), NULL);
			(config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL));
			(config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL));
			(dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL));
			(db == NULL) ? NULL : (db = (sqlite3_close (db), NULL));
			return result;
		}
		_tmp12_ = NULL;
		_tmp10_ = NULL;
		val = (_tmp11_ = sqlite3_exec (db, DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_TIMERS, NULL, NULL, &_tmp10_), errormsg = (_tmp12_ = _tmp10_, errormsg = (g_free (errormsg), NULL), _tmp12_), _tmp11_);
		if (val != SQLITE_OK) {
			g_critical ("SqliteConfigTimersStore.vala:473: SQLite error: %s", errormsg);
			result = NULL;
			errormsg = (g_free (errormsg), NULL);
			(config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL));
			(config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL));
			(dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL));
			(db == NULL) ? NULL : (db = (sqlite3_close (db), NULL));
			return result;
		}
		errormsg = (g_free (errormsg), NULL);
	}
	result = db;
	(config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL));
	(config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL));
	(dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL));
	return result;
}


DVBSqliteConfigTimersStore* dvb_sqlite_config_timers_store_construct (GType object_type) {
	DVBSqliteConfigTimersStore * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBSqliteConfigTimersStore* dvb_sqlite_config_timers_store_new (void) {
	return dvb_sqlite_config_timers_store_construct (DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE);
}


static GObject * dvb_sqlite_config_timers_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBSqliteConfigTimersStoreClass * klass;
	GObjectClass * parent_class;
	DVBSqliteConfigTimersStore * self;
	klass = DVB_SQLITE_CONFIG_TIMERS_STORE_CLASS (g_type_class_peek (DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_SQLITE_CONFIG_TIMERS_STORE (obj);
	{
		sqlite3* _tmp0_;
		sqlite3_stmt* _tmp3_;
		gint _tmp2_;
		sqlite3_stmt* _tmp1_;
		sqlite3_stmt* _tmp6_;
		gint _tmp5_;
		sqlite3_stmt* _tmp4_;
		sqlite3_stmt* _tmp9_;
		gint _tmp8_;
		sqlite3_stmt* _tmp7_;
		sqlite3_stmt* _tmp12_;
		gint _tmp11_;
		sqlite3_stmt* _tmp10_;
		sqlite3_stmt* _tmp15_;
		gint _tmp14_;
		sqlite3_stmt* _tmp13_;
		sqlite3_stmt* _tmp18_;
		gint _tmp17_;
		sqlite3_stmt* _tmp16_;
		sqlite3_stmt* _tmp21_;
		gint _tmp20_;
		sqlite3_stmt* _tmp19_;
		sqlite3_stmt* _tmp24_;
		gint _tmp23_;
		sqlite3_stmt* _tmp22_;
		sqlite3_stmt* _tmp27_;
		gint _tmp26_;
		sqlite3_stmt* _tmp25_;
		sqlite3_stmt* _tmp30_;
		gint _tmp29_;
		sqlite3_stmt* _tmp28_;
		sqlite3_stmt* _tmp33_;
		gint _tmp32_;
		sqlite3_stmt* _tmp31_;
		sqlite3_stmt* _tmp36_;
		gint _tmp35_;
		sqlite3_stmt* _tmp34_;
		_tmp0_ = NULL;
		self->priv->db = (_tmp0_ = dvb_sqlite_config_timers_store_get_db_handler (), (self->priv->db == NULL) ? NULL : (self->priv->db = (sqlite3_close (self->priv->db), NULL)), _tmp0_);
		_tmp3_ = NULL;
		_tmp1_ = NULL;
		_tmp2_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_DEVICES, -1, &_tmp1_, NULL);
		self->priv->select_devices_statement = (_tmp3_ = _tmp1_, (self->priv->select_devices_statement == NULL) ? NULL : (self->priv->select_devices_statement = (sqlite3_finalize (self->priv->select_devices_statement), NULL)), _tmp3_);
		_tmp2_;
		_tmp6_ = NULL;
		_tmp4_ = NULL;
		_tmp5_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_GROUP, -1, &_tmp4_, NULL);
		self->priv->delete_group_statement = (_tmp6_ = _tmp4_, (self->priv->delete_group_statement == NULL) ? NULL : (self->priv->delete_group_statement = (sqlite3_finalize (self->priv->delete_group_statement), NULL)), _tmp6_);
		_tmp5_;
		_tmp9_ = NULL;
		_tmp7_ = NULL;
		_tmp8_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_GROUP, -1, &_tmp7_, NULL);
		self->priv->insert_group_statement = (_tmp9_ = _tmp7_, (self->priv->insert_group_statement == NULL) ? NULL : (self->priv->insert_group_statement = (sqlite3_finalize (self->priv->insert_group_statement), NULL)), _tmp9_);
		_tmp8_;
		_tmp12_ = NULL;
		_tmp10_ = NULL;
		_tmp11_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_UPDATE_GROUP, -1, &_tmp10_, NULL);
		self->priv->update_group_statement = (_tmp12_ = _tmp10_, (self->priv->update_group_statement == NULL) ? NULL : (self->priv->update_group_statement = (sqlite3_finalize (self->priv->update_group_statement), NULL)), _tmp12_);
		_tmp11_;
		_tmp15_ = NULL;
		_tmp13_ = NULL;
		_tmp14_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_DEVICE, -1, &_tmp13_, NULL);
		self->priv->delete_device_statement = (_tmp15_ = _tmp13_, (self->priv->delete_device_statement == NULL) ? NULL : (self->priv->delete_device_statement = (sqlite3_finalize (self->priv->delete_device_statement), NULL)), _tmp15_);
		_tmp14_;
		_tmp18_ = NULL;
		_tmp16_ = NULL;
		_tmp17_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_GROUP_DEVICES, -1, &_tmp16_, NULL);
		self->priv->delete_group_devices_statement = (_tmp18_ = _tmp16_, (self->priv->delete_group_devices_statement == NULL) ? NULL : (self->priv->delete_group_devices_statement = (sqlite3_finalize (self->priv->delete_group_devices_statement), NULL)), _tmp18_);
		_tmp17_;
		_tmp21_ = NULL;
		_tmp19_ = NULL;
		_tmp20_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_DEVICE, -1, &_tmp19_, NULL);
		self->priv->insert_device_statement = (_tmp21_ = _tmp19_, (self->priv->insert_device_statement == NULL) ? NULL : (self->priv->insert_device_statement = (sqlite3_finalize (self->priv->insert_device_statement), NULL)), _tmp21_);
		_tmp20_;
		_tmp24_ = NULL;
		_tmp22_ = NULL;
		_tmp23_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_TIMERS, -1, &_tmp22_, NULL);
		self->priv->select_timers_statement = (_tmp24_ = _tmp22_, (self->priv->select_timers_statement == NULL) ? NULL : (self->priv->select_timers_statement = (sqlite3_finalize (self->priv->select_timers_statement), NULL)), _tmp24_);
		_tmp23_;
		_tmp27_ = NULL;
		_tmp25_ = NULL;
		_tmp26_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_TIMER, -1, &_tmp25_, NULL);
		self->priv->delete_timer_statement = (_tmp27_ = _tmp25_, (self->priv->delete_timer_statement == NULL) ? NULL : (self->priv->delete_timer_statement = (sqlite3_finalize (self->priv->delete_timer_statement), NULL)), _tmp27_);
		_tmp26_;
		_tmp30_ = NULL;
		_tmp28_ = NULL;
		_tmp29_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_TIMER, -1, &_tmp28_, NULL);
		self->priv->insert_timer_statement = (_tmp30_ = _tmp28_, (self->priv->insert_timer_statement == NULL) ? NULL : (self->priv->insert_timer_statement = (sqlite3_finalize (self->priv->insert_timer_statement), NULL)), _tmp30_);
		_tmp29_;
		_tmp33_ = NULL;
		_tmp31_ = NULL;
		_tmp32_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_CONTAINS_GROUP, -1, &_tmp31_, NULL);
		self->priv->contains_group_statement = (_tmp33_ = _tmp31_, (self->priv->contains_group_statement == NULL) ? NULL : (self->priv->contains_group_statement = (sqlite3_finalize (self->priv->contains_group_statement), NULL)), _tmp33_);
		_tmp32_;
		_tmp36_ = NULL;
		_tmp34_ = NULL;
		_tmp35_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_CONTAINS_TIMER, -1, &_tmp34_, NULL);
		self->priv->contains_timer_statement = (_tmp36_ = _tmp34_, (self->priv->contains_timer_statement == NULL) ? NULL : (self->priv->contains_timer_statement = (sqlite3_finalize (self->priv->contains_timer_statement), NULL)), _tmp36_);
		_tmp35_;
	}
	return obj;
}


static void dvb_sqlite_config_timers_store_class_init (DVBSqliteConfigTimersStoreClass * klass) {
	dvb_sqlite_config_timers_store_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBSqliteConfigTimersStorePrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_sqlite_config_timers_store_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_sqlite_config_timers_store_finalize;
}


static void dvb_sqlite_config_timers_store_dvb_config_store_interface_init (DVBConfigStoreIface * iface) {
	dvb_sqlite_config_timers_store_dvb_config_store_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_all_device_groups = dvb_sqlite_config_timers_store_real_get_all_device_groups;
	iface->add_device_group = dvb_sqlite_config_timers_store_real_add_device_group;
	iface->remove_device_group = dvb_sqlite_config_timers_store_real_remove_device_group;
	iface->add_device_to_group = dvb_sqlite_config_timers_store_real_add_device_to_group;
	iface->remove_device_from_group = dvb_sqlite_config_timers_store_real_remove_device_from_group;
	iface->update_from_group = dvb_sqlite_config_timers_store_real_update_from_group;
}


static void dvb_sqlite_config_timers_store_dvb_timers_store_interface_init (DVBTimersStoreIface * iface) {
	dvb_sqlite_config_timers_store_dvb_timers_store_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_all_timers_of_device_group = dvb_sqlite_config_timers_store_real_get_all_timers_of_device_group;
	iface->add_timer_to_device_group = dvb_sqlite_config_timers_store_real_add_timer_to_device_group;
	iface->remove_timer_from_device_group = dvb_sqlite_config_timers_store_real_remove_timer_from_device_group;
}


static void dvb_sqlite_config_timers_store_instance_init (DVBSqliteConfigTimersStore * self) {
	self->priv = DVB_SQLITE_CONFIG_TIMERS_STORE_GET_PRIVATE (self);
}


static void dvb_sqlite_config_timers_store_finalize (GObject* obj) {
	DVBSqliteConfigTimersStore * self;
	self = DVB_SQLITE_CONFIG_TIMERS_STORE (obj);
	(self->priv->select_devices_statement == NULL) ? NULL : (self->priv->select_devices_statement = (sqlite3_finalize (self->priv->select_devices_statement), NULL));
	(self->priv->delete_group_statement == NULL) ? NULL : (self->priv->delete_group_statement = (sqlite3_finalize (self->priv->delete_group_statement), NULL));
	(self->priv->insert_group_statement == NULL) ? NULL : (self->priv->insert_group_statement = (sqlite3_finalize (self->priv->insert_group_statement), NULL));
	(self->priv->update_group_statement == NULL) ? NULL : (self->priv->update_group_statement = (sqlite3_finalize (self->priv->update_group_statement), NULL));
	(self->priv->delete_device_statement == NULL) ? NULL : (self->priv->delete_device_statement = (sqlite3_finalize (self->priv->delete_device_statement), NULL));
	(self->priv->delete_group_devices_statement == NULL) ? NULL : (self->priv->delete_group_devices_statement = (sqlite3_finalize (self->priv->delete_group_devices_statement), NULL));
	(self->priv->insert_device_statement == NULL) ? NULL : (self->priv->insert_device_statement = (sqlite3_finalize (self->priv->insert_device_statement), NULL));
	(self->priv->select_timers_statement == NULL) ? NULL : (self->priv->select_timers_statement = (sqlite3_finalize (self->priv->select_timers_statement), NULL));
	(self->priv->delete_timer_statement == NULL) ? NULL : (self->priv->delete_timer_statement = (sqlite3_finalize (self->priv->delete_timer_statement), NULL));
	(self->priv->insert_timer_statement == NULL) ? NULL : (self->priv->insert_timer_statement = (sqlite3_finalize (self->priv->insert_timer_statement), NULL));
	(self->priv->contains_group_statement == NULL) ? NULL : (self->priv->contains_group_statement = (sqlite3_finalize (self->priv->contains_group_statement), NULL));
	(self->priv->contains_timer_statement == NULL) ? NULL : (self->priv->contains_timer_statement = (sqlite3_finalize (self->priv->contains_timer_statement), NULL));
	(self->priv->db == NULL) ? NULL : (self->priv->db = (sqlite3_close (self->priv->db), NULL));
	G_OBJECT_CLASS (dvb_sqlite_config_timers_store_parent_class)->finalize (obj);
}


GType dvb_sqlite_config_timers_store_get_type (void) {
	static GType dvb_sqlite_config_timers_store_type_id = 0;
	if (dvb_sqlite_config_timers_store_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBSqliteConfigTimersStoreClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_sqlite_config_timers_store_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBSqliteConfigTimersStore), 0, (GInstanceInitFunc) dvb_sqlite_config_timers_store_instance_init, NULL };
		static const GInterfaceInfo dvb_config_store_info = { (GInterfaceInitFunc) dvb_sqlite_config_timers_store_dvb_config_store_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo dvb_timers_store_info = { (GInterfaceInitFunc) dvb_sqlite_config_timers_store_dvb_timers_store_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_sqlite_config_timers_store_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBSqliteConfigTimersStore", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_sqlite_config_timers_store_type_id, DVB_TYPE_CONFIG_STORE, &dvb_config_store_info);
		g_type_add_interface_static (dvb_sqlite_config_timers_store_type_id, DVB_TYPE_TIMERS_STORE, &dvb_timers_store_info);
	}
	return dvb_sqlite_config_timers_store_type_id;
}




