/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>
#include <gio/gio.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>


#define DVB_TYPE_ID_BUS_CHANNEL_LIST (dvb_id_bus_channel_list_get_type ())
#define DVB_ID_BUS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_ID_BUS_CHANNEL_LIST, DVBIDBusChannelList))
#define DVB_IS_ID_BUS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_ID_BUS_CHANNEL_LIST))
#define DVB_ID_BUS_CHANNEL_LIST_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_ID_BUS_CHANNEL_LIST, DVBIDBusChannelListIface))

typedef struct _DVBIDBusChannelList DVBIDBusChannelList;
typedef struct _DVBIDBusChannelListIface DVBIDBusChannelListIface;

#define DVB_TYPE_CHANNEL_INFO (dvb_channel_info_get_type ())
typedef struct _DVBChannelInfo DVBChannelInfo;

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;
typedef struct _DVBChannelListPrivate DVBChannelListPrivate;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_ADAPTER_TYPE (dvb_adapter_type_get_type ())

#define DVB_TYPE_CHANNEL_LIST_READER (dvb_channel_list_reader_get_type ())
#define DVB_CHANNEL_LIST_READER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST_READER, DVBChannelListReader))
#define DVB_CHANNEL_LIST_READER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST_READER, DVBChannelListReaderClass))
#define DVB_IS_CHANNEL_LIST_READER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST_READER))
#define DVB_IS_CHANNEL_LIST_READER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST_READER))
#define DVB_CHANNEL_LIST_READER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST_READER, DVBChannelListReaderClass))

typedef struct _DVBChannelListReader DVBChannelListReader;
typedef struct _DVBChannelListReaderClass DVBChannelListReaderClass;
typedef struct _DBusObjectVTable _DBusObjectVTable;

struct _DVBChannelInfo {
	guint id;
	char* name;
};

struct _DVBIDBusChannelListIface {
	GTypeInterface parent_iface;
	guint* (*GetChannels) (DVBIDBusChannelList* self, int* result_length1);
	guint* (*GetRadioChannels) (DVBIDBusChannelList* self, int* result_length1);
	guint* (*GetTVChannels) (DVBIDBusChannelList* self, int* result_length1);
	char* (*GetChannelName) (DVBIDBusChannelList* self, guint channel_id);
	char* (*GetChannelNetwork) (DVBIDBusChannelList* self, guint channel_id);
	gboolean (*IsRadioChannel) (DVBIDBusChannelList* self, guint channel_id);
	char* (*GetChannelURL) (DVBIDBusChannelList* self, guint channel_id);
	DVBChannelInfo* (*GetChannelInfos) (DVBIDBusChannelList* self, int* result_length1);
};

struct _DVBChannelList {
	GObject parent_instance;
	DVBChannelListPrivate * priv;
	GeeHashMap* channels;
};

struct _DVBChannelListClass {
	GObjectClass parent_class;
};

struct _DVBChannelListPrivate {
	GFile* _channels_file;
	GStaticRecMutex __lock_channels;
};

typedef enum  {
	DVB_ADAPTER_TYPE_UNKNOWN,
	DVB_ADAPTER_TYPE_DVB_T,
	DVB_ADAPTER_TYPE_DVB_S,
	DVB_ADAPTER_TYPE_DVB_C
} DVBAdapterType;

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};


static gpointer dvb_channel_list_parent_class = NULL;
static GeeIterableIface* dvb_channel_list_gee_iterable_parent_iface = NULL;
static DVBIDBusChannelListIface* dvb_channel_list_dvb_id_bus_channel_list_parent_iface = NULL;

GType dvb_channel_info_get_type (void);
DVBChannelInfo* dvb_channel_info_dup (const DVBChannelInfo* self);
void dvb_channel_info_free (DVBChannelInfo* self);
void dvb_channel_info_copy (const DVBChannelInfo* self, DVBChannelInfo* dest);
void dvb_channel_info_destroy (DVBChannelInfo* self);
GType dvb_id_bus_channel_list_get_type (void);
GType dvb_channel_list_get_type (void);
GType dvb_channel_get_type (void);
#define DVB_CHANNEL_LIST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_CHANNEL_LIST, DVBChannelListPrivate))
enum  {
	DVB_CHANNEL_LIST_DUMMY_PROPERTY,
	DVB_CHANNEL_LIST_CHANNELS_FILE,
	DVB_CHANNEL_LIST_SIZE
};
static void dvb_channel_list_set_channels_file (DVBChannelList* self, GFile* value);
DVBChannelList* dvb_channel_list_new (GFile* channels);
DVBChannelList* dvb_channel_list_construct (GType object_type, GFile* channels);
DVBChannel* dvb_channel_list_get_channel (DVBChannelList* self, guint sid);
guint dvb_channel_get_Sid (DVBChannel* self);
void dvb_channel_list_add (DVBChannelList* self, DVBChannel* channel);
void dvb_channel_list_remove (DVBChannelList* self, guint sid);
gboolean dvb_channel_list_contains (DVBChannelList* self, guint sid);
void dvb_channel_list_clear (DVBChannelList* self);
static GType dvb_channel_list_real_get_element_type (GeeIterable* base);
static GeeIterator* dvb_channel_list_real_iterator (GeeIterable* base);
GType dvb_adapter_type_get_type (void);
DVBChannelListReader* dvb_channel_list_reader_new (GFile* file, DVBAdapterType type, guint group_id);
DVBChannelListReader* dvb_channel_list_reader_construct (GType object_type, GFile* file, DVBAdapterType type, guint group_id);
GType dvb_channel_list_reader_get_type (void);
DVBChannelList* dvb_channel_list_reader_read (DVBChannelListReader* self, GError** error);
DVBChannelList* dvb_channel_list_restore_from_file (GFile* channelsfile, DVBAdapterType type, guint group_id, GError** error);
gint dvb_channel_list_get_size (DVBChannelList* self);
static guint* dvb_channel_list_real_GetChannels (DVBIDBusChannelList* base, int* result_length1);
guint dvb_channel_get_VideoPID (DVBChannel* self);
static guint* dvb_channel_list_real_GetRadioChannels (DVBIDBusChannelList* base, int* result_length1);
static guint* dvb_channel_list_real_GetTVChannels (DVBIDBusChannelList* base, int* result_length1);
const char* dvb_channel_get_Name (DVBChannel* self);
static char* dvb_channel_list_real_GetChannelName (DVBIDBusChannelList* base, guint channel_id);
const char* dvb_channel_get_Network (DVBChannel* self);
static char* dvb_channel_list_real_GetChannelNetwork (DVBIDBusChannelList* base, guint channel_id);
static gboolean dvb_channel_list_real_IsRadioChannel (DVBIDBusChannelList* base, guint channel_id);
char* dvb_channel_get_URL (DVBChannel* self);
static char* dvb_channel_list_real_GetChannelURL (DVBIDBusChannelList* base, guint channel_id);
static void _vala_DVBChannelInfo_array_free (DVBChannelInfo* array, gint array_length);
static DVBChannelInfo* dvb_channel_list_real_GetChannelInfos (DVBIDBusChannelList* base, int* result_length1);
GFile* dvb_channel_list_get_channels_file (DVBChannelList* self);
static GObject * dvb_channel_list_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
void dvb_channel_list_dbus_register_object (DBusConnection* connection, const char* path, void* object);
void _dvb_channel_list_dbus_unregister (DBusConnection* connection, void* user_data);
DBusHandlerResult dvb_channel_list_dbus_message (DBusConnection* connection, DBusMessage* message, void* object);
static DBusMessage* _dbus_dvb_channel_list_introspect (DVBChannelList* self, DBusConnection* connection, DBusMessage* message);
static void dvb_channel_list_finalize (GObject* obj);
static void dvb_channel_list_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_channel_list_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);
static void _vala_dbus_unregister_object (gpointer connection, GObject* object);

static const DBusObjectPathVTable _dvb_channel_list_dbus_path_vtable = {_dvb_channel_list_dbus_unregister, dvb_channel_list_dbus_message};
static const _DBusObjectVTable _dvb_channel_list_dbus_vtable = {dvb_channel_list_dbus_register_object};


DVBChannelList* dvb_channel_list_construct (GType object_type, GFile* channels) {
	GParameter * __params;
	GParameter * __params_it;
	DVBChannelList * self;
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "channels-file";
	g_value_init (&__params_it->value, G_TYPE_FILE);
	g_value_set_object (&__params_it->value, channels);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBChannelList* dvb_channel_list_new (GFile* channels) {
	return dvb_channel_list_construct (DVB_TYPE_CHANNEL_LIST, channels);
}


DVBChannel* dvb_channel_list_get_channel (DVBChannelList* self, guint sid) {
	DVBChannel* result;
	DVBChannel* val;
	g_return_val_if_fail (self != NULL, NULL);
	val = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		if (gee_map_contains ((GeeMap*) self->channels, GUINT_TO_POINTER (sid))) {
			DVBChannel* _tmp0_;
			_tmp0_ = NULL;
			val = (_tmp0_ = (DVBChannel*) gee_map_get ((GeeMap*) self->channels, GUINT_TO_POINTER (sid)), (val == NULL) ? NULL : (val = (g_object_unref (val), NULL)), _tmp0_);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	result = val;
	return result;
}


void dvb_channel_list_add (DVBChannelList* self, DVBChannel* channel) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (channel != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		gee_map_set ((GeeMap*) self->channels, GUINT_TO_POINTER (dvb_channel_get_Sid (channel)), channel);
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
}


void dvb_channel_list_remove (DVBChannelList* self, guint sid) {
	g_return_if_fail (self != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		gee_map_remove ((GeeMap*) self->channels, GUINT_TO_POINTER (sid));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
}


gboolean dvb_channel_list_contains (DVBChannelList* self, guint sid) {
	gboolean result;
	gboolean val;
	g_return_val_if_fail (self != NULL, FALSE);
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		val = gee_map_contains ((GeeMap*) self->channels, GUINT_TO_POINTER (sid));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	result = val;
	return result;
}


void dvb_channel_list_clear (DVBChannelList* self) {
	g_return_if_fail (self != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		gee_map_clear ((GeeMap*) self->channels);
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
}


static GType dvb_channel_list_real_get_element_type (GeeIterable* base) {
	DVBChannelList * self;
	GType result;
	self = (DVBChannelList*) base;
	result = DVB_TYPE_CHANNEL;
	return result;
}


static GeeIterator* dvb_channel_list_real_iterator (GeeIterable* base) {
	DVBChannelList * self;
	GeeIterator* result;
	GeeCollection* _tmp0_;
	GeeIterator* _tmp1_;
	self = (DVBChannelList*) base;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	result = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_values ((GeeMap*) self->channels))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
	return result;
}


DVBChannelList* dvb_channel_list_restore_from_file (GFile* channelsfile, DVBAdapterType type, guint group_id, GError** error) {
	DVBChannelList* result;
	GError * _inner_error_;
	DVBChannelListReader* reader;
	DVBChannelList* _tmp0_;
	g_return_val_if_fail (channelsfile != NULL, NULL);
	_inner_error_ = NULL;
	reader = dvb_channel_list_reader_new (channelsfile, type, group_id);
	_tmp0_ = dvb_channel_list_reader_read (reader, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		(reader == NULL) ? NULL : (reader = (g_object_unref (reader), NULL));
		return NULL;
	}
	result = _tmp0_;
	(reader == NULL) ? NULL : (reader = (g_object_unref (reader), NULL));
	return result;
}


/**
         * @returns: List of channel IDs aka SIDs
         */
static guint* dvb_channel_list_real_GetChannels (DVBIDBusChannelList* base, int* result_length1) {
	DVBChannelList * self;
	guint* result;
	guint* _tmp1_;
	gint ids_size;
	gint ids_length1;
	gint _tmp0_;
	guint* ids;
	gint i;
	guint* _tmp4_;
	self = (DVBChannelList*) base;
	_tmp1_ = NULL;
	ids = (_tmp1_ = g_new0 (guint, _tmp0_ = dvb_channel_list_get_size (self)), ids_length1 = _tmp0_, ids_size = ids_length1, _tmp1_);
	i = 0;
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		{
			GeeSet* _tmp2_;
			GeeIterator* _tmp3_;
			GeeIterator* _id_it;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_id_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = gee_map_get_keys ((GeeMap*) self->channels))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (g_object_unref (_tmp2_), NULL)), _tmp3_);
			while (TRUE) {
				guint id;
				if (!gee_iterator_next (_id_it)) {
					break;
				}
				id = GPOINTER_TO_UINT (gee_iterator_get (_id_it));
				ids[i] = id;
				i++;
			}
			(_id_it == NULL) ? NULL : (_id_it = (g_object_unref (_id_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	_tmp4_ = NULL;
	result = (_tmp4_ = ids, *result_length1 = ids_length1, _tmp4_);
	return result;
	ids = (g_free (ids), NULL);
}


/**
         * @returns: List of channel IDs aka SIDs of radio channels
         */
static guint* dvb_channel_list_real_GetRadioChannels (DVBIDBusChannelList* base, int* result_length1) {
	DVBChannelList * self;
	guint* result;
	GSList* radio_channels;
	guint* _tmp3_;
	gint ids_size;
	gint ids_length1;
	gint _tmp2_;
	guint* ids;
	guint* _tmp5_;
	self = (DVBChannelList*) base;
	radio_channels = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _id_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_id_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_keys ((GeeMap*) self->channels))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				guint id;
				DVBChannel* chan;
				if (!gee_iterator_next (_id_it)) {
					break;
				}
				id = GPOINTER_TO_UINT (gee_iterator_get (_id_it));
				chan = (DVBChannel*) gee_map_get ((GeeMap*) self->channels, GUINT_TO_POINTER (id));
				if (dvb_channel_get_VideoPID (chan) == 0) {
					radio_channels = g_slist_prepend (radio_channels, GUINT_TO_POINTER (id));
				}
				(chan == NULL) ? NULL : (chan = (g_object_unref (chan), NULL));
			}
			(_id_it == NULL) ? NULL : (_id_it = (g_object_unref (_id_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	radio_channels = g_slist_reverse (radio_channels);
	_tmp3_ = NULL;
	ids = (_tmp3_ = g_new0 (guint, _tmp2_ = g_slist_length (radio_channels)), ids_length1 = _tmp2_, ids_size = ids_length1, _tmp3_);
	{
		gint i;
		i = 0;
		{
			gboolean _tmp4_;
			_tmp4_ = TRUE;
			while (TRUE) {
				if (!_tmp4_) {
					i++;
				}
				_tmp4_ = FALSE;
				if (!(i < g_slist_length (radio_channels))) {
					break;
				}
				ids[i] = GPOINTER_TO_UINT (g_slist_nth_data (radio_channels, (guint) i));
			}
		}
	}
	_tmp5_ = NULL;
	result = (_tmp5_ = ids, *result_length1 = ids_length1, _tmp5_);
	(radio_channels == NULL) ? NULL : (radio_channels = (g_slist_free (radio_channels), NULL));
	return result;
	(radio_channels == NULL) ? NULL : (radio_channels = (g_slist_free (radio_channels), NULL));
	ids = (g_free (ids), NULL);
}


/**
         * @returns: List of channel IDs aka SIDs of TV channels
         */
static guint* dvb_channel_list_real_GetTVChannels (DVBIDBusChannelList* base, int* result_length1) {
	DVBChannelList * self;
	guint* result;
	GSList* video_channels;
	guint* _tmp3_;
	gint ids_size;
	gint ids_length1;
	gint _tmp2_;
	guint* ids;
	guint* _tmp5_;
	self = (DVBChannelList*) base;
	video_channels = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _id_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_id_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_keys ((GeeMap*) self->channels))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				guint id;
				DVBChannel* chan;
				if (!gee_iterator_next (_id_it)) {
					break;
				}
				id = GPOINTER_TO_UINT (gee_iterator_get (_id_it));
				chan = (DVBChannel*) gee_map_get ((GeeMap*) self->channels, GUINT_TO_POINTER (id));
				if (dvb_channel_get_VideoPID (chan) != 0) {
					video_channels = g_slist_prepend (video_channels, GUINT_TO_POINTER (id));
				}
				(chan == NULL) ? NULL : (chan = (g_object_unref (chan), NULL));
			}
			(_id_it == NULL) ? NULL : (_id_it = (g_object_unref (_id_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	video_channels = g_slist_reverse (video_channels);
	_tmp3_ = NULL;
	ids = (_tmp3_ = g_new0 (guint, _tmp2_ = g_slist_length (video_channels)), ids_length1 = _tmp2_, ids_size = ids_length1, _tmp3_);
	{
		gint i;
		i = 0;
		{
			gboolean _tmp4_;
			_tmp4_ = TRUE;
			while (TRUE) {
				if (!_tmp4_) {
					i++;
				}
				_tmp4_ = FALSE;
				if (!(i < g_slist_length (video_channels))) {
					break;
				}
				ids[i] = GPOINTER_TO_UINT (g_slist_nth_data (video_channels, (guint) i));
			}
		}
	}
	_tmp5_ = NULL;
	result = (_tmp5_ = ids, *result_length1 = ids_length1, _tmp5_);
	(video_channels == NULL) ? NULL : (video_channels = (g_slist_free (video_channels), NULL));
	return result;
	(video_channels == NULL) ? NULL : (video_channels = (g_slist_free (video_channels), NULL));
	ids = (g_free (ids), NULL);
}


/**
         * @channel_id: ID of channel
         * @returns: Name of channel if channel with id exists
         * otherwise an empty string
         */
static char* dvb_channel_list_real_GetChannelName (DVBIDBusChannelList* base, guint channel_id) {
	DVBChannelList * self;
	char* result;
	char* val;
	self = (DVBChannelList*) base;
	val = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		if (gee_map_contains ((GeeMap*) self->channels, GUINT_TO_POINTER (channel_id))) {
			const char* _tmp1_;
			DVBChannel* _tmp0_;
			char* _tmp2_;
			char* name;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			_tmp2_ = NULL;
			name = (_tmp2_ = (_tmp1_ = dvb_channel_get_Name (_tmp0_ = (DVBChannel*) gee_map_get ((GeeMap*) self->channels, GUINT_TO_POINTER (channel_id))), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp2_);
			if (name != NULL) {
				char* _tmp4_;
				const char* _tmp3_;
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				val = (_tmp4_ = (_tmp3_ = name, (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_)), val = (g_free (val), NULL), _tmp4_);
			}
			name = (g_free (name), NULL);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	result = val;
	return result;
}


/**
         * @channel_id: ID of channel
         * @returns: Name of network the channel belongs to
         * if the channel with id exists, otherwise an empty
         * string
         */
static char* dvb_channel_list_real_GetChannelNetwork (DVBIDBusChannelList* base, guint channel_id) {
	DVBChannelList * self;
	char* result;
	char* val;
	self = (DVBChannelList*) base;
	val = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		if (gee_map_contains ((GeeMap*) self->channels, GUINT_TO_POINTER (channel_id))) {
			const char* _tmp1_;
			DVBChannel* _tmp0_;
			char* _tmp2_;
			char* network;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			_tmp2_ = NULL;
			network = (_tmp2_ = (_tmp1_ = dvb_channel_get_Network (_tmp0_ = (DVBChannel*) gee_map_get ((GeeMap*) self->channels, GUINT_TO_POINTER (channel_id))), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp2_);
			if (network != NULL) {
				char* _tmp4_;
				const char* _tmp3_;
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				val = (_tmp4_ = (_tmp3_ = network, (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_)), val = (g_free (val), NULL), _tmp4_);
			}
			network = (g_free (network), NULL);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	result = val;
	return result;
}


/**
         * @channel_id: ID of channel
         * @returns: Whether the channel is a radio channel or not
         */
static gboolean dvb_channel_list_real_IsRadioChannel (DVBIDBusChannelList* base, guint channel_id) {
	DVBChannelList * self;
	gboolean result;
	gboolean val;
	self = (DVBChannelList*) base;
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		if (gee_map_contains ((GeeMap*) self->channels, GUINT_TO_POINTER (channel_id))) {
			DVBChannel* _tmp0_;
			_tmp0_ = NULL;
			val = dvb_channel_get_VideoPID (_tmp0_ = (DVBChannel*) gee_map_get ((GeeMap*) self->channels, GUINT_TO_POINTER (channel_id))) == 0;
			(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	result = val;
	return result;
}


/**
         * @channel_id: ID of channel
         * @returns: URL to watch the channel
         */
static char* dvb_channel_list_real_GetChannelURL (DVBIDBusChannelList* base, guint channel_id) {
	DVBChannelList * self;
	char* result;
	char* url;
	self = (DVBChannelList*) base;
	url = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		if (gee_map_contains ((GeeMap*) self->channels, GUINT_TO_POINTER (channel_id))) {
			DVBChannel* channel;
			char* _tmp0_;
			channel = (DVBChannel*) gee_map_get ((GeeMap*) self->channels, GUINT_TO_POINTER (channel_id));
			_tmp0_ = NULL;
			url = (_tmp0_ = dvb_channel_get_URL (channel), url = (g_free (url), NULL), _tmp0_);
			(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	result = url;
	return result;
}


static void _vala_DVBChannelInfo_array_free (DVBChannelInfo* array, gint array_length) {
	if (array != NULL) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			dvb_channel_info_destroy (&array[i]);
		}
	}
	g_free (array);
}


static DVBChannelInfo* dvb_channel_list_real_GetChannelInfos (DVBIDBusChannelList* base, int* result_length1) {
	DVBChannelList * self;
	DVBChannelInfo* result;
	DVBChannelInfo* _tmp1_;
	gint channels_size;
	gint channels_length1;
	gint _tmp0_;
	DVBChannelInfo* channels;
	gint i;
	DVBChannelInfo* _tmp10_;
	self = (DVBChannelList*) base;
	_tmp1_ = NULL;
	channels = (_tmp1_ = g_new0 (DVBChannelInfo, _tmp0_ = gee_map_get_size ((GeeMap*) self->channels)), channels_length1 = _tmp0_, channels_size = channels_length1, _tmp1_);
	i = 0;
	g_static_rec_mutex_lock (&self->priv->__lock_channels);
	{
		{
			GeeSet* _tmp2_;
			GeeIterator* _tmp3_;
			GeeIterator* _id_it;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_id_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = gee_map_get_keys ((GeeMap*) self->channels))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (g_object_unref (_tmp2_), NULL)), _tmp3_);
			while (TRUE) {
				guint id;
				DVBChannelInfo _tmp4_ = {0};
				DVBChannelInfo channel;
				char* _tmp7_;
				const char* _tmp6_;
				DVBChannel* _tmp5_;
				DVBChannelInfo _tmp9_ = {0};
				DVBChannelInfo _tmp8_ = {0};
				if (!gee_iterator_next (_id_it)) {
					break;
				}
				id = GPOINTER_TO_UINT (gee_iterator_get (_id_it));
				channel = (memset (&_tmp4_, 0, sizeof (DVBChannelInfo)), _tmp4_);
				channel.id = id;
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				_tmp5_ = NULL;
				channel.name = (_tmp7_ = (_tmp6_ = dvb_channel_get_Name (_tmp5_ = (DVBChannel*) gee_map_get ((GeeMap*) self->channels, GUINT_TO_POINTER (id))), (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_)), channel.name = (g_free (channel.name), NULL), _tmp7_);
				(_tmp5_ == NULL) ? NULL : (_tmp5_ = (g_object_unref (_tmp5_), NULL));
				channels[i] = (_tmp9_ = (dvb_channel_info_copy (&channel, &_tmp8_), _tmp8_), dvb_channel_info_destroy (&channels[i]), _tmp9_);
				i++;
				dvb_channel_info_destroy (&channel);
			}
			(_id_it == NULL) ? NULL : (_id_it = (g_object_unref (_id_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_channels);
	_tmp10_ = NULL;
	result = (_tmp10_ = channels, *result_length1 = channels_length1, _tmp10_);
	return result;
	channels = (_vala_DVBChannelInfo_array_free (channels, channels_length1), NULL);
}


GFile* dvb_channel_list_get_channels_file (DVBChannelList* self) {
	GFile* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_channels_file;
	return result;
}


static void dvb_channel_list_set_channels_file (DVBChannelList* self, GFile* value) {
	GFile* _tmp1_;
	GFile* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_channels_file = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_channels_file == NULL) ? NULL : (self->priv->_channels_file = (g_object_unref (self->priv->_channels_file), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "channels-file");
}


gint dvb_channel_list_get_size (DVBChannelList* self) {
	gint result;
	g_return_val_if_fail (self != NULL, 0);
	result = gee_map_get_size ((GeeMap*) self->channels);
	return result;
}


static GObject * dvb_channel_list_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBChannelListClass * klass;
	GObjectClass * parent_class;
	DVBChannelList * self;
	klass = DVB_CHANNEL_LIST_CLASS (g_type_class_peek (DVB_TYPE_CHANNEL_LIST));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_CHANNEL_LIST (obj);
	{
		GeeHashMap* _tmp0_;
		_tmp0_ = NULL;
		self->channels = (_tmp0_ = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_CHANNEL, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->channels == NULL) ? NULL : (self->channels = (g_object_unref (self->channels), NULL)), _tmp0_);
	}
	return obj;
}


void _dvb_channel_list_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_channel_list_introspect (DVBChannelList* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.ChannelList\">\n  <method name=\"GetChannels\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetRadioChannels\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetTVChannels\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelName\">\n    <arg name=\"channel_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelNetwork\">\n    <arg name=\"channel_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"IsRadioChannel\">\n    <arg name=\"channel_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelURL\">\n    <arg name=\"channel_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelInfos\">\n    <arg name=\"result\" type=\"a(us)\" direction=\"out\"/>\n  </method>\n  <signal name=\"Changed\">\n    <arg name=\"channel_id\" type=\"u\"/>\n    <arg name=\"type\" type=\"u\"/>\n  </signal>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	dbus_free_string_array (children);
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


DBusHandlerResult dvb_channel_list_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_channel_list_introspect (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if (dvb_id_bus_channel_list_dbus_message (connection, message, object) == DBUS_HANDLER_RESULT_HANDLED) {
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


void dvb_channel_list_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_channel_list_dbus_path_vtable, object);
		g_object_weak_ref (object, _vala_dbus_unregister_object, connection);
	}
	dvb_id_bus_channel_list_dbus_register_object (connection, path, object);
}


static void dvb_channel_list_class_init (DVBChannelListClass * klass) {
	dvb_channel_list_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBChannelListPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_channel_list_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_channel_list_set_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_channel_list_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_channel_list_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_CHANNEL_LIST_CHANNELS_FILE, g_param_spec_object ("channels-file", "channels-file", "channels-file", G_TYPE_FILE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_CHANNEL_LIST_SIZE, g_param_spec_int ("size", "size", "size", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_type_set_qdata (DVB_TYPE_CHANNEL_LIST, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_channel_list_dbus_vtable));
}


static void dvb_channel_list_gee_iterable_interface_init (GeeIterableIface * iface) {
	dvb_channel_list_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = dvb_channel_list_real_get_element_type;
	iface->iterator = dvb_channel_list_real_iterator;
}


static void dvb_channel_list_dvb_id_bus_channel_list_interface_init (DVBIDBusChannelListIface * iface) {
	dvb_channel_list_dvb_id_bus_channel_list_parent_iface = g_type_interface_peek_parent (iface);
	iface->GetChannels = dvb_channel_list_real_GetChannels;
	iface->GetRadioChannels = dvb_channel_list_real_GetRadioChannels;
	iface->GetTVChannels = dvb_channel_list_real_GetTVChannels;
	iface->GetChannelName = dvb_channel_list_real_GetChannelName;
	iface->GetChannelNetwork = dvb_channel_list_real_GetChannelNetwork;
	iface->IsRadioChannel = dvb_channel_list_real_IsRadioChannel;
	iface->GetChannelURL = dvb_channel_list_real_GetChannelURL;
	iface->GetChannelInfos = dvb_channel_list_real_GetChannelInfos;
}


static void dvb_channel_list_instance_init (DVBChannelList * self) {
	self->priv = DVB_CHANNEL_LIST_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_channels);
}


static void dvb_channel_list_finalize (GObject* obj) {
	DVBChannelList * self;
	self = DVB_CHANNEL_LIST (obj);
	(self->priv->_channels_file == NULL) ? NULL : (self->priv->_channels_file = (g_object_unref (self->priv->_channels_file), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_channels);
	(self->channels == NULL) ? NULL : (self->channels = (g_object_unref (self->channels), NULL));
	G_OBJECT_CLASS (dvb_channel_list_parent_class)->finalize (obj);
}


GType dvb_channel_list_get_type (void) {
	static GType dvb_channel_list_type_id = 0;
	if (dvb_channel_list_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBChannelListClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_channel_list_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBChannelList), 0, (GInstanceInitFunc) dvb_channel_list_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) dvb_channel_list_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo dvb_id_bus_channel_list_info = { (GInterfaceInitFunc) dvb_channel_list_dvb_id_bus_channel_list_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_channel_list_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBChannelList", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_channel_list_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (dvb_channel_list_type_id, DVB_TYPE_ID_BUS_CHANNEL_LIST, &dvb_id_bus_channel_list_info);
	}
	return dvb_channel_list_type_id;
}


static void dvb_channel_list_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBChannelList * self;
	gpointer boxed;
	self = DVB_CHANNEL_LIST (object);
	switch (property_id) {
		case DVB_CHANNEL_LIST_CHANNELS_FILE:
		g_value_set_object (value, dvb_channel_list_get_channels_file (self));
		break;
		case DVB_CHANNEL_LIST_SIZE:
		g_value_set_int (value, dvb_channel_list_get_size (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_channel_list_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBChannelList * self;
	self = DVB_CHANNEL_LIST (object);
	switch (property_id) {
		case DVB_CHANNEL_LIST_CHANNELS_FILE:
		dvb_channel_list_set_channels_file (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}


static void _vala_dbus_unregister_object (gpointer connection, GObject* object) {
	char* path;
	path = g_object_steal_data ((GObject*) object, "dbus_object_path");
	dbus_connection_unregister_object_path (connection, path);
	g_free (path);
}




