
#include <src/Manager.h>
#include <gee/hashmap.h>
#include <gee/hashset.h>
#include <gee/map.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/collection.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib.h>
#include <gio/gio.h>
#include "src/Scanner.h"
#include "src/Constants.h"
#include "src/Device.h"
#include "src/Terrestrial/TerrestrialScanner.h"
#include "src/Satellite/SatelliteScanner.h"
#include "src/Cable/CableScanner.h"
#include "src/ChannelList.h"
#include "src/Factory.h"
#include "src/ConfigStore.h"
#include "src/ChangeType.h"
#include "src/Channel.h"
#include "src/Schedule.h"

typedef struct _DBusObjectVTable _DBusObjectVTable;

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};



struct _DVBManagerPrivate {
	GeeHashMap* scanners;
	GeeHashMap* recorders;
	GeeHashMap* channellists;
	GeeHashMap* devices;
	GeeHashMap* epgscanners;
	GeeHashSet* schedules;
	guint device_group_counter;
};

#define DVB_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_MANAGER, DVBManagerPrivate))
enum  {
	DVB_MANAGER_DUMMY_PROPERTY
};
static DVBManager* dvb_manager_instance = NULL;
static GStaticRecMutex dvb_manager_instance_mutex = {0};
static void _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed (DVBScanner* _sender, gpointer self);
static DVBDevice* dvb_manager_create_device (guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir);
static DBusGConnection* dvb_manager_get_dbus_connection (void);
static gboolean dvb_manager_device_is_in_any_group (DVBManager* self, DVBDevice* device);
static void dvb_manager_on_scanner_destroyed (DVBManager* self, DVBScanner* scanner);
static DVBDevice* dvb_manager_get_registered_device (DVBManager* self, guint adapter, guint frontend);
static DVBDeviceGroup* dvb_manager_get_device_group_of_device (DVBManager* self, DVBDevice* device);
static GObject * dvb_manager_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer dvb_manager_parent_class = NULL;
void _dvb_manager_dbus_unregister (DBusConnection* connection, void* user_data);
static DBusMessage* _dbus_dvb_manager_introspect (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_property_get_all (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetScannerForDevice (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetRegisteredDeviceGroups (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetTypeOfDeviceGroup (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetRecorder (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_AddDeviceToNewGroup (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_AddDeviceToExistingGroup (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_RemoveDeviceFromGroup (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetDeviceGroupName (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetChannelList (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetDeviceGroupMembers (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetNameOfRegisteredDevice (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_manager_GetSchedule (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static void _dbus_dvb_manager_changed (GObject* _sender, guint group_id, guint change_type, DBusConnection* _connection);
static void _dbus_dvb_manager_group_changed (GObject* _sender, guint group_id, guint adapter, guint frontend, guint change_type, DBusConnection* _connection);
static void dvb_manager_finalize (GObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);

static const DBusObjectPathVTable _dvb_manager_dbus_path_vtable = {_dvb_manager_dbus_unregister, dvb_manager_dbus_message};
static const _DBusObjectVTable _dvb_manager_dbus_vtable = {dvb_manager_dbus_register_object};

static void g_cclosure_user_marshal_VOID__UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_VOID__UINT_UINT_UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);

DVBManager* dvb_manager_get_instance (void) {
	g_static_rec_mutex_lock (&dvb_manager_instance_mutex);
	if (dvb_manager_instance == NULL) {
		DVBManager* _tmp0;
		_tmp0 = NULL;
		dvb_manager_instance = (_tmp0 = dvb_manager_new (), (dvb_manager_instance == NULL) ? NULL : (dvb_manager_instance = (g_object_unref (dvb_manager_instance), NULL)), _tmp0);
	}
	g_static_rec_mutex_unlock (&dvb_manager_instance_mutex);
	return dvb_manager_instance;
}


void dvb_manager_shutdown (void) {
	DVBManager* _tmp0;
	DVBManager* m;
	g_static_rec_mutex_lock (&dvb_manager_instance_mutex);
	_tmp0 = NULL;
	m = (_tmp0 = dvb_manager_instance, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	if (dvb_manager_instance != NULL) {
		DVBManager* _tmp7;
		{
			GeeCollection* _tmp1;
			GeeIterator* _tmp2;
			GeeIterator* _scanner_it;
			_tmp1 = NULL;
			_tmp2 = NULL;
			_scanner_it = (_tmp2 = gee_iterable_iterator ((GeeIterable*) (_tmp1 = gee_map_get_values ((GeeMap*) m->priv->scanners))), (_tmp1 == NULL) ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)), _tmp2);
			while (gee_iterator_next (_scanner_it)) {
				DVBScanner* scanner;
				scanner = (DVBScanner*) gee_iterator_get (_scanner_it);
				g_debug ("Manager.vala:69: Stopping scanner");
				dvb_scanner_Destroy (scanner);
				(scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL));
			}
			(_scanner_it == NULL) ? NULL : (_scanner_it = (g_object_unref (_scanner_it), NULL));
		}
		gee_map_clear ((GeeMap*) m->priv->scanners);
		{
			GeeCollection* _tmp3;
			GeeIterator* _tmp4;
			GeeIterator* _rec_it;
			_tmp3 = NULL;
			_tmp4 = NULL;
			_rec_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = gee_map_get_values ((GeeMap*) m->priv->recorders))), (_tmp3 == NULL) ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)), _tmp4);
			while (gee_iterator_next (_rec_it)) {
				DVBRecorder* rec;
				rec = (DVBRecorder*) gee_iterator_get (_rec_it);
				g_debug ("Manager.vala:75: Stopping recorder");
				dvb_recorder_stop (rec);
				(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
			}
			(_rec_it == NULL) ? NULL : (_rec_it = (g_object_unref (_rec_it), NULL));
		}
		gee_map_clear ((GeeMap*) m->priv->recorders);
		{
			GeeCollection* _tmp5;
			GeeIterator* _tmp6;
			GeeIterator* _escanner_it;
			_tmp5 = NULL;
			_tmp6 = NULL;
			_escanner_it = (_tmp6 = gee_iterable_iterator ((GeeIterable*) (_tmp5 = gee_map_get_values ((GeeMap*) m->priv->epgscanners))), (_tmp5 == NULL) ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)), _tmp6);
			while (gee_iterator_next (_escanner_it)) {
				DVBEPGScanner* escanner;
				escanner = (DVBEPGScanner*) gee_iterator_get (_escanner_it);
				g_debug ("Manager.vala:81: Stopping EPG scanner");
				dvb_epg_scanner_stop (escanner);
				(escanner == NULL) ? NULL : (escanner = (g_object_unref (escanner), NULL));
			}
			(_escanner_it == NULL) ? NULL : (_escanner_it = (g_object_unref (_escanner_it), NULL));
		}
		gee_map_clear ((GeeMap*) m->priv->epgscanners);
		gee_collection_clear ((GeeCollection*) m->priv->schedules);
		gee_map_clear ((GeeMap*) m->priv->channellists);
		gee_map_clear ((GeeMap*) m->priv->devices);
		_tmp7 = NULL;
		dvb_manager_instance = (_tmp7 = NULL, (dvb_manager_instance == NULL) ? NULL : (dvb_manager_instance = (g_object_unref (dvb_manager_instance), NULL)), _tmp7);
	}
	g_static_rec_mutex_unlock (&dvb_manager_instance_mutex);
	(m == NULL) ? NULL : (m = (g_object_unref (m), NULL));
}


static void _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed (DVBScanner* _sender, gpointer self) {
	dvb_manager_on_scanner_destroyed (self, _sender);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @returns: Object path and interface to the scanner service
         *
         * Get the object path of the channel scanner for this device.
         */
char** dvb_manager_GetScannerForDevice (DVBManager* self, guint adapter, guint frontend, int* result_length1) {
	char* path;
	DVBDevice* device;
	DVBDevice* reg_dev;
	char* dbusiface;
	char** _tmp22;
	const char* _tmp21;
	const char* _tmp20;
	char** _tmp23;
	char** _tmp24;
	g_return_val_if_fail (self != NULL, NULL);
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCANNER_PATH, adapter, frontend);
	device = NULL;
	reg_dev = dvb_manager_get_registered_device (self, adapter, frontend);
	if (reg_dev == NULL) {
		DVBDevice* _tmp0;
		/* Create new device*/
		_tmp0 = NULL;
		device = (_tmp0 = dvb_device_new (adapter, frontend, TRUE), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp0);
	} else {
		DVBDeviceGroup* _tmp1;
		DVBEPGScanner* _tmp2;
		DVBEPGScanner* epgscanner;
		DVBDevice* _tmp4;
		DVBDevice* _tmp3;
		/* Stop epgscanner for device if there's any*/
		_tmp1 = NULL;
		_tmp2 = NULL;
		epgscanner = (_tmp2 = dvb_manager_get_epg_scanner (self, _tmp1 = dvb_manager_get_device_group_of_device (self, reg_dev)), (_tmp1 == NULL) ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)), _tmp2);
		if (epgscanner != NULL) {
			dvb_epg_scanner_stop (epgscanner);
		}
		/* Assign existing device*/
		_tmp4 = NULL;
		_tmp3 = NULL;
		device = (_tmp4 = (_tmp3 = reg_dev, (_tmp3 == NULL) ? NULL : g_object_ref (_tmp3)), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp4);
		(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
	}
	dbusiface = NULL;
	switch (dvb_device_get_Type (device)) {
		case DVB_ADAPTER_TYPE_DVB_T:
		{
			char* _tmp5;
			_tmp5 = NULL;
			dbusiface = (_tmp5 = g_strdup ("org.gnome.DVB.Scanner.Terrestrial"), dbusiface = (g_free (dbusiface), NULL), _tmp5);
			break;
		}
		case DVB_ADAPTER_TYPE_DVB_S:
		{
			char* _tmp6;
			_tmp6 = NULL;
			dbusiface = (_tmp6 = g_strdup ("org.gnome.DVB.Scanner.Satellite"), dbusiface = (g_free (dbusiface), NULL), _tmp6);
			break;
		}
		case DVB_ADAPTER_TYPE_DVB_C:
		{
			char* _tmp7;
			_tmp7 = NULL;
			dbusiface = (_tmp7 = g_strdup ("org.gnome.DVB.Scanner.Cable"), dbusiface = (g_free (dbusiface), NULL), _tmp7);
			break;
		}
	}
	if (dbusiface == NULL) {
		char** _tmp8;
		char** _tmp9;
		char** _tmp10;
		g_critical ("Manager.vala:138: Unknown adapter type");
		_tmp8 = NULL;
		_tmp9 = NULL;
		_tmp10 = NULL;
		return (_tmp10 = (_tmp9 = (_tmp8 = g_new0 (char*, 2 + 1), _tmp8[0] = g_strdup (""), _tmp8[1] = g_strdup (""), _tmp8), *result_length1 = 2, _tmp9), path = (g_free (path), NULL), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL)), dbusiface = (g_free (dbusiface), NULL), _tmp10);
	}
	if (!gee_map_contains ((GeeMap*) self->priv->scanners, path)) {
		DVBScanner* scanner;
		DBusGConnection* conn;
		scanner = NULL;
		switch (dvb_device_get_Type (device)) {
			case DVB_ADAPTER_TYPE_DVB_T:
			{
				DVBScanner* _tmp11;
				_tmp11 = NULL;
				scanner = (_tmp11 = (DVBScanner*) dvb_terrestrial_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp11);
				break;
			}
			case DVB_ADAPTER_TYPE_DVB_S:
			{
				DVBScanner* _tmp12;
				_tmp12 = NULL;
				scanner = (_tmp12 = (DVBScanner*) dvb_satellite_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp12);
				break;
			}
			case DVB_ADAPTER_TYPE_DVB_C:
			{
				DVBScanner* _tmp13;
				_tmp13 = NULL;
				scanner = (_tmp13 = (DVBScanner*) dvb_cable_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp13);
				break;
			}
		}
		if (scanner == NULL) {
			char** _tmp14;
			char** _tmp15;
			char** _tmp16;
			g_critical ("Manager.vala:159: Unknown adapter type");
			_tmp14 = NULL;
			_tmp15 = NULL;
			_tmp16 = NULL;
			return (_tmp16 = (_tmp15 = (_tmp14 = g_new0 (char*, 2 + 1), _tmp14[0] = g_strdup (""), _tmp14[1] = g_strdup (""), _tmp14), *result_length1 = 2, _tmp15), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), path = (g_free (path), NULL), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL)), dbusiface = (g_free (dbusiface), NULL), _tmp16);
		}
		g_signal_connect_object (scanner, "destroyed", (GCallback) _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed, self, 0);
		gee_map_set ((GeeMap*) self->priv->scanners, path, scanner);
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char** _tmp17;
			char** _tmp18;
			char** _tmp19;
			_tmp17 = NULL;
			_tmp18 = NULL;
			_tmp19 = NULL;
			return (_tmp19 = (_tmp18 = (_tmp17 = g_new0 (char*, 0 + 1), _tmp17), *result_length1 = 0, _tmp18), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), path = (g_free (path), NULL), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL)), dbusiface = (g_free (dbusiface), NULL), _tmp19);
		}
		_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) scanner);
		g_debug ("Manager.vala:174: Created new Scanner D-Bus service for adapter %u, frontend %u (%s)", adapter, frontend, dbusiface);
		(scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL));
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
	}
	_tmp22 = NULL;
	_tmp21 = NULL;
	_tmp20 = NULL;
	_tmp23 = NULL;
	_tmp24 = NULL;
	return (_tmp24 = (_tmp23 = (_tmp22 = g_new0 (char*, 2 + 1), _tmp22[0] = (_tmp20 = path, (_tmp20 == NULL) ? NULL : g_strdup (_tmp20)), _tmp22[1] = (_tmp21 = dbusiface, (_tmp21 == NULL) ? NULL : g_strdup (_tmp21)), _tmp22), *result_length1 = 2, _tmp23), path = (g_free (path), NULL), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL)), dbusiface = (g_free (dbusiface), NULL), _tmp24);
}


/**
         * @returns: Device groups' ID
         */
guint* dvb_manager_GetRegisteredDeviceGroups (DVBManager* self, int* result_length1) {
	guint* _tmp1;
	gint devs_size;
	gint devs_length1;
	gint _tmp0;
	guint* devs;
	gint i;
	guint* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp1 = NULL;
	devs = (_tmp1 = g_new0 (guint, _tmp0 = gee_map_get_size ((GeeMap*) self->priv->devices)), devs_length1 = _tmp0, devs_size = devs_length1, _tmp1);
	i = 0;
	{
		GeeSet* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* _key_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		_key_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = gee_map_get_keys ((GeeMap*) self->priv->devices))), (_tmp2 == NULL) ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (_key_it)) {
			guint key;
			key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
			devs[i] = key;
			i++;
		}
		(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
	}
	_tmp4 = NULL;
	return (_tmp4 = devs, *result_length1 = devs_length1, _tmp4);
}


/**
         * @group_id: ID of device group
         * @returns: Name of adapter type the group holds
         * or an empty string when group with given id doesn't exist.
         */
char* dvb_manager_GetTypeOfDeviceGroup (DVBManager* self, guint group_id) {
	g_return_val_if_fail (self != NULL, NULL);
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		char* type_str;
		char* _tmp4;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		type_str = NULL;
		switch (dvb_device_group_get_Type (devgroup)) {
			case DVB_ADAPTER_TYPE_DVB_T:
			{
				char* _tmp0;
				_tmp0 = NULL;
				type_str = (_tmp0 = g_strdup ("DVB-T"), type_str = (g_free (type_str), NULL), _tmp0);
				break;
			}
			case DVB_ADAPTER_TYPE_DVB_S:
			{
				char* _tmp1;
				_tmp1 = NULL;
				type_str = (_tmp1 = g_strdup ("DVB-S"), type_str = (g_free (type_str), NULL), _tmp1);
				break;
			}
			case DVB_ADAPTER_TYPE_DVB_C:
			{
				char* _tmp2;
				_tmp2 = NULL;
				type_str = (_tmp2 = g_strdup ("DVB-C"), type_str = (g_free (type_str), NULL), _tmp2);
				break;
			}
			default:
			{
				char* _tmp3;
				_tmp3 = NULL;
				type_str = (_tmp3 = g_strdup (""), type_str = (g_free (type_str), NULL), _tmp3);
				break;
			}
		}
		_tmp4 = NULL;
		return (_tmp4 = type_str, (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp4);
	}
	return g_strdup ("");
}


/**
         * @group_id: ID of device group
         * @returns: Object path of the device's recorder
         * 
         * Returns the object path to the device's recorder.
         * The device group must be registered with AddDeviceToNewGroup () first.
         */
char* dvb_manager_GetRecorder (DVBManager* self, guint group_id) {
	char* path;
	g_return_val_if_fail (self != NULL, NULL);
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_RECORDER_PATH, group_id);
	if (!gee_map_contains ((GeeMap*) self->priv->recorders, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		DVBRecorder* recorder;
		DBusGConnection* conn;
		g_debug ("Manager.vala:226: Creating new Recorder for group %u", group_id);
		devgroup = dvb_manager_get_device_group_if_exists (self, group_id);
		if (devgroup == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = g_strdup (""), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), path = (g_free (path), NULL), _tmp0);
		}
		recorder = dvb_recorder_new (devgroup);
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char* _tmp1;
			_tmp1 = NULL;
			return (_tmp1 = g_strdup (""), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (recorder == NULL) ? NULL : (recorder = (g_object_unref (recorder), NULL)), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), path = (g_free (path), NULL), _tmp1);
		}
		_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) recorder);
		gee_map_set ((GeeMap*) self->priv->recorders, GUINT_TO_POINTER (group_id), recorder);
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		(recorder == NULL) ? NULL : (recorder = (g_object_unref (recorder), NULL));
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
	}
	return path;
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @channels_conf: Path to channels.conf for this device
         * @recordings_dir: Path where the recordings should be stored
         * @name: Name of group
         * @returns: TRUE when the device has been registered successfully
         *
         * Creates a new DeviceGroup and new DVB device whereas the
         * DVB device is the reference device of this group (i.e.
         * all other devices of this group will inherit the settings
         * of the reference device).
         */
gboolean dvb_manager_AddDeviceToNewGroup (DVBManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name) {
	DVBDevice* device;
	DVBDeviceGroup* devgroup;
	gboolean _tmp2;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (channels_conf != NULL, FALSE);
	g_return_val_if_fail (recordings_dir != NULL, FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	device = dvb_manager_create_device (adapter, frontend, channels_conf, recordings_dir);
	if (device == NULL) {
		gboolean _tmp0;
		return (_tmp0 = FALSE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp0);
	}
	/* Check if device is already assigned to other group*/
	if (dvb_manager_device_is_in_any_group (self, device)) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp1);
	}
	self->priv->device_group_counter++;
	devgroup = dvb_device_group_new (self->priv->device_group_counter, device);
	dvb_device_group_set_Name (devgroup, name);
	dvb_manager_add_device_group (self, devgroup);
	g_signal_emit_by_name (self, "changed", self->priv->device_group_counter, (guint) DVB_CHANGE_TYPE_ADDED);
	return (_tmp2 = TRUE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp2);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @group_id: ID of device group
         * @returns: TRUE when the device has been registered successfully
         *
         * Creates a new device and adds it to the specified DeviceGroup.
         * The new device will inherit all settings from the group's
         * reference device.
         */
gboolean dvb_manager_AddDeviceToExistingGroup (DVBManager* self, guint adapter, guint frontend, guint group_id) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDevice* device;
		DVBDeviceGroup* devgroup;
		/* When the device is already registered we
		 might see some errors if the device is
		 currently in use*/
		device = dvb_device_new (adapter, frontend, TRUE);
		if (device == NULL) {
			gboolean _tmp0;
			return (_tmp0 = FALSE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp0);
		}
		if (dvb_manager_device_is_in_any_group (self, device)) {
			gboolean _tmp1;
			g_debug ("Device with adapter %u, frontend %u is" "already part of a group", adapter, frontend);
			return (_tmp1 = FALSE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp1);
		}
		g_debug ("Manager.vala:310: Adding device with adapter %u, frontend %u to group %u", adapter, frontend, group_id);
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		if (dvb_device_group_add (devgroup, device)) {
			gboolean _tmp2;
			dvb_config_store_add_device_to_group (dvb_factory_get_config_store (), device, devgroup);
			g_signal_emit_by_name (self, "group-changed", group_id, adapter, frontend, (guint) DVB_CHANGE_TYPE_ADDED);
			return (_tmp2 = TRUE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp2);
		}
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	}
	return FALSE;
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @group_id: ID of device group
         * @returns: TRUE when device has been removed successfully
         *
         * Removes the device from the specified group. If the group contains
         * no devices after the removal it's removed as well.
         */
gboolean dvb_manager_RemoveDeviceFromGroup (DVBManager* self, guint adapter, guint frontend, guint group_id) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		DVBDevice* dev;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		dev = dvb_device_new (adapter, frontend, FALSE);
		if (dvb_device_group_contains (devgroup, dev)) {
			if (dvb_device_group_remove (devgroup, dev)) {
				DVBEPGScanner* epgscanner;
				gboolean _tmp0;
				/* Stop epgscanner, because it might use the
				 device we want to unregister*/
				epgscanner = dvb_manager_get_epg_scanner (self, devgroup);
				if (epgscanner != NULL) {
					dvb_epg_scanner_stop (epgscanner);
				}
				dvb_config_store_remove_device_from_group (dvb_factory_get_config_store (), dev, devgroup);
				g_signal_emit_by_name (self, "group-changed", group_id, adapter, frontend, (guint) DVB_CHANGE_TYPE_DELETED);
				/* Group has no devices anymore, delete it*/
				if (dvb_device_group_get_size (devgroup) == 0) {
					if (gee_map_remove ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
						/* Remove EPG scanner, too*/
						if (epgscanner != NULL) {
							gee_map_remove ((GeeMap*) self->priv->epgscanners, GUINT_TO_POINTER (dvb_device_group_get_Id (devgroup)));
						}
						dvb_config_store_remove_device_group (dvb_factory_get_config_store (), devgroup);
						g_signal_emit_by_name (self, "changed", group_id, (guint) DVB_CHANGE_TYPE_DELETED);
					}
				} else {
					/* We still have a device, start EPG scanner again*/
					if (epgscanner != NULL) {
						dvb_epg_scanner_start (epgscanner);
					}
				}
				return (_tmp0 = TRUE, (epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL)), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL)), _tmp0);
			}
		}
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
	}
	return FALSE;
}


/**
         * @group_id: ID of device group
         * @returns: Name of specified device group or
         * empty string if group with given ID doesn't exist
         */
char* dvb_manager_GetDeviceGroupName (DVBManager* self, guint group_id) {
	char* val;
	g_return_val_if_fail (self != NULL, NULL);
	val = g_strdup ("");
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		char* _tmp1;
		const char* _tmp0;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		_tmp1 = NULL;
		_tmp0 = NULL;
		val = (_tmp1 = (_tmp0 = dvb_device_group_get_Name (devgroup), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), val = (g_free (val), NULL), _tmp1);
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	}
	return val;
}


/**
         * @group_id: ID of device group
         * @returns: Object path to the ChannelList service for this device
         *
         * The device group must be registered with AddDeviceToNewGroup () first.
         */
char* dvb_manager_GetChannelList (DVBManager* self, guint group_id) {
	char* path;
	g_return_val_if_fail (self != NULL, NULL);
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_CHANNEL_LIST_PATH, group_id);
	if (!gee_map_contains ((GeeMap*) self->priv->channellists, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		DVBChannelList* _tmp1;
		DVBChannelList* channels;
		DBusGConnection* conn;
		g_debug ("Manager.vala:403: Creating new ChannelList D-Bus service for group %u", group_id);
		devgroup = dvb_manager_get_device_group_if_exists (self, group_id);
		if (devgroup == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = g_strdup (""), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), path = (g_free (path), NULL), _tmp0);
		}
		_tmp1 = NULL;
		channels = (_tmp1 = dvb_device_group_get_Channels (devgroup), (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char* _tmp2;
			_tmp2 = NULL;
			return (_tmp2 = g_strdup (""), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), path = (g_free (path), NULL), _tmp2);
		}
		_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) channels);
		gee_map_set ((GeeMap*) self->priv->channellists, GUINT_TO_POINTER (group_id), channels);
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
	}
	return path;
}


/**
         * @group_id: ID of device group
         * @returns: List of paths to the devices that are part of
         * the specified group (e.g. /dev/dvb/adapter0/frontend0)
         */
char** dvb_manager_GetDeviceGroupMembers (DVBManager* self, guint group_id, int* result_length1) {
	gint groupdevs_size;
	gint groupdevs_length1;
	char** groupdevs;
	char** _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	groupdevs = (groupdevs_length1 = 0, NULL);
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		char** _tmp1;
		gint _tmp0;
		gint i;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		_tmp1 = NULL;
		groupdevs = (_tmp1 = g_new0 (char*, (_tmp0 = dvb_device_group_get_size (devgroup)) + 1), groupdevs = (_vala_array_free (groupdevs, groupdevs_length1, (GDestroyNotify) g_free), NULL), groupdevs_length1 = _tmp0, groupdevs_size = groupdevs_length1, _tmp1);
		i = 0;
		{
			GeeIterator* _dev_it;
			_dev_it = gee_iterable_iterator ((GeeIterable*) devgroup);
			while (gee_iterator_next (_dev_it)) {
				DVBDevice* dev;
				char* _tmp2;
				dev = (DVBDevice*) gee_iterator_get (_dev_it);
				_tmp2 = NULL;
				groupdevs[i] = (_tmp2 = g_strdup_printf (DVB_CONSTANTS_DVB_DEVICE_PATH, dvb_device_get_Adapter (dev), dvb_device_get_Frontend (dev)), groupdevs[i] = (g_free (groupdevs[i]), NULL), _tmp2);
				i++;
				(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
			}
			(_dev_it == NULL) ? NULL : (_dev_it = (g_object_unref (_dev_it), NULL));
		}
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	} else {
		char** _tmp3;
		_tmp3 = NULL;
		groupdevs = (_tmp3 = g_new0 (char*, 0 + 1), groupdevs = (_vala_array_free (groupdevs, groupdevs_length1, (GDestroyNotify) g_free), NULL), groupdevs_length1 = 0, groupdevs_size = groupdevs_length1, _tmp3);
	}
	_tmp4 = NULL;
	return (_tmp4 = groupdevs, *result_length1 = groupdevs_length1, _tmp4);
}


/**
         * @adapter: Adapter of device
         * @frontend: Frontend of device
         * @returns: The name of the device or "Unknown"
         *
         * The device must be part of group, otherwise "Unknown"
         * is returned.
         */
char* dvb_manager_GetNameOfRegisteredDevice (DVBManager* self, guint adapter, guint frontend) {
	DVBDevice* dev;
	g_return_val_if_fail (self != NULL, NULL);
	dev = dvb_manager_get_registered_device (self, adapter, frontend);
	if (dev == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = g_strdup ("Unknown"), (dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL)), _tmp0);
	} else {
		const char* _tmp1;
		char* _tmp2;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = dvb_device_get_Name (dev), (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), (dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL)), _tmp2);
	}
	(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
}


char* dvb_manager_GetSchedule (DVBManager* self, guint group_id, guint channel_sid) {
	g_return_val_if_fail (self != NULL, NULL);
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		if (dvb_channel_list_contains (dvb_device_group_get_Channels (devgroup), channel_sid)) {
			char* path;
			char* _tmp4;
			path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCHEDULE_PATH, group_id, channel_sid);
			if (!gee_collection_contains ((GeeCollection*) self->priv->schedules, path)) {
				DBusGConnection* conn;
				DVBSchedule* _tmp2;
				DVBChannel* _tmp1;
				DVBSchedule* _tmp3;
				DVBSchedule* schedule;
				conn = dvb_manager_get_dbus_connection ();
				if (conn == NULL) {
					char* _tmp0;
					_tmp0 = NULL;
					return (_tmp0 = g_strdup (""), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), path = (g_free (path), NULL), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp0);
				}
				_tmp2 = NULL;
				_tmp1 = NULL;
				_tmp3 = NULL;
				schedule = (_tmp3 = (_tmp2 = dvb_channel_get_Schedule (_tmp1 = dvb_channel_list_get (dvb_device_group_get_Channels (devgroup), channel_sid)), (_tmp2 == NULL) ? NULL : g_object_ref (_tmp2)), (_tmp1 == NULL) ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)), _tmp3);
				_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) schedule);
				gee_collection_add ((GeeCollection*) self->priv->schedules, path);
				(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
				(schedule == NULL) ? NULL : (schedule = (g_object_unref (schedule), NULL));
			}
			_tmp4 = NULL;
			return (_tmp4 = path, (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp4);
		}
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	}
	return g_strdup ("");
}


/**
         * @returns: Whether the device has been added successfully
         *
         * Register device, create Recorder and ChannelList D-Bus service
         */
gboolean dvb_manager_add_device_group (DVBManager* self, DVBDeviceGroup* devgroup) {
	char* rec_path;
	char* channels_path;
	gboolean _tmp2;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (devgroup != NULL, FALSE);
	g_debug ("Manager.vala:500: Adding device group %u with %d devices", dvb_device_group_get_Id (devgroup), dvb_device_group_get_size (devgroup));
	gee_map_set ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (dvb_device_group_get_Id (devgroup)), devgroup);
	rec_path = dvb_manager_GetRecorder (self, dvb_device_group_get_Id (devgroup));
	if (_vala_strcmp0 (rec_path, "") == 0) {
		gboolean _tmp0;
		return (_tmp0 = FALSE, rec_path = (g_free (rec_path), NULL), _tmp0);
	}
	channels_path = dvb_manager_GetChannelList (self, dvb_device_group_get_Id (devgroup));
	if (_vala_strcmp0 (channels_path, "") == 0) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, rec_path = (g_free (rec_path), NULL), channels_path = (g_free (channels_path), NULL), _tmp1);
	}
	dvb_config_store_add_device_group (dvb_factory_get_config_store (), devgroup);
	if (dvb_device_group_get_Id (devgroup) > self->priv->device_group_counter) {
		self->priv->device_group_counter = dvb_device_group_get_Id (devgroup);
	}
	return (_tmp2 = TRUE, rec_path = (g_free (rec_path), NULL), channels_path = (g_free (channels_path), NULL), _tmp2);
}


DVBRecorder* dvb_manager_get_recorder_for_device_group (DVBManager* self, DVBDeviceGroup* devgroup) {
	guint id;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (devgroup != NULL, NULL);
	id = dvb_device_group_get_Id (devgroup);
	if (gee_map_contains ((GeeMap*) self->priv->recorders, GUINT_TO_POINTER (id))) {
		return (DVBRecorder*) gee_map_get ((GeeMap*) self->priv->recorders, GUINT_TO_POINTER (id));
	} else {
		return NULL;
	}
}


DVBEPGScanner* dvb_manager_get_epg_scanner (DVBManager* self, DVBDeviceGroup* devgroup) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (devgroup != NULL, NULL);
	return (DVBEPGScanner*) gee_map_get ((GeeMap*) self->priv->epgscanners, GUINT_TO_POINTER (dvb_device_group_get_Id (devgroup)));
}


void dvb_manager_create_and_start_epg_scanner (DVBManager* self, DVBDeviceGroup* devgroup) {
	DVBEPGScanner* epgscanner;
	g_return_if_fail (self != NULL);
	g_return_if_fail (devgroup != NULL);
	g_debug ("Manager.vala:534: Creating new EPG scanner for device group %u", dvb_device_group_get_Id (devgroup));
	epgscanner = dvb_epg_scanner_new (devgroup);
	dvb_epg_scanner_start (epgscanner);
	gee_map_set ((GeeMap*) self->priv->epgscanners, GUINT_TO_POINTER (dvb_device_group_get_Id (devgroup)), epgscanner);
	(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
}


static DVBDevice* dvb_manager_create_device (guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir) {
	GError * inner_error;
	GFile* channelsfile;
	GFile* recdir;
	DVBDevice* device;
	DVBChannelList* channels;
	DVBDevice* _tmp3;
	g_return_val_if_fail (channels_conf != NULL, NULL);
	g_return_val_if_fail (recordings_dir != NULL, NULL);
	inner_error = NULL;
	/* TODO Check if adapter and frontend exists*/
	channelsfile = g_file_new_for_path (channels_conf);
	recdir = g_file_new_for_path (recordings_dir);
	device = dvb_device_new (adapter, frontend, TRUE);
	dvb_device_set_RecordingsDirectory (device, recdir);
	channels = NULL;
	{
		DVBChannelList* _tmp0;
		DVBChannelList* _tmp1;
		_tmp0 = dvb_channel_list_restore_from_file (channelsfile, dvb_device_get_Type (device), &inner_error);
		if (inner_error != NULL) {
			goto __catch7_g_error;
			goto __finally7;
		}
		_tmp1 = NULL;
		channels = (_tmp1 = _tmp0, (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp1);
	}
	goto __finally7;
	__catch7_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			DVBDevice* _tmp2;
			g_critical ("Manager.vala:555: Could not create channels list from %s: %s", channels_conf, e->message);
			_tmp2 = NULL;
			return (_tmp2 = NULL, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL)), (recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL)), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp2);
		}
	}
	__finally7:
	if (inner_error != NULL) {
		(channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL));
		(recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL));
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
	dvb_device_set_Channels (device, channels);
	_tmp3 = NULL;
	return (_tmp3 = device, (channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL)), (recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL)), (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp3);
}


static DBusGConnection* dvb_manager_get_dbus_connection (void) {
	GError * inner_error;
	DBusGConnection* conn;
	inner_error = NULL;
	conn = NULL;
	{
		DBusGConnection* _tmp0;
		DBusGConnection* _tmp1;
		_tmp0 = dbus_g_bus_get (DBUS_BUS_SESSION, &inner_error);
		if (inner_error != NULL) {
			goto __catch8_g_error;
			goto __finally8;
		}
		_tmp1 = NULL;
		conn = (_tmp1 = _tmp0, (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), _tmp1);
	}
	goto __finally8;
	__catch8_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			DBusGConnection* _tmp2;
			g_error ("Manager.vala:570: Could not get D-Bus session bus: %s", e->message);
			_tmp2 = NULL;
			return (_tmp2 = NULL, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), _tmp2);
		}
	}
	__finally8:
	if (inner_error != NULL) {
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
	return conn;
}


DVBDeviceGroup* dvb_manager_get_device_group_if_exists (DVBManager* self, guint group_id) {
	g_return_val_if_fail (self != NULL, NULL);
	;
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		return (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
	} else {
		return NULL;
	}
}


static gboolean dvb_manager_device_is_in_any_group (DVBManager* self, DVBDevice* device) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (device != NULL, FALSE);
	{
		GeeSet* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _group_id_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_group_id_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = gee_map_get_keys ((GeeMap*) self->priv->devices))), (_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_group_id_it)) {
			guint group_id;
			DVBDeviceGroup* devgroup;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
			devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
			if (dvb_device_group_contains (devgroup, device)) {
				gboolean _tmp2;
				return (_tmp2 = TRUE, (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL)), _tmp2);
			}
			(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		}
		(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
	}
	return FALSE;
}


static void dvb_manager_on_scanner_destroyed (DVBManager* self, DVBScanner* scanner) {
	guint adapter;
	guint frontend;
	char* path;
	DVBDeviceGroup* devgroup;
	g_return_if_fail (self != NULL);
	g_return_if_fail (scanner != NULL);
	adapter = dvb_device_get_Adapter (dvb_scanner_get_Device (scanner));
	frontend = dvb_device_get_Frontend (dvb_scanner_get_Device (scanner));
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCANNER_PATH, adapter, frontend);
	gee_map_remove ((GeeMap*) self->priv->scanners, path);
	g_debug ("Manager.vala:598: Destroying scanner for adapter %u, frontend %u (%s)", adapter, frontend, path);
	/* Start epgscanner for device again if there was one*/
	devgroup = dvb_manager_get_device_group_of_device (self, dvb_scanner_get_Device (scanner));
	if (devgroup != NULL) {
		DVBEPGScanner* epgscanner;
		epgscanner = dvb_manager_get_epg_scanner (self, devgroup);
		if (epgscanner != NULL) {
			dvb_epg_scanner_start (epgscanner);
		}
		(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
	}
	path = (g_free (path), NULL);
	(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
}


static DVBDevice* dvb_manager_get_registered_device (DVBManager* self, guint adapter, guint frontend) {
	DVBDevice* fake_device;
	DVBDevice* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	fake_device = dvb_device_new (adapter, frontend, FALSE);
	{
		GeeSet* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _group_id_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_group_id_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = gee_map_get_keys ((GeeMap*) self->priv->devices))), (_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_group_id_it)) {
			guint group_id;
			DVBDeviceGroup* devgroup;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
			devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
			if (dvb_device_group_contains (devgroup, fake_device)) {
				{
					GeeIterator* _device_it;
					_device_it = gee_iterable_iterator ((GeeIterable*) devgroup);
					while (gee_iterator_next (_device_it)) {
						DVBDevice* device;
						device = (DVBDevice*) gee_iterator_get (_device_it);
						if (dvb_device_equal (fake_device, device)) {
							DVBDevice* _tmp2;
							_tmp2 = NULL;
							return (_tmp2 = device, (_device_it == NULL) ? NULL : (_device_it = (g_object_unref (_device_it), NULL)), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL)), (fake_device == NULL) ? NULL : (fake_device = (g_object_unref (fake_device), NULL)), _tmp2);
						}
						(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
					}
					(_device_it == NULL) ? NULL : (_device_it = (g_object_unref (_device_it), NULL));
				}
			}
			(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		}
		(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
	}
	_tmp3 = NULL;
	return (_tmp3 = NULL, (fake_device == NULL) ? NULL : (fake_device = (g_object_unref (fake_device), NULL)), _tmp3);
}


static DVBDeviceGroup* dvb_manager_get_device_group_of_device (DVBManager* self, DVBDevice* device) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (device != NULL, NULL);
	{
		GeeSet* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _group_id_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_group_id_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = gee_map_get_keys ((GeeMap*) self->priv->devices))), (_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_group_id_it)) {
			guint group_id;
			DVBDeviceGroup* devgroup;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
			devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
			if (dvb_device_group_contains (devgroup, device)) {
				{
					GeeIterator* _grp_device_it;
					_grp_device_it = gee_iterable_iterator ((GeeIterable*) devgroup);
					while (gee_iterator_next (_grp_device_it)) {
						DVBDevice* grp_device;
						grp_device = (DVBDevice*) gee_iterator_get (_grp_device_it);
						if (dvb_device_equal (grp_device, device)) {
							DVBDeviceGroup* _tmp2;
							_tmp2 = NULL;
							return (_tmp2 = devgroup, (grp_device == NULL) ? NULL : (grp_device = (g_object_unref (grp_device), NULL)), (_grp_device_it == NULL) ? NULL : (_grp_device_it = (g_object_unref (_grp_device_it), NULL)), (_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL)), _tmp2);
						}
						(grp_device == NULL) ? NULL : (grp_device = (g_object_unref (grp_device), NULL));
					}
					(_grp_device_it == NULL) ? NULL : (_grp_device_it = (g_object_unref (_grp_device_it), NULL));
				}
			}
			(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		}
		(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
	}
	return NULL;
}


DVBManager* dvb_manager_construct (GType object_type) {
	DVBManager * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBManager* dvb_manager_new (void) {
	return dvb_manager_construct (DVB_TYPE_MANAGER);
}


static GObject * dvb_manager_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBManagerClass * klass;
	GObjectClass * parent_class;
	DVBManager * self;
	klass = DVB_MANAGER_CLASS (g_type_class_peek (DVB_TYPE_MANAGER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_MANAGER (obj);
	{
		GeeHashMap* _tmp1;
		GeeHashMap* _tmp2;
		GeeHashMap* _tmp3;
		GeeHashMap* _tmp4;
		GeeHashMap* _tmp5;
		GeeHashSet* _tmp6;
		_tmp1 = NULL;
		self->priv->scanners = (_tmp1 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, DVB_TYPE_SCANNER, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_str_hash, g_str_equal, g_direct_equal), (self->priv->scanners == NULL) ? NULL : (self->priv->scanners = (g_object_unref (self->priv->scanners), NULL)), _tmp1);
		_tmp2 = NULL;
		self->priv->recorders = (_tmp2 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_RECORDER, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->recorders == NULL) ? NULL : (self->priv->recorders = (g_object_unref (self->priv->recorders), NULL)), _tmp2);
		_tmp3 = NULL;
		self->priv->channellists = (_tmp3 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_CHANNEL_LIST, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->channellists == NULL) ? NULL : (self->priv->channellists = (g_object_unref (self->priv->channellists), NULL)), _tmp3);
		_tmp4 = NULL;
		self->priv->devices = (_tmp4 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_DEVICE_GROUP, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->devices == NULL) ? NULL : (self->priv->devices = (g_object_unref (self->priv->devices), NULL)), _tmp4);
		_tmp5 = NULL;
		self->priv->epgscanners = (_tmp5 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_EPG_SCANNER, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->epgscanners == NULL) ? NULL : (self->priv->epgscanners = (g_object_unref (self->priv->epgscanners), NULL)), _tmp5);
		_tmp6 = NULL;
		self->priv->schedules = (_tmp6 = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->priv->schedules == NULL) ? NULL : (self->priv->schedules = (g_object_unref (self->priv->schedules), NULL)), _tmp6);
		self->priv->device_group_counter = (guint) 0;
	}
	return obj;
}


void _dvb_manager_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_manager_introspect (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.Manager\">\n  <method name=\"GetScannerForDevice\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"as\" direction=\"out\"/>\n  </method>\n  <method name=\"GetRegisteredDeviceGroups\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetTypeOfDeviceGroup\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetRecorder\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"AddDeviceToNewGroup\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"channels_conf\" type=\"s\" direction=\"in\"/>\n    <arg name=\"recordings_dir\" type=\"s\" direction=\"in\"/>\n    <arg name=\"name\" type=\"s\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"AddDeviceToExistingGroup\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"RemoveDeviceFromGroup\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDeviceGroupName\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelList\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDeviceGroupMembers\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"as\" direction=\"out\"/>\n  </method>\n  <method name=\"GetNameOfRegisteredDevice\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetSchedule\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"channel_sid\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <signal name=\"Changed\">\n    <arg name=\"group_id\" type=\"u\"/>\n    <arg name=\"change_type\" type=\"u\"/>\n  </signal>\n  <signal name=\"GroupChanged\">\n    <arg name=\"group_id\" type=\"u\"/>\n    <arg name=\"adapter\" type=\"u\"/>\n    <arg name=\"frontend\" type=\"u\"/>\n    <arg name=\"change_type\" type=\"u\"/>\n  </signal>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_property_get_all (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter, reply_iter, subiter;
	char* interface_name;
	const char* _tmp7;
	if (strcmp (dbus_message_get_signature (message), "s")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &reply_iter);
	dbus_message_iter_get_basic (&iter, &_tmp7);
	dbus_message_iter_next (&iter);
	interface_name = g_strdup (_tmp7);
	if (strcmp (interface_name, "org.gnome.DVB.Manager") == 0) {
		dbus_message_iter_open_container (&reply_iter, DBUS_TYPE_ARRAY, "{sv}", &subiter);
		dbus_message_iter_close_container (&reply_iter, &subiter);
	} else {
		return NULL;
	}
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetScannerForDevice (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint adapter;
	dbus_uint32_t _tmp8;
	guint frontend;
	dbus_uint32_t _tmp9;
	int result_length1;
	char** result;
	DBusMessage* reply;
	char** _tmp10;
	DBusMessageIter _tmp11;
	int _tmp12;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uu")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	adapter = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp8);
	dbus_message_iter_next (&iter);
	adapter = _tmp8;
	frontend = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp9);
	dbus_message_iter_next (&iter);
	frontend = _tmp9;
	result_length1 = 0;
	result = dvb_manager_GetScannerForDevice (self, adapter, frontend, &result_length1);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp10 = result;
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, "s", &_tmp11);
	for (_tmp12 = 0; _tmp12 < result_length1; _tmp12++) {
		const char* _tmp13;
		_tmp13 = *(_tmp10++);
		dbus_message_iter_append_basic (&_tmp11, DBUS_TYPE_STRING, &_tmp13);
	}
	dbus_message_iter_close_container (&iter, &_tmp11);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetRegisteredDeviceGroups (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	int result_length1;
	guint* result;
	DBusMessage* reply;
	guint* _tmp14;
	DBusMessageIter _tmp15;
	int _tmp16;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	result_length1 = 0;
	result = dvb_manager_GetRegisteredDeviceGroups (self, &result_length1);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp14 = result;
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, "u", &_tmp15);
	for (_tmp16 = 0; _tmp16 < result_length1; _tmp16++) {
		dbus_uint32_t _tmp17;
		_tmp17 = *(_tmp14++);
		dbus_message_iter_append_basic (&_tmp15, DBUS_TYPE_UINT32, &_tmp17);
	}
	dbus_message_iter_close_container (&iter, &_tmp15);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetTypeOfDeviceGroup (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint group_id;
	dbus_uint32_t _tmp18;
	char* result;
	DBusMessage* reply;
	const char* _tmp19;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "u")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp18);
	dbus_message_iter_next (&iter);
	group_id = _tmp18;
	result = dvb_manager_GetTypeOfDeviceGroup (self, group_id);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp19 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &_tmp19);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetRecorder (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint group_id;
	dbus_uint32_t _tmp20;
	char* result;
	DBusMessage* reply;
	const char* _tmp21;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "u")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp20);
	dbus_message_iter_next (&iter);
	group_id = _tmp20;
	result = dvb_manager_GetRecorder (self, group_id);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp21 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &_tmp21);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_AddDeviceToNewGroup (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint adapter;
	dbus_uint32_t _tmp22;
	guint frontend;
	dbus_uint32_t _tmp23;
	const char* channels_conf;
	const char* _tmp24;
	const char* recordings_dir;
	const char* _tmp25;
	const char* name;
	const char* _tmp26;
	gboolean result;
	DBusMessage* reply;
	dbus_bool_t _tmp27;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uusss")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	adapter = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp22);
	dbus_message_iter_next (&iter);
	adapter = _tmp22;
	frontend = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp23);
	dbus_message_iter_next (&iter);
	frontend = _tmp23;
	channels_conf = NULL;
	dbus_message_iter_get_basic (&iter, &_tmp24);
	dbus_message_iter_next (&iter);
	channels_conf = g_strdup (_tmp24);
	recordings_dir = NULL;
	dbus_message_iter_get_basic (&iter, &_tmp25);
	dbus_message_iter_next (&iter);
	recordings_dir = g_strdup (_tmp25);
	name = NULL;
	dbus_message_iter_get_basic (&iter, &_tmp26);
	dbus_message_iter_next (&iter);
	name = g_strdup (_tmp26);
	result = dvb_manager_AddDeviceToNewGroup (self, adapter, frontend, channels_conf, recordings_dir, name);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp27 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_BOOLEAN, &_tmp27);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_AddDeviceToExistingGroup (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint adapter;
	dbus_uint32_t _tmp28;
	guint frontend;
	dbus_uint32_t _tmp29;
	guint group_id;
	dbus_uint32_t _tmp30;
	gboolean result;
	DBusMessage* reply;
	dbus_bool_t _tmp31;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uuu")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	adapter = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp28);
	dbus_message_iter_next (&iter);
	adapter = _tmp28;
	frontend = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp29);
	dbus_message_iter_next (&iter);
	frontend = _tmp29;
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp30);
	dbus_message_iter_next (&iter);
	group_id = _tmp30;
	result = dvb_manager_AddDeviceToExistingGroup (self, adapter, frontend, group_id);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp31 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_BOOLEAN, &_tmp31);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_RemoveDeviceFromGroup (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint adapter;
	dbus_uint32_t _tmp32;
	guint frontend;
	dbus_uint32_t _tmp33;
	guint group_id;
	dbus_uint32_t _tmp34;
	gboolean result;
	DBusMessage* reply;
	dbus_bool_t _tmp35;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uuu")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	adapter = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp32);
	dbus_message_iter_next (&iter);
	adapter = _tmp32;
	frontend = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp33);
	dbus_message_iter_next (&iter);
	frontend = _tmp33;
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp34);
	dbus_message_iter_next (&iter);
	group_id = _tmp34;
	result = dvb_manager_RemoveDeviceFromGroup (self, adapter, frontend, group_id);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp35 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_BOOLEAN, &_tmp35);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetDeviceGroupName (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint group_id;
	dbus_uint32_t _tmp36;
	char* result;
	DBusMessage* reply;
	const char* _tmp37;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "u")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp36);
	dbus_message_iter_next (&iter);
	group_id = _tmp36;
	result = dvb_manager_GetDeviceGroupName (self, group_id);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp37 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &_tmp37);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetChannelList (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint group_id;
	dbus_uint32_t _tmp38;
	char* result;
	DBusMessage* reply;
	const char* _tmp39;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "u")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp38);
	dbus_message_iter_next (&iter);
	group_id = _tmp38;
	result = dvb_manager_GetChannelList (self, group_id);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp39 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &_tmp39);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetDeviceGroupMembers (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint group_id;
	dbus_uint32_t _tmp40;
	int result_length1;
	char** result;
	DBusMessage* reply;
	char** _tmp41;
	DBusMessageIter _tmp42;
	int _tmp43;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "u")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp40);
	dbus_message_iter_next (&iter);
	group_id = _tmp40;
	result_length1 = 0;
	result = dvb_manager_GetDeviceGroupMembers (self, group_id, &result_length1);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp41 = result;
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, "s", &_tmp42);
	for (_tmp43 = 0; _tmp43 < result_length1; _tmp43++) {
		const char* _tmp44;
		_tmp44 = *(_tmp41++);
		dbus_message_iter_append_basic (&_tmp42, DBUS_TYPE_STRING, &_tmp44);
	}
	dbus_message_iter_close_container (&iter, &_tmp42);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetNameOfRegisteredDevice (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint adapter;
	dbus_uint32_t _tmp45;
	guint frontend;
	dbus_uint32_t _tmp46;
	char* result;
	DBusMessage* reply;
	const char* _tmp47;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uu")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	adapter = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp45);
	dbus_message_iter_next (&iter);
	adapter = _tmp45;
	frontend = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp46);
	dbus_message_iter_next (&iter);
	frontend = _tmp46;
	result = dvb_manager_GetNameOfRegisteredDevice (self, adapter, frontend);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp47 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &_tmp47);
	return reply;
}


static DBusMessage* _dbus_dvb_manager_GetSchedule (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint group_id;
	dbus_uint32_t _tmp48;
	guint channel_sid;
	dbus_uint32_t _tmp49;
	char* result;
	DBusMessage* reply;
	const char* _tmp50;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uu")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp48);
	dbus_message_iter_next (&iter);
	group_id = _tmp48;
	channel_sid = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp49);
	dbus_message_iter_next (&iter);
	channel_sid = _tmp49;
	result = dvb_manager_GetSchedule (self, group_id, channel_sid);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp50 = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &_tmp50);
	return reply;
}


DBusHandlerResult dvb_manager_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_manager_introspect (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Properties", "GetAll")) {
		reply = _dbus_dvb_manager_property_get_all (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetScannerForDevice")) {
		reply = _dbus_dvb_manager_GetScannerForDevice (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetRegisteredDeviceGroups")) {
		reply = _dbus_dvb_manager_GetRegisteredDeviceGroups (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetTypeOfDeviceGroup")) {
		reply = _dbus_dvb_manager_GetTypeOfDeviceGroup (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetRecorder")) {
		reply = _dbus_dvb_manager_GetRecorder (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "AddDeviceToNewGroup")) {
		reply = _dbus_dvb_manager_AddDeviceToNewGroup (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "AddDeviceToExistingGroup")) {
		reply = _dbus_dvb_manager_AddDeviceToExistingGroup (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "RemoveDeviceFromGroup")) {
		reply = _dbus_dvb_manager_RemoveDeviceFromGroup (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetDeviceGroupName")) {
		reply = _dbus_dvb_manager_GetDeviceGroupName (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetChannelList")) {
		reply = _dbus_dvb_manager_GetChannelList (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetDeviceGroupMembers")) {
		reply = _dbus_dvb_manager_GetDeviceGroupMembers (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetNameOfRegisteredDevice")) {
		reply = _dbus_dvb_manager_GetNameOfRegisteredDevice (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetSchedule")) {
		reply = _dbus_dvb_manager_GetSchedule (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


static void _dbus_dvb_manager_changed (GObject* _sender, guint group_id, guint change_type, DBusConnection* _connection) {
	const char * _path;
	DBusMessage *_message;
	DBusMessageIter _iter;
	dbus_uint32_t _tmp51;
	dbus_uint32_t _tmp52;
	_path = g_object_get_data (_sender, "dbus_object_path");
	_message = dbus_message_new_signal (_path, "org.gnome.DVB.Manager", "Changed");
	dbus_message_iter_init_append (_message, &_iter);
	_tmp51 = group_id;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp51);
	_tmp52 = change_type;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp52);
	dbus_connection_send (_connection, _message, NULL);
	dbus_message_unref (_message);
}


static void _dbus_dvb_manager_group_changed (GObject* _sender, guint group_id, guint adapter, guint frontend, guint change_type, DBusConnection* _connection) {
	const char * _path;
	DBusMessage *_message;
	DBusMessageIter _iter;
	dbus_uint32_t _tmp53;
	dbus_uint32_t _tmp54;
	dbus_uint32_t _tmp55;
	dbus_uint32_t _tmp56;
	_path = g_object_get_data (_sender, "dbus_object_path");
	_message = dbus_message_new_signal (_path, "org.gnome.DVB.Manager", "GroupChanged");
	dbus_message_iter_init_append (_message, &_iter);
	_tmp53 = group_id;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp53);
	_tmp54 = adapter;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp54);
	_tmp55 = frontend;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp55);
	_tmp56 = change_type;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp56);
	dbus_connection_send (_connection, _message, NULL);
	dbus_message_unref (_message);
}


void dvb_manager_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_manager_dbus_path_vtable, object);
	}
	g_signal_connect (object, "changed", (GCallback) _dbus_dvb_manager_changed, connection);
	g_signal_connect (object, "group-changed", (GCallback) _dbus_dvb_manager_group_changed, connection);
}


static void dvb_manager_class_init (DVBManagerClass * klass) {
	GStaticRecMutex _tmp0 = {0};
	dvb_manager_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBManagerPrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_manager_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_manager_finalize;
	g_signal_new ("changed", DVB_TYPE_MANAGER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT, G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
	g_signal_new ("group_changed", DVB_TYPE_MANAGER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT_UINT_UINT, G_TYPE_NONE, 4, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT);
	g_type_set_qdata (DVB_TYPE_MANAGER, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_manager_dbus_vtable));
	dvb_manager_instance_mutex = (g_static_rec_mutex_init (&_tmp0), _tmp0);
}


static void dvb_manager_instance_init (DVBManager * self) {
	self->priv = DVB_MANAGER_GET_PRIVATE (self);
}


static void dvb_manager_finalize (GObject* obj) {
	DVBManager * self;
	self = DVB_MANAGER (obj);
	(self->priv->scanners == NULL) ? NULL : (self->priv->scanners = (g_object_unref (self->priv->scanners), NULL));
	(self->priv->recorders == NULL) ? NULL : (self->priv->recorders = (g_object_unref (self->priv->recorders), NULL));
	(self->priv->channellists == NULL) ? NULL : (self->priv->channellists = (g_object_unref (self->priv->channellists), NULL));
	(self->priv->devices == NULL) ? NULL : (self->priv->devices = (g_object_unref (self->priv->devices), NULL));
	(self->priv->epgscanners == NULL) ? NULL : (self->priv->epgscanners = (g_object_unref (self->priv->epgscanners), NULL));
	(self->priv->schedules == NULL) ? NULL : (self->priv->schedules = (g_object_unref (self->priv->schedules), NULL));
	G_OBJECT_CLASS (dvb_manager_parent_class)->finalize (obj);
}


GType dvb_manager_get_type (void) {
	static GType dvb_manager_type_id = 0;
	if (dvb_manager_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBManagerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_manager_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBManager), 0, (GInstanceInitFunc) dvb_manager_instance_init, NULL };
		dvb_manager_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBManager", &g_define_type_info, 0);
	}
	return dvb_manager_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}



static void g_cclosure_user_marshal_VOID__UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__UINT_UINT) (gpointer data1, guint arg_1, guint arg_2, gpointer data2);
	register GMarshalFunc_VOID__UINT_UINT callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = (GCClosure *) closure;
	g_return_if_fail (n_param_values == 3);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = (GMarshalFunc_VOID__UINT_UINT) (marshal_data ? marshal_data : cc->callback);
	callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), data2);
}


static void g_cclosure_user_marshal_VOID__UINT_UINT_UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__UINT_UINT_UINT_UINT) (gpointer data1, guint arg_1, guint arg_2, guint arg_3, guint arg_4, gpointer data2);
	register GMarshalFunc_VOID__UINT_UINT_UINT_UINT callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = (GCClosure *) closure;
	g_return_if_fail (n_param_values == 5);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = (GMarshalFunc_VOID__UINT_UINT_UINT_UINT) (marshal_data ? marshal_data : cc->callback);
	callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), g_value_get_uint (param_values + 3), g_value_get_uint (param_values + 4), data2);
}



