/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <memory.h>
#include <ctype.h>
#include "gda-error.h"

Gda_ODBC_Error*
gda_odbc_error_new(void)
{
  Gda_ODBC_Error* e = g_new0(Gda_ODBC_Error, 1);
  return e;
}

void
gda_odbc_error_free(Gda_ODBC_Error* e)
{
  if (e->description)
    g_free(e->description);
  if (e->source)
    g_free(e->source);
  if (e->helpfile)
    g_free(e->helpfile);
  if (e->helpctxt)
    g_free(e->helpctxt);
  if (e->sqlstate)
    g_free(e->sqlstate);
  if (e->native)
    g_free(e->native);
  g_free(e);
}


static gchar*
find_end_of_location(gchar* error)
{
  gint in_brackets = 0;
  
  for (;*error; error++)
    {
      if (*error == '[')
	{
	  in_brackets = 1;
	  continue;
	}
      if (*error == ']')
	{
	  in_brackets = 0;
	  continue;
	}
      if (!isspace(*error) && !in_brackets)
	return error;
    }
  return 0;
}
      

void
gda_odbc_error_make(Gda_ODBC_Recordset*  rec,
		    Gda_ODBC_Connection* cnc,
		    gchar*               where)
{
  HENV  henv;
  HDBC  hdbc;
  HSTMT hstmt;
  gint  rc;
  Gda_ODBC_Error* e;

  gchar* eol;
  gchar* bol;
  
  static UCHAR  sqlstate[SQL_MAX_MESSAGE_LENGTH];
  static UCHAR  errMsg[SQL_MAX_MESSAGE_LENGTH];
  SWORD  actLength;

  
  if (rec)
    {
      cnc = rec->cnc;
      hstmt = rec->hstmt;
      henv  = rec->cnc->henv;
      hdbc  = rec->cnc->hdbc;
    }
  else if (cnc)
    {
      hstmt = 0;
      henv  = cnc->henv;
      hdbc  = cnc->hdbc;
    }
  else 
    {
      g_log("GDA ODBC", G_LOG_LEVEL_ERROR, "Invalid parameters passed to gda_odbc_error_make\n");
      abort();
    }
  e = gda_odbc_error_new();
  rc = SQLError(henv, hdbc, hstmt, sqlstate, &e->number, errMsg,
		sizeof(errMsg)-1, &actLength);


  fprintf(stderr,"Error at '%s': ", where);
  fprintf(stderr,"error->description = '%s'\n", errMsg);
  fprintf(stderr,"   end of location = '%s'\n", find_end_of_location(errMsg));
  fprintf(stderr,"   sqlstate = '%s'\n", sqlstate);

  bol = errMsg;
  e->description = g_strdup(eol = find_end_of_location(errMsg));
  e->source      = g_strdup_printf("[gda-odbc]%0.*s", (eol - bol), errMsg);

  e->helpfile    = g_strdup("error");
  e->helpctxt    = g_strdup("error");
  e->sqlstate    = g_strdup(sqlstate);
  e->native      = g_strdup(eol);
  cnc->errors = g_list_append(cnc->errors, e);
}

  
  
