/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#include <memory.h>
#include <glib.h>

#include "odbc.h"
#include "sqldriver.h"

#include "gda-types.h"
#include "gda-connection.h"
#include "gda-command.h"
#include "gda-error.h"
#include "gda-command.h"
#include "gda-odbc-misc.h"

typedef Gda_ODBC_Recordset* (*schema_ops_fn)(Gda_ODBC_Recordset*,
					     Gda_ODBC_Connection* ,
					     GDA_Connection_Constraint* constraints,
					     gint length);

schema_ops_fn schema_ops[GDNC_SCHEMA_LAST] =
{
  0,
};

static void            init_gda_odbc_lib                (void) __attribute__((constructor));
static void            initialize_schema_ops            (void);
static Gda_ODBC_Recordset*  schema_tables(Gda_ODBC_Recordset*  recset,
					  Gda_ODBC_Connection* cnc,
					  GDA_Connection_Constraint* constraints,
					  gint length);

static Gda_ODBC_Recordset*  schema_columns(Gda_ODBC_Recordset*  recset,
					   Gda_ODBC_Connection* cnc,
					   GDA_Connection_Constraint* constraints,
					   gint length);
static Gda_ODBC_Recordset*  schema_types(Gda_ODBC_Recordset*   recset,
					 Gda_ODBC_Connection*  cnc,
					 GDA_Connection_Constraint*  constraint,
					 gint length);

static Gda_ODBC_Recordset*  schema_views(Gda_ODBC_Recordset* recset,
					 Gda_ODBC_Connection* cnc,
					 GDA_Connection_Constraint* constraint,
					 gint length);

static Gda_ODBC_Recordset*  schema_indexes(Gda_ODBC_Recordset* recset,
					   Gda_ODBC_Connection* cnc,
					   GDA_Connection_Constraint* constraint,
					   gint length);

static Gda_ODBC_Recordset*  schema_procedures(Gda_ODBC_Recordset* recset,
					      Gda_ODBC_Connection* cnc,
					      GDA_Connection_Constraint* constraint,
					      gint length);

static void
initialize_schema_ops(void)
{
  schema_ops[GDCN_SCHEMA_TABLES] = schema_tables;
  schema_ops[GDCN_SCHEMA_COLS]   = schema_columns;
  schema_ops[GDCN_SCHEMA_PROV_TYPES] = schema_types;
  schema_ops[GDCN_SCHEMA_VIEWS] = schema_views;
  schema_ops[GDCN_SCHEMA_INDEXES] = schema_indexes;
  schema_ops[GDCN_SCHEMA_PROCS] = schema_procedures;
}


static void
init_gda_odbc_lib(void)
{
  fprintf(stderr,"init_gda_odbc_lib called\n");
  initialize_schema_ops();
}


static Gda_ODBC_Recordset*
schema_tables(Gda_ODBC_Recordset*   recset,
	      Gda_ODBC_Connection*  cnc,
	      GDA_Connection_Constraint* constraints,
	      gint length)
{
  gint                rc;
  
  recset->cnc = cnc;
  rc = SQLAllocStmt(cnc->hdbc, &recset->hstmt);
  fprintf(stderr,"schema_tables: Step1: cnc = %p, cnc->hdbc = %p, recset->hstmt = %p\n", cnc, cnc->hdbc, recset->hstmt);
  if (rc != SQL_SUCCESS)
    {
      fprintf(stderr,"SQLAllocSrmr failed\n");
      return 0;
    }
  rc = SQLTables(recset->hstmt,
		 "", SQL_NTS,
		 "", SQL_NTS,
		 "", SQL_NTS,
		 "", SQL_NTS);
  if (rc != SQL_SUCCESS)
    {
      fprintf(stderr,"schema_tables: Step2: cnc = %p, cnc->hdbc = %p\n", cnc, cnc->hdbc);
      gda_odbc_error_make(recset,
			  0,
			  "SQLTables");
      SQLFreeStmt(recset->hstmt, SQL_DROP);
      return 0;
    }
  return gda_odbc_describe_recset(recset);
}

static gint
filter_views(Gda_ODBC_Recordset* in, gpointer data)
{
  GList*          field_ptr;
  Gda_ODBC_Field* field;
  gint            rc;
  
  field_ptr = g_list_nth(in->fields, 3);
  field = field_ptr->data;
  rc = strcmp(field->value->_u.lvc, "VIEW") == 0;
  return rc;
}


static Gda_ODBC_Recordset*
schema_views(Gda_ODBC_Recordset*   recset,
	     Gda_ODBC_Connection*  cnc,
	     GDA_Connection_Constraint* constraints,
	     gint length)
{
  gint                rc;
  
  recset->cnc = cnc;
  rc = SQLAllocStmt(cnc->hdbc, &recset->hstmt);
  if (rc != SQL_SUCCESS)
    {
      fprintf(stderr,"SQLAllocStmt failed\n");
      return 0;
    }
  rc = SQLTables(recset->hstmt,
		 "", SQL_NTS,
		 "", SQL_NTS,
		 "", SQL_NTS,
		 "", SQL_NTS);
  if (rc != SQL_SUCCESS)
    {
      gda_odbc_error_make(recset,
			  0,
			  "SQLTables");
      SQLFreeStmt(recset->hstmt, SQL_DROP);
      return 0;
    }
  recset->internal_filter_function = filter_views;
  return gda_odbc_describe_recset(recset);
}

static Gda_ODBC_Recordset*
schema_indexes(Gda_ODBC_Recordset*   recset,
	     Gda_ODBC_Connection*  cnc,
	     GDA_Connection_Constraint* constraints,
	     gint length)
{
  return 0;
}

static Gda_ODBC_Recordset*
schema_procedures(Gda_ODBC_Recordset*   recset,
		  Gda_ODBC_Connection*  cnc,
		  GDA_Connection_Constraint* constraints,
		  gint length)
{
  Gda_ODBC_Command* cmd;
  gint              rc;
  
  cmd = gda_odbc_cmd_new();
  cmd->cnc = cnc;
  rc = SQLAllocStmt(cnc->hdbc, &recset->hstmt);
  if (rc != SQL_SUCCESS)
    {
      fprintf(stderr,"SQLAllocStmt failed\n");
      return 0;
    }
  rc = SQLProcedures(recset->hstmt,
		     "", SQL_NTS, /* proc Qualifier */
		     "", SQL_NTS, /* proc Owner  */
		     "", SQL_NTS); /* proc Name */
  if (rc != SQL_SUCCESS)
    {
      gda_odbc_error_make(recset,
			  0,
			  "SQLProcedures");
      SQLFreeStmt(recset->hstmt, SQL_DROP);
      recset->hstmt = 0;
      return 0;
    }
  return gda_odbc_describe_recset(recset);
}



static Gda_ODBC_Recordset*
schema_columns(Gda_ODBC_Recordset*  recset,
	       Gda_ODBC_Connection* cnc,
	       GDA_Connection_Constraint* constraints,
	       gint            length)
{
  gint             rc;
  gchar*           table_qualifier = 0;
  gchar*           table_owner     = 0;
  gchar*           table_name      = 0;
  gchar*           column_name     = 0;
  GDA_Connection_Constraint*  ptr;
  
  recset->cnc = cnc;
  rc = SQLAllocStmt(cnc->hdbc, &recset->hstmt);
  if (rc != SQL_SUCCESS)
    {
      return 0;
    }
  ptr = constraints;
  while(length)
    {
      switch (ptr->ctype)
	{
	case GDA_Connection_TABLE_CATALOG:
	  table_qualifier = ptr->value;
	  fprintf(stderr,"schema_columns: table_qualifier = '%s'\n", table_qualifier);
	  break;
	case GDA_Connection_TABLE_SCHEMA:
	  table_owner = ptr->value;
	  fprintf(stderr,"schema_columns: table_owner = '%s'\n", table_owner);
	  break;
	case GDA_Connection_TABLE_NAME:
	  table_name = ptr->value;
	  fprintf(stderr,"schema_columns: table_name = '%s'\n", table_name);
	  break;
	case GDA_Connection_COLUMN_NAME:
	  column_name = ptr->value;
	  fprintf(stderr,"schema_columns: column_name = '%s'\n", column_name);
	  break;
	default:
	  fprintf(stderr,"schema_columns: invalid constraint type %d\n", ptr->ctype);
	  SQLFreeStmt(recset->hstmt, SQL_DROP);
	  return 0;
	  break;
	}
      length--;
      ptr++;
    }
  rc = SQLColumns(recset->hstmt,
		  table_qualifier, table_qualifier ? SQL_NTS : 0,
		  table_owner,     table_owner ? SQL_NTS : 0, 
		  table_name,      table_name ? SQL_NTS : 0,
		  column_name,     column_name ? SQL_NTS : 0);
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(recset, 0, "SQLColumns");
      SQLFreeStmt(recset->hstmt, SQL_DROP);
      return 0;
    }
  return gda_odbc_describe_recset(recset);
}

static Gda_ODBC_Recordset*
schema_types(Gda_ODBC_Recordset*   recset,
	     Gda_ODBC_Connection*  cnc,
	     GDA_Connection_Constraint*  constraints,
	     gint             length)
{
  gint              rc;

  recset->cnc = cnc;
  rc = SQLAllocStmt(cnc->hdbc, &recset->hstmt);
  if (rc == SQL_ERROR)
    return 0;
  rc = SQLGetTypeInfo(recset->hstmt, 0);
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(recset, 0, "SQLColumns");
      SQLFreeStmt(recset->hstmt, SQL_DROP);
      return 0;
    }
  return gda_odbc_describe_recset(recset);
}




Gda_ODBC_Connection*
gda_odbc_connection_new()
{

  Gda_ODBC_Connection* c = g_new(Gda_ODBC_Connection, 1);
  memset(c, 0, sizeof(Gda_ODBC_Connection));
  return c;
}


void
gda_odbc_connection_free(Gda_ODBC_Connection* c)
{
  g_free(c);
}






gint
gda_odbc_connection_open(Gda_ODBC_Connection* cnc,
			 const gchar*               cnc_string,
			 const gchar*               user,
			 const gchar*               passwd)
{
  int rc;

  fprintf(stderr,"gda_odbc_connection_open: enter\n");
  rc = SQLAllocEnv(&cnc->henv);
  if (rc != SQL_SUCCESS)
    return -1;

  rc = SQLAllocConnect(cnc->henv, &cnc->hdbc);
  if (rc != SQL_SUCCESS)
    {
      SQLFreeEnv(cnc->henv);
      return -1;
    }
  g_warning("gda_odbc_connection_open: cnc_string = '%s'\n", cnc_string);
  g_warning("gda_odbc_connection_open: user = '%s'\n", user);
  g_warning("gda_odbc_connection_open: passwd = '%s'\n", passwd);
  rc = SQLConnect(cnc->hdbc, (char*)cnc_string, SQL_NTS, (char*) user, SQL_NTS,
		  (char*)passwd, SQL_NTS);
  if (rc != SQL_SUCCESS)
    {
      gda_odbc_error_make(0,
			  cnc,
			  "SQLConnect");
      SQLFreeEnv(cnc->henv);
      SQLFreeConnect(cnc->hdbc);
      return -1;
    }
  fprintf(stderr,"gda_odbc_connection_open: left\n");
  return 0;
}


Gda_ODBC_Recordset*
gda_odbc_connection_open_schema(Gda_ODBC_Connection*      cnc,
				GDA_Connection_QType t,
				GDA_Connection_Constraint*      constraints,
				gint                 length)
{
  Gda_ODBC_Recordset* rc = gda_odbc_recset_new();
  schema_ops_fn       fn = schema_ops[t];

  if (fn)
    {
      return fn(rc, cnc, constraints, length);
    }
  else
    {
      g_log("GDA ODBC", G_LOG_LEVEL_INFO,"schema_open_schema: Unhandled SCHEMA_QTYPE %d\n", t);
    }
  gda_odbc_recset_free(rc);
  return 0;
}



gint
gda_odbc_connection_close(Gda_ODBC_Connection* cnc)
{
  while (cnc->commands)
    {
      gda_odbc_cmd_set_connection(cnc->commands->data, 0);
      cnc->commands = g_list_remove_link(cnc->commands, cnc->commands);
    }
  while(cnc->errors)
    {
      gda_odbc_error_free(cnc->errors->data);
      cnc->errors = g_list_remove_link(cnc->errors, cnc->errors);
    }
  SQLDisconnect(cnc->hdbc);
#if 0
  SQLFreeConnect(cnc->hdbc);
  SQLFreeEnv(cnc->henv);
#endif
  cnc->hdbc = 0;
  cnc->henv = 0;
  return 0;
}

struct type_info
{
  gint    type;
  gchar*  type_name;
  gchar*  create_params;
};

static GList* needed_types = 0;

static void
add_type(gint type)
{
 GList* ptr;
 struct type_info* current;
 
 ptr = needed_types;
 while(ptr)
   {
     current = ptr->data;
     if (current->type == type)
       return;
     ptr = g_list_next(ptr);
   }
 current = g_new0(struct type_info, 1);
 current->type = type;
 needed_types = g_list_append(needed_types, current);
}

static struct type_info*
lookup_typeinfo(gint type)
{
  GList* ptr;

  ptr = needed_types;
  while(ptr)
    {
      struct type_info* current;

      current = ptr->data;
      if (current->type == type)
	return current;
      ptr = g_list_next(ptr);
    }
  return 0;
}

static void
free_typeinfo(void)
{
  GList* ptr = needed_types;
  while(ptr)
    {
      struct type_info* type_info;

      type_info = ptr->data;
      if (type_info->type_name)
	g_free(type_info->type_name);
      if (type_info->create_params)
	g_free(type_info->create_params);
      ptr = g_list_next(ptr);
    }
  g_list_free(needed_types);
  needed_types = 0;
}


CORBA_char*
gda_odbc_connection_create_recset(Gda_ODBC_Connection* cnc, GDA_RowAttributes* columns)
{

  gint                colidx;
  Gda_ODBC_Recordset* types;
  GList*              ptr;
  Gda_ODBC_Field*     field;
  gint                rc;
  gint                sqltype;
  GString*            stmt;
  gint                need_sep = 0;
  CORBA_char*         retval;
  struct type_info*   type_info;
  
  stmt = g_string_new("create table XXXXX (");
  fprintf(stderr,"gda_odbc_connection_create_recset: entered\n");
  for (colidx = 0; colidx < columns->_length; colidx++)
    {
      sqltype =	gda_odbc_gda2sqltype(columns->_buffer[colidx].gdaType);
      fprintf(stderr,"Adding sqltype %d[0x%x] from gda type %d\n",
	      sqltype, sqltype, columns->_buffer[colidx].gdaType);
      add_type(sqltype);
    }
  types = gda_odbc_connection_open_schema(cnc, GDA_Connection_GDCN_SCHEMA_PROV_TYPES, 0, 0);
  ptr = types->fields;
  while(ptr)
    {

      field = ptr->data;
      field->value = g_new0(GDA_Value, 1);
      ptr = g_list_next(ptr);
    }

  while ((rc = gda_odbc_recset_move_next(types)) == 0)
    {
      
      field = g_list_nth(types->fields, 1)->data;
      sqltype = (gint)field->value->_u.si;
      if ((type_info = lookup_typeinfo(sqltype)))
	{
	  field = g_list_nth(types->fields, 0)->data;
	  type_info->type_name = g_strdup(field->value->_u.lvc);
	  fprintf(stderr,"Got type name '%s'\n", type_info->type_name);
	  field = g_list_nth(types->fields, 5)->data;
	  type_info->create_params = g_strdup(field->value->_u.lvc);
	  fprintf(stderr,"Got create params '%s'\n", type_info->create_params);
	}
    }
  if (rc < 0)
    g_error("%s: Fatal error while fetching type info", __PRETTY_FUNCTION__);

  for (colidx = 0; colidx < columns->_length; colidx++)
    {
      type_info = lookup_typeinfo(gda_odbc_gda2sqltype(columns->_buffer[colidx].gdaType));
      if (need_sep)
	g_string_append(stmt, ", ");
      else
	need_sep = 1;
      g_string_append(stmt, columns->_buffer[colidx].name);
      g_string_append(stmt, " ");
      g_string_append(stmt, type_info->type_name);
      if (strcasecmp(type_info->create_params, "length") == 0)
	g_string_sprintfa(stmt, "(%d)", columns->_buffer[colidx].definedSize);
      if (strcasecmp(type_info->create_params, "precision,scale") == 0)
	g_string_sprintfa(stmt, "(%d,%d)", columns->_buffer[colidx].definedSize, columns->_buffer[colidx].scale);
    }
  free_typeinfo();
  g_string_append(stmt, ")");
  retval = CORBA_string_dup(stmt->str);
  fprintf(stderr,"statement = '%s'\n", retval);
  g_string_free(stmt, 1);
  return retval;
}
  
