/* GNOME DB libary
 * Copyright (C) 1999, 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __gda_xml_h__
#define __gda_xml_h__ 1

#include <glib.h>

#include <gda.h>
#include <gda-field.h>
#include <gda-recordset.h>

#include <gnome-xml/parser.h>
#include <gnome-xml/tree.h>

/*
 * Data structures and utility functions for reading and writing
 * GDA objects from/to XML
 *
 * GDA xml uses a custom DTD to represent the entire database in
 * XML format. Currently, the format is as follows:
 *
 * <database>
 *   <recordset name="SELECT or TABLE NAME" command="CREATE...">
 *     <field name="" EMPTY>
 *     ...
 *     <row>
 *       <value EMPTY>
 *       ...
 *     </row>
 *     ...
 *   </recordset>
 *   ...
 *   <view name="VIEW NAME" command="CREATE...">
 *   ...
 * </database>
 */
 
typedef struct _Gda_XmlDatabase      Gda_XmlDatabase;
typedef struct _Gda_XmlDatabaseClass Gda_XmlDatabaseClass;

#define GDA_TYPE_XML_DATABASE            (gda_xml_database_get_type())
#define GDA_XML_DATABASE(obj)            GTK_CHECK_CAST(obj, GDA_TYPE_XML_DATABASE, Gda_XmlDatabase)
#define GDA_XML_DATABASE_CLASS(klass)    GTK_CHECK_CLASS_CAST(klass, GDA_TYPE_XML_DATABASE, Gda_XmlDatabaseClass)
#define IS_GDA_XML_DATABASE(obj)         GTK_CHECK_TYPE(obj, GDA_TYPE_XML_DATABASE)
#define IS_GDA_XML_DATABASE_CLASS(klass) (GTK_CHECK_CLASS_TYPE((klass), GDA_TYPE_XML_DATABASE))

struct _Gda_XmlDatabase
{
  GtkObject object;
  xmlDocPtr xmldoc;
  gchar*    filename;
};

struct _Gda_XmlDatabaseClass
{
  GtkObjectClass parent_class;
  void           (*changed)(Gda_XmlDatabase *db);
};

guint            gda_xml_database_get_type (void);
Gda_XmlDatabase* gda_xml_database_new (const gchar *name);
Gda_XmlDatabase* gda_xml_database_load_from_file (const gchar *filename);
void             gda_xml_database_free (Gda_XmlDatabase *db);
gint             gda_xml_database_add_recordset (Gda_XmlDatabase *,
                                                 Gda_Recordset *recset,
                                                 const gchar *name);
gint             gda_xml_database_save_to_file (Gda_XmlDatabase *db, 
                                                const gchar *filename);
                                                
gchar*           gda_xml_database_get_name (Gda_XmlDatabase *db);
void             gda_xml_database_set_name (Gda_XmlDatabase *db,
                                            const gchar *new_name);
gchar*           gda_xml_database_get_filename (Gda_XmlDatabase *db);
void             gda_xml_database_set_filename (Gda_XmlDatabase *db,
                                                const gchar *new_name);
gchar*           gda_xml_database_get_date (Gda_XmlDatabase *db);
void             gda_xml_database_set_date (Gda_XmlDatabase *db,
                                            const gchar *new_date);

void             gda_xml_database_list_recordsets (Gda_XmlDatabase *db, GFunc callback);

/*
 * Adding recordsets and fields
 */
gint             gda_xml_recordset_new (Gda_XmlDatabase *db, 
                                        const gchar *name);
gint             gda_xml_recordset_remove (Gda_XmlDatabase *db, 
                                           const gchar *name);
gint             gda_xml_recordset_add_field (Gda_XmlDatabase *db,
                                              const gchar *recset,
                                              const gchar *fieldname,
                                              gint size,
                                              gint scale,
                                              gint type);
gint             gda_xml_recordset_add_gda_field (Gda_XmlDatabase *db,
                                                  const gchar *recset,
                                                  Gda_Field *);
gint             gda_xml_recordset_remove_field (Gda_XmlDatabase *db,
                                                 const gchar *recset,
                                                 const gchar *fieldname);
                                                 
typedef void (*Gda_XmlListFieldFunc)(Gda_XmlDatabase *db,
                                     const gchar *recset,
                                     const gchar *fieldname,
                                     gint size,
                                     gint scale,
                                     const gchar *type,
                                     gpointer user_data);
void             gda_xml_recordset_list_fields (Gda_XmlDatabase *db,
                                                const gchar *recset,
                                                Gda_XmlListFieldFunc callback,
                                                gpointer user_data);

/*
 * Adding views
 */
gint             gda_xml_view_new    (Gda_XmlDatabase *db, const gchar *view, const gchar *cmd);
gint             gda_xml_view_remove (Gda_XmlDatabase *db, const gchar *view);

#endif
