/* GNOME DB library
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(HAVE_CONFIG_H)
#  include <config.h>
#endif
#include "gda-ui.h"

static void gnome_db_list_class_init (GnomeDbListClass *klass);
static void gnome_db_list_init       (GnomeDbList *dblist);

void gnome_db_list_destroy (GnomeDbList *dblist);

/*
 * GnomeDbList widget signals
 */
enum
{
  GNOME_DB_LIST_SELECT_ROW,
  GNOME_DB_LIST_LAST_SIGNAL
};

static gint gnome_db_list_signals[GNOME_DB_LIST_LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
static void
row_selected_cb (GtkCList *clist, gint row, gint col, GdkEvent *event, GnomeDbList *dblist)
{
  gchar *str = 0;

  g_return_if_fail(GNOME_DB_IS_LIST(dblist));

  gtk_clist_get_text(clist, row, col, &str);
  if (str != 0)
    {
      gtk_signal_emit_by_name(GTK_OBJECT(dblist), "select_row", str);
    }
}

/*
 * GnomeDbList widget interface
 */
static void
gnome_db_list_class_init (GnomeDbListClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  gnome_db_list_signals[GNOME_DB_LIST_SELECT_ROW] =
          gtk_signal_new("select_row", GTK_RUN_FIRST, object_class->type,
                         GTK_SIGNAL_OFFSET(GnomeDbListClass, select_row),
                         gtk_signal_default_marshaller,
                         GTK_TYPE_NONE, 1, GTK_TYPE_STRING);
  gtk_object_class_add_signals(object_class, gnome_db_list_signals,
                               GNOME_DB_LIST_LAST_SIGNAL);

  object_class->destroy = gnome_db_list_destroy;
}

static void
gnome_db_list_init (GnomeDbList *dblist)
{
  GtkWidget *scroll;

  dblist->recset = 0;
  dblist->total_rows = 0;

  scroll = gda_ui_new_scrolled_window_widget();
  dblist->list = gda_ui_new_clist_widget(0, 2);
  gtk_signal_connect(GTK_OBJECT(dblist->list), "select_row",
                     GTK_SIGNAL_FUNC(row_selected_cb), (gpointer) dblist);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), 
                                        dblist->list);
  gtk_box_pack_start(GTK_BOX(dblist), scroll, 1, 1, 0);
}

guint
gnome_db_list_get_type (void)
{
  static guint db_list_type = 0;

  if (!db_list_type)
    {
      GtkTypeInfo db_list_info =
      {
        "GnomeDbList",
        sizeof (GnomeDbList),
        sizeof (GnomeDbListClass),
        (GtkClassInitFunc) gnome_db_list_class_init,
        (GtkObjectInitFunc) gnome_db_list_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_list_type = gtk_type_unique(gtk_vbox_get_type(), &db_list_info);
    }
  return (db_list_type);
}

/**
 * gnome_db_list_new
 * @recset: recordset to be shown
 * @pos: position of the field being shown
 *
 * Creates a new GnomeDbList widget
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_list_new (Gda_Recordset *recset, gint pos)
{
  GtkWidget *dblist = gtk_type_new(gnome_db_list_get_type());

  if (recset != 0)
    gnome_db_list_set_recordset(GNOME_DB_LIST(dblist), recset, pos);
  return (dblist);
}

void
gnome_db_list_destroy (GnomeDbList *dblist)
{
  g_return_if_fail(GNOME_DB_IS_LIST(dblist));

  if (dblist->recset != 0)
    {
      gda_recordset_free(dblist->recset);
    }
}

/**
 * gnome_db_list_clear
 * @dblist: GnomeDbList widget
 *
 * Clears the DbList widget. It clears all entries in the list and frees
 * the recordset associated
 */
void
gnome_db_list_clear (GnomeDbList *dblist)
{
  g_return_if_fail(GNOME_DB_IS_LIST(dblist));

  if (dblist->recset != 0)
    {
      gda_recordset_free(dblist->recset);
      dblist->recset = 0;
    }
  dblist->total_rows = 0;
  gda_ui_clear_clist(GTK_CLIST(dblist->list));
}

/**
 * gnome_db_list_get_row_count
 * @dblist: GnomeDbList widget
 *
 * Returns number of rows actually loaded in the recordset. As the grid widget loads
 * data asynchronously, you may call this function before all rows have been fetched
 * from the database, in which case, it won't reflect the "real" number of rows
 * returned in the recordset
 *
 * Returns: number of rows in the recordset , or -1 on error
 */
gint
gnome_db_list_get_row_count (GnomeDbList *dblist)
{
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), -1);
  return (dblist->total_rows);
}

/**
 * gnome_db_list_get_string
 * @dblist: GnomeDbList widget
 *
 * Returns the currently selected item's text. That is, the literal of the first
 * selected item
 *
 * Returns: a string containing the literal selected, or NULL if none is
 * selected or if there is an error
 */
gchar *
gnome_db_list_get_string (GnomeDbList *dblist)
{
  GList *selected;
  
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
  
  selected = GTK_CLIST(dblist->list)->selection;
  if (selected)
    {
      gchar *name = 0;
      guint row = GPOINTER_TO_UINT(selected->data);
      gtk_clist_get_text(GTK_CLIST(dblist->list), row, 1, &name);
      return (name);
    }
  return (0);
}

/**
 * gnome_db_list_get_recordset
 * @dblist: GnomeDbList widget
 *
 * Returns the Gda_Recordset object associated with the given GnomeDbList
 * widget
 *
 * Returns: a pointer to the Gda_Recordset object being used, or NULL if none
 * or if there is an error
 */
Gda_Recordset *
gnome_db_list_get_recordset (GnomeDbList *dblist)
{
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
  return (dblist->recset);
}

/**
 * gnome_db_list_set_recordset
 * @dblist: GnomeDbList widget
 * @recset: recordset to be shown
 * @pos: position of the field being shown
 *
 * Associates a Gda_Recordset object with the given GnomeDbList
 * widget and shows its contents
 */
void
gnome_db_list_set_recordset (GnomeDbList *dblist, Gda_Recordset *recset, gint pos)
{
  gulong position;

  g_return_if_fail(GNOME_DB_IS_LIST(dblist));
  g_return_if_fail(IS_GDA_RECORDSET(recset));
  g_return_if_fail(pos >= 0);

  /* close previous recordset */
  if (dblist->recset != 0)
    {
      gda_recordset_free(dblist->recset);
      dblist->recset = 0;
    }

  gda_ui_clear_clist(GTK_CLIST(dblist->list));
  gtk_clist_freeze(GTK_CLIST(dblist->list));
  dblist->total_rows = 0;

  position = gda_recordset_move(recset, 1, 0);
  while (position != GDA_RECORDSET_INVALID_POSITION &&
         !gda_recordset_eof(dblist->recset))
    {
      gchar *row[2], *empty_string = "";
      Gda_Field *field = gda_recordset_field_idx(dblist->recset, pos);

      row[0] = empty_string;
      row[1] = gda_stringify_value(0, 0, field);
      gtk_clist_append(GTK_CLIST(dblist->list), row);

      /* fetch next row */
      dblist->total_rows++;
      position = gda_recordset_move(recset, 1, 0);
    }
  gtk_clist_thaw(GTK_CLIST(dblist->list));
}
