/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

/*
 * Import/Export Utility
 * This module makes use of the XML functions provided by the GDA client
 * library for importing/exporting data both from/to XML and currently
 * open connections
 */

enum
{
  FE_IMPEXP_TYPE_IMPORT,
  FE_IMPEXP_TYPE_EXPORT
};

typedef struct
{
  gint               type;
  FE_ConnectionInfo* cnc_info;
  GtkWidget*         view_window;
  GtkWidget*         object_combo;
  GtkWidget*         object_list;
  GtkWidget*         selected_list;
  GtkWidget*         to_file;
  GtkWidget*         file_entry;
  GtkWidget*         to_sql;
  GtkWidget*         sql_entry;
  GtkWidget*         to_cnc;
  GtkWidget*         cnc_list;
  GtkWidget*         log_text;
  Gda_XmlDatabase*   xml_db;
} FE_ImportExportInfo;

static void add_all_cb (GtkWidget *w, gpointer data);
static void add_object_cb (GtkWidget *w, gpointer data);
static void clear_cb (GtkWidget *w, gpointer data);
static void close_connection_cb (Gda_Connection *cnc, GtkWidget *view);
static void close_window_cb (GtkWidget *w, gpointer data);
static void object_combo_select_cb (GtkWidget *w, gpointer data);
static void remove_all_cb (GtkWidget *w, gpointer data);
static void remove_object_cb (GtkWidget *w, gpointer data);
static void start_cb (GtkWidget *w, gpointer data);

static void                 export_to_connection (FE_ImportExportInfo *);
static void                 export_to_sql (FE_ImportExportInfo *);
static void                 export_to_xml (FE_ImportExportInfo *);
static FE_ImportExportInfo* get_current_view (void);
static void                 real_close (GtkWidget *w, gpointer data);

static GnomeUIInfo impexptoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Start"), N_("Start Selected Operation"),
    start_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear all entries"),
    clear_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close this window"),
    close_window_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Callbacks
 */
static void
add_all_cb (GtkWidget *w, gpointer data)
{
  FE_ImportExportInfo *imp_info = get_current_view();
  if (imp_info != NULL)
    {
      GList *sel = g_list_first(GTK_LIST(imp_info->object_list)->children);
      while (sel != NULL)
        {
          gchar *name = gtk_object_get_data(GTK_OBJECT(sel->data),
                                            FE_LIST_ITEM_DATA_KEY);
          if (name != NULL)
            {
              GtkWidget *li = fe_add_list_item(GTK_LIST(imp_info->selected_list), name);
              gtk_object_set_data(GTK_OBJECT(li), "FE_ImpExp_ObjectType",
                                                                   fe_get_combo_current_string(GTK_COMBO(imp_info->object_combo)));
            }
          sel = g_list_next(sel);
        }
    }
}

static void
add_object_cb (GtkWidget *w, gpointer data)
{
  FE_ImportExportInfo *imp_info = get_current_view();
  if (imp_info != NULL)
    {
      GList *sel = g_list_first(GTK_LIST(imp_info->object_list)->selection);
      while (sel != NULL)
        {
          gchar *name = gtk_object_get_data(GTK_OBJECT(sel->data), FE_LIST_ITEM_DATA_KEY);
          if (name != NULL)
            {
              GtkWidget *li = fe_add_list_item(GTK_LIST(imp_info->selected_list), name);
              gtk_object_set_data(GTK_OBJECT(li), "FE_ImpExp_ObjectType",
                                  fe_get_combo_current_string(GTK_COMBO(imp_info->object_combo)));
            }
          sel = g_list_next(sel);
        }
    }
}

static void
clear_cb (GtkWidget *w, gpointer data)
{
  FE_ImportExportInfo *imp_info = get_current_view();
  if (imp_info != NULL)
    {
      const gchar *empty_string = "";
      gtk_list_clear_items(GTK_LIST(imp_info->object_list), 0, -1);
      gtk_list_clear_items(GTK_LIST(imp_info->selected_list), 0, -1);
      gtk_entry_set_text(GTK_ENTRY(imp_info->file_entry), empty_string);
      gtk_entry_set_text(GTK_ENTRY(imp_info->sql_entry), empty_string);
      gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(imp_info->cnc_list)->entry), empty_string);
      gtk_editable_delete_text(GTK_EDITABLE(imp_info->log_text), 0, -1);
    }
}

static void
close_connection_cb (Gda_Connection *cnc, GtkWidget *view)
{
  fe_destroy_view(view);
}

static void
close_window_cb (GtkWidget *w, gpointer data)
{
  GtkWidget *view = fe_get_current_view();
  if (view != NULL)
    {
      if (fe_get_view_data(view, "FE_ImpExp_Info"))
        fe_destroy_view(view);
    }
}

static void
object_combo_select_cb (GtkWidget *w, gpointer data)
{
  FE_ImportExportInfo *imp_info = (FE_ImportExportInfo *) data;
  if (imp_info != NULL)
    {
      Gda_Recordset *recset;
      gchar *object_type = fe_get_combo_current_string(GTK_COMBO(imp_info->object_combo));
      g_print("determining object type = %s\n", object_type);
      if (!strcmp(object_type, _("Tables")))
        {
          recset = gda_connection_open_schema(imp_info->cnc_info->cnc,
                                              GDA_Connection_GDCN_SCHEMA_TABLES,
                                              GDA_Connection_no_CONSTRAINT, NULL);
        }
      else recset = NULL;
      gtk_list_clear_items(GTK_LIST(imp_info->object_list), 0, -1);
      if (recset != NULL)
        {
          gchar  bfr[128];
          gulong position = gda_recordset_move(recset, 1, 0);
          while (position != GDA_RECORDSET_INVALID_POSITION &&
                 !gda_recordset_eof(recset))
            {
              Gda_Field *field = gda_recordset_field_idx(recset, 2);
              if (field != NULL)
                {
                  fe_add_list_item(GTK_LIST(imp_info->object_list),
                                   gda_stringify_value(bfr, sizeof(bfr) - 1, field));
                }
              position = gda_recordset_move(recset, 1, 0);
            }
          /* free the recordset */
          gda_recordset_close(recset);
          gda_recordset_free(recset);
        }
    }
}

static void
remove_all_cb (GtkWidget *w, gpointer data)
{
  FE_ImportExportInfo *imp_info = get_current_view();
  if (imp_info != NULL)
    {
      gtk_list_clear_items(GTK_LIST(imp_info->selected_list), 0, -1);
    }
}

static void
remove_object_cb (GtkWidget *w, gpointer data)
{
  FE_ImportExportInfo *imp_info = get_current_view();
  if (imp_info != NULL)
    {
      gtk_list_remove_items(GTK_LIST(imp_info->selected_list),
                            GTK_LIST(imp_info->selected_list)->selection);
    }
}

static void
start_cb (GtkWidget *w, gpointer data)
{
  FE_ImportExportInfo *imp_info = get_current_view();
  if (imp_info != NULL)
    {
      gtk_editable_delete_text(GTK_EDITABLE(imp_info->log_text), 0, -1);
      if (imp_info->type == FE_IMPEXP_TYPE_EXPORT)
        {
          if (fe_is_radio_button_active(GTK_RADIO_BUTTON(imp_info->to_file)))
            export_to_xml(imp_info);
          else if (fe_is_radio_button_active(GTK_RADIO_BUTTON(imp_info->to_sql)))
            export_to_sql(imp_info);
          else if (fe_is_radio_button_active(GTK_RADIO_BUTTON(imp_info->to_cnc)))
            export_to_connection(imp_info);
        }
      else if (imp_info->type == FE_IMPEXP_TYPE_IMPORT)
        {
        }
    }
}

/*
 * Gda_Connection export
 */
static void
export_to_connection (FE_ImportExportInfo *imp_info)
{
}

/*
 * SQL export
 */
static void
export_to_sql (FE_ImportExportInfo *imp_info)
{
  gchar *file = gtk_entry_get_text(GTK_ENTRY(imp_info->sql_entry));
  if (file != NULL)
    {
    }
  else gda_ui_show_error(_("You must provide a file name"));
}

/*
 * XML export
 */
static void
real_xml_export (GtkWidget *list_item, FE_ImportExportInfo *imp_info)
{
  gchar *name, *type;
  
  g_return_if_fail(list_item != NULL);
  g_return_if_fail(GTK_IS_WIDGET(list_item));
  g_return_if_fail(imp_info != NULL);
  
  name = (gchar *) gtk_object_get_data(GTK_OBJECT(list_item), FE_LIST_ITEM_DATA_KEY);
  if (name != NULL)
    {
      type = (gchar *) gtk_object_get_data(GTK_OBJECT(list_item), "FE_ImpExp_ObjectType");
      if (!strcmp(type, _("Tables")))
        {
          gulong reccount;
          Gda_Recordset *recset;
          gchar *str = g_strdup_printf(_("Adding table %s..."), name);
          fe_status_bar_message(str);
          gtk_text_insert(GTK_TEXT(imp_info->log_text), 0, 0, 0, str, strlen(str));
          g_free((gpointer) str);
          
          str = g_strdup_printf("select * from %s", name);
          recset = gda_connection_execute(imp_info->cnc_info->cnc, str, &reccount, 0);
          g_free((gpointer) str);
          if (recset != NULL)
            {
              gda_xml_database_add_recordset(imp_info->xml_db, recset, name);
              gda_recordset_close(recset);
              gda_recordset_free(recset);
              gtk_text_insert(GTK_TEXT(imp_info->log_text), 0, 0, 0, _("ok\n"),
                              strlen(_("ok\n")));
            }
          else gtk_text_insert(GTK_TEXT(imp_info->log_text), 0, 0, 0, _("no\n"),
                               strlen(_("no\n")));
        }
    }
}

static void
export_to_xml (FE_ImportExportInfo *imp_info)
{
  gchar *file = gtk_entry_get_text(GTK_ENTRY(imp_info->file_entry));
  if (file != NULL)
    {
      imp_info->xml_db = gda_xml_database_new(imp_info->cnc_info->dsn);
      if (imp_info->xml_db != NULL)
        {
          gtk_text_insert(GTK_TEXT(imp_info->log_text), 0, 0, 0, _("Exporting database\n"),
                          strlen(_("Exporting database\n")));
          gtk_container_foreach(GTK_CONTAINER(imp_info->selected_list), 
                                (GtkCallback) real_xml_export, (gpointer) imp_info);
          if (gda_xml_database_save_to_file(imp_info->xml_db, file) == -1)
            gda_ui_show_error(_("Error saving XML database to disk"));
          else fe_status_bar_message(_("Export finished successfully"));
          gda_xml_database_free(imp_info->xml_db);
          imp_info->xml_db = NULL;
        }
    }
  else gda_ui_show_error(_("You must provide a file name"));
}

/*
 * Private functions
 */
static FE_ImportExportInfo *
get_current_view (void)
{
  GtkWidget *view = fe_get_current_view();
  if (view != NULL)
    {
      FE_ImportExportInfo *imp_info = fe_get_view_data(view, "FE_ImpExp_Info");
      return (imp_info);
    }
  return (NULL);
}

static void
real_close (GtkWidget *w, gpointer data)
{
  FE_ImportExportInfo *imp_info = (FE_ImportExportInfo *) data;
  g_print("real_close() called!\n");
  if (imp_info != NULL)
    {
      g_free((gpointer) imp_info);
    }
}

/*
 * Public functions
 */
void
fe_open_export (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      FE_ImportExportInfo *imp_info;
      GtkWidget *container, *label, *scrolled_window, *box, *button, *box2, *table;
      if ((imp_info = g_new0(FE_ImportExportInfo, 1)))
        {
          gchar *str;
          GList *objects = 0;
          imp_info->cnc_info = cnc_info;
          imp_info->type = FE_IMPEXP_TYPE_EXPORT;
          str = g_strdup_printf(_("Export: %s@%s"), gda_connection_get_user(cnc_info->cnc),
                                cnc_info->dsn);
          imp_info->view_window = fe_new_view(str, 4, 5, impexptoolbar);
          fe_set_view_data(imp_info->view_window, "FE_ImpExp_Info", (gpointer) imp_info);
          fe_set_view_data(imp_info->view_window, "FE_ConnectionInfo", 
                           (gpointer) imp_info->cnc_info);
          fe_set_view_destroy_func(imp_info->view_window, (FE_DestroyViewFunc) real_close, 
                                   (gpointer) imp_info);
          g_free((gpointer) str);
          
          /* connect to Gda_Connection's close signal */
          gtk_signal_connect(GTK_OBJECT(cnc_info->cnc), "close",
                             GTK_SIGNAL_FUNC(close_connection_cb), 
                             (gpointer) imp_info->view_window);

          container = gtk_vbox_new(FALSE, 0);
          gtk_widget_show(container);
          fe_add_widget_to_view(imp_info->view_window, 0, 1, 4, 2, container);
          
          /* create children */
          box = gtk_hbox_new(FALSE, 0);
          gtk_box_pack_start(GTK_BOX(container), box, 0, 0, 0);
          gtk_widget_show(box);
          
          label = gda_ui_new_label_widget(_("Objects"));
          gtk_box_pack_start(GTK_BOX(box), label, 0, 0, GNOME_PAD);
          
          imp_info->object_combo = gda_ui_new_combo_widget();
          gtk_signal_connect(GTK_OBJECT(GTK_COMBO(imp_info->object_combo)->entry), "activate",
                             GTK_SIGNAL_FUNC(object_combo_select_cb), (gpointer) imp_info);
          gtk_box_pack_start(GTK_BOX(box), imp_info->object_combo, 0, 0, GNOME_PAD);
          
          /* add object type list to combo box */
          objects = g_list_append(objects, g_strdup(_("Tables")));
          objects = g_list_append(objects, g_strdup(_("Views")));
          gtk_combo_set_popdown_strings(GTK_COMBO(imp_info->object_combo), objects);

          box = gtk_hbox_new(FALSE, 0);
          box2 = gtk_vbox_new(FALSE, 0);
          gtk_widget_show(box2);
          label = gda_ui_new_label_widget(_("Object List"));
          gtk_box_pack_start(GTK_BOX(box2), label, 0, 0, GNOME_PAD);
          scrolled_window = gda_ui_new_scrolled_window_widget();
          imp_info->object_list = fe_new_list_widget();
          gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
                                                imp_info->object_list);
          gtk_list_set_selection_mode(GTK_LIST(imp_info->object_list), GTK_SELECTION_MULTIPLE);
          gtk_widget_show(scrolled_window);
          gtk_box_pack_start(GTK_BOX(box2), scrolled_window, 1, 1, GNOME_PAD);
          gtk_box_pack_start(GTK_BOX(box), box2, 1, 1, GNOME_PAD);
          gtk_box_pack_start(GTK_BOX(container), box, 1, 1, GNOME_PAD);
          gtk_widget_show(box);

          box2 = gtk_vbox_new(FALSE, 0);
          button = gda_ui_new_button_widget_with_pixmap(_("Add"), GNOME_STOCK_MENU_FORWARD);
          gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(add_object_cb),
                             (gpointer) imp_info);
          gtk_box_pack_start(GTK_BOX(box2), button, 0, 0, 0);
          button = gda_ui_new_button_widget_with_pixmap(_("Remove"), GNOME_STOCK_MENU_BACK);
          gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(remove_object_cb),
                             (gpointer) imp_info);
          gtk_box_pack_start(GTK_BOX(box2), button, 0, 0, 0);
          button = gda_ui_new_button_widget_with_pixmap(_("Add all"), GNOME_STOCK_MENU_REDO);
          gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(add_all_cb),
                             (gpointer) imp_info);
          gtk_box_pack_start(GTK_BOX(box2), button, 0, 0, 0);
          button = gda_ui_new_button_widget_with_pixmap(_("Remove All"), GNOME_STOCK_MENU_UNDO);
          gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(remove_all_cb),
                             (gpointer) imp_info);
          gtk_box_pack_start(GTK_BOX(box2), button, 0, 0, 0);
          gtk_widget_show(box2);
          gtk_box_pack_start(GTK_BOX(box), box2, 0, 0, GNOME_PAD);

          box2 = gtk_vbox_new(FALSE, 0);
          gtk_widget_show(box2);
          label = gda_ui_new_label_widget(_("Selection List"));
          gtk_box_pack_start(GTK_BOX(box2), label, 0, 0, GNOME_PAD);
          scrolled_window = gda_ui_new_scrolled_window_widget();
          imp_info->selected_list = fe_new_list_widget();
          gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window),
                                                imp_info->selected_list);
          gtk_widget_show(imp_info->selected_list);
          gtk_box_pack_start(GTK_BOX(box2), scrolled_window, 1, 1, GNOME_PAD);
          gtk_box_pack_start(GTK_BOX(box), box2, 1, 1, GNOME_PAD);
          
          /* log window */
          table = gtk_table_new(2, 1, FALSE);
          fe_add_widget_to_view(imp_info->view_window, 0, 2, 3, 4, table);
          gtk_widget_show(table);
          
          label = gda_ui_new_label_widget(_("Log"));
          gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                           GTK_FILL,
                           GTK_FILL, 3, 3);
          scrolled_window = gda_ui_new_scrolled_window_widget();
          gtk_table_attach(GTK_TABLE(table), scrolled_window, 0, 1, 1, 2,
                           GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                           GTK_FILL | GTK_EXPAND | GTK_SHRINK, 3, 3);
          imp_info->log_text = fe_new_text_widget();
          gtk_text_set_editable(GTK_TEXT(imp_info->log_text), FALSE);
          gtk_container_add(GTK_CONTAINER(scrolled_window), imp_info->log_text);

          /* "To File" radio button and friends */
          table = gtk_table_new(3, 3, FALSE);
          fe_add_widget_to_view(imp_info->view_window, 0, 4, 1, 5, table);
          
          imp_info->to_file = gda_ui_new_radio_button_widget(_("To XML"), NULL);
          gtk_table_attach(GTK_TABLE(table), imp_info->to_file, 0, 1, 0, 1,
                           GTK_FILL,
                           GTK_FILL, 3, 3);

          imp_info->file_entry = gda_ui_new_entry_widget(0, TRUE);
          gtk_table_attach(GTK_TABLE(table), imp_info->file_entry, 1, 2, 0, 1,
                           GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                           GTK_FILL, 3, 3);
          button = fe_new_browse_button_widget(GTK_ENTRY(imp_info->file_entry));
          gtk_table_attach(GTK_TABLE(table), button, 2, 3, 0, 1,
                           GTK_FILL,
                           GTK_FILL, 3, 3);
          
          /* "To SQL" radio button and friends */
          imp_info->to_sql = gda_ui_new_radio_button_widget(_("To SQL"), imp_info->to_file);
          gtk_table_attach(GTK_TABLE(table), imp_info->to_sql, 0, 1, 1, 2,
                           GTK_FILL,
                           GTK_FILL, 3, 3);

          imp_info->sql_entry = gda_ui_new_entry_widget(0, TRUE);
          gtk_table_attach(GTK_TABLE(table), imp_info->sql_entry, 1, 2, 1, 2,
                           GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                           GTK_FILL, 3, 3);
          button = fe_new_browse_button_widget(GTK_ENTRY(imp_info->sql_entry));
          gtk_table_attach(GTK_TABLE(table), button, 2, 3, 1, 2,
                           GTK_FILL,
                           GTK_FILL, 3, 3);
          
          /* "To Connection" radio button and friends */
          imp_info->to_cnc = gda_ui_new_radio_button_widget(_("To Connection"), imp_info->to_file);
          gtk_table_attach(GTK_TABLE(table), imp_info->to_cnc, 0, 1, 2, 3,
                           GTK_FILL,
                           GTK_FILL, 3, 3);
          
          imp_info->cnc_list = gda_ui_new_combo_widget();
          gtk_table_attach(GTK_TABLE(table), imp_info->cnc_list, 1, 2, 2, 3,
                           GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                           GTK_FILL, 3, 3);
          
          fe_display_view(imp_info->view_window);
        }
      else gda_ui_show_error(_("Cannot allocate %d bytes"), sizeof(FE_ImportExportInfo));
    }
}
