/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* board.c
 *
 * Copyright (C) 2001  JP Rosevear
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <libgnomecanvas/gnome-canvas-pixbuf.h>

#include "makros.h"
#include "prefs.h"
#include "board.h"

struct _BoardPrivate {
	GtkWidget *frame;
	GnomeCanvas *canvas;
	
	/* Movement information */
	gboolean moving;
	double orig_x;
	double orig_y;
	double curr_x;
	double curr_y;
	Square from;
	Square to;
	Square selected_from;
	GnomeCanvasItem *selected;

	BoardLockType white_lock;
	BoardLockType black_lock;
	
	/* Position information */
	Position *pos;
	gboolean flip;	

	/* Board information */
	double size;

	char db[120];
	GdkPixbuf *pieces[12];
	GnomeCanvasItem *db_image[120];
	GnomeCanvasItem *squares[64];
	guint light_rgba;
	guint dark_rgba;

	/* Gutter stuff */
	GnomeCanvasItem *white_gutter;
	GnomeCanvasItem *black_gutter;

	GnomeCanvasItem *gutter_pieces[12];
	GnomeCanvasItem *gutter_text[12];
	
	gint gutter_count[12];
};

/* Class signals */
enum {
	MOVE_SIGNAL,
	FLIP_SIGNAL,
	LAST_SIGNAL
};
static gint board_signals[LAST_SIGNAL] = { 0 };

enum {
  ARG_0,
  ARG_LIGHT_COLOR_RGBA,
  ARG_DARK_COLOR_RGBA,
};

/* Prototypes */
static void set_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void get_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void destroy (GtkObject *obj);
static void finalize (GObject *obj);

static GdkPixbuf *get_piece (Board *board, Piece piece);
static void draw_gutter (Board *board);
static void draw_background (Board *board);
static void draw_position (Board *board);

static void size_allocate_cb (GtkWidget *widget, GtkAllocation *allocation, gpointer data);
static gboolean event_cb (GtkWidget *widget, GdkEvent *event, gpointer data);

static GtkVBoxClass *parent_class = NULL;

static void
class_init (BoardClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	gobject_class = G_OBJECT_CLASS (klass);
	object_class = GTK_OBJECT_CLASS (klass);
	widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = gtk_type_class (GTK_TYPE_VBOX);
	
	board_signals[MOVE_SIGNAL]
		= gtk_signal_new ("move",
				  GTK_RUN_FIRST,
				  GTK_CLASS_TYPE (object_class),
				  GTK_SIGNAL_OFFSET (BoardClass, move),
				  gtk_marshal_NONE__INT_INT,
				  GTK_TYPE_NONE, 2,
				  GTK_TYPE_INT, GTK_TYPE_INT);

	board_signals[FLIP_SIGNAL]
		= gtk_signal_new ("flip",
				  GTK_RUN_FIRST,
				  GTK_CLASS_TYPE (object_class),
				  GTK_SIGNAL_OFFSET (BoardClass, flip),
				  gtk_marshal_NONE__BOOL,
				  GTK_TYPE_NONE, 1,
				  GTK_TYPE_BOOL);

	 gtk_object_add_arg_type ("Board::light_color_rgba",
				  GTK_TYPE_UINT,
				  GTK_ARG_READWRITE,
				  ARG_LIGHT_COLOR_RGBA);
	 gtk_object_add_arg_type ("Board::dark_color_rgba",
				  GTK_TYPE_UINT,
				  GTK_ARG_READWRITE,
				  ARG_DARK_COLOR_RGBA);
	
	 gobject_class->finalize = finalize;
	 object_class->set_arg = set_arg;
	 object_class->get_arg = get_arg;
	 object_class->destroy = destroy;
	 
         klass->move = NULL;
}


static void
load_pieces (Board *board)
{
	BoardPrivate *priv;
       	GdkPixbuf *pb;
	
	int i;
	
	char *filenames [] =
	{
		"P.png",
		"N.png",
		"B.png",
		"R.png",
		"Q.png",
		"K.png",
		"p.png",
		"n.png",
		"b.png",
		"r.png",
		"q.png",
		"k.png",
		NULL
	};
	
	priv = board->priv;
	
	for (i = 0; filenames[i] != NULL; i++) {
		char *fname;
		
		fname = g_build_filename (prefs_get_piecedir (), filenames[i], NULL);
		pb = gdk_pixbuf_new_from_file (fname, NULL);
		g_free (fname);
	
		/* If not installed */
		if (pb == NULL) {
			fname = g_build_filename ("../Sets/", filenames[i], NULL);
			pb = gdk_pixbuf_new_from_file (fname, NULL);
			g_free (fname);
		}
		priv->pieces[i] = pb;
	}

	priv->size = 8 * gdk_pixbuf_get_width (pb);
}

static void
init (Board *board)
{
	BoardPrivate *priv;
	int i;
	
	priv = g_new0 (BoardPrivate, 1);

	board->priv = priv;

	priv->frame = gtk_aspect_frame_new (NULL, 0.5, 0.5, 1.0, FALSE);
	gtk_box_pack_end (GTK_BOX (board), priv->frame, TRUE, TRUE, 0);
	gtk_widget_show (GTK_WIDGET (priv->frame));
	
	priv->canvas = GNOME_CANVAS (gnome_canvas_new ());
	gtk_container_add (GTK_CONTAINER (priv->frame), GTK_WIDGET (priv->canvas));
	gtk_widget_show (GTK_WIDGET (priv->canvas));

	gtk_signal_connect (GTK_OBJECT (priv->canvas), "size_allocate",
			    GTK_SIGNAL_FUNC (size_allocate_cb),
			    board);
	
	gtk_widget_show (GTK_WIDGET (board));

	priv->white_lock = BOARD_LOCK_COMPLETE;
	priv->black_lock = BOARD_LOCK_COMPLETE;
	
	priv->light_rgba = GNOME_CANVAS_COLOR (144, 238, 144);
	priv->dark_rgba = GNOME_CANVAS_COLOR (0, 100, 0);
	
	priv->flip = FALSE;
	for (i = 0; i < 120; i++)
		priv->db[i] = -1;
	for (i = 0; i < 120; i++)
		priv->db_image[i] = NULL;

	priv->pos = POSITION (position_new_initial ());
	
	load_pieces (board);
	draw_background (board);
	draw_position (board);
}



static void
set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	switch (arg_id) {
	case ARG_LIGHT_COLOR_RGBA:
		board_set_light_color (BOARD (object), GTK_VALUE_UINT (*arg));
		break;
	case ARG_DARK_COLOR_RGBA:
		board_set_dark_color (BOARD (object), GTK_VALUE_UINT (*arg));
		break;
	}
}

static void
get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	switch (arg_id) {
	case ARG_LIGHT_COLOR_RGBA:
		GTK_VALUE_UINT (*arg) = board_get_light_color (BOARD (object));
		break;
	case ARG_DARK_COLOR_RGBA:
		GTK_VALUE_UINT (*arg) = board_get_dark_color (BOARD (object));
		break;
	}
}

static void
destroy (GtkObject *obj)
{
	Board *board;
	BoardPrivate *priv;

	board = BOARD (obj);
	priv = board->priv;

	if (priv->pos != NULL)
		g_object_unref (priv->pos);
	priv->pos = NULL;

	if (GTK_OBJECT_CLASS (parent_class)->destroy)	
		GTK_OBJECT_CLASS (parent_class)->destroy (obj);
}

static void
finalize (GObject *obj)
{
	Board *board;
	BoardPrivate *priv;

	board = BOARD (obj);
	priv = board->priv;

	g_free (priv);
}

GtkType
board_get_type (void)
{
	static GtkType type = 0;

	if (type == 0) {
		static const GtkTypeInfo info =
		{
			"Board",
			sizeof (Board),
			sizeof (BoardClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		type = gtk_type_unique (GTK_TYPE_VBOX, &info);
	}

	return type;
}

GtkWidget *
board_new () 
{
	GtkWidget *w;
	
	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());

	w = GTK_WIDGET (gtk_type_new (TYPE_BOARD));

	gtk_widget_pop_colormap ();
	gtk_widget_pop_visual ();

	return w;
}

Position *
board_get_position (Board *board) 
{
	BoardPrivate *priv;
	
	priv = board->priv;
	
	return priv->pos;
}

void 
board_set_position (Board *board, Position *pos) 
{
	BoardPrivate *priv;
	Position     *old_pos;
	
	priv = board->priv;

	old_pos = priv->pos;
	priv->pos = position_copy (pos);
	g_object_unref (old_pos);

	draw_position (board);
}

static void
set_colors (Board *board) 
{
	BoardPrivate *priv;
	int i, j;
	
	priv = board->priv;

	for (i = 0; i < 8; i++)
		for (j = 0; j < 8; j++)
			gnome_canvas_item_set (priv->squares[i * 8 +j],
					       "fill_color_rgba", 
					       ((i + j) % 2) ? priv->dark_rgba 
					       : priv->light_rgba,
					       NULL);
}

guint32 
board_get_light_color (Board *board)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	return priv->light_rgba;
}

void 
board_set_light_color (Board *board, guint32 rgba)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	priv->light_rgba = rgba;

	set_colors (board);

	if (GTK_WIDGET_REALIZED (board))
		gtk_widget_queue_draw (GTK_WIDGET (board));
}

guint32 
board_get_dark_color (Board *board)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	return priv->dark_rgba;
}

void 
board_set_dark_color (Board *board, guint32 rgba)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	priv->dark_rgba = rgba;

	set_colors (board);

	if (GTK_WIDGET_REALIZED (board))
		gtk_widget_queue_draw (GTK_WIDGET (board));
}

BoardLockType
board_get_white_lock (Board *board)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	return priv->white_lock;
}

void
board_set_white_lock (Board *board, BoardLockType lock)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	priv->white_lock = lock;	
}


BoardLockType
board_get_black_lock (Board *board)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	return priv->black_lock;
}

void
board_set_black_lock (Board *board, BoardLockType lock)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	priv->black_lock = lock;
}


void
board_gutter_show (Board *board)
{
	draw_gutter (board);
}

void
board_gutter_hide (Board *board)
{
}

void
board_gutter_add (Board *board, Piece piece)
{
	BoardPrivate *priv;
	GdkPixbuf *pb = NULL;
        GnomeCanvasGroup *root;
	double start = 0.0;
	int index, pos;

	priv = board->priv;
	
	if (piece == EMPTY)
		return;
	
	if (WPIECE (piece))
		index =  piece - WP;
	else
		index =  piece - BP + 6;

	priv->gutter_count[index]++;

	/* Draw the piece if necessary */
	if (priv->gutter_count[index] >= 2)
		return;
	
	root = GNOME_CANVAS_GROUP (gnome_canvas_root (priv->canvas));

	pb = get_piece (board, piece);
	
	pos = index;
	if ((WPIECE (piece) && !priv->flip) || (BPIECE (piece) && priv->flip))
		start = 4.0;
	if (BPIECE (piece))
		pos -= 6;

	priv->gutter_pieces[index] =
		gnome_canvas_item_new (root,
				       gnome_canvas_pixbuf_get_type (),
				       "x", 8.20, "y", start + (0.66 * pos),
				       "width", 0.66, "height", 0.66,
				       "width_set", TRUE, "height_set", TRUE,
				       "pixbuf", pb,
				       NULL);	
}

void
board_gutter_remove (Board *board, Piece piece)
{
	BoardPrivate *priv;
	int index;

	priv = board->priv;
	
	if (WPIECE (piece))
		index =  piece - WP;
	else
		index =  piece - BP + 6;

	if (priv->gutter_count[index] > 0)
		priv->gutter_count[index]--;

	if (priv->gutter_count[index] == 0) {
		gtk_object_destroy (GTK_OBJECT (priv->gutter_pieces[index]));
		priv->gutter_pieces[index] = NULL;
	}
}

void 
board_set_flip (Board *board, gboolean flipped) 
{
	BoardPrivate *priv;
	
	priv = board->priv;

	priv->flip = flipped;

	gtk_signal_emit (GTK_OBJECT (board),
			 board_signals[FLIP_SIGNAL],
			 priv->flip);
}

void 
board_flip (Board *board) 
{
	BoardPrivate *priv;
	
	priv = board->priv;

	if (priv->flip)
		priv->flip = FALSE;
	else
		priv->flip = TRUE;

	gtk_signal_emit (GTK_OBJECT (board),
			 board_signals[FLIP_SIGNAL],
			 priv->flip);
}

void
board_move (Board *board, Square from, Square to)
{
	BoardPrivate *priv;
	
	priv = board->priv;

	position_move (priv->pos, from, to);
	draw_position (board);
}

static int
get_square (double x, double y, gboolean flip)
{
	Square from;

	if (!flip)
		from = A1 + (Square)x  + 10 * (7 - (Square)y);
	else
		from = H8 - (Square)x - 10 * (7 - (Square)y);

	return from;
}

static GdkPixbuf *
get_piece (Board *board, Piece piece) 
{
	BoardPrivate *priv;
	
	priv = board->priv;
	
	if (piece == EMPTY)
		return NULL;
	
	if (WPIECE (piece))
		return priv->pieces[piece - WP];
	else
		return priv->pieces[piece - BP + 6];
}

static void
clear_square (GnomeCanvasItem **square)
{
	gnome_canvas_item_set (*square, "outline_color", NULL, NULL);
	*square = NULL;
}

static void
mark_square (Board *board, double drop_x, double drop_y)
{
	BoardPrivate *priv;
	
	priv = board->priv;
	
	if (priv->selected)
		clear_square (&priv->selected);
	priv->selected = priv->squares [(int)drop_x * 8 + (int)drop_y];
	gnome_canvas_item_raise_to_top (priv->selected);
	gnome_canvas_item_set (priv->selected, "outline_color", "red", NULL);
}

static gboolean
try_move (Board *board, GnomeCanvasItem *item)
{
	BoardPrivate *priv;
	
	priv = board->priv;
	
	priv->to = position_move_normalize (priv->pos, priv->from, priv->to);
	if (priv->to) {
		position_move (priv->pos, priv->from, priv->to);
		draw_position (board);
		gtk_signal_emit (GTK_OBJECT (board),
				 board_signals [MOVE_SIGNAL],
				 priv->from,
				 priv->to);
		return TRUE;		
	} else if (item != NULL) {
		gnome_canvas_item_move (item,
					priv->orig_x - priv->curr_x,
					priv->orig_y - priv->curr_y);
		return FALSE;
	}

	return FALSE;
}

static GnomeCanvasItem *
draw_piece (Board *board, Square sq, Piece piece)
{
	BoardPrivate *priv;
	GdkPixbuf *pb = NULL;
	GnomeCanvasItem *image;
        GnomeCanvasGroup *root;
	double x, y;
        int i,j;

	priv = board->priv;

	root = GNOME_CANVAS_GROUP (gnome_canvas_root (priv->canvas));
	
	i= 9 - sq / 10 ;
	j= sq % 10 - 1;
 
	x = j;
	y = i;
	
	pb = get_piece (board, piece);
	if (pb == NULL)
		return NULL;
	
	image = gnome_canvas_item_new (root,
				       gnome_canvas_pixbuf_get_type (),
				       "x", x, "y", y,
				       "width", 1.0, "height", 1.0,
				       "width_set", TRUE, "height_set", TRUE,
				       "pixbuf", pb,
				       NULL);

	g_signal_connect (G_OBJECT (image), "event", 
			  G_CALLBACK (event_cb), NULL);
	
	return image;
}

static GnomeCanvasItem *
draw_square (Board *board, double x, double y)
{
	BoardPrivate *priv;
	GnomeCanvasItem *image = NULL;
        GnomeCanvasGroup *root;
	gint f;

	priv = board->priv;
	f = get_square (x, y, priv->flip);

	root = GNOME_CANVAS_GROUP (gnome_canvas_root (priv->canvas));

	image = gnome_canvas_item_new (root,
				       gnome_canvas_rect_get_type (),
				       "x1", x,
				       "y1", y,
				       "x2", x + 1.0,
				       "y2", y + 1.0,
				       "width_pixels", 0,
				       "fill_color_rgba", 
				       (((int)(x+y)) % 2) ? priv->dark_rgba : priv->light_rgba,
				       NULL);

	g_signal_connect (G_OBJECT (image), "event", 
			  G_CALLBACK (event_cb), GINT_TO_POINTER (f));

	return image;
}

static void
draw_gutter (Board *board) 
{
	BoardPrivate *priv;
	GnomeCanvasGroup *root;
	double wstart = 0, bstart = 0;
	
	priv = board->priv;

	root = GNOME_CANVAS_GROUP (gnome_canvas_root (priv->canvas));

	if (priv->flip)
		bstart = 4.0;
	else
		wstart = 4.0;
	
	priv->white_gutter  = gnome_canvas_item_new (root,
						     gnome_canvas_rect_get_type (),
						     "x1", 8.1,
						     "y1", wstart,
						     "x2", 8.99,
						     "y2", wstart + 3.99,
						     "fill_color_rgba", 
						     priv->dark_rgba,
						     NULL);
	
	priv->black_gutter  = gnome_canvas_item_new (root,
						     gnome_canvas_rect_get_type (),
						     "x1", 8.1,
						     "y1", bstart,
						     "x2", 8.99,
						     "y2", bstart + 3.99,
						     "fill_color_rgba", 
						     priv->light_rgba,
						     NULL);

}

static void 
draw_background (Board *board)
{
	BoardPrivate *priv;
	int i,j;

	priv = board->priv;

	for (i = 0; i < 8; i++)
		for (j = 0; j < 8; j++)
			priv->squares[i * 8 + j] = draw_square (board, i, j);

	gnome_canvas_set_scroll_region (priv->canvas, 0, 0, priv->size, priv->size);
}

static void
draw_position (Board *board) 
{
	BoardPrivate *priv;
	int n, n2, i, j;

	priv = board->priv;
	
	for (i = 0; i < 8; i++)
		for (j = 0; j < 8; j++) {
			n  = H8 - i * 10 - j;
			if (!priv->flip) 
				n2 = n;
			else
				n2 = A1 + i*10 + j;

			if (priv->pos->square[n2] != priv->db[n]) {
				if (priv->db_image[n])
					gtk_object_destroy (GTK_OBJECT (priv->db_image[n]));
				priv->db_image[n] = draw_piece (board, n, priv->pos->square[n2]);
				priv->db[n] = priv->pos->square[n2];
			}
		}
}


static void
size_allocate_cb (GtkWidget *widget, GtkAllocation *allocation, gpointer data)
{
	Board *board;
	BoardPrivate *priv;
	
	board = BOARD (data);
	priv = board->priv;
	
	priv->size = 8 * ((double)allocation->height / 8);
	
	gnome_canvas_set_pixels_per_unit (priv->canvas, priv->size / 8);
	gnome_canvas_scroll_to (priv->canvas, 0, 0);
}

static gboolean
event_cb (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	Board *board;
	BoardPrivate *priv;
	GnomeCanvasItem *item;
	Square from;
	short to_move;
	double drop_x, drop_y;
	double new_x, new_y;

	item = GNOME_CANVAS_ITEM (widget);
	board = BOARD (gtk_widget_get_ancestor (GTK_WIDGET (item->canvas), TYPE_BOARD));
	priv = board->priv;
	
	/* If it was a square, find the piece on it */
	if (GNOME_IS_CANVAS_RE (widget)) {
		item = priv->db_image [GPOINTER_TO_INT (data)];
		if (item == NULL && event->type != GDK_BUTTON_RELEASE) 
			return FALSE;
	}
	
	switch (event->type) {
	case GDK_BUTTON_PRESS:
		if (event->button.button != 1)
			break;		

		priv->orig_x = priv->curr_x = event->button.x;
		priv->orig_y = priv->curr_y = event->button.y;

		from = get_square (priv->curr_x,
				   priv->curr_y,
				   priv->flip);

		if (WPIECE (priv->pos->square[from]) && priv->white_lock == BOARD_LOCK_COMPLETE)
			break;
		if (BPIECE (priv->pos->square[from]) && priv->black_lock == BOARD_LOCK_COMPLETE)
			break;

		gnome_canvas_item_raise_to_top (item);
		gnome_canvas_item_grab (item,
					GDK_POINTER_MOTION_MASK | 
					GDK_BUTTON_RELEASE_MASK,
					NULL,
					event->button.time);
		
		priv->from = from;		
		priv->to = from;
		priv->moving = TRUE;
		break;

	case GDK_MOTION_NOTIFY:
 		if (priv->moving && (event->motion.state & GDK_BUTTON1_MASK)) {
			new_x = event->motion.x;
			new_y = event->motion.y;
			gnome_canvas_item_move (item, 
						new_x - priv->curr_x,
						new_y - priv->curr_y);
			
			if (priv->selected != NULL)
				clear_square (&priv->selected);

			priv->curr_x = new_x;
			priv->curr_y = new_y;
		}
		break;

	case GDK_BUTTON_RELEASE:
		if (event->button.button != 1)
			break;

		if (item != NULL)
			gnome_canvas_item_ungrab (item, event->button.time);

		drop_x = event->button.x;
		drop_y = event->button.y;
		
		/* Click to move handling */
		priv->to = get_square (drop_x, drop_y, priv->flip);
		if (priv->to == priv->from) {
			/* Press and release in the same square */
			if (priv->selected == NULL) {
				/* Nothing was currently marked, mark the 
				   square, move the piece back to its original spot */
				mark_square (board, drop_x, drop_y);
				if (item != NULL) {
					gnome_canvas_item_raise_to_top (item);					
					gnome_canvas_item_move (item, 
								priv->orig_x - priv->curr_x,
								priv->orig_y - priv->curr_y);
				}				
				priv->selected_from = priv->from;
				priv->moving = FALSE;
				break;
			} else {
				/* Something was marked previously */
				to_move = position_get_color_to_move (priv->pos);
				
				if (priv->selected && priv->to == priv->selected_from) {
					/* Thing same thing was clicked, unmark it */
					clear_square (&priv->selected);
					priv->from = -priv->to;
					priv->moving = FALSE;
					break;
				} else if (PIECE_IS (priv->pos->square[priv->to]) == to_move) {
					/* A piece of the tomove color was clicked, mark it instead */
					mark_square (board, drop_x, drop_y);
					gnome_canvas_item_raise_to_top (item);
					priv->from = priv->to;
					priv->selected_from = priv->to;
					break;
				} else {
					/* The user clicked an empty space or a piece of the other
					   color so try to move */
					priv->from = priv->selected_from;
				}
			}
		}
	
		if (try_move (board, item)) {
			if (priv->selected)
				clear_square (&priv->selected);
			priv->moving = FALSE;
		} else if (!priv->selected) {
			priv->moving = FALSE;
		}
		
		break;
	default:
		break;
	}
	
	return FALSE;
}
