/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2000 Dave Smith & Julian Missig
 */


 
#include "WelcomeDruid.hh"
#include "GladeHelper.hh"
#include "GabberApp.hh"
#include "TCPTransmitter.hh"
#include "GabberUtility.hh"

using namespace jabberoo;
using namespace GabberUtil;

WelcomeDruid* WelcomeDruid::_Dialog = NULL;

WelcomeDruid::WelcomeDruid()
     : BaseGabberWindow("Welcome_win")
{
     // Connect buttons to handlers
     _druid = getWidget<Gnome::Druid>("Welcome_Druid");
     _druid->cancel.connect(slot(this, &WelcomeDruid::OnCancel));

     // Connect the pages
     Gnome::DruidPage* page = getWidget<Gnome::DruidPage>("Welcome_Password");
     page->prepare.connect(slot(this, &WelcomeDruid::OnPasswordPrepare));
     page = getWidget<Gnome::DruidPage>("Welcome_Confirm");
     page->prepare.connect(slot(this, &WelcomeDruid::OnConfirmPrepare));
     page = getWidget<Gnome::DruidPage>("Welcome_LoggingIn");
     page->prepare.connect(slot(this, &WelcomeDruid::OnLoggingInPrepare));
     page = getWidget<Gnome::DruidPage>("Welcome_Finish");
     page->finish.connect(slot(this, &WelcomeDruid::OnFinish));

     // Connect widgets
     _barStatus    = getWidget<Gnome::AppBar>("Welcome_Status_bar");
     _entFirstName = getEntry("Welcome_FirstName_txt");
     _entFirstName ->changed.connect(slot(this, &WelcomeDruid::OnNameChanged));
     _entLastName  = getEntry("Welcome_LastName_txt");
     _entLastName  ->changed.connect(slot(this, &WelcomeDruid::OnNameChanged));
     _entFullName  = getEntry("Welcome_FullName_txt");
     _entUsername  = getEntry("Welcome_Username_txt");
     _entUsername  ->changed.connect(slot(this, &WelcomeDruid::OnUsernameChanged));
     _entServer    = getEntry("Welcome_Server_txt");
     _entServer    ->changed.connect(slot(this, &WelcomeDruid::OnServerChanged));
     _entResource  = getEntry("Welcome_Resource_txt");
     _entResource  ->changed.connect(slot(this, &WelcomeDruid::OnResourceChanged));
     _entPassword  = getEntry("Welcome_Password_txt");
     _entPassword  ->changed.connect(slot(this, &WelcomeDruid::OnPasswordChanged));
     _chkSavePassword = getCheckButton("Welcome_SavePassword_chk");
     _entConfirmPassword = getEntry("Welcome_ConfirmPassword_txt");
     _entConfirmPassword->changed.connect(slot(this, &WelcomeDruid::OnPasswordChanged));

     _tried_connecting = false;

     // Display
     show();
}

WelcomeDruid::~WelcomeDruid()
{
     _Dialog = NULL;
}    

void WelcomeDruid::execute()
{
     // Create a welcome window
     _Dialog = manage(new WelcomeDruid());
}

bool WelcomeDruid::isRunning()
{
     // Return whether or not we're in the welcome druid
     return (_Dialog != NULL);
}

bool WelcomeDruid::hasTriedConnecting()
{
     // If we're currently running
     if (_Dialog != NULL)
     {
	  // if we haven't tried, set it so we have and return
	  if (!_Dialog->_tried_connecting)
	  {
	       _Dialog->_tried_connecting = true;
	       return false;
	  }
	  else
	       return true;
     }
     // Else they're not in the welcome druid, so they have
     else
	  return true;
}

void WelcomeDruid::Connected()
{
     g_assert( _Dialog != NULL );
     g_assert( _Dialog->_druid != NULL );

     // Be *sure* it's running
     if (_Dialog != NULL)
     {
	       // Send them to the finish page, and don't let them go back
	       _Dialog->_druid->set_page(*_Dialog->getWidget<Gnome::DruidPage>("Welcome_Finish"));
	       _Dialog->_druid->set_buttons_sensitive(0,1,1);
	       
	       // Status bar message
	       _Dialog->_barStatus->pop();
	       _Dialog->_barStatus->push(_("Login successful."));
	       _Dialog->_barStatus->get_progress()->set_activity_mode(false);
     }
}

void WelcomeDruid::OnCancel()
{
     close();
}

void WelcomeDruid::OnNameChanged()
{
     g_assert( _entFullName != NULL );

     // Make the full name equal First Name plus Last Name
     _entFullName->set_text(_entFirstName->get_text() + " " + _entLastName->get_text());
}

void WelcomeDruid::OnUsernameChanged()
{
     g_assert( _entUsername != NULL );
     g_assert( _druid != NULL );
     g_assert( _barStatus != NULL );

     // If the username is empty, don't let them click next
     if (_entUsername->get_text().empty())
     {
	  _druid->set_buttons_sensitive(1,0,1);
	  _barStatus->pop();
	  _barStatus->push(_("Enter your desired username."));
     }
     else
     {
	  _druid->set_buttons_sensitive(1,1,1);
	  _barStatus->pop();
     }
}

void WelcomeDruid::OnServerChanged()
{
     g_assert( _entServer != NULL );
     g_assert( _druid != NULL );
     g_assert( _barStatus != NULL );

     // If the server is empty, don't let them click next
     if (_entServer->get_text().empty())
     {
	  _druid->set_buttons_sensitive(1,0,1);
	  _barStatus->pop();
	  _barStatus->push(_("Select a Jabber server."));
     }
     else
     {
	  _druid->set_buttons_sensitive(1,1,1);
	  _barStatus->pop();
     }
}

void WelcomeDruid::OnResourceChanged()
{
     g_assert( _entResource != NULL );
     g_assert( _druid != NULL );
     g_assert( _barStatus != NULL );

     // If the resource is empty, don't let them click next
     if (_entResource->get_text().empty())
     {
	  _druid->set_buttons_sensitive(1,0,1);
	  _barStatus->pop();
	  _barStatus->push(_("Enter a resource."));
     }
     else
     {
	  _druid->set_buttons_sensitive(1,1,1);
	  _barStatus->pop();
     }
}

void WelcomeDruid::OnPasswordPrepare()
{
     g_assert( _entPassword != NULL );
     g_assert( _druid != NULL );
     g_assert( _barStatus != NULL );

     // If the password hasn't been set, or the passwords aren't equal
     // Don't let them click next
     if (_entPassword->get_text().empty() || _entPassword->get_text() != _entConfirmPassword->get_text())
     {
	  _druid->set_buttons_sensitive(1,0,1);
	  _barStatus->pop();
	  _barStatus->push(_("Both passwords must match."));
     }
}

void WelcomeDruid::OnPasswordChanged()
{
     g_assert( _entPassword != NULL );
     g_assert( _druid != NULL );
     g_assert( _barStatus != NULL );

     // If the passwords are equal, allow them to click next
     if (_entPassword->get_text() == _entConfirmPassword->get_text() && !_entPassword->get_text().empty())
     {
	  _druid->set_buttons_sensitive(1,1,1);
	  getLabel("Welcome_Password_warning_lbl")->set_text(_("Passwords match."));
	  _barStatus->pop();
     }
     else
     {
	  _druid->set_buttons_sensitive(1,0,1);
	  getLabel("Welcome_Password_warning_lbl")->set_text(_("Passwords do not match!"));
	  _barStatus->pop();
	  _barStatus->push(_("Passwords do not match!"));
     }
}

void WelcomeDruid::OnConfirmPrepare()
{
     // Show them their user@server and user@server/resource JIDs
     // I think it's helpful for new users
     getEntry("Welcome_JabberID_txt")->set_text(_entUsername->get_text() + "@" + _entServer->get_text());
     getEntry("Welcome_JIDResource_txt")->set_text(_entUsername->get_text() + "@" + _entServer->get_text() + "/" + _entResource->get_text());
}

void WelcomeDruid::OnLoggingInPrepare()
{
     g_assert( _druid != NULL );
     g_assert( _entFirstName != NULL );
     g_assert( _entLastName != NULL );
     g_assert( _entFullName != NULL );
     g_assert( _entUsername != NULL );
     g_assert( _entServer != NULL );
     g_assert( _entPassword != NULL );
     g_assert( _chkSavePassword != NULL );
     g_assert( _entResource != NULL );
     g_assert( _barStatus != NULL );

     // Don't let them click next
     _druid->set_buttons_sensitive(1,0,1);

     // After they have filled out new account information
     // we try saving config and logging in
     Gtk::Entry* txtEMail     = getEntry("Welcome_eMail_txt");
     Gtk::Entry* txtNickname  = getEntry("Welcome_Nickname_txt");
     Gtk::Entry* txtPort      = getEntry("Welcome_Port_txt");

     g_assert( txtEMail != NULL );
     g_assert( txtNickname != NULL );
     g_assert( txtPort != NULL );

     if (!_entUsername->get_text().empty() && !_entServer->get_text().empty() && !txtPort->get_text().empty() && !_entPassword->get_text().empty() && !_entResource->get_text().empty())
     {
	  // Save current config
	  ConfigManager& c = G_App->getCfg();
	  // User Info
	  c.putValue("User/FirstName", toUTF8(_entFirstName, _entFirstName->get_text()));
	  c.putValue("User/LastName", toUTF8(_entLastName, _entLastName->get_text()));
	  c.putValue("User/FullName", toUTF8(_entFullName, _entFullName->get_text()));
	  c.putValue("User/eMail", toUTF8(txtEMail, txtEMail->get_text()));
	  c.set_nick(toUTF8(txtNickname, txtNickname->get_text()));
	  // Account Info
	  c.putValue("Server/Username", toUTF8(_entUsername, _entUsername->get_text()));
	  c.putValue("Server/Server", _entServer->get_text());
	  c.putValue("Server/Port", txtPort->get_text());
	  c.putPrivValue("Server/Password", toUTF8(_entPassword, _entPassword->get_text()));
	  c.putValue("Server/SavePassword", _chkSavePassword->get_active());
	  c.putValue("Server/Resource", toUTF8(_entResource, _entResource->get_text()));
	  c.putValue("Wizards/LoggedIn", false);
	  c.sync();

	  // Pop up the GabberWin
	  G_App->init_win();

	  // Try logging in
	  G_App->login();
	  _barStatus->pop();
	  _barStatus->push(_("Logging in..."));
	  _barStatus->get_progress()->set_activity_mode(true);
     }
}

void WelcomeDruid::OnFinish()
{
     // Exit 
     close();
}


