/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2000 Dave Smith & Julian Missig
 */


#ifndef INCL_TCP_TRANSMITTER_HH
#define INCL_TCP_TRANSMITTER_HH


#include <glib.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>

#include "Gabber.hh"

#ifdef WITH_SSL
#include "SSLAdapter.hh"
#endif

class TCPTransmitter 
     : public SigC::Object
{
public:
     TCPTransmitter();
     ~TCPTransmitter();

     // General connection methods
     void connect(const string& host, guint port);
     void disconnect();
	  
     // Signals
     Signal0<void>			evtConnected;
     Signal0<void>			evtDisconnected;
     Signal1<void, const char*> 	evtDataAvailable;
     Signal1<void, const string&>	evtError;
     
     
     // Transmitter methods
     void send(const char* data);

protected:
     enum Error {
       errAddressLookup, errSocket
     };

     // Socket related callbacks
     static gboolean on_socket_event(GIOChannel* iochannel, GIOCondition cond, gpointer data);
     static gboolean on_socket_connect(GIOChannel* iochannel, GIOCondition cond, gpointer data);
     static gboolean on_host_resolved(GIOChannel* iochannel, GIOCondition cond, gpointer data);
     
     void handleError(const Error e);
     void handleError(const string & emsg);
     const string getSocketError();

     // Members
     gint				_socketfd;
     GIOChannel*			_IOChannel;

private:
     void _async_resolve(const gchar* hostname);
     void _async_connect();
#ifdef WITH_IPV6
     bool _gethostbyname(const gchar* hostname, struct in6_addr* result);
#else
     bool _gethostbyname(const gchar* hostname, struct in_addr* result);
#endif
     static gboolean _connection_poll(gpointer data);

     // Members
     bool				_hostResolved;
     bool				_connecting;
     bool				_reconnecting;
     bool				_connected;
#ifdef WITH_IPV6
     struct sockaddr_in6		_host_sockaddr;
#else
     struct sockaddr_in			_host_sockaddr;
#endif
     guint				_resolver_watchid;
     guint				_poll_eventid;
     gint				_socket_flags;
     pid_t				_resolver_pid;
     guint				_socket_watchid;
#ifdef WITH_SSL
     SSLAdapter*			_adapter;
#endif
#ifdef HAVE_GETHOSTBYNAME_R_GLIB_MUTEX
     static GStaticMutex		_gethostbyname_mutex;
#endif
};


#endif /* INCL_TCP_TRANSMITTER_HH */
